/**
 * Baijiahulian.com Inc. Copyright (c) 2015-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.wechat.helper.statistics;

import com.baijia.tianxiao.sal.wechat.dto.statistics.FansSCDto;
import com.baijia.tianxiao.sal.wechat.dto.statistics.FansSummaryDto;
import com.baijia.tianxiao.sal.wechat.dto.wechatapi.WechatApiResponse;
import com.baijia.tianxiao.util.date.DateUtil;

import com.google.common.collect.Lists;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletionService;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

/**
 * @author : zhenyujian
 * @title : DataStatisticsApiHelper
 * @description : 微信公众平台 数据统计相关接口
 * @date : 2015年12月17日 下午1:59:14
 */
@Slf4j
public class DataStatisticsApiHelper {
    private static final ExecutorService executor = Executors.newFixedThreadPool(5);

    public static FansSCDto getFansSCDto(String accessToken, Date beginDate, Date endDate, Date dateForCumulate) throws ParseException {
        long start = System.currentTimeMillis();
        FansSCDto dto = new FansSCDto();
        dto.setStartDate(DateUtil.getStrByDate(beginDate));
        dto.setEndDate(DateUtil.getStrByDate(endDate));

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

        //查询昨日数据统计
        //累计数据
        WechatApiResponse ytCumulateResp = DataStatisticsApiCaller.getFansCumulate(accessToken, dateForCumulate, dateForCumulate);
        //分日数据
        WechatApiResponse ytSummaryResp = DataStatisticsApiCaller.getFansSummary(accessToken, dateForCumulate, dateForCumulate);

        int cancelFansForCumulate = 0;
        int newFansForCumulate = 0;
        int totalFansForCumulate = 0;

        JSONObject ytSummaryRootNode = ytSummaryResp.getRootJSONObj();
        JSONArray ytSummaryArray = ytSummaryRootNode.getJSONArray("list");
        if (ytSummaryArray != null && ytSummaryArray.size() > 0) {
            JSONObject obj;
            for (int i = 0; i < ytSummaryArray.size(); i++) {
                obj = ytSummaryArray.getJSONObject(i);
                cancelFansForCumulate += obj.getInt("cancel_user");
                newFansForCumulate += obj.getInt("new_user");
            }
        }

        JSONObject ytCumulateRootNode = ytCumulateResp.getRootJSONObj();
        JSONArray ytCumulateArray = ytCumulateRootNode.getJSONArray("list");
        if (ytCumulateArray != null && !ytCumulateArray.isEmpty()) {
            totalFansForCumulate = ytCumulateArray.getJSONObject(0).getInt("cumulate_user");
            dto.setYtDate(sdf.format(dateForCumulate));
            dto.setYtCancelFans(cancelFansForCumulate);
            dto.setYtNewFans(newFansForCumulate);
            dto.setYtFansDiff(newFansForCumulate - cancelFansForCumulate);
            dto.setYtCumulateFans(totalFansForCumulate);
        }
        

        //查询分日数据统计
        List<FansSummaryDto> fansSummaryDtoList = new ArrayList<FansSummaryDto>();
        List<Date> dateList = DateUtil.getDateBetween(beginDate, endDate);

        if (dateList != null && !dateList.isEmpty()) {
            LinkedHashMap<String, FansSummaryDto> symmaryMap = new LinkedHashMap<String, FansSummaryDto>();
            FansSummaryDto fansSummaryDto;
            for (Date date : dateList) {
                fansSummaryDto = new FansSummaryDto(sdf.format(date));
                symmaryMap.put(sdf.format(date), fansSummaryDto);
            }

            //传来的时间段有可能大于7天，需要将时间端按7天来割分，然后向微信发出请求
            boolean bContinue = DateUtil.compare(DateUtil.getDiffDateTime(beginDate, 6), endDate) || DateUtil.compare(beginDate, endDate);
            List<CallableTask> tasks = new ArrayList<>();
            while (bContinue) {
                Date tempEndDate = DateUtil.getDiffDateTime(beginDate, 6);
                if(tempEndDate.compareTo(endDate) >=0){
                    tasks.add(new CallableTask(beginDate, endDate, accessToken));
                    break;
                }else{
                    tasks.add(new CallableTask(beginDate, tempEndDate, accessToken));
                }
                //beginDate 向后移动7天
                beginDate = DateUtil.getDiffDateTime(tempEndDate, 1);
                bContinue = DateUtil.compare(DateUtil.getDiffDateTime(beginDate, 6), endDate) || DateUtil.compare(beginDate, endDate);
            }
            
            //批量处理线程,以及线程返回结果
            if(tasks.size()>0){
                List<JSONArray> jsonArrayList = executeTasks(tasks);
                log.info("[Fans] Cost time = " + (System.currentTimeMillis() - start) + "按时间段获取微信粉丝列表");
                for(JSONArray jsonArray : jsonArrayList){
                    setFansSummaryDtoMap(jsonArray, symmaryMap);
                }
                for (FansSummaryDto fsdto : symmaryMap.values()) {
                    fansSummaryDtoList.add(fsdto);
                }
            }
        }
        dto.setSummary(fansSummaryDtoList);
        return dto;
    }
    
    static class CallableTask implements Callable<JSONArray>{
        private Date startDate;
        private Date endDate;
        private String accessToken;
        
        public CallableTask(Date startDate, Date endDate, String accessToken){
            this.startDate = startDate;
            this.endDate = endDate;
            this.accessToken = accessToken;
        }

        @Override
        public JSONArray call() throws Exception {
            JSONArray summaryArray = getJSONArray(accessToken, startDate, endDate);
            return summaryArray;
        }
        
    }
    
    public static JSONArray getJSONArray(String accessToken, Date beginDate, Date endDate){
        WechatApiResponse summaryResp = DataStatisticsApiCaller.getFansSummary(accessToken, beginDate, endDate);
        JSONObject summaryRootNode = summaryResp.getRootJSONObj();
        JSONArray summaryArray = summaryRootNode.getJSONArray("list");
        return summaryArray;
    }
    
    private static void setFansSummaryDtoMap(JSONArray summaryArray, LinkedHashMap<String, FansSummaryDto> symmaryMap){
        if (summaryArray != null & !summaryArray.isEmpty()) {
            JSONObject obj;
            FansSummaryDto fsdto;
            for (int i = 0; i < summaryArray.size(); i++) {
                obj = summaryArray.getJSONObject(i);
                fsdto = symmaryMap.get(obj.getString("ref_date"));
                fsdto.setCancelFans(fsdto.getCancelFans() + obj.getInt("cancel_user"));
                
                fsdto.setNewFans(fsdto.getNewFans() + obj.getInt("new_user"));
                fsdto.setFansDiff(fsdto.getNewFans() - fsdto.getCancelFans());
                fsdto.setFlag(true);
            }
        }
    }
    
    //批量处理多线程
    public static List<JSONArray> executeTasks(Collection<? extends Callable<JSONArray>> tasks){
        CompletionService<JSONArray> completionService = new ExecutorCompletionService<>(executor);
        for(Callable<JSONArray> task : tasks){
            completionService.submit(task);
        }
        
        List<JSONArray> results = Lists.newArrayListWithCapacity(tasks.size());
        try {
            for (int t = 0, n = tasks.size(); t < n; t++) {
                Future<JSONArray> f = completionService.take();
                JSONArray retT = f.get();
                results.add(retT);
            }
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
        } catch (Exception e) {
            log.error("cause error : {} ", e);
            throw new RuntimeException(e);
        }
        return results;
    }
}

