
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.sal.wechat.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.commons.lang.utils.collection.CollectionUtils;
import com.baijia.tianxiao.constants.org.BizConf;
import com.baijia.tianxiao.dal.commons.UnifiedWechatAccountConfigDto;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgBranchsDao;
import com.baijia.tianxiao.dal.org.dao.OrgStorageDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.dao.OrgSubAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgTeacherDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgBranchs;
import com.baijia.tianxiao.dal.org.po.OrgInfo;
import com.baijia.tianxiao.dal.org.po.OrgStorage;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.org.po.OrgSubAccount;
import com.baijia.tianxiao.dal.org.po.OrgTeacher;
import com.baijia.tianxiao.dal.wechat.constant.WechatAppServiceType;
import com.baijia.tianxiao.dal.wechat.constant.WechatStatusEnum;
import com.baijia.tianxiao.dal.wechat.dao.AuthorizerInfoDao;
import com.baijia.tianxiao.dal.wechat.dao.UnifiedWechatAccountDao;
import com.baijia.tianxiao.dal.wechat.po.AuthorizerInfo;
import com.baijia.tianxiao.dal.wechat.po.UnifiedWechatAccount;
import com.baijia.tianxiao.dto.smstoken.StudentSmsTokenDto;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.enums.OrgErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.common.api.OrganizationInfoAPIService;
import com.baijia.tianxiao.sal.common.utils.OrgInfoCampare;
import com.baijia.tianxiao.sal.wechat.api.CustomMenuService;
import com.baijia.tianxiao.sal.wechat.api.UnifiedWechatAccountService;
import com.baijia.tianxiao.sal.wechat.constant.webauth.WebAuthScope;
import com.baijia.tianxiao.sal.wechat.dto.unifiedWechatAccount.CampusInfoDto;
import com.baijia.tianxiao.sal.wechat.dto.unifiedWechatAccount.CampusWechatInfoDto;
import com.baijia.tianxiao.sal.wechat.helper.WechatProperties;
import com.baijia.tianxiao.sal.wechat.helper.webauthlink.WechatWebAuthLinkBuilder;
import com.baijia.tianxiao.util.Distance;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.properties.PropertiesReader;
import com.baijia.tianxiao.util.storage.StorageUtil;


import lombok.extern.slf4j.Slf4j;

/**
 * @say little Boy, don't be sad.
 * @name Rezar
 * @time Nov 21, 2016
 * @Desc this guy is too lazy, nothing left.
 */
@Service
@Slf4j
public class UnifiedWechatAccountServiceImpl implements UnifiedWechatAccountService {

    @Autowired
    private UnifiedWechatAccountDao unifiedWechatAccountDao;
    @Autowired
    private AuthorizerInfoDao authorizerInfoDao;
    @Autowired
    private OrgSubAccountDao orgSubAccountDao;
    @Autowired
    private OrgAccountDao orgAccountDao;
    @Autowired
    private CustomMenuService customMenuService;
    @Autowired
    private OrganizationInfoAPIService organizationInfoAPIService;
    @Autowired
    private OrgBranchsDao orgBranchsDao;
    @Autowired
    private OrgStorageDao orgStorageDao;
    @Autowired
    private OrgStudentDao orgStudentDao;
    @Autowired
    private OrgTeacherDao orgTeacherDao;

    /**
     * 
     * 这里分校区的orgId应该由orgNumber进行换取得到.
     * masterOrgId可以从authtoken中获取到
     */
    @Override
    public UnifiedWechatAccount addUnifiedWechatAccount(Long orgId, Long masterOrgId) {
        OrgAccount orgAccount = this.organizationInfoAPIService.findOrgAccountWithOrgId(orgId);
        if (orgAccount == null) {
            throw new BussinessException(CommonErrorCode.NOT_FOUND,
                "can not find any organization with orgId :" + orgId);
        }
        AuthorizerInfo masterWechatAccount = this.authorizerInfoDao.getByOrgId(masterOrgId.intValue());
        log.info("master authorizerInfo is :{} ", masterWechatAccount);
        boolean masterHasBound = true;
        // 主校区不存在公众号,不进行主校区公众号的相关初始处理
        if (masterWechatAccount == null) {
            masterHasBound = false;
            log.info(
                "master organization:{} has not bind a wechat account , but want to add a slave campus {}  as a unifiedWechatAccount ",
                masterOrgId, orgId);
        }
        // 分校区已绑定公众号,现有逻辑不允许该分校区公众号被纳入统一公众号管理的范畴,
        AuthorizerInfo slaveCampusAccount = this.authorizerInfoDao.getByOrgId(orgId.intValue(), false);
        if (slaveCampusAccount != null) {
            throw new BussinessException(OrgErrorCode.ORG_STILL_BIND_WECHAT);
        }
        UnifiedWechatAccount subCampusRecord = this.unifiedWechatAccountDao.findUnifiedWechatRecord(orgId, true, null);
        log.info("subCampusRecord is :{} ", subCampusRecord);
        if (subCampusRecord != null) {
            if (subCampusRecord.getDelStatus() == DeleteStatus.NORMAL.getValue()) {
                log.info("subCampusRecord has already init :{}", subCampusRecord);
                return subCampusRecord;
            } else {
                subCampusRecord.setDelStatus(DeleteStatus.NORMAL.getValue());
                subCampusRecord.setUpdateTime(new Date());
            }
        }
        log.info("need init master  ");
        initMasterAccount(masterOrgId, masterWechatAccount, masterHasBound);
        Integer orgNumber = orgAccount.getNumber();
        subCampusRecord = subCampusRecord == null
            ? UnifiedWechatAccount.newInstance(orgId, orgNumber.longValue(), masterOrgId.intValue()) : subCampusRecord;
        subCampusRecord.setDelStatus(DeleteStatus.NORMAL.getValue());
        this.unifiedWechatAccountDao.saveOrUpdate(subCampusRecord);
        return subCampusRecord;
    }

    /**
     * @param orgId
     * @param masterOrgId
     * @param masterWechatAccount
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public void initMasterAccount(Long masterOrgId, AuthorizerInfo masterWechatAccount, boolean masterHasBound) {
        OrgAccount orgAccount = this.organizationInfoAPIService.findOrgAccountWithOrgId(masterOrgId);
        if (orgAccount == null) {
            throw new BussinessException(CommonErrorCode.NOT_FOUND,
                "can not find any organization with orgId :" + masterOrgId);
        }
        boolean needInit = masterHasBound;
        log.info("masterUnifiedWechatAccount is need init ? {} ", needInit);
        // TODO 这里需要进行锁控制
        UnifiedWechatAccount masterRecord =
            this.unifiedWechatAccountDao.findUnifiedWechatRecord(masterOrgId, true, null);
        if (masterRecord == null) {
            // 初始化主账号微信相关
            masterRecord = UnifiedWechatAccount.newInstance(masterOrgId, orgAccount.getNumber().longValue(),
                masterOrgId.intValue());
            masterRecord.setMasterOrgId(0);// 设置为主账号
        } else {
            if (masterRecord.getDelStatus() == DeleteStatus.NORMAL.getValue() && masterRecord.isInitOver()) {
                log.info("need not init master account :{} ", masterRecord);
                return;
            }
        }
        if (masterHasBound) {
            masterRecord.setInitStatus(UnifiedWechatAccount.IS_INIT_OVER);
        }
        masterRecord.setDelStatus(DeleteStatus.NORMAL.getValue());
        try {
            this.unifiedWechatAccountDao.saveOrUpdate(masterRecord);
        } catch (Exception e) {
            GenericsUtils.logErrorAndInfo(this.getClass(), e, "can not saveOrUpdate unifiedWechatAccount:{} ",
                masterRecord);
        }
        log.info("init unifiedWechatAccount master over :{} ", masterRecord);
        if (needInit) {
            this.initMasterAccount(masterWechatAccount);
        }
    }

    /**
     * 根据当前主账号获取所有分校区公众号设置状态信息,后期如果添加权限控制的话,可以扩展dto的属性
     * 
     * @param masterOrgId
     * @return
     */
    @Override
    public List<UnifiedWechatAccountConfigDto> orgCampusUnifiedWechatAccountStatus(Long masterOrgId) {

        List<UnifiedWechatAccountConfigDto> status = null;

        List<OrgSubAccount> orgSubaccounts = this.orgSubAccountDao.getSlavesByMasterOrgId(masterOrgId.intValue(), null);
        if (GenericsUtils.isNullOrEmpty(orgSubaccounts)) {
            return GenericsUtils.emptyList();
        }
        status = new ArrayList<>(orgSubaccounts.size());
        Map<Long, OrgSubAccount> map =
            CollectionUtils.extractMap(orgSubaccounts, new CollectionUtils.Extracter<Long, OrgSubAccount>() {
                @Override
                public Long extract(OrgSubAccount arg0) {
                    return arg0.getOrgId().longValue();
                }
            });
        Map<Long, Integer> getorgNumbersByOrgIds = this.orgAccountDao.getorgNumbersByOrgIds(map.keySet());

        List<UnifiedWechatAccount> findUnifiedWechatRecords =
            this.unifiedWechatAccountDao.findUnifiedWechatRecords(map.keySet(), true, DeleteStatus.NORMAL.getValue());

        for (UnifiedWechatAccount uwa : findUnifiedWechatRecords) {
            Long orgId = uwa.getOrgId();
            if (map.get(orgId) != null) {
                status.add(UnifiedWechatAccountConfigDto.newInstance(getorgNumbersByOrgIds.get(orgId).longValue(), 1));
                map.remove(orgId);
            }
        }
        for (Long orgId : map.keySet()) {
            status.add(UnifiedWechatAccountConfigDto.newInstance(getorgNumbersByOrgIds.get(orgId).longValue(), 0));
        }
        log.info("find all campus's status are :{} for masterOrgId:{} ", status, masterOrgId);

        return status;
    }

    /**
     * @param masterWechatAccount
     */
    @Transactional(propagation = Propagation.REQUIRED)
    public void initMasterAccount(AuthorizerInfo masterWechatAccount) {
        String quickMenuJson = this.customMenuService.getQuickMenuJson(masterWechatAccount.getOrgId());
        this.customMenuService.updateOrgWechatMenu(masterWechatAccount, quickMenuJson, true);
    }

    @Override
    public String createCampusListUrl(Integer orgNumber, String appId) {
        return this.customMenuService.createCampusListUrl(orgNumber, appId);
    }

    /**
     * 校区列表展示页,包含数据:机构名称/地址展示页面/在线咨询的链接
     * 
     * @param orgNumber
     * @return
     */
    @Override
    public List<CampusInfoDto> listAllCampusUseUnifiedAccount(Long orgNumber, String openId) {
        OrgAccount orgAccount = this.orgAccountDao.getAccountByNumber(orgNumber.intValue());
        if (orgAccount == null) {
            return GenericsUtils.emptyList();
        }
        List<CampusInfoDto> campusInfoDtos = Lists.newArrayList();
        Map<Integer, OrgInfo> orgInfoMap = Maps.newHashMap();
        Map<Integer, Integer> orgNumberMapper = Maps.newHashMap();
        Integer orgId = orgAccount.getId();
        AuthorizerInfo authorizerInfo = this.authorizerInfoDao.getByOrgId(orgId);
        fillAllCampusOrgInfo(orgInfoMap, orgNumberMapper, orgId);
        // 未参与到统一公众号管理的校区机构
        if (GenericsUtils.isNullOrEmpty(orgInfoMap)) {
            orgInfoMap.put(orgId, this.organizationInfoAPIService.getOrgInfoMap(Arrays.asList(orgId)).get(orgId));
            orgNumberMapper.put(orgId, orgAccount.getNumber());
        }

        List<OrgInfo> orgInfos = new ArrayList<>(orgInfoMap.values());
        Collections.sort(orgInfos, new Comparator<OrgInfo>() {
            @Override
            public int compare(OrgInfo o1, OrgInfo o2) {
                return compareUsePinyin(o1, o2);
            }
        });

        for (OrgInfo orgInfo : orgInfos) {
            Integer orgId_ = orgInfo.getOrgId();
            Integer number = orgNumberMapper.get(orgId_);
            String orgName = orgInfo.getShortName();
            Double latitude = orgInfo.getLatitude();
            Double longitude = orgInfo.getLongitude();
            String chatUrl = "";
            if (authorizerInfo != null) {
                chatUrl = createChatUrl(authorizerInfo.getAuthorizerAppId(), orgId_, number, openId);
            }
            CampusInfoDto campusInfoDto =
                CampusInfoDto.newInstance(orgName, number, null, null, null, latitude, longitude, null, chatUrl);
            appendOrgInfo(orgInfo, number, campusInfoDto);
            campusInfoDtos.add(campusInfoDto);
        }
        return campusInfoDtos;
    }

    /**
     * @param o1
     * @param o2
     * @return
     */
    protected int compareUsePinyin(OrgInfo o1, OrgInfo o2) {
        return OrgInfoCampare.compareUsePinyin(o1.getShortName(), o2.getShortName());
    }

    /**
     * @param orgInfoMap
     * @param orgNumberMapper
     * @param orgId
     * @return
     */
    private void fillAllCampusOrgInfo(Map<Integer, OrgInfo> orgInfoMap, Map<Integer, Integer> orgNumberMapper,
        Integer orgId) {
        if (orgId == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "invalid orgId :" + orgId);
        }
        List<UnifiedWechatAccount> findUnifiedWechatRecords =
            this.unifiedWechatAccountDao.listAllCampusWithAnyOneOrgInfo(orgId.longValue(), true);
        List<Integer> orgIds = GenericsUtils.toFieldList(findUnifiedWechatRecords, "orgId");
        log.info("orgIds is :{} ", orgIds);
        orgInfoMap.putAll(this.organizationInfoAPIService.getOrgInfoMap(orgIds));
        List<OrgAccount> allOrgAccounts = this.orgAccountDao.getByIds(orgInfoMap.keySet(), "id", "number");
        // 所有机构id和机构number的映射
        orgNumberMapper.putAll(
            GenericsUtils.<List<OrgAccount>, OrgAccount, Integer, Integer> toFieldMap(allOrgAccounts, "id", "number"));
    }

    // TODO , 不能将名称作为主键返回,应该使用number,可能存在校区重名的情况
    @Override
    public Map<String, String> listAllCampusCoursesUrl(Long orgNumber) {
        OrgAccount orgAccount = this.organizationInfoAPIService.findOrgAccountWithOrgNumber(orgNumber);
        if (orgAccount == null) {
            throw new BussinessException(CommonErrorCode.NOT_FOUND,
                String.format("can not find any orgAccount with orgNumber:%s", orgNumber));
        }
        Map<String, String> orgNameAndCourseUrlMapper = Maps.newLinkedHashMap();
        Integer orgId = orgAccount.getId();
        Map<Integer, OrgInfo> orgInfoMap = Maps.newHashMap();
        Map<Integer, Integer> orgNumberMapper = Maps.newHashMap();
        this.fillAllCampusOrgInfo(orgInfoMap, orgNumberMapper, orgId);

        log.info("all orgInfo are :{} ", orgInfoMap.values());
        List<OrgInfo> orgInfos = new ArrayList<>(orgInfoMap.values());
        Collections.sort(orgInfos, new Comparator<OrgInfo>() {
            @Override
            public int compare(OrgInfo o1, OrgInfo o2) {
                return compareUsePinyin(o1, o2);
            }
        });

        log.info("keySorted are :{} ", orgInfos);
        for (OrgInfo orgInfo : orgInfos) {
            String orgName = orgInfo.getShortName();
            String courseUrl = getCourseUrl(orgNumberMapper.get(orgInfo.getOrgId()));
            orgNameAndCourseUrlMapper.put(orgName, courseUrl);
        }
        return orgNameAndCourseUrlMapper;
    }

    /**
     * @param orgNumber
     * @return
     */
    private String getCourseUrl(Integer orgNumber) {
        String fmt2 = "%s%s%s";
        return String.format(fmt2, WechatProperties.getWebMTianXiaoUrlPrefix(), "course/", orgNumber);
    }

    /**
     * 
     * 生成学员同机构在线聊天的url
     * 
     * @param orgId
     * @param number
     * @param openId
     * @return
     */
    @Override
    public String createChatUrl(String appId, Integer orgId, Integer number, String openId) {
        if (GenericsUtils.isNullOrEmpty(openId)) {
            return "";
        }
        StudentSmsTokenDto token = new StudentSmsTokenDto(orgId.longValue(), appId, null, openId);
        String imSite = null;
        try {
            imSite = WechatWebAuthLinkBuilder.imSite(WebAuthScope.BASE, appId, token.toTokenStr());
        } catch (Exception e) {
            GenericsUtils.logErrorAndInfo(log, e, "can not crate chat url for openId :{} with orgId:{} ", openId,
                orgId);
        }
        return imSite;
    }

    public void appendOrgInfo(OrgInfo orgInfo, Integer number, CampusInfoDto campusInfoDto) {
        try {
            Integer orgId = orgInfo.getOrgId();
            String _400_phone = orgInfo.getExtension();
            campusInfoDto.setMobile(
                GenericsUtils.isNullOrEmpty(_400_phone) ? _400_phone : BizConf._400_PHONE_PREFIX + "," + _400_phone);
            // 获取机构地址
            List<OrgBranchs> branchsList = orgBranchsDao.listBranchsByOrgId(orgId);
            String address = "";
            if (GenericsUtils.notNullAndEmpty(branchsList)) {
                OrgBranchs branchs = branchsList.get(0);
                address = buildAddress(branchs.getProvince(), branchs.getCounty(), branchs.getDistrict(),
                    branchs.getAddress(), branchs.getBranchName());
            }
            campusInfoDto.setAddress(address);
            campusInfoDto.setMUrl(getMUrl(number));
            String lhost = PropertiesReader.getValue("wxm", "m.server");
            if (lhost != null && lhost.startsWith("https:")) {
                lhost = lhost.replace("https", "http");
            }
            campusInfoDto.setLocationUrl(lhost + "/branch/" + number);
        } catch (Exception e) {
            log.warn("[OrgInfo] Exception and orgInfo : {} and campusInfoDto is :{} ", e.getMessage(), orgInfo,
                campusInfoDto);
        }
    }

    /**
     * @return
     */
    private String buildAddress(String...strs) {
        if (GenericsUtils.isNullOrEmpty(strs)) {
            return "";
        }
        StringBuilder sb = new StringBuilder();
        for (String str : strs) {
            if (GenericsUtils.notNullAndEmpty(str)) {
                sb.append(str);
            }
        }
        return sb.toString();
    }

    @Override
    public CampusInfoDto findCloseOrgInfo(Long orgNumber, double longitude, double latitude) {
        OrgAccount orgAccount = this.organizationInfoAPIService.findOrgAccountWithOrgNumber(orgNumber);
        if (orgAccount == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "invalid orgNumber : " + orgNumber);
        }
        Map<Integer, Integer> orgNumberMapper = Maps.newHashMap(); // orgId -- > orgNumber
        Map<Integer, OrgInfo> orgInfoMap = Maps.newHashMap();// orgId -- > orgInfo
        this.fillAllCampusOrgInfo(orgInfoMap, orgNumberMapper, orgAccount.getId());
        Map<Integer, OrgBranchs> orgBranchsMapper = queryBranchs(orgNumberMapper.keySet());
        CampusInfoDto campusInfoDto = null;
        OrgInfo mostlyCloseOrgInfo = null;
        double minDistance = Double.MAX_VALUE;
        for (Map.Entry<Integer, OrgInfo> entry : orgInfoMap.entrySet()) {
            OrgInfo orgInfo = entry.getValue();
            OrgBranchs orgBranchs = orgBranchsMapper.get(orgInfo.getOrgId());
            if (orgBranchs == null) {
                continue;
            }
            try {
                double getDistance = Distance.GetDistance(longitude, latitude,
                    GenericsUtils.getNumValueFromStr(Double.class, orgBranchs.getLongitude()),
                    GenericsUtils.getNumValueFromStr(Double.class, orgBranchs.getLatitude()));
                if (getDistance < minDistance) {
                    minDistance = getDistance;
                    mostlyCloseOrgInfo = orgInfo;
                }
            } catch (Exception e) {
                GenericsUtils.logErrorAndInfo(log, e,
                    "can not compare for : {} and current mostlyCloseOrgInfo is :{} , minDistance is :{} ", orgBranchs,
                    mostlyCloseOrgInfo, minDistance);
            }
        }

        if (mostlyCloseOrgInfo != null) {
            Integer number = orgNumberMapper.get(mostlyCloseOrgInfo.getOrgId());
            String mUrl = getMUrl(number);
            campusInfoDto =
                CampusInfoDto.newInstance(mostlyCloseOrgInfo.getName(), number, "", mUrl, "", null, null, "", "");
        } else {
            log.info(
                "Positioning failure, jump to the master account by default ,orgNumber:{} ,and longitude:{} ,latitude:{}",
                orgNumber, longitude, latitude);
            campusInfoDto = CampusInfoDto.newInstance("", orgNumber.intValue(), "", getMUrl(orgNumber.intValue()), "",
                null, null, "", "");
        }
        return campusInfoDto;
    }

    /**
     * @param
     */
    private Map<Integer, OrgBranchs> queryBranchs(Set<Integer> orgIds) {
        List<OrgBranchs> branchsList = orgBranchsDao.listBranchsByOrgIds(orgIds);
        Map<Integer, OrgBranchs> orgBranchsMapper = Maps.newHashMap();
        for (OrgBranchs ob : branchsList) {
            if (orgBranchsMapper.get(ob.getOrgId()) == null) {
                orgBranchsMapper.put(ob.getOrgId(), ob);
            }
        }
        return orgBranchsMapper;
    }

    /**
     * 根据orgNumber获取到机构的微官网主页
     * 
     * @param number
     * @return
     */
    private String getMUrl(Integer number) {
        String mhost = PropertiesReader.getValue("wxm", "m.server");
        String mUrl = mhost + "/i/" + number;
        return mUrl;
    }

    /**
     * 根据机构id信息获取机构
     * 
     * @param orgIds , 所有的机构ID(校区)
     * @param masterOrgId , 主校区的机构Id,简化当前方法查询主账号微信公众号的逻辑
     */
    @Override
    public Map<Integer, CampusWechatInfoDto> findCampusWechatStatus(Collection<Integer> orgIds, Integer masterOrgId) {
        log.info("orgIds is :{} ", orgIds);
        if (GenericsUtils.isNullOrEmpty(orgIds)) {
            return GenericsUtils.emptyMap();
        }
        Map<Integer, CampusWechatInfoDto> campusWechatInfoDtos = Maps.newHashMap();
        if (!orgIds.contains(masterOrgId)) {
            orgIds.add(masterOrgId);
        }
        Set<Long> orgIdsInLong = Sets.newHashSet();
        for (Integer orgId : orgIds) {
            orgIdsInLong.add(orgId.longValue());
        }
        List<AuthorizerInfo> authorizerInfosByOrgIds = this.authorizerInfoDao.getAuthorizerInfosByOrgIds(orgIdsInLong);
        List<UnifiedWechatAccount> findUnifiedWechatRecords =
            this.unifiedWechatAccountDao.findUnifiedWechatRecords(orgIdsInLong, true, DeleteStatus.NORMAL.getValue());
        Map<Integer, AuthorizerInfo> orgAuthorizerInfoMapper =
            GenericsUtils.toFieldMap(authorizerInfosByOrgIds, "orgId");
        Map<Long, UnifiedWechatAccount> orgUnifiedWechatAccountMapper =
            GenericsUtils.toFieldMap(findUnifiedWechatRecords, "orgId");
        AuthorizerInfo masterWechatAccount = orgAuthorizerInfoMapper.get(masterOrgId);
        log.info("masterWechatAccount is :{} ", masterWechatAccount);
        for (Integer orgId : orgIds) {
            AuthorizerInfo ai = orgAuthorizerInfoMapper.get(orgId);
            boolean isMaster = (orgId.intValue() == masterOrgId.intValue());
            UnifiedWechatAccount uw = orgUnifiedWechatAccountMapper.get(orgId.longValue());
            CampusWechatInfoDto campusWechatInfoDto = buildCampusWechatInfoDto(ai, uw, masterWechatAccount, isMaster);
            log.info("orgId:{} and ai :{} ,uw is :{} , and masterWechatAccount:{}", orgId, ai, uw, masterWechatAccount);
            campusWechatInfoDtos.put(orgId, campusWechatInfoDto);
        }
        return campusWechatInfoDtos;
    }

    /**
     * @param ai
     * @param masterWechatAccount
     * @param uw
     * @param isMaster
     * @return
     */
    private CampusWechatInfoDto buildCampusWechatInfoDto(AuthorizerInfo ai, UnifiedWechatAccount uw,
        AuthorizerInfo masterWechatAccount, boolean isMaster) {
        CampusWechatInfoDto campusWechatInfoDto = new CampusWechatInfoDto();
        if (ai != null) {
            log.info("find is like ,ai:{} ,master:{} ", ai.getId(), masterWechatAccount);
            if (masterWechatAccount != null && ai.getId().intValue() == masterWechatAccount.getId().intValue()) {
                if (isMaster) {
                    campusWechatInfoDto.setWechatName(ai.getNickName());
                    campusWechatInfoDto.setWechatStatus(WechatStatusEnum.MASTER_BIND.status);
                    campusWechatInfoDto.setWechatStatusStr(WechatStatusEnum.MASTER_BIND.desc);
                    campusWechatInfoDto.setWechatUr(findAvatarUrl(ai.getQrcodeStorageId(), ai.getQrcodeUrl()));
                    campusWechatInfoDto.setOrgWehatType(findOrgWechatType(ai));
                } else {
                    campusWechatInfoDto.setWechatName(masterWechatAccount.getNickName());
                    campusWechatInfoDto.setWechatStatus(WechatStatusEnum.UNIFIED_ACCOUNT_WITH_MASTER.status);
                    campusWechatInfoDto.setWechatStatusStr(WechatStatusEnum.UNIFIED_ACCOUNT_WITH_MASTER.desc);
                    campusWechatInfoDto.setWechatUr(
                        findAvatarUrl(masterWechatAccount.getQrcodeStorageId(), masterWechatAccount.getQrcodeUrl()));
                    campusWechatInfoDto.setOrgWehatType(findOrgWechatType(masterWechatAccount));
                }
            } else {
                campusWechatInfoDto.setWechatName(ai.getNickName());
                campusWechatInfoDto.setWechatStatus(WechatStatusEnum.INDEPENDENT.status);
                campusWechatInfoDto.setWechatStatusStr(WechatStatusEnum.INDEPENDENT.desc);
                campusWechatInfoDto.setWechatUr(findAvatarUrl(ai.getQrcodeStorageId(), ai.getQrcodeUrl()));
                campusWechatInfoDto.setOrgWehatType(findOrgWechatType(ai));
            }
        } else {
            if (masterWechatAccount != null) {
                campusWechatInfoDto.setWechatName("");
                campusWechatInfoDto.setWechatStatus(WechatStatusEnum.INDEPENDENT_UN_BIND.status);
                campusWechatInfoDto.setWechatStatusStr(WechatStatusEnum.INDEPENDENT_UN_BIND.desc);
                campusWechatInfoDto.setWechatUr("");
            } else {
                if (uw != null) {
                    campusWechatInfoDto.setWechatName("");
                    campusWechatInfoDto.setWechatStatus(WechatStatusEnum.UNIFIED_ACCOUNT_WITH_MASTER_UN_BIND.status);
                    campusWechatInfoDto.setWechatStatusStr(WechatStatusEnum.UNIFIED_ACCOUNT_WITH_MASTER_UN_BIND.desc);
                    campusWechatInfoDto.setWechatUr("");
                } else {
                    if (isMaster) {
                        campusWechatInfoDto.setWechatName("");
                        campusWechatInfoDto.setWechatStatus(WechatStatusEnum.MASTER_UN_BIND.status);
                        campusWechatInfoDto.setWechatStatusStr(WechatStatusEnum.MASTER_UN_BIND.desc);
                        campusWechatInfoDto.setWechatUr("");
                    } else {
                        campusWechatInfoDto.setWechatName("");
                        campusWechatInfoDto.setWechatStatus(WechatStatusEnum.INDEPENDENT_UN_BIND.status);
                        campusWechatInfoDto.setWechatStatusStr(WechatStatusEnum.INDEPENDENT_UN_BIND.desc);
                    }
                }
            }
        }
        return campusWechatInfoDto;
    }

    /**
     * @param ai
     * @return
     */
    private String findOrgWechatType(AuthorizerInfo ai) {
        String serviceType = WechatAppServiceType.getNote(ai.getServiceType());
        serviceType =
            WechatAppServiceType.SUBSCRIBE_APP_FROM_OLD.getValue() == ai.getServiceType() ? "订阅号" : serviceType;
        String verifyType = ai.getVerifyType() == -1 ? "未认证" : "认证";
        return verifyType + " " + serviceType;
    }

    /**
     * @param qrcodeStorageId
     * @return
     */
    private String findAvatarUrl(Integer qrcodeStorageId, String defaultValue) {
        String avatarUrl = null;
        OrgStorage storage = this.orgStorageDao.getById(qrcodeStorageId.longValue());
        log.info("storage id is : {} and storage is :{} ", qrcodeStorageId.longValue(), storage);
        if (storage != null) {
            avatarUrl = StorageUtil.constructUrl(storage.getFid(), storage.getSn(), storage.getMimeType());
        } else {
            avatarUrl = defaultValue;
        }
        return avatarUrl;
    }

    @Override
    public void clearUnifiedWechatAccountRecord(Long masterOrgId) {
        this.unifiedWechatAccountDao.deleteByOrgId(masterOrgId);
    }

    @Override
    public UnifiedWechatAccount findUnifiedWechatAccountWithOrgId(Integer orgId) {
        return this.unifiedWechatAccountDao.findUnifiedWechatRecord(orgId.longValue(), true,
            DeleteStatus.NORMAL.getValue());
    }

    @Override
    public UnifiedWechatAccount findMasterAccountWithAnyCampusOrgId(Integer orgId, Boolean isInit) {
        return this.unifiedWechatAccountDao.findMasterAccountWithAnyCampusOrgId(orgId, isInit);
    }

    /**
     * @param findOrgAccountWithOrgNumber
     * @return
     */
    public CampusInfoDto singleCampusInfoDto(OrgAccount findOrgAccountWithOrgNumber) {
        Integer orgId = findOrgAccountWithOrgNumber.getId();
        Map<Integer, OrgInfo> orgInfoMap = this.organizationInfoAPIService.getOrgInfoMap(Arrays.asList(orgId));
        CampusInfoDto cif = null;
        if (GenericsUtils.notNullAndEmpty(orgInfoMap)) {
            OrgInfo orgInfo = orgInfoMap.get(orgId);
            String name = orgInfo.getShortName();
            cif = new CampusInfoDto();
            cif.setOrgName(name);
            cif.setOrgNumber(findOrgAccountWithOrgNumber.getNumber());
            cif.setOrgNumber(findOrgAccountWithOrgNumber.getNumber());
        }
        return cif;
    }

    @Override
    public void deleteSlaveCampus(Integer slaveOrgId, Integer masterOrgId) {
        this.unifiedWechatAccountDao.deleteSlaveCampusByOrgId(slaveOrgId.longValue());
        // reset master if only left master self
        resetOrgWechatTypeIfNecessary(masterOrgId);
    }

    /**
     * @param
     */
    private void resetOrgWechatTypeIfNecessary(Integer masterOrgId) {
        boolean ifNesetOrgWechatTypeIfNecessary =
            this.unifiedWechatAccountDao.resetOrgWechatTypeIfNecessary(masterOrgId);
        log.info("ifNesetOrgWechatTypeIfNecessary:{} for orgId :{} ", ifNesetOrgWechatTypeIfNecessary, masterOrgId);
        try {
            if (ifNesetOrgWechatTypeIfNecessary) {
                AuthorizerInfo authorizerInfo = this.authorizerInfoDao.getByOrgId(masterOrgId);
                if (authorizerInfo == null) {
                    log.info("can not find authorizerInfo for masterOrgId:{} ", masterOrgId);
                    return;
                }
                String quickMenuJson = this.customMenuService.getQuickMenuJson(masterOrgId);
                log.info("to prepare reset organization's customMenu to :{} ", quickMenuJson);
                this.customMenuService.updateOrgWechatMenu(authorizerInfo, quickMenuJson, false);
                log.info("reset over for orgId:{} ", masterOrgId);
            }
        } catch (BussinessException e) {
            log.info(" 1 can not reset cause by :{} ", e.getMessage());
        } catch (Exception ex) {
            log.info(" 2 can not reset cause by :{} ", ex.getCause());
        }
    }

    @Override
    public String findWeixinFromCampusOrgWithMobile(Long orgId, String mobile, boolean isStudent) {
        List<UnifiedWechatAccount> listAllCampusWithAnyOneOrgInfo =
            this.unifiedWechatAccountDao.listAllCampusWithAnyOneOrgInfo(orgId, true);
        if (GenericsUtils.isNullOrEmpty(listAllCampusWithAnyOneOrgInfo)) {
            return "";
        }
        List<Long> orgIds = GenericsUtils.toFieldList(listAllCampusWithAnyOneOrgInfo, "orgId");
        String weixin = "";
        if (isStudent) {
            List<OrgStudent> studentsByMobileAndOrgIds =
                this.orgStudentDao.getStudentsByMobileAndOrgIds(orgIds, mobile, "weixin");
            log.info("find student are :{} ", studentsByMobileAndOrgIds);
            if (GenericsUtils.isNullOrEmpty(studentsByMobileAndOrgIds)) {
                return "";
            }
            for (OrgStudent orgStudent : studentsByMobileAndOrgIds) {
                if (GenericsUtils.notNullAndEmpty(orgStudent.getWeixin())) {
                    weixin = orgStudent.getWeixin();
                    log.info("find weixin is :{} ", weixin);
                    break;
                }
            }
        } else {
            List<OrgTeacher> getTeachersByMobileAndOrgIds =
                this.orgTeacherDao.getTeachersByMobileAndOrgIds(orgIds, mobile, "weixin");
            if (GenericsUtils.isNullOrEmpty(getTeachersByMobileAndOrgIds)) {
                return "";
            }
            for (OrgTeacher orgTeacher : getTeachersByMobileAndOrgIds) {
                if (GenericsUtils.notNullAndEmpty(orgTeacher.getWeixin())) {
                    weixin = orgTeacher.getWeixin();
                    log.info("find weixin is :{} ", weixin);
                    break;
                }
            }
        }
        return weixin;
    }

}
