/**
 * Baijiahulian.com Inc. Copyright (c) 2015-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.wechat.impl;

import java.io.File;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.common.service.WechatMsgRenderService;
import com.baijia.tianxiao.constants.MIMEType;
import com.baijia.tianxiao.dal.org.dao.OrgInfoDao;
import com.baijia.tianxiao.dal.org.dao.OrgStorageDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgBaseInfoDto;
import com.baijia.tianxiao.dal.org.po.OrgStorage;
import com.baijia.tianxiao.dal.push.constant.MsgType;
import com.baijia.tianxiao.dal.roster.dao.TxConsultUserDao;
import com.baijia.tianxiao.dal.roster.po.TxConsultUser;
import com.baijia.tianxiao.dal.wechat.constant.WechatOpenIdEntityType;
import com.baijia.tianxiao.dal.wechat.constant.WechateTemplateMsgType;
import com.baijia.tianxiao.dal.wechat.dao.FansDao;
import com.baijia.tianxiao.dal.wechat.dao.OrgWechatOpenIdRecordDao;
import com.baijia.tianxiao.dal.wechat.po.AuthorizationInfo;
import com.baijia.tianxiao.dal.wechat.po.AuthorizerInfo;
import com.baijia.tianxiao.dal.wechat.po.Fans;
import com.baijia.tianxiao.dal.wechat.po.OrgWechatOpenIdRecord;
import com.baijia.tianxiao.dto.WebResponse;
import com.baijia.tianxiao.dto.smstoken.StudentSmsTokenDto;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.exception.ParameterException;
import com.baijia.tianxiao.exception.WebServiceException;
import com.baijia.tianxiao.exception.WechatException;
import com.baijia.tianxiao.sal.common.api.OrganizationInfoAPIService;
import com.baijia.tianxiao.sal.common.api.RedisDefaultService;
import com.baijia.tianxiao.sal.wechat.api.AuthorizationInfoService;
import com.baijia.tianxiao.sal.wechat.api.AuthorizerInfoService;
import com.baijia.tianxiao.sal.wechat.api.MessagePushService;
import com.baijia.tianxiao.sal.wechat.api.UnifiedWechatAccountService;
import com.baijia.tianxiao.sal.wechat.api.WechatFreeVersionService;
import com.baijia.tianxiao.sal.wechat.api.WechatOpenIdRecordService;
import com.baijia.tianxiao.sal.wechat.constant.MediaType;
import com.baijia.tianxiao.sal.wechat.constant.SalWechatErrorCode;
import com.baijia.tianxiao.sal.wechat.constant.webauth.WebAuthScope;
import com.baijia.tianxiao.sal.wechat.dto.common.WechatMessagePayload;
import com.baijia.tianxiao.sal.wechat.dto.msgfromapp.MessageContentDto;
import com.baijia.tianxiao.sal.wechat.dto.msgfromapp.MessageDto;
import com.baijia.tianxiao.sal.wechat.dto.wechatapi.TempMediaApiDto;
import com.baijia.tianxiao.sal.wechat.helper.WechatProperties;
import com.baijia.tianxiao.sal.wechat.helper.media.WechatMediaApiHelper;
import com.baijia.tianxiao.sal.wechat.helper.sendmsg.customer.CustomerServiceApiHelper;
import com.baijia.tianxiao.sal.wechat.helper.webauthlink.WechatWebAuthLinkBuilder;
import com.baijia.tianxiao.sal.wechat.util.LocalFileHelper;
import com.baijia.tianxiao.sal.wechat.util.StorageUtil;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.ShortUrlUtil;
import com.baijia.tianxiao.util.WebResponseHelper;
import com.baijia.tianxiao.util.httpclient.HttpClientUtils;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * @title : MessagePushServiceImpl
 * @description : 接收来自app的消息并推送给微信端
 * @author : zhenyujian
 * @date : 2015年12月6日 下午10:39:33
 */
@Slf4j
@Service
public class MessagePushServiceImpl implements MessagePushService {

    @Autowired
    private AuthorizationInfoService authorizationInfoService;
    @Autowired
    private AuthorizerInfoService authorizerInfoService;
    @Autowired
    private OrgStorageDao orgStorageDao;
    @Autowired
    private WechatMsgRenderService wechatMsgRenderService;
    @Autowired
    private WechatFreeVersionService freeVersionService;
    @Autowired
    private WechatOpenIdRecordService wechatOpenIdRecordService;
    @Autowired
    private OrgWechatOpenIdRecordDao orgWechatOpenIdRecordDao;
    @Autowired
    private FansDao fansDao;
    @Autowired
    private TxConsultUserDao txConsultUserDao;
    @Autowired
    private OrgInfoDao orgInfoDao;
    @Autowired
    private UnifiedWechatAccountService unifiedWechatAccountService;
    @Autowired
    private OrganizationInfoAPIService organizationInfoAPIService;
    @Autowired
    private RedisDefaultService redisDefaultService;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public WebResponse<?> handle(MessageDto message) {
        log.info("handle - message from app to wechat - message:{}", message);

        String unifiedWechatStrAdditatioin = null;
        try {
            AuthorizerInfo authorizerInfo = authorizerInfoService.getByOrgId(message.getOrgId());

            Integer orgId = message.getOrgId();
            String appId = null;
            String openId = null;
            OrgWechatOpenIdRecord record = null;

            if (authorizerInfo == null || freeVersionService.isFreeAccount(orgId)) {
                // 免费版 或 普通版但没绑定公众号
                appId = WechatProperties.getWechatAppidForFreeVersion();
                record = orgWechatOpenIdRecordDao.getBy(appId, message.getOrgId().longValue(),
                    message.getConsultUserId(), WechatOpenIdEntityType.CONSULT);
                if (authorizerInfo == null) {
                    authorizerInfo = authorizerInfoService.getByAuthorizerAppId(appId);
                }
            } else {
                // 普通版绑定了公众号
                appId = authorizerInfo.getAuthorizerAppId();
                record = orgWechatOpenIdRecordDao.getBy(appId, message.getOrgId().longValue(),
                    message.getConsultUserId(), WechatOpenIdEntityType.CONSULT);
            }

            if (record != null) {
                openId = record.getOpenId();
            } else {
                TxConsultUser consulterUser = txConsultUserDao.getById(message.getConsultUserId());
                if (StringUtils.isNotBlank(consulterUser.getWeixinOpenId())
                    && consulterUser.getWeixinAppId().equals(appId)) {
                    openId = consulterUser.getWeixinOpenId();
                }
            }

            if (StringUtils.isNotBlank(openId)) {
                Fans fans = fansDao.getByOpenId(openId);
                if (fans == null) {
                    throw new BussinessException(SalWechatErrorCode.WECHAT_SEND_MESSAGE_FAIL_BY_SUBSCRIBE);
                } else if (!fans.isSubscribed()) {
                    throw new BussinessException(SalWechatErrorCode.WECHAT_SEND_MESSAGE_FAIL_BY_SUBSCRIBE);
                } else {
                    message.setWechatOpenId(fans.getOpenId());
                    message.setWechatAppid(fans.getAuthorizerAppId());
                }
            } else {
                throw new BussinessException(SalWechatErrorCode.WECHAT_SEND_MESSAGE_FAIL_BY_SUBSCRIBE);
            }

            // 免费版处理
            String additional = null;
            if (freeVersionService.isFreeAccount(appId)) {
                additional = permissionCheckAndGetAdditionalMsg4FreeVersion(message);
            }
            boolean isUnifiedWechatAccount =
                unifiedWechatAccountService.findMasterAccountWithAnyCampusOrgId(orgId, true) != null;
            String unifiedChatUrl = null;
            // 统一公众号处理
            if (isUnifiedWechatAccount) {
                String chatPayload = redisDefaultService.get(WechatMessagePayload.getKey(message.getWechatOpenId()));
                if (GenericsUtils.notNullAndEmpty(chatPayload) && !checkChatPayLoad(chatPayload, message.getOrgId())) {
                    return WebResponseHelper.success();
                }
                log.info("current org:{} is a type of unifiedWechatAccount", orgId);
                OrgAccount orgAccount = this.organizationInfoAPIService.findOrgAccountWithOrgId(orgId.longValue());
                unifiedChatUrl = this.unifiedWechatAccountService.createChatUrl(authorizerInfo.getAuthorizerAppId(),
                    orgId, orgAccount.getNumber(), openId);
                unifiedChatUrl = ShortUrlUtil.getShortUrl(unifiedChatUrl);
                String fromat = "<a href='%s'>%s</a>";
                additional = String.format(fromat, unifiedChatUrl, "机构回复消息了,点我与TA沟通吧");
                unifiedWechatStrAdditatioin = additional;
            }

            AuthorizationInfo authorizationInfo = authorizationInfoService.refreshAccessToken(message.getWechatAppid());
            try {
                if (message.getMsgT().intValue() == MsgType.TEXT.getValue()) {
                    handleTextMessage(authorizationInfo, message, additional);

                } else if (message.getMsgT().intValue() == MsgType.PICTURE.getValue()) {
                    handleImageMessage(authorizationInfo, message, additional);

                } else if (message.getMsgT().intValue() == MsgType.VOICE.getValue()) {
                    handleVoiceMessage(authorizationInfo, message, additional);

                } else if (message.getMsgT().intValue() == MsgType.CARD.getValue()) {
                    handleTextMessage(authorizationInfo, message, additional);

                } else {
                    return WebResponseHelper.error(SalWechatErrorCode.MESSAGE_PUSH_TO_WECHAT_NONSUUPORT_TYPE);
                }
            } catch (WechatException e) {
                if (authorizerInfo.isServiceApp() && authorizerInfo.isPassedVerify()
                    && (e.getErrorCode().getSubsystemErrorCode() == SalWechatErrorCode.WECHAT_CANNOT_SENDMSG_TO_FANS
                        .getSubsystemErrorCode()
                        || e.getErrorCode()
                            .getSubsystemErrorCode() == SalWechatErrorCode.MESSAGE_PUSH_TO_WECHAT_FAIL_45047
                                .getSubsystemErrorCode())) {
                    boolean sendTry = false;
                    try {
                        // 发送模板消息
                        Map<String, Object> params = Maps.newHashMap();

                        // 免费版处理
                        if (freeVersionService.isFreeAccount(authorizationInfo.getAuthorizerAppId())) {
                            params.put("first", "机构请求与您进行沟通，若同意请点击此消息进入聊天页面。\n");
                            params.put("url", permissionCheckAndGetImSite4FreeVersion(message));
                        } else if (isUnifiedWechatAccount) {
                            params.put("first", "机构请求与您进行沟通，若同意请点击此消息进入聊天页面。\n");
                            params.put("url", unifiedChatUrl);
                            if (message.getMsgT().intValue() == MsgType.TEXT.getValue()) {
                                String content = message.getContent().getText();
                                if (GenericsUtils.notNullAndEmpty(unifiedWechatStrAdditatioin)) {
                                    content = content.replace(unifiedWechatStrAdditatioin, "");
                                    log.info("content is :{} ", content);
                                    message.getContent().setText(content);
                                }
                            }
                        } else {
                            params.put("first", "机构请求与您进行沟通，若同意请在公众号回复任意文字。\n");
                            params.put("url", null);
                            sendTry = true;
                        }

                        params.put("remark", "");
                        params.put("keyword1", "沟通请求");
                        params.put("keyword2", message.getMsgId() + "");
                        if (message.getMsgT().intValue() == MsgType.TEXT.getValue()) {
                            params.put("keyword3", message.getContent().getText());

                        } else if (message.getMsgT().intValue() == MsgType.PICTURE.getValue()) {
                            params.put("keyword3", "[图片]");

                        } else if (message.getMsgT().intValue() == MsgType.VOICE.getValue()) {
                            params.put("keyword3", "[语音]");

                        }
                        params.put("touser", message.getWechatOpenId());

                        Object msg = wechatMsgRenderService.render(WechateTemplateMsgType.BATCH.getValue(), params);

                        wechatMsgRenderService.sendMsg(message.getOrgId().longValue(), msg);
                    } catch (Exception e2) {
                        sendTry = false;
                        log.warn("WECHAT_SEND_MESSAGE_FAIL_BUT_TRY - fail - e:{}", e);
                        throw e;
                    }
                    if (sendTry) {
                        throw new BussinessException(SalWechatErrorCode.WECHAT_SEND_MESSAGE_FAIL_BUT_TRY);
                    }
                } else {
                    throw e;
                }
            }

        } catch (WechatException | BussinessException | WebServiceException e) {
            log.warn("1handle - message from app to wechat - CustomException - message:{},e", message, e);
            throw e;
        } catch (Exception e) {
            log.error("1handle - message from app to wechat - Exception - message:{}", message);
            log.error("1handle - message from app to wechat - Exception - e", e);
            throw new WechatException(SalWechatErrorCode.MESSAGE_PUSH_TO_WECHAT_FAIL);
        }

        return WebResponseHelper.success();
    }

    /**
     * 
     * IM聊天页面限制,判断当前消息是否允许发送到微信公众号原生聊天页面<br/>
     * 
     * <统一公众号需求>
     * 
     * @param chatPayload
     * @param markPayload
     * @return false:代表不再允许发送到原生微信页 ; true:允许发送到原生页
     */
    private boolean checkChatPayLoad(String chatPayload, Integer orgId) {
        log.info("[chatPayload]  json str is :  :{} ", chatPayload);
        WechatMessagePayload wmp = WechatMessagePayload.createWechatMessagePayload(chatPayload);
        log.info("[chatPayload] WechatMessagePayload is :{} ", wmp);
        if (wmp != null && (wmp.getFromType() == WechatMessagePayload.FROM_WECHAT
            || (System.currentTimeMillis() - wmp.getLastChatTime()) > 4000
            || checkLastOrgIsNotMatch(wmp.getLastChatOrgId(), orgId))) {
            log.info(
                "message from im or user leave im page last four second or lastChatOrgId not match the message from ");
            return true;
        }
        return false;
    }

    /**
     * @param lastChatOrgId
     * @param orgId
     * @return
     */
    private boolean checkLastOrgIsNotMatch(Integer lastChatOrgId, Integer orgId) {
        return lastChatOrgId.intValue() != orgId.intValue();
    }

    public static final long TEM_MIN = 600000;

    // 文本消息
    void handleTextMessage(AuthorizationInfo authorizationInfo, MessageDto message, String additional) {
        String accessToken = authorizationInfo.getAuthorizerAccessToken();
        MessageContentDto content = message.getContent();
        String openId = message.getWechatOpenId();

        if (content.getText() == null) {
            content.setText("null");
        }

        // 免费版
        if (additional != null) {
            String newText = "%s\n\n%s";
            newText = String.format(newText, content.getText(), additional);
            content.setText(newText);
        }
        CustomerServiceApiHelper.sendTextMsg(accessToken, openId, content.getText());
    }

    // 图片消息
    void handleImageMessage(AuthorizationInfo authorizationInfo, MessageDto message, String additional)
        throws Exception {
        String accessToken = authorizationInfo.getAuthorizerAccessToken();
        String openId = message.getWechatOpenId();
        MessageContentDto content = message.getContent();

        File localFile = null;
        try {
            OrgStorage orgStorage = orgStorageDao.getById(content.getStorageId());

            byte[] data = HttpClientUtils.download(StorageUtil.constructUrl(orgStorage));
            localFile = LocalFileHelper.saveToLocal(data, MIMEType.getMimeType(orgStorage.getMimeType()));

            // 上传文件到微信
            TempMediaApiDto mediaApiDto = WechatMediaApiHelper.uploadTempMedia(accessToken, MediaType.IMAGE, localFile);
            // 发送微信消息
            CustomerServiceApiHelper.sendImageMsg(accessToken, openId, mediaApiDto.getMediaId());

            if (additional != null) {
                CustomerServiceApiHelper.sendTextMsg(accessToken, openId, additional);
            }
        } catch (Exception e) {
            throw e;
        } finally {
            if (localFile != null) {
                LocalFileHelper.deleteFile(localFile);
            }
        }
    }

    // 语音消息
    void handleVoiceMessage(AuthorizationInfo authorizationInfo, MessageDto message, String additional)
        throws Exception {
        String accessToken = authorizationInfo.getAuthorizerAccessToken();
        String openId = message.getWechatOpenId();
        MessageContentDto content = message.getContent();

        File localFile = null;
        try {
            OrgStorage orgStorage = orgStorageDao.getById(content.getStorageId());

            byte[] data = HttpClientUtils.download(StorageUtil.constructUrl(orgStorage));
            localFile = LocalFileHelper.saveToLocal(data, MIMEType.getMimeType(orgStorage.getMimeType()));
            // 上传文件到微信
            TempMediaApiDto mediaApiDto = WechatMediaApiHelper.uploadTempMedia(accessToken, MediaType.VOICE, localFile);
            // 发送微信消息
            CustomerServiceApiHelper.sendVoiceMsg(accessToken, openId, mediaApiDto.getMediaId());

            if (additional != null) {
                CustomerServiceApiHelper.sendTextMsg(accessToken, openId, additional);
            }
        } catch (Exception e) {
            throw e;
        } finally {
            if (localFile != null) {
                LocalFileHelper.deleteFile(localFile);
            }
        }
    }

    String permissionCheckAndGetAdditionalMsg4FreeVersion(MessageDto message) throws Exception {
        String url = permissionCheckAndGetImSite4FreeVersion(message);
        OrgBaseInfoDto orgInfo = orgInfoDao.getBaseInfo(message.getOrgId());
        String fmt = "<a href=\"%s\">[%s]%s</a>";
        return String.format(fmt, url, orgInfo.getShortName(), "回复你消息了，点击我与他沟通吧！");
    }

    String permissionCheckAndGetImSite4FreeVersion(MessageDto message) throws Exception {
        Long studentId = wechatOpenIdRecordService.getStudentIdBy(message.getWechatAppid(),
            message.getOrgId().longValue(), message.getWechatOpenId());
        if (studentId == null) {
            throw new ParameterException("freeversion app msg push - studentId is null");
        }
        StudentSmsTokenDto token = new StudentSmsTokenDto(message.getOrgId().longValue(), message.getWechatAppid(),
            studentId, message.getWechatOpenId());
        return WechatWebAuthLinkBuilder.imSite(WebAuthScope.BASE, message.getWechatAppid(), token.toTokenStr());
    }
}
