/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.vzhibo.service.impl;

import com.baijia.commons.lang.utils.PropertiesReader;
import com.baijia.tianxiao.constant.Flag;
import com.baijia.tianxiao.constant.TxVZhiBoLessonStatusEnums;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.vzhibo.constant.MessageTypeEnums;
import com.baijia.tianxiao.dal.vzhibo.dao.TxVZhiBoLessonDao;
import com.baijia.tianxiao.dal.vzhibo.po.TxVZhiBoLesson;
import com.baijia.tianxiao.dal.vzhibo.po.TxVZhiBoRoom;
import com.baijia.tianxiao.dal.wechat.po.AuthorizationInfo;
import com.baijia.tianxiao.dal.wechat.po.AuthorizerInfo;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.enums.RedisKeyEnums;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.common.dto.InviteCardTaskDto;
import com.baijia.tianxiao.sal.marketing.commons.exceptions.BusinessException;
import com.baijia.tianxiao.sal.organization.api.OrgAccountService;
import com.baijia.tianxiao.sal.organization.org.dto.OrgInfoSimpleDto;
import com.baijia.tianxiao.sal.organization.org.dto.TxCascadeCredentialDto;
import com.baijia.tianxiao.sal.organization.org.service.OrgInfoService;
import com.baijia.tianxiao.sal.organization.org.service.TxCascadeCredentialService;
import com.baijia.tianxiao.sal.vzhibo.constant.AreaTypeEnums;
import com.baijia.tianxiao.sal.vzhibo.constant.InviteCardStatus;
import com.baijia.tianxiao.sal.vzhibo.constant.OwnerTypeEnums;
import com.baijia.tianxiao.sal.vzhibo.constant.TxVZhiBoEventType;
import com.baijia.tianxiao.sal.vzhibo.constant.TxVZhiBoLessonFieldEnums;
import com.baijia.tianxiao.sal.vzhibo.constant.TxVZhiBoLessonListTypeEnums;
import com.baijia.tianxiao.sal.vzhibo.constant.TxVZhiBoUserType;
import com.baijia.tianxiao.sal.vzhibo.constant.UserTypeEnums;
import com.baijia.tianxiao.sal.vzhibo.service.AudienceService;
import com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoEventLogService;
import com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoInviteCardService;
import com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoLessonService;
import com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoLessonStudentService;
import com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoMessageService;
import com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoRoomService;
import com.baijia.tianxiao.sal.vzhibo.vo.AudienceDto;
import com.baijia.tianxiao.sal.vzhibo.vo.FieldUpdateVO;
import com.baijia.tianxiao.sal.vzhibo.vo.LessonDetailVO;
import com.baijia.tianxiao.sal.vzhibo.vo.ListRequestDto;
import com.baijia.tianxiao.sal.vzhibo.vo.MessageInfo;
import com.baijia.tianxiao.sal.vzhibo.vo.ShareDataVO;
import com.baijia.tianxiao.sal.vzhibo.vo.TxVZhiBoLessonListVO;
import com.baijia.tianxiao.sal.wechat.api.AuthorizationInfoService;
import com.baijia.tianxiao.sal.wechat.api.AuthorizerInfoService;
import com.baijia.tianxiao.sal.wechat.constant.WechatApi;
import com.baijia.tianxiao.sal.wechat.constant.webauth.WebAuthScope;
import com.baijia.tianxiao.sal.wechat.helper.webauthlink.WechatWebAuthLinkBuilder;
import com.baijia.tianxiao.sal.wechat.validator.WechatApiValidator;
import com.baijia.tianxiao.util.date.DateUtil;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import lombok.extern.slf4j.Slf4j;

/**
 * @title LiveServiceImpl
 * @desc TODO
 * @author he11o
 * @date 2016年11月25日
 * @version 1.0
 */
@Slf4j
@Service
public class TxVZhiBoLessonServiceImpl implements TxVZhiBoLessonService {

    @Autowired
    private TxVZhiBoInviteCardService txVZhiBoInviteCardService;

    @Autowired
    private AudienceService audienceService;

    @Autowired
    private AuthorizerInfoService authorizerInfoService;

    @Autowired
    AuthorizationInfoService authorizationInfoService;

    @Autowired
    OrgAccountService orgAccountService;

    @Autowired
    TxVZhiBoLessonDao txVZhiBoLessonDao;

    @Autowired
    TxVZhiBoLessonStudentService txVZhiBoLessonStudentService;

    @Autowired
    TxVZhiBoEventLogService txVZhiBoEventLogService;

    @Autowired
    TxVZhiBoRoomService txVZhiBoRoomService;

    @Value("${lesson_detail_url}")
    private String LESSON_DETAIL_URL; // 直播课简介url前缀

    @Value("${room_detail_url}")
    private String ROOM_DETAIL_URL;

    @Value("${room_expire_time}")
    private String roomExpireTime; // 直播间创建后过期时间，单位：天 l

    @Value("${tianxiaojia_app_id}")
    private String tianXiaoJiaAppId;

    @Autowired
    TxCascadeCredentialService txCascadeCredentialService;

    @Autowired
    OrgInfoService orgInfoService;

    @Autowired
    TxVZhiBoMessageService txVZhiBoMessageService;

    @Autowired
    StringRedisTemplate redisTemplate;

    @Value("${msg_queue_pos}")
    String MSG_QUEUE_POS;

    // private LoadingCache<Integer, AuthorizationInfo> authorizationInfoCache =
    // CacheBuilder.newBuilder().maximumSize(5000)
    // .expireAfterWrite(10, TimeUnit.HOURS).build(new CacheLoader<Integer, AuthorizationInfo>() {
    // @Override
    // public AuthorizationInfo load(Integer key) throws Exception {
    // Preconditions.checkNotNull(key);
    // return authorizationInfoService.getByOrgId(key);
    // }
    // });

    private static final String YET_START_SHARE_TITLE = "微直播 | %s将于%s开课，别忘记上课哦~";
    private static final String ONGOING_MANY_PEOPLE_SHARE_TITLE = "直播中 | %s%s人正在听课，马上进入教室吧！";
    private static final String ONGOING_FEW_PEOPLE_SHARE_TITLE = "直播中 | %s正在上课，马上进入教室吧！";
    private static final int MANY_FEW_THRESHOLD = 5;

    public boolean validateLessonPassword(Integer lessonId, String password) {
        TxVZhiBoLesson txVZhiBoLesson = this.txVZhiBoLessonDao.getById(lessonId);
        if (null == txVZhiBoLesson) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "直播课节不存在");
        } else {
            if (StringUtils.isBlank(txVZhiBoLesson.getPassword())) {
                return true;
            } else {
                return StringUtils.equals(txVZhiBoLesson.getPassword(), password);
            }
        }
    }

    @Override
    public String getLiveLessonUrl(Integer OrgNumber, Integer lessonId) {
        log.info("[getLiveLessonUrl]OrgNumber:{},lessonId:{}", OrgNumber, lessonId);
        OrgAccount orgAccount = orgAccountService.getOrgAccountByNumber(OrgNumber);
        if (orgAccount == null) {
            return "";
        }
        AuthorizationInfo authorizationInfo = authorizationInfoService.getByOrgId(orgAccount.getId());
        AuthorizerInfo authorizerInfo = authorizerInfoService.getByOrgId(orgAccount.getId());
        // try {
        // authorizationInfo = authorizationInfoCache.get(orgAccount.getId());
        // } catch (Exception e) {
        // log.info("get authorizationInfo faild,orgId:{}",orgAccount.getId());
        // }
        String appId = "";
        if (authorizationInfo == null || StringUtils.isBlank(authorizationInfo.getAuthorizerAppId())) {
            // 机构还没有绑定公众号时，用天校家的公众号
            appId = tianXiaoJiaAppId;
        } else {
            // 机构绑定的公众号权限不够时, 也用天校家.
            try {
                WechatApiValidator._4CallApi(authorizationInfo, authorizerInfo, WechatApi.USER_WEBAUTH);
                appId = authorizationInfo.getAuthorizerAppId();
            } catch (Exception e) {
                log.warn("orgId:{}, bind wechat no webauth permission. try to use tianxiaojia exception:{}",
                    orgAccount.getId(), e);
                appId = tianXiaoJiaAppId;
            }
        }
        String lessonLiveUrl = PropertiesReader.getValue("config", "lesson_live_url");
        String redirectUrl = lessonLiveUrl.replace("{orgNumber}", orgAccount.getNumber().toString())
            .replace("{lessonId}", lessonId.toString());
        log.info("redirectUrl is:{} ", redirectUrl);
        return WechatWebAuthLinkBuilder.fansinfo(WebAuthScope.BASE, appId, redirectUrl);
    }

    @Override
    public List<TxVZhiBoLessonListVO> list(Integer roomId, Integer ownerId, Integer ownerType, Integer status,
        Integer hidden, String key, ListRequestDto pageDto, boolean all) {
        log.info("[list]roomId:{},ownerId:{},pageDto:{}, all:{}", roomId, ownerId, pageDto, all);
        List<TxVZhiBoLesson> txVZhiBoLessons = null;
        if (all) {
            txVZhiBoLessons = txVZhiBoLessonDao.list(roomId, null, null, status, hidden, key, pageDto.getLastId(),
                pageDto.getPageSize());
        } else {
            txVZhiBoLessons = txVZhiBoLessonDao.list(roomId, ownerId, ownerType, status, hidden, key,
                pageDto.getLastId(), pageDto.getPageSize());
        }
        if (CollectionUtils.isEmpty(txVZhiBoLessons)) {
            return Lists.newArrayList();
        }
        return buildTxVZhiBoLessonListVO(txVZhiBoLessons, roomId, ownerId, ownerType);
    }

    private List<TxVZhiBoLessonListVO> buildTxVZhiBoLessonListVO(List<TxVZhiBoLesson> txVZhiBoLessons, Integer roomId,
        Integer ownerId, Integer ownerType) {
        // 获取由子账号创建的直播课的老师名称
        List<Integer> cascadeIdList = Lists.transform(txVZhiBoLessons, new Function<TxVZhiBoLesson, Integer>() {
            @Override
            public Integer apply(TxVZhiBoLesson input) {
                if (OwnerTypeEnums.CASCADE_ACCOUNT.getCode() == input.getOwnerType()) {
                    return input.getOwnerId();
                }
                return null;
            }
        });
        // 批量获取对应关系，key:cascadeId, value:name
        Map<Integer, String> teacherNameMap = txCascadeCredentialService.getNameMapByAccountIds(cascadeIdList);
        // 批量获取直播课学生统计信息,key:lessonId, value:studentCount
        Map<Integer, Integer> studentStat = txVZhiBoLessonStudentService.statStudentCountByRoomId(roomId);
        // 转换为VO
        return buildTxVZhiBoLessonListVO(txVZhiBoLessons, studentStat, teacherNameMap, ownerId, ownerType);

    }

    private List<TxVZhiBoLessonListVO> buildTxVZhiBoLessonListVO(List<TxVZhiBoLesson> txVZhiBoLessons,
        Map<Integer, Integer> studentStat, Map<Integer, String> teacherNameMap, Integer ownerId, Integer ownerType) {
        List<TxVZhiBoLessonListVO> lessonListVOs = Lists.newArrayListWithExpectedSize(txVZhiBoLessons.size());
        for (TxVZhiBoLesson txVZhiBoLesson : txVZhiBoLessons) {
            TxVZhiBoLessonListVO txVZhiBoLessonListVO = new TxVZhiBoLessonListVO();
            lessonListVOs.add(txVZhiBoLessonListVO);

            txVZhiBoLessonListVO.setOrgId(txVZhiBoLesson.getOrgId());
            txVZhiBoLessonListVO.setOrgNumber(txVZhiBoLesson.getOrgNumber());
            txVZhiBoLessonListVO.setHidden(txVZhiBoLesson.getHidden());
            txVZhiBoLessonListVO.setId(txVZhiBoLesson.getId());
            txVZhiBoLessonListVO.setRoomId(txVZhiBoLesson.getRoomId());
            txVZhiBoLessonListVO.setName(txVZhiBoLesson.getName());
            if (txVZhiBoLesson.getStartTime() != null) {
                txVZhiBoLessonListVO.setStartTime(txVZhiBoLesson.getStartTime().getTime());
            }
            txVZhiBoLessonListVO.setCreateTime(txVZhiBoLesson.getCreateTime());
            txVZhiBoLessonListVO.setAvatar(txVZhiBoLesson.getAvatar());
            txVZhiBoLessonListVO.setDetailUrl(getDetailUrl(txVZhiBoLesson.getId()));

            String liveUrl = getLiveLessonUrl(txVZhiBoLesson.getOrgNumber(), txVZhiBoLesson.getId().intValue());
            txVZhiBoLessonListVO.setLiveUrl(liveUrl);

            txVZhiBoLessonListVO.setStatus(txVZhiBoLesson.getStatus());
            if (studentStat.containsKey(txVZhiBoLesson.getId().intValue())) {
                txVZhiBoLessonListVO.setStudentCount(studentStat.get(txVZhiBoLesson.getId().intValue()));
            }
            if (teacherNameMap.containsKey(txVZhiBoLesson.getOwnerId())) {
                txVZhiBoLessonListVO.setTeacherName(teacherNameMap.get(txVZhiBoLesson.getOwnerId()));
            } else {
                try {
                    OrgInfoSimpleDto simpleDto = orgInfoSimpleDtoCache.get(txVZhiBoLesson.getOrgId());
                    txVZhiBoLessonListVO.setTeacherName(simpleDto.getShortName());
                } catch (Exception e) {
                    log.error("Error in Get OrgInfoSimpleDto", e);
                }
            }
            if (null != ownerId && null != ownerType && ownerId.intValue() == txVZhiBoLesson.getOwnerId()
                && ownerType.intValue() == txVZhiBoLesson.getOwnerType()) {
                txVZhiBoLessonListVO.setOwnType(TxVZhiBoLessonListTypeEnums.OWN.getCode());
            }
        }
        return lessonListVOs;
    }

    private LoadingCache<Integer, OrgInfoSimpleDto> orgInfoSimpleDtoCache = CacheBuilder.newBuilder().maximumSize(5000)
        .expireAfterWrite(120, TimeUnit.MINUTES).build(new CacheLoader<Integer, OrgInfoSimpleDto>() {
            @Override
            public OrgInfoSimpleDto load(Integer orgId) throws Exception {
                Preconditions.checkNotNull(orgId);
                return orgInfoService.getOrgInfo(new Long(orgId));
            }
        });

    @Override
    public List<TxVZhiBoLessonListVO> listOthers(Integer roomId, Integer ownerId, Integer ownerType, Integer status,
        Integer hidden, String key, ListRequestDto pageDto) {
        log.info("[listOthers]roomId:{},ownerId:{},pageDto:{}", roomId, ownerId, pageDto);
        List<TxVZhiBoLesson> txVZhiBoLessons = txVZhiBoLessonDao.listOthers(roomId, ownerId, ownerType, status, hidden,
            key, pageDto.getLastId(), pageDto.getPageSize());
        if (CollectionUtils.isEmpty(txVZhiBoLessons)) {
            return Lists.newArrayList();
        }
        List<TxVZhiBoLessonListVO> boLessonListVOs =
            buildTxVZhiBoLessonListVO(txVZhiBoLessons, roomId, ownerId, ownerType);
        // 替换直播url为无授权的
        // why??? 因为授权的app访问不了
        return boLessonListVOs;
    }

    private void initWithoutAuthLiveUrl(List<TxVZhiBoLessonListVO> boLessonListVOs) {
        for (TxVZhiBoLessonListVO lessonListVO : boLessonListVOs) {
            String lessonLiveUrl = PropertiesReader.getValue("config", "lesson_live_url");
            String liveUrl = lessonLiveUrl.replace("{orgNumber}", lessonListVO.getOrgNumber().toString())
                .replace("{lessonId}", lessonListVO.getId() + "");
            lessonListVO.setLiveUrl(liveUrl);
        }
    }

    @Override
    public Long create(Integer orgId, Integer txCascadeId, TxVZhiBoLesson txVZhiBoLesson) {
        OrgAccount orgAccount = orgAccountService.getOrgAccountById(orgId);
        // 检测orgId的真实性
        if (orgAccount == null) {
            return null;
        }
        txVZhiBoLesson.setOrgId(orgId);
        txVZhiBoLesson.setOrgNumber(orgAccount.getNumber());
        if (txCascadeId == null) {
            txVZhiBoLesson.setOwnerId(orgId);
            txVZhiBoLesson.setOwnerType(OwnerTypeEnums.MAIN.getCode());
        } else {
            txVZhiBoLesson.setOwnerId(txCascadeId);
            txVZhiBoLesson.setOwnerType(OwnerTypeEnums.CASCADE_ACCOUNT.getCode());
        }
        txVZhiBoLessonDao.save(txVZhiBoLesson);
        // 入库后初始化一条消息
        try {
            MessageInfo messageInfo = initLessonMessage(txVZhiBoLesson.getId(), orgAccount);
            txVZhiBoMessageService.send(messageInfo, null);
        } catch (Exception e) {
            log.info("Error in 初始化直播间消息失败", e);
            throw new BusinessException("初始化直播间消息失败!");
        }

        // 生成邀请卡
        InviteCardTaskDto taskDto = new InviteCardTaskDto(txVZhiBoLesson.getId().intValue(), "", false);
        redisTemplate.opsForList().rightPush(RedisKeyEnums.VZB.VZHIBO_INVITE_CARD_TASK_QUEUE.getRedisKey(),
            taskDto.toJson());

        return txVZhiBoLesson.getId();
    }

    private MessageInfo initLessonMessage(Long lessonId, OrgAccount orgAccount) {
        MessageInfo messageInfo = new MessageInfo();
        messageInfo.setAreaType(AreaTypeEnums.ZHIBO.getCode());
        messageInfo.setMsgType(MessageTypeEnums.TEXT.getCode());
        messageInfo.setLessonId(lessonId.intValue());
        messageInfo.setOrgId(orgAccount.getId());
        messageInfo.setContent(PropertiesReader.getValue("config-common", "lesson_init_msg"));
        return messageInfo;
    }

    @Override
    public long getExpireTime(Integer lessonId) {
        TxVZhiBoLesson txVZhiBoLesson = txVZhiBoLessonDao.getById(lessonId);
        if (txVZhiBoLesson == null) {
            return -1;
        }
        Calendar endTime = Calendar.getInstance();
        endTime.setTime(txVZhiBoLesson.getStartTime());
        endTime.add(Calendar.DAY_OF_MONTH, Integer.parseInt(roomExpireTime));
        endTime.add(Calendar.HOUR_OF_DAY, 2);
        Calendar currentTime = Calendar.getInstance();
        return endTime.getTimeInMillis() - currentTime.getTimeInMillis();
    }

    @Override
    public TxVZhiBoLesson getByOrgIdAndId(Integer orgId, int lessonId) {
        log.info("[getByOrgIdAndId]orgId:{},lessonId:{}", orgId, lessonId);
        return txVZhiBoLessonDao.getByOrgIdAndId(orgId, lessonId);
    }

    @Override
    public boolean end(Integer orgId, Integer txCascadeId, int lessonId) {
        TxVZhiBoLesson txVZhiBoLesson = getByOrgIdAndId(orgId, lessonId);
        if (txVZhiBoLesson != null) {
            int mod =
                txVZhiBoLessonDao.updateColumnValueById(lessonId, "status", TxVZhiBoLessonStatusEnums.DONE.getCode());
            if (mod > 0) {
                // 1.记录主动结束直播间Event
                TxVZhiBoUserType userType = TxVZhiBoUserType.CASCADE_ACCOUNT;
                Integer uId = txCascadeId;
                if (txCascadeId == null) {
                    uId = orgId;
                    userType = TxVZhiBoUserType.JIGOU;
                }
                // 2.持久化消息队列的数据
                txVZhiBoMessageService.persistMessage(lessonId);
                txVZhiBoEventLogService.saveEvent(lessonId, uId, userType, TxVZhiBoEventType.SELF_EXIT);
                txVZhiBoMessageService.persistRecallAll(lessonId);
                // 3.移除消息队列
                removeMessageQuque(lessonId);
                // 4.更新缓存中直播间状态
                txVZhiBoLesson.setStatus(TxVZhiBoLessonStatusEnums.DONE.getCode());
                lessonCache.put(txVZhiBoLesson.getId().intValue(), txVZhiBoLesson);
                return true;
            }
        }
        return false;
    }

    /**
     * 各种删除
     * 
     * @param lessonId 直播课节id
     */
    @Override
    public void removeMessageQuque(int lessonId) {
        // 删除直播区消息
        redisTemplate.delete(AreaTypeEnums.ZHIBO.getPrefix() + lessonId);
        // 删除直播区消息持久化计数器
        redisTemplate.delete(AreaTypeEnums.ZHIBO.getPrefix() + lessonId + MSG_QUEUE_POS);
        // 删除直播区撤回list
        redisTemplate.delete(AreaTypeEnums.ZHIBO.getDeleteIndexListPrefix() + lessonId);
        redisTemplate.delete(AreaTypeEnums.ZHIBO.getDeleteIndexListPrefix() + lessonId + MSG_QUEUE_POS);
        // 删除讨论区消息
        redisTemplate.delete(AreaTypeEnums.DISCUSS.getPrefix() + lessonId);
        // 删除讨论区消息持久化计数器
        redisTemplate.delete(AreaTypeEnums.DISCUSS.getPrefix() + lessonId + MSG_QUEUE_POS);
        // 删除讨论区撤回list
        redisTemplate.delete(AreaTypeEnums.DISCUSS.getDeleteIndexListPrefix() + lessonId);
        redisTemplate.delete(AreaTypeEnums.DISCUSS.getDeleteIndexListPrefix() + lessonId + MSG_QUEUE_POS);
        // 删除课节微信绑定set
        // redisTemplate.delete(RedisKeyEnums.VZB.STUDENT_COUNT_PREFFIX.getRedisKey() + lessonId);
    }

    @Override
    public TxVZhiBoLesson getById(long lessonId) {
        log.info("[getById]id:{}", lessonId);
        return txVZhiBoLessonDao.getById(lessonId);
    }

    @Override
    public int syncLessonStatus() {
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(System.currentTimeMillis());
        log.info("[syncLessonStatus]currentTime:{}", calendar.getTime());
        calendar.add(Calendar.DAY_OF_MONTH, -7);
        calendar.add(Calendar.HOUR_OF_DAY, -2);
        log.info("[syncLessonStatus]time before:{}", calendar.getTime());
        return txVZhiBoLessonDao.syncLessonStatus(calendar.getTime(), TxVZhiBoLessonStatusEnums.DONE.getCode());
    }

    @Override
    public ShareDataVO getShareData(int lessonId) {
        TxVZhiBoLesson txVZhiBoLesson = txVZhiBoLessonDao.getById(lessonId);
        if (txVZhiBoLesson == null) {
            return null;
        }
        ShareDataVO shareDataVO = new ShareDataVO();
        shareDataVO.setTitle(txVZhiBoLesson.getName());
        shareDataVO.setImageUrl(txVZhiBoLesson.getAvatar());
        shareDataVO.setContent(txVZhiBoLesson.getDescription());
        String liveUrl = getLiveLessonUrl(txVZhiBoLesson.getOrgNumber(), lessonId);
        shareDataVO.setShareUrl(liveUrl);
        return shareDataVO;
    }

    @Override
    public ShareDataVO getShareData(int lessonId, long studentCount) {
        TxVZhiBoLesson txVZhiBoLesson = txVZhiBoLessonDao.getById(lessonId);
        if (txVZhiBoLesson == null) {
            return null;
        }
        ShareDataVO shareDataVO = new ShareDataVO();
        shareDataVO.setImageUrl(txVZhiBoLesson.getAvatar());
        shareDataVO.setContent(txVZhiBoLesson.getDescription());
        String liveUrl = getLiveLessonUrl(txVZhiBoLesson.getOrgNumber(), lessonId);
        shareDataVO.setShareUrl(liveUrl);

        if (txVZhiBoLesson.getStartTime().after(new Date())) {
            String dateStr = DateUtil.getStrByDateFormate(txVZhiBoLesson.getStartTime(), "M月d日HH:mm");
            shareDataVO.setTitle(String.format(YET_START_SHARE_TITLE, txVZhiBoLesson.getName(), dateStr));
        } else if (studentCount < MANY_FEW_THRESHOLD) {
            shareDataVO.setTitle(String.format(ONGOING_FEW_PEOPLE_SHARE_TITLE, txVZhiBoLesson.getName()));
        } else {
            shareDataVO
                .setTitle(String.format(ONGOING_MANY_PEOPLE_SHARE_TITLE, txVZhiBoLesson.getName(), studentCount));
        }
        return shareDataVO;
    }

    // private LoadingCache<Integer, LessonDetailVO> lessonDetailCache = CacheBuilder.newBuilder().maximumSize(2000)
    // .expireAfterWrite(120, TimeUnit.MINUTES).build(new CacheLoader<Integer, LessonDetailVO>() {
    // @Override
    // public LessonDetailVO load(Integer key) throws Exception {
    // TxVZhiBoLesson zhiBoLesson = getById(key);
    // if(zhiBoLesson == null){
    // return null;
    // }
    // return convert2DetailVO(zhiBoLesson);
    // }
    // });

    @Override
    public LessonDetailVO getDetailById(Integer lessonId) throws BussinessException, NumberFormatException, Exception {
        // try {
        // LessonDetailVO detailVO = lessonDetailCache.get(lessonId);
        // return detailVO;
        // } catch (Exception e) {
        // log.warn("Error In getDetailById[LessonDetailVO]",e);
        // }
        // return null;
        TxVZhiBoLesson zhiBoLesson = getById(lessonId);
        if (zhiBoLesson == null) {
            return null;
        }
        // 检查直播课状态
        if (zhiBoLesson.getStatus() == TxVZhiBoLessonStatusEnums.LIVE.getCode()) {
            // 若消息队列的key已经不存在了，说明直播课已经结束了
            String msgQuqueKey = AreaTypeEnums.ZHIBO.getPrefix() + lessonId;
            if (!redisTemplate.hasKey(msgQuqueKey)) {
                log.info("[getDetailById]lessonId:{} 消息队列已经不存在了，key:{}", lessonId, msgQuqueKey);
                // 1.修正直播课状态
                zhiBoLesson.setStatus(TxVZhiBoLessonStatusEnums.DONE.getCode());
                // 2.更新数据库
                txVZhiBoLessonDao.updateColumnValueById(zhiBoLesson.getId(), "status",
                    TxVZhiBoLessonStatusEnums.DONE.getCode());
            }
        }
        return convert2DetailVO(zhiBoLesson);
    }

    private LessonDetailVO convert2DetailVO(TxVZhiBoLesson zhiBoLesson)
        throws BussinessException, NumberFormatException, Exception {
        LessonDetailVO lessonDetailVO = new LessonDetailVO();
        lessonDetailVO.setId(zhiBoLesson.getId());
        lessonDetailVO.setAvatar(zhiBoLesson.getAvatar());
        lessonDetailVO.setDescription(zhiBoLesson.getDescription());
        lessonDetailVO.setName(zhiBoLesson.getName());
        lessonDetailVO.setOrgNumber(zhiBoLesson.getOrgNumber());
        lessonDetailVO.setPassword(zhiBoLesson.getPassword());
        lessonDetailVO.setHidden(zhiBoLesson.getHidden());
        if (zhiBoLesson.getStartTime() != null) {
            lessonDetailVO.setStartTime(zhiBoLesson.getStartTime().getTime());
        }
        lessonDetailVO.setRoomId(zhiBoLesson.getRoomId());

        OrgInfoSimpleDto orgInfo = orgInfoService.getOrgInfo(Long.parseLong(zhiBoLesson.getOrgId().toString()));
        lessonDetailVO.setOrgName(orgInfo.getShortName());
        lessonDetailVO.setOrgAvatar(orgInfo.getLogo());
        // 机构创建的直播课，老师名称填机构的名称
        if (orgInfo != null && OwnerTypeEnums.MAIN.getCode() == zhiBoLesson.getOwnerType()) {
            lessonDetailVO.setOrgName(orgInfo.getShortName());
            lessonDetailVO.setTeacherName(orgInfo.getShortName());
            lessonDetailVO.setTeacherAvatar(orgInfo.getLogo());
        } else {
            TxCascadeCredentialDto credentialDto = txCascadeCredentialService
                .getByTxCasCade(Long.parseLong(zhiBoLesson.getOrgId().toString()), zhiBoLesson.getOwnerId());
            if (credentialDto != null) {
                lessonDetailVO.setTeacherName(credentialDto.getName());
                if (StringUtils.isNotBlank(credentialDto.getAvatarUrl())) {
                    lessonDetailVO.setTeacherAvatar(credentialDto.getAvatarUrl());
                } else {
                    lessonDetailVO
                        .setTeacherAvatar(com.baijia.tianxiao.biz.www.constant.BizConstant.DEFAULT_CASCADE_AVATAR);
                }
            }
        }

        TxVZhiBoRoom txVZhiBoRoom = txVZhiBoRoomService.getById(zhiBoLesson.getRoomId());
        if (txVZhiBoRoom != null) {
            lessonDetailVO.setRoomName(
                StringUtils.isNotBlank(txVZhiBoRoom.getName()) ? txVZhiBoRoom.getName() : lessonDetailVO.getOrgName());
            lessonDetailVO.setRoomAvatar(StringUtils.isNotBlank(txVZhiBoRoom.getAvatar()) ? txVZhiBoRoom.getAvatar()
                : lessonDetailVO.getOrgAvatar());
        }

        lessonDetailVO.setStatus(zhiBoLesson.getStatus());
        // 获取直播课地址
        String liveUrl = getLiveLessonUrl(zhiBoLesson.getOrgNumber(), zhiBoLesson.getId().intValue());
        lessonDetailVO.setLiveUrl(liveUrl);
        // 获取直播间地址
        String roomUrl = ROOM_DETAIL_URL + zhiBoLesson.getOrgNumber();
        lessonDetailVO.setRoomUrl(roomUrl);

        ShareDataVO shareDataVO = new ShareDataVO();
        shareDataVO.setTitle(lessonDetailVO.getName());
        shareDataVO.setImageUrl(lessonDetailVO.getAvatar());
        shareDataVO.setContent(lessonDetailVO.getDescription());
        shareDataVO.setShareUrl(LESSON_DETAIL_URL + lessonDetailVO.getId());
        lessonDetailVO.setShareData(shareDataVO);

        List<AudienceDto> audienceList =
            audienceService.latestAudienceByCourseId(zhiBoLesson.getId().intValue(), zhiBoLesson.getOrgId(), 5);

        try {
            AuthorizationInfo authorizationInfo = authorizationInfoService.refreshAccessToken(zhiBoLesson.getOrgId());
            AuthorizerInfo authorizerInfo = authorizerInfoService.getByOrgId(zhiBoLesson.getOrgId());
            WechatApiValidator._4CallApi(authorizationInfo, authorizerInfo, WechatApi.USER_WEBAUTH);
        } catch (Exception e) {
            log.info("lesson:{} find no auth wechat. no invite card! e:{}", zhiBoLesson, e.getMessage());
            lessonDetailVO.setCardStatus(InviteCardStatus.WECHAT_NO_AUTH.getCode());
            lessonDetailVO.setCardStatusStr(InviteCardStatus.WECHAT_NO_AUTH.getDesc());
        }

        lessonDetailVO
            .setAudienceNum(txVZhiBoLessonStudentService.getStudentCountbyLessonId(zhiBoLesson.getId().intValue()));

        lessonDetailVO.setAudienceList(audienceList);
        return lessonDetailVO;
    }

    @Override
    public int update(Integer orgId, FieldUpdateVO updateVO) {
        Map<String, Object> countCondition = Maps.newHashMap();
        countCondition.put("orgId", orgId);
        countCondition.put("id", updateVO.getId());
        // 判断直播课是否属于当前机构
        int count = txVZhiBoLessonDao.countByCondition(countCondition, "id", true);
        if (count < 1) {
            return -1;
        }
        TxVZhiBoLessonFieldEnums fieldEnums = TxVZhiBoLessonFieldEnums.parse(updateVO.getField());
        if (fieldEnums == null) {
            return -1;
        }
        Map<String, Object> updateCondtion = Maps.newHashMap();
        if (TxVZhiBoLessonFieldEnums.STARTTIME.equals(fieldEnums)) {
            updateCondtion.put(fieldEnums.getName(), new Date(Long.parseLong(updateVO.getValue())));
        } else if (TxVZhiBoLessonFieldEnums.HIDDEN.equals(fieldEnums)) {
            Preconditions.checkArgument(updateVO.getValue().toString().equals("1") || updateVO.getValue().equals("0"),
                "hidden字段传值错误");
            updateCondtion.put(fieldEnums.getName(), updateVO.getValue());
        } else if (TxVZhiBoLessonFieldEnums.PASSWORD.equals(fieldEnums)) {
            String newPwd = updateVO.getValue();
            Preconditions.checkArgument(
                null == newPwd
                    || (!StringUtils.containsWhitespace(newPwd) && (newPwd.length() == 6 || newPwd.length() == 0)),
                "请输入六位不含空格密码");
            updateCondtion.put(fieldEnums.getName(), updateVO.getValue());
        } else {
            updateCondtion.put(fieldEnums.getName(), updateVO.getValue());
        }
        List<String> updateProps = Lists.newArrayList();
        updateProps.add(fieldEnums.getName());
        if (TxVZhiBoLessonFieldEnums.AVATAR.equals(fieldEnums)) {
            if (updateVO != null && updateVO.getExt() != null && NumberUtils.isDigits(updateVO.getExt())) {
                updateCondtion.put("storageId", updateVO.getExt());
                updateProps.add("storageId");
            }
        }
        updateCondtion.put("id", updateVO.getId());
        int mod = txVZhiBoLessonDao.update(updateCondtion, updateProps.toArray(new String[updateProps.size()]));
        if (mod > 0) {
            // 刷新缓存
            // refreshCache(updateVO,fieldEnums);
        }
        return mod;
    }

    private void refreshCache(FieldUpdateVO updateVO, TxVZhiBoLessonFieldEnums fieldEnums) {
        try {
            // LessonDetailVO detailVO = lessonDetailCache.get(updateVO.getId());
            // if(detailVO != null){
            // if(updateVO.getValue() == null){
            // return;
            // }
            // String value = updateVO.getValue().toString();
            // switch(fieldEnums){
            // case AVATAR:
            // detailVO.setAvatar(value);
            // break;
            // case NAME:
            // detailVO.setName(value);
            // break;
            // case DESCRIPTION:
            // detailVO.setDescription(value);
            // break;
            // case STARTTIME:
            // detailVO.setStartTime(Long.parseLong(value));
            // break;
            // }
            // lessonDetailCache.put(updateVO.getId(), detailVO);
            // }
        } catch (Exception e) {
            log.error("Error in refreshCache", e);
        }
    }

    @Override
    public Map<String, List<TxVZhiBoLessonListVO>> listAll(Integer roomId, Integer ownerId, int lessonListType,
        Integer hidden, String key, ListRequestDto pageDto) {
        List<TxVZhiBoLessonListVO> boLessonListVOs = null;
        if (TxVZhiBoLessonListTypeEnums.ALL.getCode() == lessonListType) {
            boLessonListVOs =
                list(roomId, null, null, TxVZhiBoLessonStatusEnums.ALL.getCode(), hidden, key, pageDto, true);
        } else if (TxVZhiBoLessonListTypeEnums.OWN.getCode() == lessonListType) {
            boLessonListVOs =
                list(roomId, ownerId, null, TxVZhiBoLessonStatusEnums.ALL.getCode(), hidden, key, pageDto, false);
        } else {
            boLessonListVOs =
                listOthers(roomId, ownerId, null, TxVZhiBoLessonStatusEnums.ALL.getCode(), hidden, key, pageDto);
            initWithoutAuthLiveUrl(boLessonListVOs);
        }

        if (CollectionUtils.isEmpty(boLessonListVOs)) {
            return Maps.newHashMap();
        }

        List<TxVZhiBoLessonListVO> liveList = Lists.newArrayListWithExpectedSize(boLessonListVOs.size());
        List<TxVZhiBoLessonListVO> doneList = Lists.newArrayListWithExpectedSize(boLessonListVOs.size());

        for (TxVZhiBoLessonListVO listVO : boLessonListVOs) {
            // 对于直播中的课程, 如果消息队列都没了, 就修正直播课状态.
            if (TxVZhiBoLessonStatusEnums.LIVE.getCode() == listVO.getStatus()) {
                String msgQuqueKey = AreaTypeEnums.ZHIBO.getPrefix() + listVO.getId();
                if (DateUtil.getMinuteDiff(new Date(), listVO.getCreateTime()) > 5
                    && !redisTemplate.hasKey(msgQuqueKey)) {
                    log.info("[getDetailById]lessonId:{} 消息队列已经不存在了，key:{}", listVO.getId(), msgQuqueKey);
                    // 1.修正直播课状态
                    listVO.setStatus(TxVZhiBoLessonStatusEnums.DONE.getCode());
                    // 2.更新数据库
                    txVZhiBoLessonDao.updateColumnValueById(listVO.getId(), "status",
                        TxVZhiBoLessonStatusEnums.DONE.getCode());
                }
            }

            if (TxVZhiBoLessonStatusEnums.LIVE.getCode() == listVO.getStatus()) {
                liveList.add(listVO);
            } else {
                doneList.add(listVO);
            }
        }

        Map<String, List<TxVZhiBoLessonListVO>> statusGroupedMap = Maps.newHashMap();
        statusGroupedMap.put("live", liveList);
        statusGroupedMap.put("done", doneList);
        return statusGroupedMap;
    }

    @Override
    public List<TxVZhiBoLessonListVO> listLessons(int lessonListType, Integer ownerId, Integer ownerType,
        Integer roomId, Integer hidden, Integer status, String key, ListRequestDto pageDto) {
        List<TxVZhiBoLessonListVO> boLessonListVOs = null;
        if (TxVZhiBoLessonListTypeEnums.ALL.getCode() == lessonListType) {
            boLessonListVOs = list(roomId, ownerId, ownerType, status, hidden, key, pageDto, true);
        } else if (TxVZhiBoLessonListTypeEnums.OWN.getCode() == lessonListType) {
            boLessonListVOs = list(roomId, ownerId, ownerType, status, hidden, key, pageDto, false);
        } else if (TxVZhiBoLessonListTypeEnums.OTHERS.getCode() == lessonListType) {
            boLessonListVOs = listOthers(roomId, ownerId, ownerType, status, hidden, key, pageDto);
        }
        initWithoutAuthLiveUrl(boLessonListVOs);
        return boLessonListVOs;
    }

    @Override
    public String getDetailUrl(Long id) {
        return LESSON_DETAIL_URL + id;
    }

    @Override
    public int checkLessonStatus(long lessonId) {
        TxVZhiBoLesson detailVO = txVZhiBoLessonDao.getById(lessonId);
        if (detailVO == null) {
            return TxVZhiBoLessonStatusEnums.DELETED.getCode();
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(System.currentTimeMillis());
        calendar.add(Calendar.DAY_OF_MONTH, -7);
        calendar.add(Calendar.HOUR_OF_DAY, -2);

        Date startTime = detailVO.getStartTime();
        if (startTime != null) {
            Long startT = startTime.getTime();
            if (startT > calendar.getTimeInMillis()) {
                return TxVZhiBoLessonStatusEnums.LIVE.getCode();
            } else {
                return TxVZhiBoLessonStatusEnums.DONE.getCode();
            }
        }
        return TxVZhiBoLessonStatusEnums.DELETED.getCode();
    }

    @Override
    public boolean delete(Integer orgId, Integer txCascadeId, int lessonId) {
        TxVZhiBoLesson detailVO = txVZhiBoLessonDao.getByOrgIdAndId(orgId, lessonId);
        if (detailVO == null) {
            return false;
        }

        int mod =
            txVZhiBoLessonDao.updateColumnValueById(lessonId, "status", TxVZhiBoLessonStatusEnums.DELETED.getCode());
        return mod > 0 ? true : false;
    }

    @Override
    public Map<String, Integer> countByStatus(Integer roomId) {
        Map<String, Object> countCondition = Maps.newHashMap();
        countCondition.put("roomId", roomId);
        countCondition.put("hidden", Flag.FALSE.getInt());
        Map<Integer, Integer> map = txVZhiBoLessonDao.groupCount(countCondition, "id", "status", true, Integer.class);
        Map<String, Integer> resultMap = Maps.newHashMap();
        if (map == null || map.size() == 0) {
            return resultMap;
        }
        for (Map.Entry<Integer, Integer> entry : map.entrySet()) {
            if (entry.getKey().intValue() == TxVZhiBoLessonStatusEnums.LIVE.getCode()) {
                resultMap.put("live", entry.getValue());
            } else if (entry.getKey().intValue() == TxVZhiBoLessonStatusEnums.DONE.getCode()) {
                resultMap.put("done", entry.getValue());
            }
        }
        return resultMap;
    }

    /**
     * 直播课缓存
     */
    private LoadingCache<Integer, TxVZhiBoLesson> lessonCache = CacheBuilder.newBuilder().maximumSize(10000)
        .expireAfterWrite(2, TimeUnit.HOURS).build(new CacheLoader<Integer, TxVZhiBoLesson>() {
            @Override
            public TxVZhiBoLesson load(Integer lessonId) throws Exception {
                Preconditions.checkNotNull(lessonId);
                return getById(lessonId);
            }
        });

    @Override
    public boolean checkLessonIsLiving(Integer lessonId) {
        // 检查直播课状态，已结束或已删除的不能再发消息了
        try {
            TxVZhiBoLesson boLesson = lessonCache.get(lessonId);
            log.info("[send]boLesson:{}", boLesson);
            if (boLesson == null || TxVZhiBoLessonStatusEnums.DONE.getCode() == boLesson.getStatus()
                || TxVZhiBoLessonStatusEnums.DELETED.getCode() == boLesson.getStatus()) {
                log.info("[send]lesson status is illegel!");
                return false;
            }
        } catch (ExecutionException e) {
            log.error("Error In send msg", e);
            return false;
        }
        return true;
    }

    @Override
    public UserTypeEnums checkUserType(Integer lessonId, Integer orgId, Integer txCascadeId, String openId) {
        log.info("[checkUserType]lessonId:{},orgId:{},txCascadeId:{},openId:{}", lessonId, orgId, txCascadeId, openId);
        if (lessonId == null) {
            return UserTypeEnums.HACKER;
        }
        // 非法访问
        if (orgId == null && txCascadeId == null && StringUtils.isBlank(openId)) {
            return UserTypeEnums.HACKER;
        }
        // 学生
        if (orgId == null && txCascadeId == null && StringUtils.isNoneBlank(openId)) {
            return UserTypeEnums.STUDENT;
        }
        TxVZhiBoLesson lesson = null;
        try {
            lesson = getById(new Long(lessonId));
        } catch (Exception e) {
            log.info("Cann`t find lesson", e);
        }
        if (lesson == null) {
            return UserTypeEnums.HACKER;
        }
        // 判断是否为同一个机构
        if (lesson.getOrgId().equals(orgId)) {
            if (lesson.getOwnerType() == OwnerTypeEnums.MAIN.getCode()) {
                if (txCascadeId == null && orgId == lesson.getOwnerId().intValue()) {
                    return UserTypeEnums.TEACHER;
                } else {
                    return UserTypeEnums.OTHER_TEACHER;
                }
            } else {
                if (lesson.getOwnerId().equals(txCascadeId)) {
                    return UserTypeEnums.TEACHER;
                } else {
                    return UserTypeEnums.OTHER_TEACHER;
                }
            }
        } else if (null != orgId) {
            return UserTypeEnums.OTHER_TEACHER;
        } else {
            return UserTypeEnums.HACKER;
        }
    }

    @Override
    public List<TxVZhiBoLesson> getByRoomId(Integer roomId) {
        if (roomId == null) {
            return Lists.newArrayList();
        }
        return txVZhiBoLessonDao.getByRoomId(roomId);
    }

    @Override
    public LessonDetailVO getOnlyDetailById(Integer lessonId)
        throws BussinessException, NumberFormatException, Exception {
        TxVZhiBoLesson zhiBoLesson = getById(lessonId);
        if (zhiBoLesson == null) {
            return null;
        }
        return convert2DetailVO(zhiBoLesson);
    }

}
