/**
 * Baijiahulian.com Inc.
 * Copyright (c) 2014-2016 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.vzhibo.service.impl;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.vzhibo.dao.TxVZhiBoRoomDao;
import com.baijia.tianxiao.dal.vzhibo.po.TxVZhiBoRoom;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.organization.api.OrgAccountService;
import com.baijia.tianxiao.sal.organization.org.dto.OrgInfoSimpleDto;
import com.baijia.tianxiao.sal.organization.org.service.OrgInfoService;
import com.baijia.tianxiao.sal.vzhibo.constant.OrgFieldTypeEnums;
import com.baijia.tianxiao.sal.vzhibo.constant.TxVZhiBoRoomFieldEnums;
import com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoRoomService;
import com.baijia.tianxiao.sal.vzhibo.vo.FieldUpdateVO;
import com.baijia.tianxiao.sal.vzhibo.vo.ShareDataVO;
import com.baijia.tianxiao.sal.vzhibo.vo.TxVZhiBoRoomBaseVO;
import com.baijia.tianxiao.sal.vzhibo.vo.TxVZhiBoRoomDetailVO;
import com.baijia.tianxiao.sal.wechat.api.AuthorizerInfoService;
import com.baijia.tianxiao.sal.wechat.api.BindingService;
import com.baijia.tianxiao.sal.wechat.dto.binding.BindingInfoDto;
import com.baijia.tianxiao.sal.wechat.helper.WechatProperties;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.ShortUrlUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * @title TxVZhiBoRoomServiceImpl
 * @desc 直播间service实现
 * @author fubaokui
 * @date 2016年11月28日
 * @version 1.0
 */
@Slf4j
@Service
public class TxVZhiBoRoomServiceImpl implements TxVZhiBoRoomService {

    @Autowired
    TxVZhiBoRoomDao txVZhiBoRoomDao;
    
    @Autowired
    OrgAccountService orgAccountService;
    
    @Autowired
    OrgInfoService orgInfoService;

    @Value("${room_detail_url}")
    String ROOM_DETAIL_URL;
    
    @Value("${gsx_qrcode_url}")
    String GSX_QRCODE_URL;
    
    @Autowired
    BindingService bindingInfoService;
    
    @Autowired
    AuthorizerInfoService authorizerInfoService;

    @Value("${welcome_url}")
    String WELCOME_URL;
    
    @Value("${room_default_description}")
    String ROOM_DEFAULT_DESCRIPTION;
    
    /* (non-Javadoc)
     * @see com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoRoomService#detail(long)
     */
    @Override
    public TxVZhiBoRoomDetailVO detail(int fieldValue,OrgFieldTypeEnums fieldType,String authToken) {
        log.info("[detail]fieldValue:{},fieldType:{}",fieldValue,fieldType.getCode());
        
        TxVZhiBoRoomBaseVO  roomBaseVO = getBaseInfo(fieldValue, fieldType);
        if(roomBaseVO == null){
            return null;
        }
        TxVZhiBoRoomDetailVO detailVO = new TxVZhiBoRoomDetailVO(roomBaseVO);
        
        //获取直播间url
        String roomUrl = ROOM_DETAIL_URL + roomBaseVO.getOrgNumber();
        String shortUrl = ShortUrlUtil.getHttpsShortUrl(roomUrl);
        detailVO.setRoomUrl(shortUrl);
        
        //获取直播间Qrcode
        String qrcodeUrl = GSX_QRCODE_URL + roomUrl;
        detailVO.setQrcodeUrl(qrcodeUrl);
        
        //获取已关联的公众号名称
        Map<String, Object> orgWechatMap = null;
        try {
            orgWechatMap = getWechatBindingInfo(roomBaseVO.getOrgId(), authToken);
            log.info("[room.detail]getWechatBindingInfo:{}",orgWechatMap);
            if(orgWechatMap != null){
                Object wechatInfo = orgWechatMap.get("info");
                detailVO.setBingPageUrl((String)orgWechatMap.get("bingPageUrl"));
                if(wechatInfo != null){
                    detailVO.setWechat((BindingInfoDto)wechatInfo);
                }
            }
        } catch (UnsupportedEncodingException e) {
            log.error("[detail]Error In getWechatBindingInfo,orgId:{},authToken:{}",roomBaseVO.getOrgId(), authToken);
        }
        return detailVO;
    }

    /* (non-Javadoc)
     * @see com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoRoomService#update(java.lang.Integer, int, java.lang.String)
     */
    @Override
    public int update(Integer orgId, FieldUpdateVO updateVO){
        TxVZhiBoRoomFieldEnums fieldEnums = TxVZhiBoRoomFieldEnums.parse(updateVO.getField());
        if(fieldEnums == null){
            return -1;
        }
        Map<String,Object> updateCondtion = Maps.newHashMap();
        updateCondtion.put(fieldEnums.getName(), updateVO.getValue());
        List<String> updateProps = Lists.newArrayList();
        updateProps.add(fieldEnums.getName());
        if(TxVZhiBoRoomFieldEnums.AVATAR.equals(fieldEnums)){
            if(updateVO.getExt() != null && NumberUtils.isDigits(updateVO.getExt().toString())){
                updateCondtion.put("storageId", updateVO.getExt());
                updateProps.add("storageId");
            }
        }
        updateCondtion.put("orgId", orgId);
        return txVZhiBoRoomDao.update(updateCondtion, updateProps.toArray(new String[updateProps.size()]));
    }
    
    /**
     * 等雨剑提供接口，暂时这么写
     * @param orgId
     * @param authToken
     * @return
     * @throws UnsupportedEncodingException
     */
    private Map<String,Object> getWechatBindingInfo(Integer orgId,String authToken) throws UnsupportedEncodingException{
        Map<String, Object> map = new HashMap<String, Object>();
        log.info("[getWechatBindingInfo]orgId:{}",orgId);
        BindingInfoDto dto = bindingInfoService.getBindingInfo(orgId);
        log.info("[getWechatBindingInfo]BindingInfoDto:{}",dto);
        map.put("info", dto);
        if(dto == null && StringUtils.isNotBlank(authToken)){
            String urlPrefix = WechatProperties.getBindPageUrlPrefix();
            
            StringBuilder redirectBuilder = new StringBuilder();
            redirectBuilder.append(urlPrefix).append("wechat/bind.do?").append("&auth_token=")
            .append(authToken).append("&from=")
            .append(URLEncoder.encode(urlPrefix, "utf8"));
            
            String bingPageUrl = urlPrefix + "view/weixin/forward.html?redirect="
                + URLEncoder.encode(redirectBuilder.toString(), "utf8");
            map.put("bingPageUrl", bingPageUrl);
        }
        return map;
    }

    @Override
    public TxVZhiBoRoomBaseVO getBaseInfo(int fieldValue,OrgFieldTypeEnums fieldType){
        OrgAccount orgAccount = null;
        if(OrgFieldTypeEnums.NUMBER.equals(fieldType)){
            orgAccount =  orgAccountService.getOrgAccountByNumber(fieldValue);
        }else{
            orgAccount =  orgAccountService.getOrgAccountById(fieldValue);
        }
        if(orgAccount == null){
            return null;
        }
        
        TxVZhiBoRoomBaseVO vZhiBoRoomBaseVO = new TxVZhiBoRoomBaseVO();
        
        Map<String, Object> condition = Maps.newHashMap();
        if(OrgFieldTypeEnums.NUMBER.equals(fieldType)){
            condition.put(fieldType.getName(), orgAccount.getNumber());
        }else{
            condition.put(fieldType.getName(), orgAccount.getId());
        }
        List<TxVZhiBoRoom> txVZhiBoRooms =  txVZhiBoRoomDao.queryByCondition(condition, new PageDto());
        //获取机构信息
        OrgInfoSimpleDto simpleDto  = null;
        try {
            simpleDto = orgInfoService.getOrgInfo(new Long(orgAccount.getId()));
        } catch (Exception e) {
            log.error("Error In orgInfoService.getMOrgInfo,orgId:{}",e,orgAccount.getId());
        }
        if(simpleDto == null){
            return null;
        }
        vZhiBoRoomBaseVO.setOrgId(simpleDto.getOrgId());
        //还没有创建直播间的话，给创建一个空的直播间
        if(CollectionUtils.isEmpty(txVZhiBoRooms)){
            //持久化TxVZhiBoRoom
            TxVZhiBoRoom vZhiBoRoom = new TxVZhiBoRoom();
            vZhiBoRoom.setOrgId(orgAccount.getId());
            
            vZhiBoRoom.setOrgNumber(simpleDto.getOrgNumber().intValue());
            
            vZhiBoRoom.setDescription(simpleDto.getIntroduce());
            vZhiBoRoom.setName(simpleDto.getShortName()+"的直播间");
            
            txVZhiBoRoomDao.save(vZhiBoRoom);
            //默认取机构的数据
            vZhiBoRoomBaseVO.setId(vZhiBoRoom.getOrgId());
            vZhiBoRoomBaseVO.setDescription(simpleDto.getIntroduce());
            vZhiBoRoomBaseVO.setName(simpleDto.getShortName());
            vZhiBoRoomBaseVO.setOrgNumber(simpleDto.getOrgNumber().intValue());
            vZhiBoRoomBaseVO.setAvatar(simpleDto.getLogo());
        }else{
            TxVZhiBoRoom vZhiBoRoom  = txVZhiBoRooms.get(0);
            //直播课简介
            if(StringUtils.isNotBlank(vZhiBoRoom.getDescription())){
                vZhiBoRoomBaseVO.setDescription(vZhiBoRoom.getDescription());
            }else{
                vZhiBoRoomBaseVO.setDescription(simpleDto.getIntroduce());
            }
            //直播课名称
            if(StringUtils.isNotBlank(vZhiBoRoom.getName())){
                vZhiBoRoomBaseVO.setName(vZhiBoRoom.getName());
            }else{
                vZhiBoRoomBaseVO.setName(simpleDto.getShortName()+"的直播间");
            }
            //机构logo
            if(StringUtils.isNotBlank(vZhiBoRoom.getAvatar())){
                vZhiBoRoomBaseVO.setAvatar(vZhiBoRoom.getAvatar());
            }else{
                vZhiBoRoomBaseVO.setAvatar(simpleDto.getLogo());
            }
            vZhiBoRoomBaseVO.setOrgNumber(vZhiBoRoom.getOrgNumber().intValue());
            vZhiBoRoomBaseVO.setId(vZhiBoRoom.getId().intValue());
        }
        return vZhiBoRoomBaseVO;
    }

    @Override
    public Map<String,Object> create(Integer orgId) {
        //获取机构信息
        OrgAccount orgAccount = orgAccountService.getOrgAccountById(orgId);
        if(orgAccount == null){
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "token is invalid!");
        }
        
        Map<String,Object> returnMap = Maps.newHashMap();
        returnMap.put("welcomeUrl", WELCOME_URL);
        returnMap.put("roomUrl",ROOM_DETAIL_URL + orgAccount.getNumber().toString());
        
        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        List<TxVZhiBoRoom> txVZhiBoRooms =  txVZhiBoRoomDao.queryByCondition(condition, new PageDto());
        //还没有创建直播间的话，给创建一个空的直播间
        if(CollectionUtils.isEmpty(txVZhiBoRooms)){
            //持久化TxVZhiBoRoom
            TxVZhiBoRoom vZhiBoRoom = new TxVZhiBoRoom();
            vZhiBoRoom.setOrgId(orgId);
            vZhiBoRoom.setOrgNumber(orgAccount.getNumber());

            //获取机构信息
            OrgInfoSimpleDto simpleDto  = null;
            try {
                simpleDto = orgInfoService.getOrgInfo(new Long(orgAccount.getId()));
            } catch (Exception e) {
                log.error("Error In orgInfoService.getMOrgInfo,orgId:{}",e,orgAccount.getId());
            }
            if(simpleDto != null){
                log.info("[create]simpleDto:{}",simpleDto);
                if(StringUtils.isBlank(simpleDto.getIntroduce())){
                    vZhiBoRoom.setDescription(ROOM_DEFAULT_DESCRIPTION);
                }else{
                    vZhiBoRoom.setDescription(simpleDto.getIntroduce());
                }
                vZhiBoRoom.setName(simpleDto.getShortName()+"的直播间");
            }
            //直播间简介默认为机构简介
            vZhiBoRoom.setDescription(simpleDto.getIntroduce());
            //直播间默认头像是机构头像
            vZhiBoRoom.setAvatar(simpleDto.getLogo());
            txVZhiBoRoomDao.save(vZhiBoRoom);
            returnMap.put("roomId",vZhiBoRoom.getId());
            returnMap.put("name",vZhiBoRoom.getName());
        }else{
            // 2016-12-6 当前每个机构只有一个直播间，如有多个，此处当修改
            returnMap.put("roomId",txVZhiBoRooms.get(0).getId());
            returnMap.put("name", txVZhiBoRooms.get(0).getName());
        }
        return returnMap;
    }

    @Override
    public ShareDataVO getShareData(TxVZhiBoRoomBaseVO baseVO) {
        ShareDataVO dataVO = new ShareDataVO();
        dataVO.setTitle(baseVO.getName());
        if(StringUtils.isBlank(baseVO.getDescription())){
            //获取机构信息
              OrgInfoSimpleDto simpleDto  = null;
              try {
                  simpleDto = orgInfoService.getOrgInfo(new Long(baseVO.getOrgId()));
              } catch (Exception e) {
                  log.error("Error In orgInfoService.getMOrgInfo,orgId:{} e:{}",baseVO.getOrgId(), e);
              }
              if(simpleDto == null || StringUtils.isBlank(simpleDto.getIntroduce())){
                  dataVO.setContent(ROOM_DEFAULT_DESCRIPTION);
              }else{
                  dataVO.setContent(simpleDto.getIntroduce());
              }
          }else{
              dataVO.setContent(baseVO.getDescription());
          }
        dataVO.setImageUrl(baseVO.getAvatar());
        dataVO.setShareUrl(ROOM_DETAIL_URL + baseVO.getOrgNumber());
        return dataVO;
    }

    @Override
    public ShareDataVO getShareData(int roomId) {
        TxVZhiBoRoom boRoom = txVZhiBoRoomDao.getById(roomId);
        if(boRoom == null){
            return null;
        }
        ShareDataVO dataVO = new ShareDataVO();
        dataVO.setTitle(boRoom.getName());
        if(StringUtils.isBlank(boRoom.getDescription())){
          //获取机构信息
            OrgInfoSimpleDto simpleDto  = null;
            try {
                simpleDto = orgInfoService.getOrgInfo(new Long(boRoom.getOrgId()));
            } catch (Exception e) {
                log.error("Error In orgInfoService.getMOrgInfo,orgId:{}",e,boRoom.getOrgId());
            }
            if(StringUtils.isBlank(simpleDto.getIntroduce())){
                dataVO.setContent(ROOM_DEFAULT_DESCRIPTION);
            }else{
                dataVO.setContent(simpleDto.getIntroduce());
            }
        }else{
            dataVO.setContent(boRoom.getDescription());
        }
        dataVO.setImageUrl(boRoom.getAvatar());
        dataVO.setShareUrl(ROOM_DETAIL_URL + boRoom.getOrgNumber());
        return dataVO;
    }

    
    @Override
    public TxVZhiBoRoom getById(Integer roomId) {
        if(roomId == null){
            return null;
        }
        return txVZhiBoRoomDao.getById(roomId);
    }
    
    
}
