/**
 * Baijiahulian.com Inc.
 * Copyright (c) 2014-2016 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.vzhibo.service.impl;

import java.io.IOException;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import com.baijia.commons.lang.utils.JacksonUtil;
import com.baijia.tianxiao.constant.OrderEnums;
import com.baijia.tianxiao.constant.TxVZhiBoLessonStatusEnums;
import com.baijia.tianxiao.dal.vzhibo.dao.TxVZhiBoMessageDao;
import com.baijia.tianxiao.dal.vzhibo.po.TxVZhiBoLesson;
import com.baijia.tianxiao.dal.vzhibo.po.TxVZhiBoMessage;
import com.baijia.tianxiao.dal.wechat.po.Fans;
import com.baijia.tianxiao.sal.vzhibo.constant.AreaTypeEnums;
import com.baijia.tianxiao.sal.vzhibo.constant.MessageTypeEnums;
import com.baijia.tianxiao.sal.vzhibo.constant.OrderTypeEnums;
import com.baijia.tianxiao.sal.vzhibo.constant.TxVZhiBoEventType;
import com.baijia.tianxiao.sal.vzhibo.constant.TxVZhiBoUserType;
import com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoEventLogService;
import com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoLessonService;
import com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoMessageService;
import com.baijia.tianxiao.sal.vzhibo.util.WebSocketUtil;
import com.baijia.tianxiao.sal.vzhibo.vo.LessonDetailVO;
import com.baijia.tianxiao.sal.vzhibo.vo.MessageInfo;
import com.baijia.tianxiao.sal.vzhibo.vo.MessageListVO;
import com.baijia.tianxiao.sal.vzhibo.vo.MessageVO;
import com.baijia.tianxiao.sal.vzhibo.vo.UploadInfo;
import com.baijia.tianxiao.sal.wechat.api.FansService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.mashape.unirest.http.exceptions.UnirestException;

import lombok.extern.slf4j.Slf4j;

/**
 * @title MessageServiceImpl
 * @desc 直播消息message
 * @author he11o
 * @date 2016年11月23日
 * @version 1.0
 */
@Slf4j
@Service("txVZhiBoMessageService")
public class TxVZhiBoMessageServiceImpl implements TxVZhiBoMessageService {
    
    @Autowired
    TxVZhiBoMessageDao txVZhiBoMessageDao; 
    
    @Autowired
    RedisTemplate<String, String> redisTemplate;
    
    @Autowired
    FansService fansService;
    
    @Autowired
    TxVZhiBoLessonService txVZhiBoLessonService;
    
    @Autowired
    TxVZhiBoEventLogService txVZhiBoEventLogService;
    
    public static final String MSG_QUEUE_LIVE_LESSONIDS = "vzhibo_live_lessonIds";
    
    public static final int MSG_FREQUENT_LIMIT = 5000;   //讨论区消息发送频率现在5秒
    
    public static final int MSG_FREQUENT_COUNT_LIMIT = 3;   //讨论区消息发送频率内数量控制
    
    @Value("${student_default_avatar}")
    String STUDENT_DEFAULT_AVATAR;
    
    @Value("${cc_server_url}")
    String ccServerUrl;
    
    @Value("${msg_queue_pos}")
    String MSG_QUEUE_POS;
    
    @Value("${process_count}")
    String PROCESS_COUNT;
    
    @Value("${student_latest_visit}")
    String STUDENT_LATEST_VISIT;
    
    @Value("${student_latest_visit_count}")
    String STUDENT_LATEST_VISIT_COUNT;
    
    private static final String VISIT_COUNT_INIT = "1";
    
    @Override
    public String send(MessageInfo messageInfo, Set<Object> deviceIds){
        log.info("[send]msg:{}",messageInfo.toString());
        Integer lessonId = messageInfo.getLessonId();
        if(lessonId == null){
            return null;
        }
        if(!txVZhiBoLessonService.checkLessonMessageStatus(lessonId)){
            log.info("[send]checkLessonMessageStatus result false!!");
            return null;
        }
        MessageTypeEnums typeEnums = MessageTypeEnums.parse(messageInfo.getMsgType());
        return porcessMsg(messageInfo,typeEnums,deviceIds);
    }
        
    private String porcessMsg(MessageInfo messageInfo, MessageTypeEnums typeEnums, Set<Object> deviceIds){
        //有消息的话,就将当前直播课id添加到redis的“直播中”列表上
        final Integer lessonId = messageInfo.getLessonId();
        redisTemplate.opsForSet().add(MSG_QUEUE_LIVE_LESSONIDS, lessonId.toString());
        
        switch(typeEnums){
            case PIC:
                return porcessMediaMsg(messageInfo,deviceIds);
            case VOICE:
                return porcessMediaMsg(messageInfo,deviceIds);
            case TEXT:
                return processTextMsg(messageInfo,deviceIds);
            default:
                break;
        }
        return null;
    }

    private MessageVO initCommonMsg(MessageInfo messageInfo){
        //1.构成消息
        MessageVO msg = new MessageVO();
        msg.setMsgType(messageInfo.getMsgType());
        msg.setAreaType(messageInfo.getAreaType());
        msg.setContent(messageInfo.getContent());
        msg.setLessonId(messageInfo.getLessonId());
        msg.setOpenId(messageInfo.getOpenId());
        msg.setCreateTime(System.currentTimeMillis());
        msg.setFile(messageInfo.getFile());
        
        //生成uniqueId
        UUID uniqueId = UUID.randomUUID();
        msg.setUniqueId(uniqueId.toString());
        //根据openid初始化用户信息(昵称、头像)
        try {
            initUserInfo(msg,messageInfo);
            return msg;
        } catch (Exception e) {
            log.error("Error In initMsg",e);;
            return null;
        }
    }

    private boolean processCommonMsg(MessageVO messageVO, Set<Object> deviceIds) {
        AreaTypeEnums areaTypeEnums = AreaTypeEnums.parse(messageVO.getAreaType());
        String msgQuqueKey = areaTypeEnums.getPrefix() + messageVO.getLessonId();
        //1.写入消息到redis队列
        //放置到队尾
        try {
            //判断消息队列是否存在
            Boolean hasKey = redisTemplate.hasKey(msgQuqueKey);
            //第一次初始化消息队列后，设置key的超时时间
            if(!hasKey){
                Long expireTime = txVZhiBoLessonService.getExpireTime(messageVO.getLessonId());
                log.info("[processTextMsg]expireTime:{}",expireTime);
                if(expireTime < 0){
                    return false;
                }
                redisTemplate.opsForList().rightPush(msgQuqueKey, JacksonUtil.obj2Str(messageVO));
                redisTemplate.expire(msgQuqueKey, expireTime/1000, TimeUnit.SECONDS);
            }else{
                redisTemplate.opsForList().rightPush(msgQuqueKey, JacksonUtil.obj2Str(messageVO));
            }
        } catch (Exception e) {
            log.error("Error In processTextMsg",e);
            return false;
        }
        //2.给WebSocket发通知
        MessageVO newMessageNotice = new MessageVO();
        newMessageNotice.setMsgType(messageVO.getMsgType());
        newMessageNotice.setAreaType(messageVO.getAreaType());
        newMessageNotice.setContent(messageVO.getContent());
        newMessageNotice.setLessonId(messageVO.getLessonId());
        log.info("[processTextMsg] deviceIds:{}",deviceIds);
		if (CollectionUtils.isNotEmpty(deviceIds)) {
			WebSocketUtil.sendBatchWsMessage(newMessageNotice, deviceIds);
			// for(Object deviceId : deviceIds){
			// WebSocketUtil.sendWsMessage(newMessageNotice,
			// deviceId.toString());
			// }
		}
        return true;
    }
    
    private String processTextMsg(MessageInfo messageInfo, Set<Object> deviceIds) {
        //初始化持久化格式的消息
        MessageVO msg = initCommonMsg(messageInfo);
        if(msg == null){
            log.info("Error In initMsg,messageInfo:{}",messageInfo);
            return null;
        }
        boolean flag = processCommonMsg(msg, deviceIds);
        return flag ? msg.getUniqueId() : null;
    }

    /**
     * 微信粉丝本地缓存
     */
//    private LoadingCache<String, Fans> fansCache = CacheBuilder.newBuilder().maximumSize(10000)
//        .expireAfterWrite(2, TimeUnit.HOURS).build(new CacheLoader<String, Fans>() {
//            @Override
//            public Fans load(String openid) throws Exception {
//                Preconditions.checkNotNull(openid);
//                return fansService.getFans(openid);
//            }
//    });
    
    private void initUserInfo(MessageVO msg, MessageInfo messageInfo) throws ExecutionException {
        //老师
        if(messageInfo.getOrgId() != null){
            LessonDetailVO detailVO = null;
            try {
                detailVO = txVZhiBoLessonService.getOnlyDetailById(messageInfo.getLessonId());
                if(detailVO != null){
                    msg.setAvatar(detailVO.getTeacherAvatar());
                    msg.setNickname(detailVO.getTeacherName());
                }
            } catch (Exception e) {
                log.info("Error in getDetailById",e);
            }
            
        }else{//学生
            if(StringUtils.isNoneBlank(messageInfo.getOpenId())){
//                Fans fans = fansCache.get(messageInfo.getOpenId());
                Fans fans =   fansService.getFans(messageInfo.getOpenId());
              
                if(fans != null){
                    if(StringUtils.isBlank(fans.getHeadImgUrl())){
                        msg.setAvatar(STUDENT_DEFAULT_AVATAR);
                    }else{
                        msg.setAvatar(fans.getHeadImgUrl());
                    }
                    msg.setNickname(fans.getNick());
                }
            }
        }
    }

    /**
     * 处理图片、音频消息
     * @param messageInfo
     * @return
     * @throws UnirestException 
     */
    private String porcessMediaMsg(MessageInfo messageInfo, Set<Object> deviceIds) {
        //1.初始化持久化格式的消息
        MessageVO msg = initCommonMsg(messageInfo);
        if(msg == null){
            log.info("Error In initMsg,messageInfo:{}",messageInfo);
            return null;
        }
        //2.发送消息
        boolean flag = processCommonMsg(msg, deviceIds);
        return flag ? msg.getUniqueId() : null; 
    }

    @Override
    public MessageListVO get(long lessonId,long index,long count,int areaType,int order) {
        AreaTypeEnums areaTypeEnums = AreaTypeEnums.parse(areaType);
        if(areaTypeEnums == null){
            return new MessageListVO();
        }
        String msgQuqueKey = areaTypeEnums.getPrefix() + lessonId;
        
        Long length = redisTemplate.opsForList().size(msgQuqueKey); 
        List<String> messages = null;
        if(count <=0){
            count = length;
        }
        if(index < 0){
            index = 0;
        }
        if(order == 0){
            long end = index + count -1;
            if(end < 0){
                end = 0;
            }
            log.info("[get]rest.range msgQuqueKey:{},start:{},end:{}",msgQuqueKey,index,end);
            messages = redisTemplate.opsForList().range(msgQuqueKey, index, end);
        }else{
            long start = index-count - 1;
            if(start < 0){
                start = 0;
            }
            long end = index - 2;
            if(end < 0){
                end = 0;
            }
            log.info("[get]rest.range msgQuqueKey:{},start:{},end:{}",msgQuqueKey,index,end);
            messages = redisTemplate.opsForList().range(msgQuqueKey, start,end);
        }
        
        MessageListVO listVO = new MessageListVO();
        if(CollectionUtils.isEmpty(messages)){
            return listVO;
        }
        List<MessageVO> infos = convert2MessageVO(messages);
        //初始化id
        initMsgId(infos,index,order,count);
        
        //讨论群的消息要逆序一下
        if(areaType == AreaTypeEnums.DISCUSS.getCode()){
            infos = Lists.reverse(infos);
        }
        
        listVO.setMessages(infos);
        listVO.setTotal(length);
        return listVO;
    }

    private void initMsgId(List<MessageVO> infos, long index, int order, long count) {
        for(int i = 0;i<infos.size();i++){
            MessageVO messageVO = infos.get(i);
            if(order == OrderTypeEnums.POSITIVE.getCode()){
                messageVO.setId(index + i + 1); 
            }else{
                if(infos.size() == count){
                    messageVO.setId(index - count + i);
                }else{
                    messageVO.setId(i + 1L);
                }
            }
        }
    }
    

    @Override
    public MessageListVO latest(long lessonId,int areaType, int count) {
        AreaTypeEnums areaTypeEnums = AreaTypeEnums.parse(areaType);
        if(areaTypeEnums == null){
            return new MessageListVO();
        }
        //check直播课的状态
        TxVZhiBoLesson  boLesson = txVZhiBoLessonService.getById(lessonId);
        if(boLesson == null){
            return new MessageListVO();
        }
        MessageListVO listVO = new MessageListVO();
        if(TxVZhiBoLessonStatusEnums.DONE.getCode() == boLesson.getStatus()){
            listVO = getLatestMsgFromDB(areaTypeEnums,lessonId,count);
        }else if(TxVZhiBoLessonStatusEnums.LIVE.getCode() == boLesson.getStatus()){
            listVO = getLatestMsgFromRedis(areaTypeEnums,lessonId,count);
        }
        return listVO;
    }
    
    
    private MessageListVO getLatestMsgFromDB(AreaTypeEnums areaTypeEnums, long lessonId, int count) {
        MessageListVO listVO = new MessageListVO();
        Map<String,Object> countCondition = Maps.newHashMap();
        countCondition.put("areaType", areaTypeEnums.getCode());
        countCondition.put("lesson_id",lessonId);
        int total = txVZhiBoMessageDao.countByCondition(countCondition, "id", true);
        if(total == 0){
            return listVO;
        }
        listVO.setTotal(total);
        countCondition.put("count",count);
        
        List<TxVZhiBoMessage> boMessages  = txVZhiBoMessageDao.getLatestMsgFromDB(lessonId,count,areaTypeEnums.getCode());
        List<MessageVO> infos = convertMsg2MessageVO(boMessages); 
        listVO.setMessages(infos);
        return listVO;
    }

    private List<MessageVO> convertMsg2MessageVO(List<TxVZhiBoMessage> boMessages) {
        if(CollectionUtils.isEmpty(boMessages)){
            return null;
        }
        List<MessageVO> messageVOs = Lists.newArrayListWithExpectedSize(boMessages.size());
        for(TxVZhiBoMessage message : boMessages){
            MessageVO messageVO = convert2MessageVO(message);
            messageVOs.add(messageVO);
        }
        return messageVOs;
    }

    private MessageVO convert2MessageVO(TxVZhiBoMessage message) {
        MessageVO messageVO = new MessageVO(); 
        messageVO.setAreaType(message.getAreaType());
        messageVO.setContent(message.getContent());
        if(message.getCreateTime() != null){
            messageVO.setCreateTime(message.getCreateTime().getTime());
        }
        messageVO.setId(message.getId());
        messageVO.setLessonId(message.getLessonId());
        messageVO.setMsgType(message.getMsgType());
        messageVO.setOpenId(message.getOpenid());
        messageVO.setStudentId(message.getStudentId());
        messageVO.setUniqueId(message.getUniqueId());
        messageVO.setAvatar(message.getAvatar());
        messageVO.setNickname(message.getNickname());
        
        //图片、语音
        if(MessageTypeEnums.PIC.getCode() == message.getMsgType() || 
            MessageTypeEnums.VOICE.getCode() == message.getMsgType()){
            UploadInfo uploadInfo = new  UploadInfo();
            uploadInfo.setDuration(message.getDuration());
            uploadInfo.setHeight(message.getHeight());
            uploadInfo.setStorageId(message.getStorageId());
            uploadInfo.setUrl(message.getUrl());
            uploadInfo.setWidth(message.getWidth());
            messageVO.setFile(uploadInfo);
        }
        
        return messageVO;
    }

    private MessageListVO getLatestMsgFromRedis(AreaTypeEnums areaTypeEnums,long lessonId,int count){
        String msgQuqueKey = areaTypeEnums.getPrefix()+lessonId;
        Long length = redisTemplate.opsForList().size(msgQuqueKey);
            
        long index = length -  count;
        if(index < 0){
            index = 0;
        }
        List<String> messages = redisTemplate.opsForList().range(msgQuqueKey, index, length);
        MessageListVO listVO = new MessageListVO();
        if(CollectionUtils.isEmpty(messages)){
            return listVO;
        }
        List<MessageVO> infos = convert2MessageVO(messages); 
        
        //初始化消息id(redis消息队列中的index)
        initId(infos,length);
        
        //讨论群的消息要逆序一下
        if(AreaTypeEnums.DISCUSS.equals(areaTypeEnums)){
            infos = Lists.reverse(infos);
        }
        
        listVO.setMessages(infos);
        listVO.setTotal(length);
        return listVO;
    }

    private List<MessageVO> convert2MessageVO(List<String> messages) {
        List<MessageVO> messageVOs = Lists.newArrayListWithExpectedSize(messages.size());
        for(String message : messages){
            try {
                messageVOs.add(JacksonUtil.str2Obj(message, MessageVO.class));
            } catch (IOException e) {
                log.error("Error In convert2MessageVO,message:{}",message);
            }
        }
        return messageVOs;
    }

    private void initId(List<MessageVO> infos, Long length) {
        if(CollectionUtils.isEmpty(infos)){
            return ;
        }
        Long index = length - infos.size() + 1;
        for(int i = 0;i<infos.size();i++){
            MessageVO vo = infos.get(i);
            vo.setId( index  + i);
        }
    }

    @Override
    public String getCCUrl(Integer lessonId,Integer txCascadeId,Integer orgId, String deviceId) {
        TxVZhiBoLesson txVZhiBoLesson = txVZhiBoLessonService.getById(lessonId);
        //已经结束的直播间不返回cc key
        if(txVZhiBoLesson == null || TxVZhiBoLessonStatusEnums.DONE.getCode() == txVZhiBoLesson.getStatus()){
            return "";
        }
        String token = WebSocketUtil.getUserToken(lessonId, deviceId);
        log.info("[getWSKey]lessonId:{},token:{}",lessonId,token);

        //记录老师进直播间的event
        if(orgId != null){
            Integer uId = txCascadeId;
            TxVZhiBoUserType userType = TxVZhiBoUserType.CASCADE_ACCOUNT;
            if(txCascadeId == null){
                uId = orgId;
                userType = TxVZhiBoUserType.JIGOU;
            }
            txVZhiBoEventLogService.saveEvent(lessonId, uId, userType, TxVZhiBoEventType.ENTER);
        }
        return ccServerUrl + token;
    }

    @Override
    public List<TxVZhiBoMessage> pagingByLessonIdAndAreaType(Long lessonId, int areaType, PageDto pageDto) {
        Map<String,Object> condition = Maps.newHashMap();
        condition.put("lessonId", lessonId);
        condition.put("areaType", areaType);
        return txVZhiBoMessageDao.queryByCondition(condition, pageDto, new String[]{"lessonId","areaType"});
    }

    @Override
    public int getCountByLessonIdAndAreaType(Long lessonId, int areaType) {
        Map<String,Object> condition = Maps.newHashMap();
        condition.put("lessonId", lessonId);
        condition.put("areaType", areaType);
        return txVZhiBoMessageDao.countByCondition(condition, "id", false);
    }

    @Override
    public void batchSave(List<MessageVO> messagesList) {
        if(CollectionUtils.isEmpty(messagesList)){
            return;
        }
        List<TxVZhiBoMessage> vZhiBoMessages = 
            Lists.transform(messagesList, new Function<MessageVO,TxVZhiBoMessage>(){
            @Override
            public TxVZhiBoMessage apply(MessageVO input) {
                TxVZhiBoMessage message = convert2Message(input);
                return message;
            }});
        
        txVZhiBoMessageDao.saveAll(vZhiBoMessages);
    }

    protected TxVZhiBoMessage convert2Message(MessageVO input) {
        TxVZhiBoMessage message = new TxVZhiBoMessage();
        message.setAreaType(input.getAreaType());
        message.setContent(input.getContent());
        message.setCreateTime(new Date(input.getCreateTime()));
        UploadInfo file = input.getFile();
        if(file != null){
            message.setHeight(file.getHeight());
            message.setStorageId(file.getStorageId());
            message.setWidth(file.getWidth());
            message.setUrl(file.getUrl());
            message.setDuration(file.getDuration());
        }
        message.setLessonId(input.getLessonId());
        message.setMsgType(input.getMsgType());
        message.setOpenid(input.getOpenId());
        message.setStudentId(input.getStudentId());
        message.setUniqueId(input.getUniqueId());
        message.setAvatar(input.getAvatar());
        message.setNickname(input.getNickname());
        return message;
    }

    @Override
    public List<MessageVO> history(long lessonId, long id, int count, int areaType) {
        List<TxVZhiBoMessage> messages = txVZhiBoMessageDao.history(lessonId,id,count,areaType,OrderEnums.DESC.getCode());
        if(CollectionUtils.isEmpty(messages)){
            return Lists.newArrayList();
        }
        List<MessageVO> messageVOs = convertMsg2MessageVO(messages);
        return messageVOs;
    }
    
    /**
     * 目前是5秒3条
     * 消息频控，true：可以发消息，false:不能发送消息，时间间隔在频控范围内
     * @param openid
     * @return
     */
    @Override
    public boolean visitFrequentCheck(String openid){
        String redisKey = STUDENT_LATEST_VISIT + openid;
        //${MSG_FREQUENT_LIMIT}秒${MSG_FREQUENT_COUNT_LIMIT}条
        //目前是5秒3条
        String redisCountKey = STUDENT_LATEST_VISIT_COUNT + openid;
        //判断消息队列是否存在
        Boolean hasKey = redisTemplate.hasKey(redisKey);
        //第一次初始化消息队列后，设置key的超时时间
        if(!hasKey){
            redisTemplate.opsForValue().set(redisKey,String.valueOf(System.currentTimeMillis()));
            redisTemplate.expire(redisKey, 120, TimeUnit.MINUTES);
            redisTemplate.opsForValue().set(redisCountKey,VISIT_COUNT_INIT);
            redisTemplate.expire(redisKey, 120, TimeUnit.MINUTES);
            return true;
        }
        String lastVisitStr = redisTemplate.opsForValue().get(redisKey);
        if(StringUtils.isBlank(lastVisitStr)){
            redisTemplate.opsForValue().set(redisKey,String.valueOf(System.currentTimeMillis()));
            redisTemplate.expire(redisKey, 120, TimeUnit.MINUTES);
            redisTemplate.opsForValue().set(redisCountKey,VISIT_COUNT_INIT);
            redisTemplate.expire(redisKey, 120, TimeUnit.MINUTES);
            return true;
        }else{
            Long lastVisit = Long.parseLong(lastVisitStr);
            long currentTime = System.currentTimeMillis(); 
            log.info("[visitFrequentCheck]lastVisit:{},currentTime:{}",lastVisit,currentTime);
            if(lastVisit == null || lastVisit == -1){
                return true;
            }
            //判断消息队列是否存在
            Boolean hasRedisCountKey = redisTemplate.hasKey(redisCountKey);
            //第一次初始化消息队列后，设置key的超时时间
            if(!hasRedisCountKey){
                redisTemplate.opsForValue().set(redisCountKey,VISIT_COUNT_INIT);
                redisTemplate.expire(redisKey, 120, TimeUnit.MINUTES);
                //更新访问时间
                redisTemplate.opsForValue().set(redisKey,String.valueOf(currentTime));
                return true;
            }
            if(currentTime - lastVisit < MSG_FREQUENT_LIMIT){
                //1.获取计数
                String lastVisitCountStr = redisTemplate.opsForValue().get(redisCountKey);
                log.info("[visitFrequentCheck]lastVisitCount:{}",lastVisitCountStr);
                Integer lastestVistCount = Integer.parseInt(lastVisitCountStr);
                if(lastestVistCount < MSG_FREQUENT_COUNT_LIMIT){
                    redisTemplate.opsForValue().set(redisCountKey,++lastestVistCount + "");
                    return true;
                }else{
                    return false;
                }
            }else{
                //超过5秒后计数置为1
                redisTemplate.opsForValue().set(redisCountKey,VISIT_COUNT_INIT);
                //更新访问时间
                redisTemplate.opsForValue().set(redisKey,String.valueOf(currentTime));
                return true;
            }
        }
    }
   
    @Override
    public int persistMessage(int lessonId){
        int zhiboCount = doPersistentMsg(redisTemplate,AreaTypeEnums.ZHIBO.getPrefix() + lessonId);
        log.info("[persistMessage]persist msg in [{}] count:{}",lessonId,zhiboCount);
        int discussCount = doPersistentMsg(redisTemplate,AreaTypeEnums.DISCUSS.getPrefix() + lessonId);
        log.info("[persistMessage]persist msg in [{}] count:{}",lessonId,discussCount);
        return zhiboCount + discussCount;
    }
    
    private int doPersistentMsg(RedisTemplate<String, String> redisTemplate, String key) {
        long lastPos = 0;
        String pos = redisTemplate.opsForValue().get(key + MSG_QUEUE_POS);
        if(StringUtils.isBlank(pos)){
            lastPos = 0;
        }else{
            lastPos = Long.parseLong(pos);
        }
        long length = redisTemplate.opsForList().size(key);
        long end = lastPos + Integer.parseInt(PROCESS_COUNT);
        if(end > length){
            end = length;
        }
        log.info("[persistentMsg]key:{},lastPos:{},end:{}",key,lastPos,end);
        if(lastPos == end){
            return 0;
        }
        List<String> messages = redisTemplate.opsForList().range(key, lastPos,end);
        if(CollectionUtils.isNotEmpty(messages)){
            List<MessageVO> messagesList = convert2Message(messages);
            batchSave(messagesList);
            //更新redis中的pos
            redisTemplate.opsForValue().set(key + MSG_QUEUE_POS,end+"");
            return messages.size();
        }
        log.info("[doPersistentMsg]msg queue is empty!");
        return 0;
    }

    private List<MessageVO> convert2Message(List<String> messages) {
        List<MessageVO> messagesList = Lists.newArrayListWithExpectedSize(messages.size());
        for(String messageStr : messages){
            try{
                MessageVO message = JacksonUtil.str2Obj(messageStr, MessageVO.class);
                messagesList.add(message);
            }catch(Exception e){
                log.error("error to parse Object ,message :{}",messageStr);
            }
        }
        return messagesList;
    }

    
    @Override
    public List<MessageVO> historyByUniqueId(Integer lessonId, String uniqueId, Integer count, Integer areaType) {
        TxVZhiBoMessage boMessage = getByUniqueId(uniqueId);
        if(boMessage == null){
            return null;
        }
        return history(lessonId, boMessage.getId(), count, areaType);
    }

    @Override
    public TxVZhiBoMessage getByUniqueId(String uniqueId) {
        if(StringUtils.isBlank(uniqueId)){
            return null;
        }
        return txVZhiBoMessageDao.getByUniqueId(uniqueId);
    }

    
    @Override
    public List<MessageVO> history(Long lessonId, Long id, Integer count, Integer areaType, Integer order) {
        List<TxVZhiBoMessage> messages = txVZhiBoMessageDao.history(lessonId,id,count,areaType,order);
        if(CollectionUtils.isEmpty(messages)){
            return Lists.newArrayList();
        }
        List<MessageVO> messageVOs = convertMsg2MessageVO(messages);
        return messageVOs;
    }
    

    @Override
    public List<MessageVO> historyByUniqueId(Long lessonId, String uniqueId, Integer count, Integer areaType,
        Integer order) {
        TxVZhiBoMessage boMessage = getByUniqueId(uniqueId);
        if(boMessage == null){
            return null;
        }
        return history(lessonId, boMessage.getId(), count, areaType,order);
    }
    
    
}