/**
 * Baijiahulian.com Inc.
 * Copyright (c) 2014-2016 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.vzhibo.service.impl;

import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import com.baijia.commons.lang.utils.PropertiesReader;
import com.baijia.tianxiao.constant.TxVZhiBoLessonStatusEnums;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.vzhibo.dao.TxVZhiBoLessonDao;
import com.baijia.tianxiao.dal.vzhibo.po.TxVZhiBoLesson;
import com.baijia.tianxiao.dal.vzhibo.po.TxVZhiBoRoom;
import com.baijia.tianxiao.dal.wechat.po.AuthorizationInfo;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.marketing.commons.exceptions.BusinessException;
import com.baijia.tianxiao.sal.organization.api.OrgAccountService;
import com.baijia.tianxiao.sal.organization.org.dto.OrgInfoSimpleDto;
import com.baijia.tianxiao.sal.organization.org.dto.TxCascadeCredentialDto;
import com.baijia.tianxiao.sal.organization.org.service.OrgInfoService;
import com.baijia.tianxiao.sal.organization.org.service.TxCascadeCredentialService;
import com.baijia.tianxiao.sal.vzhibo.constant.AreaTypeEnums;
import com.baijia.tianxiao.sal.vzhibo.constant.MessageTypeEnums;
import com.baijia.tianxiao.sal.vzhibo.constant.OwnerTypeEnums;
import com.baijia.tianxiao.sal.vzhibo.constant.TxVZhiBoEventType;
import com.baijia.tianxiao.sal.vzhibo.constant.TxVZhiBoLessonFieldEnums;
import com.baijia.tianxiao.sal.vzhibo.constant.TxVZhiBoLessonListTypeEnums;
import com.baijia.tianxiao.sal.vzhibo.constant.TxVZhiBoUserType;
import com.baijia.tianxiao.sal.vzhibo.constant.UserTypeEnums;
import com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoEventLogService;
import com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoLessonService;
import com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoLessonStudentService;
import com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoMessageService;
import com.baijia.tianxiao.sal.vzhibo.service.TxVZhiBoRoomService;
import com.baijia.tianxiao.sal.vzhibo.vo.FieldUpdateVO;
import com.baijia.tianxiao.sal.vzhibo.vo.LessonDetailVO;
import com.baijia.tianxiao.sal.vzhibo.vo.ListRequestDto;
import com.baijia.tianxiao.sal.vzhibo.vo.MessageInfo;
import com.baijia.tianxiao.sal.vzhibo.vo.ShareDataVO;
import com.baijia.tianxiao.sal.vzhibo.vo.TxVZhiBoLessonListVO;
import com.baijia.tianxiao.sal.wechat.api.AuthorizationInfoService;
import com.baijia.tianxiao.sal.wechat.constant.webauth.WebAuthScope;
import com.baijia.tianxiao.sal.wechat.helper.webauthlink.WechatWebAuthLinkBuilder;
import com.baijia.tianxiao.sqlbuilder.bean.Order;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * @title LiveServiceImpl
 * @desc TODO 
 * @author he11o
 * @date 2016年11月25日
 * @version 1.0
 */
@Slf4j
@Service
public class TxVZhiBoLessonServiceImpl implements TxVZhiBoLessonService{
    
    @Autowired
    AuthorizationInfoService authorizationInfoService;
    
    @Autowired
    OrgAccountService orgAccountService;
    
    @Autowired
    TxVZhiBoLessonDao txVZhiBoLessonDao;
    
    @Autowired
    TxVZhiBoLessonStudentService txVZhiBoLessonStudentService;
    
    @Autowired
    TxVZhiBoEventLogService txVZhiBoEventLogService;
    
    @Autowired
    TxVZhiBoRoomService txVZhiBoRoomService;
    
    @Value("${lesson_detail_url}")
    private String LESSON_DETAIL_URL;      //直播课简介url前缀
    
    @Value("${room_detail_url}")
    private String ROOM_DETAIL_URL;
    
    @Value("${room_expire_time}")
    private String roomExpireTime;      //直播间创建后过期时间，单位：天 l
    
    @Value("${tianxiaojia_app_id}")
    private String tianXiaoJiaAppId;
    
    @Autowired
    TxCascadeCredentialService txCascadeCredentialService;
    
    @Autowired
    OrgInfoService orgInfoService;
    
    @Autowired
    TxVZhiBoMessageService txVZhiBoMessageService;
    
    @Autowired
    StringRedisTemplate redisTemplate;
    
    @Value("${msg_queue_pos}")
    String MSG_QUEUE_POS;
    
//    private LoadingCache<Integer, AuthorizationInfo> authorizationInfoCache = CacheBuilder.newBuilder().maximumSize(5000)
//        .expireAfterWrite(10, TimeUnit.HOURS).build(new CacheLoader<Integer, AuthorizationInfo>() {
//            @Override
//            public AuthorizationInfo load(Integer key) throws Exception {
//                Preconditions.checkNotNull(key);
//                return authorizationInfoService.getByOrgId(key);
//            }
//    });
    
    @Override
    public String getLiveLessonUrl(Integer OrgNumber,Integer lessonId) {
        log.info("[getLiveLessonUrl]OrgNumber:{},lessonId:{}",OrgNumber,lessonId);
        OrgAccount orgAccount = orgAccountService.getOrgAccountByNumber(OrgNumber);
        if(orgAccount == null){
            return "";
        }
        AuthorizationInfo authorizationInfo = authorizationInfoService.getByOrgId(orgAccount.getId());
//        try {
//            authorizationInfo = authorizationInfoCache.get(orgAccount.getId());
//        } catch (Exception e) {
//            log.info("get authorizationInfo faild,orgId:{}",orgAccount.getId());
//        }
        String appId = "";
        if(authorizationInfo == null || StringUtils.isBlank(authorizationInfo.getAuthorizerAppId())){
            //机构还没有绑定公众号时，用天校家的公众号
            appId = tianXiaoJiaAppId;
        }else{
            appId = authorizationInfo.getAuthorizerAppId();
        }
        String lessonLiveUrl = PropertiesReader.getValue("config", "lesson_live_url");
        String redirectUrl = lessonLiveUrl.replace("{orgNumber}", orgAccount.getNumber().toString()).replace("{lessonId}", lessonId.toString());
        return WechatWebAuthLinkBuilder.fansinfo(WebAuthScope.BASE, appId, redirectUrl);
    }

    @Override
    public List<TxVZhiBoLessonListVO> list(Integer roomId,Integer ownerId,int status, ListRequestDto dto) {
        log.info("[list]roomId:{},ownerId:{},pageDto:{}",roomId,ownerId,dto);
        Long lastId = null;
        Integer pageSize = null;
        if(dto != null){
            lastId = dto.getLastId();
            pageSize = dto.getPageSize();
        }
        List<TxVZhiBoLesson> txVZhiBoLessons = txVZhiBoLessonDao.list(roomId,ownerId,lastId,pageSize,Order.desc("startTime"),status);
        if(CollectionUtils.isEmpty(txVZhiBoLessons)){
            return Lists.newArrayList();
        }
        return buildTxVZhiBoLessonListVO(txVZhiBoLessons,roomId);
    }

    private List<TxVZhiBoLessonListVO> buildTxVZhiBoLessonListVO(List<TxVZhiBoLesson> txVZhiBoLessons,Integer roomId){
        //获取由子账号创建的直播课的老师名称
        List<Integer> cascadeIdList = Lists.transform(txVZhiBoLessons, new Function<TxVZhiBoLesson, Integer>() {
            @Override
            public Integer apply(TxVZhiBoLesson input) {
                if(OwnerTypeEnums.CASCADE_ACCOUNT.getCode() == input.getOwnerType()){
                    return input.getOwnerId();
                }
                return null;
            }
        });
        //批量获取对应关系，key:cascadeId, value:name
        Map<Integer,String> teacherNameMap = txCascadeCredentialService.getNameMapByAccountIds(cascadeIdList);
        //批量获取直播课学生统计信息,key:lessonId, value:studentCount 
        Map<Integer,Integer> studentStat = txVZhiBoLessonStudentService.statStudentCountByRoomId(roomId);
        //转换为VO
        return buildTxVZhiBoLessonListVO(txVZhiBoLessons,studentStat,teacherNameMap);
        
    }
    
    private List<TxVZhiBoLessonListVO> buildTxVZhiBoLessonListVO(List<TxVZhiBoLesson> txVZhiBoLessons, 
        Map<Integer, Integer> studentStat, Map<Integer, String> teacherNameMap) {
        List<TxVZhiBoLessonListVO> lessonListVOs = Lists.newArrayListWithExpectedSize(txVZhiBoLessons.size());
        for(TxVZhiBoLesson txVZhiBoLesson : txVZhiBoLessons){
            TxVZhiBoLessonListVO txVZhiBoLessonListVO = new TxVZhiBoLessonListVO();
            lessonListVOs.add(txVZhiBoLessonListVO);
            
            txVZhiBoLessonListVO.setOrgId(txVZhiBoLesson.getOrgId());
            txVZhiBoLessonListVO.setOrgNumber(txVZhiBoLesson.getOrgNumber());
            
            txVZhiBoLessonListVO.setId(txVZhiBoLesson.getId());
            txVZhiBoLessonListVO.setRoomId(txVZhiBoLesson.getRoomId());
            txVZhiBoLessonListVO.setName(txVZhiBoLesson.getName());
            if(txVZhiBoLesson.getStartTime() != null){
                txVZhiBoLessonListVO.setStartTime(txVZhiBoLesson.getStartTime().getTime());
            }
            txVZhiBoLessonListVO.setAvatar(txVZhiBoLesson.getAvatar());
            txVZhiBoLessonListVO.setDetailUrl(getDetailUrl(txVZhiBoLesson.getId()));
            
            String liveUrl = getLiveLessonUrl(txVZhiBoLesson.getOrgNumber(), txVZhiBoLesson.getId().intValue());
            txVZhiBoLessonListVO.setLiveUrl(liveUrl);
            
            txVZhiBoLessonListVO.setStatus(txVZhiBoLesson.getStatus());
            if(studentStat.containsKey(txVZhiBoLesson.getId().intValue())){
                txVZhiBoLessonListVO.setStudentCount(studentStat.get(txVZhiBoLesson.getId().intValue()));
            }
            if(teacherNameMap.containsKey(txVZhiBoLesson.getOwnerId())){
                txVZhiBoLessonListVO.setTeacherName(teacherNameMap.get(txVZhiBoLesson.getOwnerId()));
            }else{
                try{
                    OrgInfoSimpleDto simpleDto = orgInfoSimpleDtoCache.get(txVZhiBoLesson.getOrgId());
                    txVZhiBoLessonListVO.setTeacherName(simpleDto.getShortName());
                }catch(Exception e){
                    log.error("Error in Get OrgInfoSimpleDto",e);
                }
            }
        }
        return lessonListVOs;
    }

    private LoadingCache<Integer, OrgInfoSimpleDto> orgInfoSimpleDtoCache = CacheBuilder.newBuilder().maximumSize(5000)
        .expireAfterWrite(120, TimeUnit.MINUTES).build(new CacheLoader<Integer, OrgInfoSimpleDto>() {
            @Override
            public OrgInfoSimpleDto load(Integer orgId) throws Exception {
                Preconditions.checkNotNull(orgId);
                return orgInfoService.getOrgInfo(new Long(orgId));
            }
    });
    
    
    @Override
    public List<TxVZhiBoLessonListVO> listOthers(Integer roomId,Integer ownerId,int status, ListRequestDto dto) {
        log.info("[listOthers]roomId:{},ownerId:{},pageDto:{}",roomId,ownerId,dto);
        Long lastId = null;
        Integer pageSize = null;
        if(dto != null){
            lastId = dto.getLastId();
            pageSize = dto.getPageSize();
        }
        List<TxVZhiBoLesson> txVZhiBoLessons = txVZhiBoLessonDao.listOthers(roomId,ownerId,lastId,pageSize,Order.desc("startTime"),status);
        if(CollectionUtils.isEmpty(txVZhiBoLessons)){
            return Lists.newArrayList();
        }
        List<TxVZhiBoLessonListVO> boLessonListVOs= buildTxVZhiBoLessonListVO(txVZhiBoLessons,roomId);
        //替换直播url为无授权的
        initWithoutAuthLiveUrl(boLessonListVOs);
        return boLessonListVOs; 
    }

    private void initWithoutAuthLiveUrl(List<TxVZhiBoLessonListVO> boLessonListVOs) {
        for(TxVZhiBoLessonListVO lessonListVO : boLessonListVOs){
            String lessonLiveUrl = PropertiesReader.getValue("config", "lesson_live_url");
            String liveUrl = lessonLiveUrl
                .replace("{orgNumber}", lessonListVO.getOrgNumber().toString())
                .replace("{lessonId}",lessonListVO.getId() + "");
            lessonListVO.setLiveUrl(liveUrl);
        }
    }

    @Override
    public Long create(Integer orgId, Integer txCascadeId,TxVZhiBoLesson txVZhiBoLesson) {
        OrgAccount orgAccount = orgAccountService.getOrgAccountById(orgId);
        //检测orgId的真实性
        if(orgAccount == null){
            return null;
        }
        txVZhiBoLesson.setOrgId(orgId);
        txVZhiBoLesson.setOrgNumber(orgAccount.getNumber());
        if(txCascadeId == null){
            txVZhiBoLesson.setOwnerId(orgId);
            txVZhiBoLesson.setOwnerType(OwnerTypeEnums.MAIN.getCode());
        }else{
            txVZhiBoLesson.setOwnerId(txCascadeId);
            txVZhiBoLesson.setOwnerType(OwnerTypeEnums.CASCADE_ACCOUNT.getCode());
        }
        txVZhiBoLessonDao.save(txVZhiBoLesson);
        //入库后初始化一条消息
        try {
            MessageInfo messageInfo = initLessonMessage(txVZhiBoLesson.getId(),orgAccount);
            txVZhiBoMessageService.send(messageInfo, null);
        } catch (Exception e) {
            log.info("Error in 初始化直播间消息失败",e);
            throw new BusinessException("初始化直播间消息失败!");
        }
        return txVZhiBoLesson.getId();
    }

    private MessageInfo initLessonMessage(Long lessonId, OrgAccount orgAccount) {
        MessageInfo messageInfo = new MessageInfo();
        messageInfo.setAreaType(AreaTypeEnums.ZHIBO.getCode());
        messageInfo.setMsgType(MessageTypeEnums.TEXT.getCode());
        messageInfo.setLessonId(lessonId.intValue());
        messageInfo.setOrgId(orgAccount.getId());
        messageInfo.setContent(PropertiesReader.getValue("config-common", "lesson_init_msg"));
        return messageInfo;
    }

    @Override
    public long getExpireTime(Integer lessonId) {
        TxVZhiBoLesson txVZhiBoLesson = txVZhiBoLessonDao.getById(lessonId);
        if(txVZhiBoLesson == null){
            return -1;
        }
        Calendar endTime = Calendar.getInstance();
        endTime.setTime(txVZhiBoLesson.getStartTime());
        endTime.add(Calendar.DAY_OF_MONTH, Integer.parseInt(roomExpireTime));
        endTime.add(Calendar.HOUR_OF_DAY, 2);
        Calendar currentTime = Calendar.getInstance();
        return  endTime.getTimeInMillis() - currentTime.getTimeInMillis();
    }

    @Override
    public TxVZhiBoLesson getByOrgIdAndId(Integer orgId, int lessonId) {
        log.info("[getByOrgIdAndId]orgId:{},lessonId:{}",orgId,lessonId);
        return txVZhiBoLessonDao.getByOrgIdAndId(orgId, lessonId);
    }


    @Override
    public boolean end(Integer orgId, Integer txCascadeId,int lessonId) {
        TxVZhiBoLesson txVZhiBoLesson = getByOrgIdAndId(orgId,lessonId);
        if(txVZhiBoLesson != null){
            int mod = txVZhiBoLessonDao.updateColumnValueById(lessonId, "status", TxVZhiBoLessonStatusEnums.DONE.getCode());
            if(mod > 0){
                //1.记录主动结束直播间Event
                TxVZhiBoUserType userType = TxVZhiBoUserType.CASCADE_ACCOUNT;
                Integer uId = txCascadeId;
                if(txCascadeId == null){
                    uId = orgId;
                    userType = TxVZhiBoUserType.JIGOU;
                }
                txVZhiBoEventLogService.saveEvent(lessonId, uId, userType, TxVZhiBoEventType.SELF_EXIT);
                //2.持久化消息队列的数据
                txVZhiBoMessageService.persistMessage(lessonId);
                //3.移除消息队列
                removeMessageQuque(lessonId);
                //4.更新缓存中直播间状态
                txVZhiBoLesson.setStatus(TxVZhiBoLessonStatusEnums.DONE.getCode());
                lessonCache.put(txVZhiBoLesson.getId().intValue(), txVZhiBoLesson);
                return true;
            }
        }
        return false;
    }
    
    @Override
    public void removeMessageQuque(int lessonId) {
        redisTemplate.delete(AreaTypeEnums.ZHIBO.getPrefix() + lessonId);
        redisTemplate.delete(AreaTypeEnums.ZHIBO.getPrefix() + lessonId + MSG_QUEUE_POS);
        redisTemplate.delete(AreaTypeEnums.DISCUSS.getPrefix() + lessonId);
        redisTemplate.delete(AreaTypeEnums.DISCUSS.getPrefix() + lessonId  + MSG_QUEUE_POS);
    }
    

    @Override
    public TxVZhiBoLesson getById(long lessonId) {
        log.info("[getById]id:{}",lessonId);
        return txVZhiBoLessonDao.getById(lessonId);
    }

    @Override
    public int syncLessonStatus() {
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(System.currentTimeMillis());
        log.info("[syncLessonStatus]currentTime:{}",calendar.getTime());
        calendar.add(Calendar.DAY_OF_MONTH, -7);
        calendar.add(Calendar.HOUR_OF_DAY, -2);
        log.info("[syncLessonStatus]time before:{}",calendar.getTime());
        return txVZhiBoLessonDao.syncLessonStatus(calendar.getTime(),TxVZhiBoLessonStatusEnums.DONE.getCode());
    }

    @Override
    public ShareDataVO getShareData(int lessonId) {
        TxVZhiBoLesson txVZhiBoLesson = txVZhiBoLessonDao.getById(lessonId);
        if(txVZhiBoLesson == null){
            return null;
        }
        ShareDataVO shareDataVO = new ShareDataVO();
        shareDataVO.setTitle(txVZhiBoLesson.getName());
        shareDataVO.setImageUrl(txVZhiBoLesson.getAvatar());
        shareDataVO.setContent(txVZhiBoLesson.getDescription());
        String liveUrl = getLiveLessonUrl(txVZhiBoLesson.getOrgNumber(), lessonId);
        shareDataVO.setShareUrl(liveUrl);
        return shareDataVO;
    }

//    private LoadingCache<Integer, LessonDetailVO> lessonDetailCache = CacheBuilder.newBuilder().maximumSize(2000)
//        .expireAfterWrite(120, TimeUnit.MINUTES).build(new CacheLoader<Integer, LessonDetailVO>() {
//            @Override
//            public LessonDetailVO load(Integer key) throws Exception {
//                TxVZhiBoLesson zhiBoLesson = getById(key);
//                if(zhiBoLesson == null){
//                    return null;
//                }
//                return convert2DetailVO(zhiBoLesson);
//            }
//    });
    
    
    @Override
    public LessonDetailVO getDetailById(Integer lessonId) throws BussinessException, NumberFormatException, Exception{
//        try {
//            LessonDetailVO detailVO =  lessonDetailCache.get(lessonId);
//            return detailVO;
//        } catch (Exception e) {
//            log.warn("Error In getDetailById[LessonDetailVO]",e);
//        }
//        return null;
        TxVZhiBoLesson zhiBoLesson = getById(lessonId);
        if(zhiBoLesson == null){
            return null;
        }
        //检查直播课状态
        if(zhiBoLesson.getStatus() == TxVZhiBoLessonStatusEnums.LIVE.getCode()){
            //若消息队列的key已经不存在了，说明直播课已经结束了
            String msgQuqueKey = AreaTypeEnums.ZHIBO.getPrefix() + lessonId;
            if(!redisTemplate.hasKey(msgQuqueKey)){
                log.info("[getDetailById]lessonId:{} 消息队列已经不存在了，key:{}",lessonId,msgQuqueKey);
                //1.修正直播课状态
                zhiBoLesson.setStatus(TxVZhiBoLessonStatusEnums.DONE.getCode());
                //2.更新数据库
                txVZhiBoLessonDao.updateColumnValueById(zhiBoLesson.getId(), "status", TxVZhiBoLessonStatusEnums.DONE.getCode());
            }
        }
        return convert2DetailVO(zhiBoLesson);
    }

    private LessonDetailVO convert2DetailVO(TxVZhiBoLesson zhiBoLesson) throws BussinessException, NumberFormatException, Exception {
        LessonDetailVO lessonDetailVO = new LessonDetailVO();
        lessonDetailVO.setId(zhiBoLesson.getId());
        lessonDetailVO.setAvatar(zhiBoLesson.getAvatar());
        lessonDetailVO.setDescription(zhiBoLesson.getDescription());
        lessonDetailVO.setName(zhiBoLesson.getName());
        lessonDetailVO.setOrgNumber(zhiBoLesson.getOrgNumber());
        if(zhiBoLesson.getStartTime() != null){
            lessonDetailVO.setStartTime(zhiBoLesson.getStartTime().getTime());
        }
        lessonDetailVO.setRoomId(zhiBoLesson.getRoomId());
        
        OrgInfoSimpleDto orgInfo = orgInfoService.getOrgInfo(Long.parseLong(zhiBoLesson.getOrgId().toString()));
        lessonDetailVO.setOrgName(orgInfo.getShortName());
        lessonDetailVO.setOrgAvatar(orgInfo.getLogo());
        //机构创建的直播课，老师名称填机构的名称
        if(orgInfo != null && OwnerTypeEnums.MAIN.getCode() == zhiBoLesson.getOwnerType()){
            lessonDetailVO.setOrgName(orgInfo.getShortName());
            lessonDetailVO.setTeacherName(orgInfo.getShortName());
            lessonDetailVO.setTeacherAvatar(orgInfo.getLogo());
        }else{
            TxCascadeCredentialDto credentialDto =  txCascadeCredentialService.getByTxCasCade(
                Long.parseLong(zhiBoLesson.getOrgId().toString()), zhiBoLesson.getOwnerId());
            if(credentialDto != null){
                lessonDetailVO.setTeacherName(credentialDto.getName());
                if (StringUtils.isNotBlank(credentialDto.getAvatarUrl())) {
                    lessonDetailVO.setTeacherAvatar(credentialDto.getAvatarUrl());
                } else {
                    lessonDetailVO.setTeacherAvatar(com.baijia.tianxiao.biz.www.constant.BizConstant.DEFAULT_CASCADE_AVATAR);
                }
            }
        }
        
        TxVZhiBoRoom txVZhiBoRoom = txVZhiBoRoomService.getById(zhiBoLesson.getRoomId());
        if(txVZhiBoRoom != null){
            lessonDetailVO.setRoomName(txVZhiBoRoom.getName());
            lessonDetailVO.setRoomAvatar(txVZhiBoRoom.getAvatar());
        }
        
        lessonDetailVO.setStatus(zhiBoLesson.getStatus());
        //获取直播课地址
        String liveUrl = getLiveLessonUrl(zhiBoLesson.getOrgNumber(), zhiBoLesson.getId().intValue());
        lessonDetailVO.setLiveUrl(liveUrl);
        //获取直播间地址
        String roomUrl = ROOM_DETAIL_URL + zhiBoLesson.getOrgNumber();
        lessonDetailVO.setRoomUrl(roomUrl);

        ShareDataVO shareDataVO = new ShareDataVO();
        shareDataVO.setTitle(lessonDetailVO.getName());
        shareDataVO.setImageUrl(lessonDetailVO.getAvatar());
        shareDataVO.setContent(lessonDetailVO.getDescription());
        shareDataVO.setShareUrl(LESSON_DETAIL_URL + lessonDetailVO.getId());
        lessonDetailVO.setShareData(shareDataVO);
        
        return lessonDetailVO;
    }
 
    @Override
    public int update(Integer orgId,FieldUpdateVO updateVO){
        Map<String,Object> countCondition = Maps.newHashMap();
        countCondition.put("orgId", orgId);
        countCondition.put("id", updateVO.getId());
        //判断直播课是否属于当前机构
        int count = txVZhiBoLessonDao.countByCondition(countCondition, "id", true);
        if(count < 1){
            return -1;
        }
        TxVZhiBoLessonFieldEnums fieldEnums = TxVZhiBoLessonFieldEnums.parse(updateVO.getField());
        if(fieldEnums == null){
            return -1;
        }
        Map<String,Object> updateCondtion = Maps.newHashMap();
        if(TxVZhiBoLessonFieldEnums.STARTTIME.equals(fieldEnums)){
            updateCondtion.put(fieldEnums.getName(), new Date(Long.parseLong(updateVO.getValue().toString())));
        }else{
            updateCondtion.put(fieldEnums.getName(), updateVO.getValue());
        }
        List<String> updateProps = Lists.newArrayList();
        updateProps.add(fieldEnums.getName());
        if(TxVZhiBoLessonFieldEnums.AVATAR.equals(fieldEnums)){
            if(updateVO != null && updateVO.getExt() != null && NumberUtils.isDigits(updateVO.getExt().toString())){
                updateCondtion.put("storageId", updateVO.getExt());
                updateProps.add("storageId");
            }
        } 
        updateCondtion.put("id", updateVO.getId());
        int mod = txVZhiBoLessonDao.update(updateCondtion, updateProps.toArray(new String[updateProps.size()]));
        if(mod > 0){
            //刷新缓存
//            refreshCache(updateVO,fieldEnums);
        }
        return mod; 
    }

    private void refreshCache(FieldUpdateVO updateVO, TxVZhiBoLessonFieldEnums fieldEnums) {
        try {
//            LessonDetailVO detailVO =  lessonDetailCache.get(updateVO.getId());
//            if(detailVO != null){
//                if(updateVO.getValue() == null){
//                    return;
//                }
//                String value = updateVO.getValue().toString();
//                switch(fieldEnums){
//                    case AVATAR:
//                        detailVO.setAvatar(value);
//                        break;
//                    case NAME:
//                        detailVO.setName(value);
//                        break;
//                    case DESCRIPTION:
//                        detailVO.setDescription(value);
//                        break;
//                    case STARTTIME:   
//                        detailVO.setStartTime(Long.parseLong(value));
//                        break;
//                }
//                lessonDetailCache.put(updateVO.getId(), detailVO);
//            }
        } catch (Exception e) {
            log.error("Error in refreshCache",e);
        }
    }

    @Override
    public Map<String, List<TxVZhiBoLessonListVO>> listAll(Integer roomId, Integer ownerId, int lessonListType) {
        List<TxVZhiBoLessonListVO> boLessonListVOs = null;
        if(TxVZhiBoLessonListTypeEnums.ALL.getCode() == lessonListType){
            boLessonListVOs = list(roomId, null,TxVZhiBoLessonStatusEnums.ALL.getCode(),null);
        }else if(TxVZhiBoLessonListTypeEnums.OWN.getCode() == lessonListType){
            boLessonListVOs = list(roomId, ownerId, TxVZhiBoLessonStatusEnums.ALL.getCode(),null);
        }else{
            boLessonListVOs = listOthers(roomId, ownerId, TxVZhiBoLessonStatusEnums.ALL.getCode(),null);
        }
        
        if(CollectionUtils.isEmpty(boLessonListVOs)){
            return Maps.newHashMap();
        }
        
        List<TxVZhiBoLessonListVO> liveList = Lists.newArrayListWithExpectedSize(boLessonListVOs.size());
        List<TxVZhiBoLessonListVO> doneList = Lists.newArrayListWithExpectedSize(boLessonListVOs.size());
        
        for(TxVZhiBoLessonListVO listVO : boLessonListVOs){
            if(TxVZhiBoLessonStatusEnums.LIVE.getCode() == listVO.getStatus()){
                liveList.add(listVO);
            }else{
                doneList.add(listVO);
            }
        }
        
        Map<String, List<TxVZhiBoLessonListVO>> statusGroupedMap = Maps.newHashMap();
        statusGroupedMap.put("live", liveList);
        statusGroupedMap.put("done", doneList);
        return statusGroupedMap;
    }

    @Override
    public String getDetailUrl(Long id) {
        return LESSON_DETAIL_URL + id;
    }

    @Override
    public int checkLessonStatus(long lessonId) {
        TxVZhiBoLesson detailVO = txVZhiBoLessonDao.getById(lessonId);
        if(detailVO == null){
            return TxVZhiBoLessonStatusEnums.DELETED.getCode();
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(System.currentTimeMillis());
        calendar.add(Calendar.DAY_OF_MONTH, -7);
        calendar.add(Calendar.HOUR_OF_DAY, -2);
        
        Date startTime = detailVO.getStartTime();
        if(startTime != null){
            Long startT = startTime.getTime();
            if(startT > calendar.getTimeInMillis()){
                return TxVZhiBoLessonStatusEnums.LIVE.getCode();
            }else{
                return TxVZhiBoLessonStatusEnums.DONE.getCode();
            }
        }
        return TxVZhiBoLessonStatusEnums.DELETED.getCode();
    }

    @Override
    public boolean delete(Integer orgId,Integer txCascadeId, int lessonId) {
        TxVZhiBoLesson detailVO = txVZhiBoLessonDao.getByOrgIdAndId(orgId, lessonId);
        if(detailVO == null){
            return false;
        }
        
        int mod = txVZhiBoLessonDao.updateColumnValueById(lessonId, "status", TxVZhiBoLessonStatusEnums.DELETED.getCode());
        return mod > 0 ? true : false;
    }
 
    @Override
    public Map<String,Integer> countByStatus(Integer roomId){
        Map<String,Object>  countCondition = Maps.newHashMap();
        countCondition.put("roomId", roomId);
        Map<Integer,Integer> map = txVZhiBoLessonDao.groupCount(countCondition, "id", "status", true, Integer.class);
        Map<String,Integer> resultMap = Maps.newHashMap();
        if(map == null || map.size() == 0){
            return resultMap;
        }
        for(Map.Entry<Integer,Integer> entry : map.entrySet()){
            if(entry.getKey().intValue() == TxVZhiBoLessonStatusEnums.LIVE.getCode()){
                resultMap.put("live", entry.getValue());
            }else if(entry.getKey().intValue() == TxVZhiBoLessonStatusEnums.DONE.getCode()){
                resultMap.put("done", entry.getValue());
            }
        }
        return resultMap;
    }
    
    /**
     * 直播课缓存
     */
    private LoadingCache<Integer, TxVZhiBoLesson> lessonCache = CacheBuilder.newBuilder().maximumSize(10000)
        .expireAfterWrite(2, TimeUnit.HOURS).build(new CacheLoader<Integer, TxVZhiBoLesson>() {
            @Override
            public TxVZhiBoLesson load(Integer lessonId) throws Exception {
                Preconditions.checkNotNull(lessonId);
                return getById(lessonId);
            }
    });
    
    @Override
    public boolean checkLessonMessageStatus(Integer lessonId){
      //检查直播课状态，已结束或已删除的不能再发消息了
        try {
            TxVZhiBoLesson boLesson = lessonCache.get(lessonId);
            log.info("[send]boLesson:{}",boLesson);
            if(boLesson == null 
                || TxVZhiBoLessonStatusEnums.DONE.getCode() == boLesson.getStatus() 
                    || TxVZhiBoLessonStatusEnums.DELETED.getCode() == boLesson.getStatus()){
                log.info("[send]lesson status is illegel!");
                return false;
            }
        } catch (ExecutionException e) {
            log.error("Error In send msg",e);
            return false;
        }
        return true;
    }
    
    @Override
    public UserTypeEnums checkUserType(Integer lessonId, Integer orgId, Integer txCascadeId, String openId) {
        log.info("[checkUserType]lessonId:{},orgId:{},txCascadeId:{},openId:{}",lessonId,orgId,txCascadeId,openId);
        if(lessonId == null){
            return UserTypeEnums.HACKER;
        }
        //非法访问
        if(orgId == null && txCascadeId == null && StringUtils.isBlank(openId)){
            return UserTypeEnums.HACKER;
        }
        //学生
        if(orgId == null && txCascadeId == null && StringUtils.isNoneBlank(openId)){
            return UserTypeEnums.STUDENT;
        }
        TxVZhiBoLesson lesson = null;
        try {
            lesson = getById(new Long(lessonId));
        } catch (Exception e) {
            log.info("Cann`t find lesson",e);
        }
        if(lesson == null){
            return UserTypeEnums.HACKER;
        }
        //判断是否为同一个机构
        if(lesson.getOrgId().equals(orgId)){
            if(lesson.getOwnerType() == OwnerTypeEnums.MAIN.getCode()){
                if(txCascadeId == null){
                    return UserTypeEnums.TEACHER;
                }else{
                    return UserTypeEnums.VISITOR;
                }
            }else{
                if(lesson.getOwnerId().equals(txCascadeId)){
                    return UserTypeEnums.TEACHER;
                }else{
                    return UserTypeEnums.VISITOR;
                }
            }
        }else{
            //不是当前机构的auth_token
            return UserTypeEnums.HACKER;
        }
    }

    
    @Override
    public List<TxVZhiBoLesson> getByRoomId(Integer roomId) {
        if(roomId == null){
            return Lists.newArrayList();
        }
        return txVZhiBoLessonDao.getByRoomId(roomId);
    }
    
    @Override
    public LessonDetailVO getOnlyDetailById(Integer lessonId) throws BussinessException, NumberFormatException, Exception{
        TxVZhiBoLesson zhiBoLesson = getById(lessonId);
        if(zhiBoLesson == null){
            return null;
        }
        return convert2DetailVO(zhiBoLesson);
    }
    
}
