
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.upload.service.impl;

import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.excel.ExcelExporterUtils;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.upload.service.UploadFileReaderService;

import com.google.common.base.Preconditions;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.io.InputStream;
import java.io.PushbackInputStream;
import java.util.List;

import lombok.extern.slf4j.Slf4j;

/**
 * @title ExcepUploadFileReaderServiceImpl
 * @desc TODO
 * @author cxm
 * @date 2016年3月16日
 * @version 1.0
 */
@Slf4j
public class ExcelUploadFileReaderServiceImpl implements UploadFileReaderService {

    private MultipartFile excelFile;

    private int currentIndex=1;

    private Sheet sheet;

    private Workbook wb;

    private boolean isComplete;

    private int count;

    private int headerCount;

    public ExcelUploadFileReaderServiceImpl(MultipartFile excelFile) {
        Preconditions.checkArgument(excelFile != null && !excelFile.isEmpty(), "excel file is not exist");
        this.excelFile = excelFile;
        initReader();
    }

    private synchronized void initReader() {
        if (sheet == null && !isComplete) {
            log.info("init to reader excel:{}", excelFile.getOriginalFilename());
            InputStream in = null;
            try {
                in = excelFile.getInputStream();

                wb = WorkbookFactory.create(new PushbackInputStream(in, 8));

                sheet = wb.getSheetAt(1);

                count = sheet.getLastRowNum() + 1;

                log.info("init read excel:{} complete", excelFile);
            }catch(IllegalArgumentException e){
                IOUtils.closeQuietly(wb);
                IOUtils.closeQuietly(in);
                log.warn("read excel catch exception:{}",e);
                throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "excel版本不匹配");
            }catch (Exception e) {
                IOUtils.closeQuietly(wb);
                IOUtils.closeQuietly(in);
                log.error("read excel catch exception:", e);
                throw new RuntimeException("read excel exception - ", e);
            }
        }
    }

    @Override
    public int getDataLength() throws IOException {
        if (count == 0 && !isComplete) {
            initReader();
            count = sheet.getLastRowNum() + 1;
        }
        return count;
    }

    /**
     * 暂时不用
     */
    @Override
    public synchronized Object[] readData() throws IOException {
        try {
            initReader();
            Row row = sheet.getRow(currentIndex);
            if (row == null ||row.getFirstCellNum()<0) {
                if(currentIndex<=sheet.getLastRowNum()){
                    currentIndex++;
                    return readData();
                }
                return null;
            }
            if (headerCount == 0) {
                headerCount = row.getLastCellNum();
            }
            Object[] rowObj = new Object[headerCount];
            boolean hasValueRow = false;
            
            for (int c = row.getFirstCellNum(); c < headerCount; c++) {
                Cell cell = row.getCell(c);
                rowObj[c] = ExcelExporterUtils.getValueFromCell(cell);
                if (!hasValueRow && StringUtils.isNoneBlank(rowObj[c].toString().trim())) {
                    hasValueRow = true;
                }
            }
            currentIndex++;
            if (currentIndex >= sheet.getLastRowNum()) {
                this.isComplete = true;
            }
            if (hasValueRow) {
                return rowObj;
            } else {
                this.count--;
                // 如果没有值,递归调用
                return readData();
            }

        } catch (Exception e) {
            log.error("read data get error:", e);
            this.isComplete = true;
            throw e;
        } finally {
            if (isComplete) {
                IOUtils.closeQuietly(wb);
            }
        }
    }

    @Override
    public List<Object[]> readDatas() throws IOException {
        List<Object[]> datas = ExcelExporterUtils.read(excelFile.getInputStream(), 0);
        this.count = datas.size();
        return datas;
    }

    @Override
    public void close() {
        IOUtils.closeQuietly(wb);
    }


}
