

/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.upload.service.impl;

import com.baijia.tianxiao.common.service.ImportDataProcessService;
import com.baijia.tianxiao.common.service.ImportDataProcessService.SingleSaveErrorResult;
import com.baijia.tianxiao.constants.DataProcType;
import com.baijia.tianxiao.dal.org.dao.OrgInfoDao;
import com.baijia.tianxiao.dal.upload.dao.TxUploadRecordsDao;
import com.baijia.tianxiao.dal.upload.po.TxUploadRecords;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.enums.CrmErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.filter.TianxiaoPCContext;
import com.baijia.tianxiao.sal.upload.dto.TaskStatus;
import com.baijia.tianxiao.sal.upload.service.CrmUploadService;
import com.baijia.tianxiao.sal.upload.service.UploadFileReaderService;
import com.baijia.tianxiao.util.SerializeUtil;
import com.baijia.tianxiao.util.bean.BizDateThreadLocalUtil;
import com.baijia.tianxiao.util.json.JacksonUtil;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.dao.DataAccessException;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;

import lombok.extern.slf4j.Slf4j;

/**
 * @title UploadServiceImpl
 * @desc TODO
 * @author cxm
 * @date 2016年3月15日
 * @version 1.0
 */
@Service
@Slf4j
public class CrmUploadServiceImpl implements CrmUploadService, InitializingBean, ApplicationContextAware {

    private Map<Integer, ImportDataProcessService> dataProcessServiceMap;
    private ApplicationContext context;

    @Autowired
    private TxUploadRecordsDao txUploadRecordsDao;
    @Autowired
    private OrgInfoDao orgInfoDao;
    @Autowired
    private RedisTemplate<String, Object> redisTemplate;
    
  
    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.context = applicationContext;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        dataProcessServiceMap = Maps.newHashMap();
        Map<String, ImportDataProcessService> serviceMap = context.getBeansOfType(ImportDataProcessService.class);
        for (ImportDataProcessService service : serviceMap.values()) {
            dataProcessServiceMap.put(service.getProcessType().getType(), service);
        }
    }
    
    /**
     * 创建一个固定10个线程大小的线程池,用来响应上传任务
     */
    private ExecutorService uploadTaskExecutor = Executors.newFixedThreadPool(10, new ThreadFactory() {
        private final AtomicInteger threadNumber = new AtomicInteger(1);

        @Override
        public Thread newThread(Runnable r) {
            Thread t = new Thread(r, "UPLOAD_TASK_" + threadNumber.getAndIncrement());
            return t;
        }
    });
    

    
    /*** redis */
    String getCacheDataKey(String taskId){
    	return new StringBuilder().append("data_").append(taskId).toString();
    }
    String getCacheTaskStatusKey(String taskId){
    	return new StringBuilder().append("task_").append(taskId).toString();
    }
    Integer getCacheExpireTime(){
    	return 60*30;
    }
    
    private void cacheData(final String taskId,final Collection<SingleSaveErrorResult> data) {
        try {
        	redisTemplate.execute(new RedisCallback<Collection<SingleSaveErrorResult>>() {  
                @Override  
                public Collection<SingleSaveErrorResult> doInRedis(RedisConnection connection)   throws DataAccessException {
                	String key = getCacheDataKey(taskId);
                    connection.set(key.getBytes(),SerializeUtil.serialize(JacksonUtil.obj2Str(data)));
                    connection.expire(key.getBytes(), getCacheExpireTime());
                    return null;  
                }  
            });
        	
        } catch (Exception e) {
            log.error("cacheData - exception:", e);
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "缓存数据信息失败");
        }
    }
    
    private Collection<SingleSaveErrorResult> getData(final String taskId){
    	try {
    		return redisTemplate.execute(new RedisCallback<Collection<SingleSaveErrorResult>>() {  
    	        @Override  
    	        public Collection<SingleSaveErrorResult> doInRedis(RedisConnection connection)   throws DataAccessException {  
    	        	String key = getCacheDataKey(taskId); 
    	            if (connection.exists(key.getBytes())) {  
    	                byte[] value = connection.get(key.getBytes());  
    	                Object obj = SerializeUtil.unserialize(value);
    	                if(obj!=null){
	    	                try {
								return JacksonUtil.str2List(obj.toString(), SingleSaveErrorResult.class);
							} catch (Exception e) {
								log.error("",e);
							}
    	                }
    	            }  
    	            return null;  
    	        }  
    	    });  
        } catch (Exception e) {
            log.error("getData exception:", e);
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "查询缓存数据信息失败");
        }
    }
    
    
    
    private void cacheTaskStatus(final String taskId, final TaskStatus taskStatus) {
        try {
        	redisTemplate.execute(new RedisCallback<Object>() {  
                @Override  
                public Object doInRedis(RedisConnection connection)   throws DataAccessException {
                	String key = getCacheTaskStatusKey(taskId);
                	connection.set(key.getBytes(), SerializeUtil.serialize(taskStatus));
					try {
						log.info("Set redis expired time.key={}",key);
						connection.expire(key.getBytes(), getCacheExpireTime());
					}catch (Exception e){
						log.error("Redis exception.",e);
					}
                    return null;
                }  
            });
        	
        } catch (Exception e) {
            log.error("cacheTaskStatus - exception:", e);
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "缓存执行进度失败");
        }
    }
    
    private TaskStatus getTaskStatus(final String taskId){
    	try {
    		return redisTemplate.execute(new RedisCallback<TaskStatus>() {  
    	        @Override  
    	        public TaskStatus doInRedis(RedisConnection connection)   throws DataAccessException {  
    	        	String key = getCacheTaskStatusKey(taskId);  
    	            if (connection.exists(key.getBytes())) {  
    	                byte[] value = connection.get(key.getBytes());  
    	                Object obj = SerializeUtil. unserialize(value);
    	                if(obj!=null){
    	                	return (TaskStatus) obj;
    	                }
    	                return null;  
    	            }  
    	            return null;  
    	        }  
    	    });  
        } catch (Exception e) {
            log.error("cacheTaskStatus - exception:", e);
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "查询执行进度失败");
        }
    }
    
    
    
	@Override
	public void downloadConsultImportTemplate(int uploadType, OutputStream os) {
		ImportDataProcessService processService = dataProcessServiceMap.get(uploadType);
		processService.downloadImportTemplate(os);
	}
    
	
	
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public String validateFile(final Long orgId, final int uploadType, final boolean override, final MultipartFile file){
    	Preconditions.checkNotNull(uploadType, "upload type can not be null");
        Preconditions.checkArgument(file != null && !file.isEmpty(), "upload file is null");

        final Integer cascadeId =  TianxiaoPCContext.getTXCascadeId();
        final String taskId = UUID.randomUUID().toString();
        final UploadFileReaderService service = getUploadFileService(file);
        
        uploadTaskExecutor.execute(new Runnable() {
            @Override
            public void run() {
            	
                //新建的线程内取不到登陆信息，需要从外部传进来
                TianxiaoPCContext.setTXCascadeId(cascadeId);
                TianxiaoPCContext.setOrgId(orgId.intValue());
                log.info("start to execute upload task:{}", taskId);
                
                ImportDataProcessService processService = dataProcessServiceMap.get(uploadType);
                List<SingleSaveErrorResult> dataList = new ArrayList<SingleSaveErrorResult>();
                
                TaskStatus taskStatus = new TaskStatus();
                taskStatus.setUploadType(uploadType);
                taskStatus.setFileName(file.getOriginalFilename());
                taskStatus.setOverride(override);
                cacheTaskStatus(taskId, taskStatus);
                
                try {
                	Object[] lineData = null;
                	
                	//******* 头部信息 检测
                	int headerLineNum = 0;
                	int maxHeaderCheckNum = 1;//最多检测10行 判断是否有匹配的头部
                	List<String> headers = null;
                	boolean validateHeaders = false;
                	
                	try{
	                	while ((lineData = service.readData()) != null && headerLineNum<maxHeaderCheckNum) {
	                		lineData = Arrays.copyOfRange(lineData, 1, lineData.length);
	                		try{
	                			headers = tranHeader(lineData);
	                		}catch(Exception e){
	                			log.info("",e);
	                		}
	                		if(DataProcType.ORG_STUDENT.getType() == uploadType||DataProcType.CONSULT.getType() == uploadType){
                                
                                if(processService.validateHeader(headers,orgId)){
                                    taskStatus.setHeaders(headers);
                                    validateHeaders = true;//检测通过
                                    break;
                                }
                                
                                
                            }
	                		
	                		else if(processService.validateHeader(headers)){
		                    	taskStatus.setHeaders(headers);
		                    	validateHeaders = true;//检测通过
		                    	break;
		                    }
		                    headerLineNum++;
	                	}
                	}catch(Exception e){
                		log.error("",e);
                	}
                	
                	if(!validateHeaders){
                    	throw new BussinessException(CrmErrorCode.EXCEL_TEMPLATE_VALIDATE_FAIL, "上传文件的表头信息不正确!");
                    }
                	
                    //*********** 数据信息
                	//数据读取
                    int lineNum = 0;
                    while ((lineData = service.readData()) != null) {
                    	lineData = Arrays.copyOfRange(lineData, 1, lineData.length);
                    	dataList.add(new SingleSaveErrorResult(lineData));
                    }
                    
                    if(dataList.size()==0){
                    	taskStatus.setException( new BussinessException(CommonErrorCode.PARAM_ERROR, "上传文件内容为空!"));
                    	cacheTaskStatus(taskId, taskStatus);
                    	return;
                    }else{
                    	taskStatus.setTotalCount(dataList.size());
                    	cacheTaskStatus(taskId, taskStatus);
                    }

                    //清空本地线程的数据，后续业务会刷新填充
                    BizDateThreadLocalUtil.clear();
                    //数据检测
                	for(SingleSaveErrorResult result:dataList){
                		try{
                			result = processService.validate(orgId, TianxiaoPCContext.getTXCascadeId()==null?0L:TianxiaoPCContext.getTXCascadeId().longValue(), headers, result, dataList, override);
                			
                			if (result.isSuccess()) {
                                taskStatus.increaseSuccessCount();
                            } else {
                                taskStatus.increaseFailCount();
                            }
                		}catch (Exception e) {
                            log.warn("vaildate data:{} ,catch error:{}", ToStringBuilder.reflectionToString(lineData), e.getMessage());
                            log.error("vaildate data catch error:", e);
                            taskStatus.increaseFailCount();
                        }
                		
                		lineNum++;
                		if(lineNum%10==0){
                			cacheTaskStatus(taskId, taskStatus);
                		}
                	}
                	BizDateThreadLocalUtil.clear();
                	
                	if(!override){
                		processService.validateResult(dataList);
                		int succNum = 0;
                		int failNum = 0;
                		for(SingleSaveErrorResult s:dataList){
                			if(s.isSuccess()){ 
                				succNum++;
                			}else{ 
                				failNum++;
                			}
                		}
                		taskStatus.setSuccessCount(succNum);
                		taskStatus.setFailCount(failNum);
                	}
                	
                    taskStatus.setCompleteCount(taskStatus.getTotalCount());
                    cacheData(taskId, dataList);
                    
                }catch (BussinessException be){
                	if(be.getErrorCode() == CrmErrorCode.EXCEL_TEMPLATE_VALIDATE_FAIL){
                		taskStatus.setException(new BussinessException(CrmErrorCode.EXCEL_TEMPLATE_VALIDATE_FAIL, be.getMessage()));
                	}else{
                		taskStatus.setException(new BussinessException(CommonErrorCode.BUSINESS_ERROR, be.getMessage()));
                	}
                }catch (Exception e) {
                    log.warn("read data catch error:", e);
                    taskStatus.setException(new BussinessException(CommonErrorCode.BUSINESS_ERROR, "读取文件错误"));
                    
                }finally {
                	cacheTaskStatus(taskId, taskStatus);
                    service.close();
                }
            }
        });
        return taskId;
    }

    
    
    @Override
	public void downloadVaildateResult(OutputStream os, Long orgId, String taskId) {
    	TaskStatus taskStatus = getTaskStatus(taskId);
    	Collection<SingleSaveErrorResult> data = getData(taskId);
    	if(data==null){
    		try{
    			os.write("文件已失效，下载失败。".getBytes());
    			return;
    		}catch(Exception e){
    		}finally{
    			if(os!=null){
    				try {
						os.close();
					} catch (IOException e) {
					}
    			}
    		}
    	}
    	
    	ImportDataProcessService processService = dataProcessServiceMap.get(taskStatus.getUploadType());
    	processService.downloadValidateResult(os, orgId, taskId, data);
	}
    
    public static void main(String args[]){
    	Object[] lineData = new Object[]{1,2};
    	SingleSaveErrorResult saveResult = new SingleSaveErrorResult();
    	saveResult.setLineData(lineData);
    	lineData=null;
    	System.out.println(saveResult.getLineData().length);
    }
    
    
	@Override
	public void doImport(final Long orgId, final String taskId) {
		  final TaskStatus taskStatus = getTaskStatus(taskId);
		  final Collection<SingleSaveErrorResult> validateData = getData(taskId);
		  
		  final Integer cascadeId =  TianxiaoPCContext.getTXCascadeId();
	      final Integer uploadType = taskStatus.getUploadType();
	      final TxUploadRecords uploadRecord = new TxUploadRecords();
	      
	      uploadRecord.setFileName(taskStatus.getFileName());
	      uploadRecord.setOrgId(orgId);
	      uploadRecord.setStatus(0);
	      uploadRecord.setUploadType(uploadType);
	      txUploadRecordsDao.save(uploadRecord);
	
	      uploadTaskExecutor.execute(new Runnable() {
			  @Override
			  public void run() {
				  //新建的线程内取不到登陆信息，需要从外部传进来
				  TianxiaoPCContext.setTXCascadeId(cascadeId);
				  TianxiaoPCContext.setOrgId(orgId.intValue());

				  ImportDataProcessService processService = dataProcessServiceMap.get(uploadType);
				  Map<String, SingleSaveErrorResult> dataMap = new LinkedHashMap<String, SingleSaveErrorResult>();

				  //清理缓存状态
				  taskStatus.clear();
				  taskStatus.setTotalCount(validateData.size());
				  cacheTaskStatus(taskId, taskStatus);
				  Collection<SingleSaveErrorResult> importData = new ArrayList<SingleSaveErrorResult>();
				  cacheData(taskId, importData);
				  
				  //清空本地线程的数据，后续业务会填充
				  BizDateThreadLocalUtil.clear();
				  try {
					  Object[] lineData = null;
					  SingleSaveErrorResult saveResult = null;

					  int lineNum = 0;

					  boolean isExistedPublicClue = false;

					  for (Iterator<SingleSaveErrorResult> iter = validateData.iterator(); iter.hasNext(); ) {
						  saveResult = iter.next();
						  try {
							  lineData = saveResult.getLineData();
							  saveResult = processService.saveSingleData(orgId, TianxiaoPCContext.getTXCascadeId() == null ? 0L : TianxiaoPCContext.getTXCascadeId().longValue(), taskStatus.getHeaders(), lineData, taskStatus.isOverride());
							  saveResult.setLineData(lineData);
							  dataMap.put(lineNum + "", saveResult);
							  if (saveResult.isPublicClue()) {
								  isExistedPublicClue = true;
							  }
							  if (saveResult.isSuccess()) {
								  taskStatus.increaseSuccessCount();
							  } else {
								  taskStatus.increaseFailCount();
							  }
						  } catch (DuplicateKeyException e) {
							  log.warn("save data:{} ,is repeat", ToStringBuilder.reflectionToString(lineData));
							  taskStatus.increaseRepeatCount();
						  } catch (Exception e) {
							  log.warn("save data:{} ,catch error:{}", ToStringBuilder.reflectionToString(lineData), e.getMessage());
							  log.error("save data catch error:", e);
							  taskStatus.increaseFailCount();
						  }

						  lineNum++;
						  // 每10次保存一次状态
						  if (lineNum % 10 == 0 && taskStatus.getCompleteCount() < taskStatus.getTotalCount()) {
							  processResultStore(uploadRecord, taskStatus, null, false);
							  cacheTaskStatus(taskId, taskStatus);
						  }
					  }
					  BizDateThreadLocalUtil.clear();
					  if (taskStatus.getCompleteCount() == 0) {
						  taskStatus.setException(new BussinessException(CommonErrorCode.PARAM_ERROR, "文件数据为空"));
					  } else {
						  if (isExistedPublicClue) {
							  processService.afterComplete();
						  }
					  }

					  cacheData(taskId, dataMap.values());
					  processResultStore(uploadRecord, taskStatus, null, true);

				  } catch (Exception e) {
					  log.warn("read data catch error:", e);
					  taskStatus.setException(new BussinessException(CommonErrorCode.BUSINESS_ERROR, "读取文件错误"));
				  } finally {
					  cacheTaskStatus(taskId, taskStatus);
				  }
			  }
		  });
	}
	
	
	
	@Override
	public void downloadImportResult(OutputStream os, Long orgId, String taskId) {
		TaskStatus taskStatus = getTaskStatus(taskId);
    	Collection<SingleSaveErrorResult> data = getData(taskId);
    	if(data==null){
    		try{
    			os.write("文件已失效，下载失败。".getBytes());
    			return;
    		}catch(Exception e){
    		}finally{
    			if(os!=null){
    				try {
						os.close();
					} catch (IOException e) {
					}
    			}
    		}
    	}
    	ImportDataProcessService processService = dataProcessServiceMap.get(taskStatus.getUploadType());
    	processService.downloadImportResult(os, orgId, taskId, data);
	}
    
 
   @Override
   public TaskStatus getTaskStatus(Long orgId, String taskId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(taskId != null, "taskId is illegal");

        TaskStatus taskStatus = getTaskStatus(taskId);
        if (taskStatus == null) {
        	TaskStatus status = new TaskStatus(null, 0);
        	//status.setException(new BussinessException(CommonErrorCode.BUSINESS_ERROR, "进度查询失败"));
        	return status;
        }
        taskStatus.getCompleteRate();
        
        return taskStatus;
   }



    private void processResultStore(TxUploadRecords uploadRecord, TaskStatus taskStatus, String fileUrl, boolean isComplete) {
        // TxUploadRecords uploadRecord = txUploadRecordsDao.getById(taskId);
        uploadRecord.setFailCount(taskStatus.getFailCount());
        uploadRecord.setRepeatCount(taskStatus.getRepeatCount());
        uploadRecord.setSuccessCount(taskStatus.getSuccessCount());
        uploadRecord.setTotalCount(taskStatus.getTotalCount() - 1);
        if (isComplete) {
            String errorFileUrl = fileUrl == null ? "" : fileUrl;
            uploadRecord.setErrorFileUrl(errorFileUrl);
            taskStatus.setErrorFileUrl(errorFileUrl);
            uploadRecord.setStatus(1);
        }
        txUploadRecordsDao.update(uploadRecord);
    }

    private List<String> tranHeader(Object[] headerObjs) {
        Preconditions.checkArgument(ArrayUtils.isNotEmpty(headerObjs), "头信息为空");
        List<String> headers = Lists.newArrayList();
        for (int i=0;i<headerObjs.length;i++) {
            if (headerObjs[i] != null) {
                headers.add(headerObjs[i].toString());
            } else {
                headers.add("-");
            }
        }
        return headers;

    }

    private UploadFileReaderService getUploadFileService(MultipartFile file) {
        UploadFileReaderService service = null;
        String fileName = file.getOriginalFilename();
        if (fileName.toLowerCase().endsWith(".xls") || fileName.toLowerCase().endsWith(".xlsx")) {
            service = new ExcelUploadFileReaderServiceImpl(file);
        } else if (fileName.toLowerCase().endsWith(".csv")) {
            // 默认不使用流的方式读取csv文件
            service = new CsvUploadFileReaderServiceImpl(file, false);
        } else {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, fileName + "文件类型不支持,只支持CSV和excel文件");
        }
        return service;
    }

	


	
}
