package com.baijia.tianxiao.sal.student.impl;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Semaphore;

import com.baijia.tianxiao.dal.enums.CourseTypeEnum;
import com.google.common.collect.Sets;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.constant.LessonStatus;
import com.baijia.tianxiao.dal.constant.ChargeUnit;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgSubAccountDao;
import com.baijia.tianxiao.dal.org.dto.StudentClassHourStatusDocument;
import com.baijia.tianxiao.dal.org.po.OrgClassLesson;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.dal.org.po.OrgStudentCourse;
import com.baijia.tianxiao.dal.org.po.OrgStudentLesson;
import com.baijia.tianxiao.dal.org.po.StudentLessonStatistics;
import com.baijia.tianxiao.dal.org.po.OrgSubAccount;
import com.baijia.tianxiao.dal.solr.enums.StudentLessonStatus;
import com.baijia.tianxiao.dal.solr.po.StudentClass;
import com.baijia.tianxiao.enums.StudentCourseStatus;
import com.baijia.tianxiao.redis.AbstractBaseRedisDao;
import com.baijia.tianxiao.sal.student.api.OrgStudentCourseService;
import com.baijia.tianxiao.sal.student.api.StudentLessonService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.CollectionHelper;
import com.baijia.tianxiao.util.ListUtil;
import com.google.common.collect.Maps;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;

/**
 * Created by liuxp on 16/7/22.
 */
@Service
@Slf4j
public class StudentLessonServiceImpl extends AbstractBaseRedisDao implements StudentLessonService {

    private static final int PAGE_SIZE = 500;
    private static final int UPDATE_TIME_TYPE = 1;
    private static final int START_TIME_TYPE = 2;

    @Autowired
    private OrgStudentDao studentDao;
    @Autowired
    private OrgClassLessonDao classLessonDao;
    @Autowired
    private OrgStudentLessonDao studentLessonDao;
    @Autowired
    private OrgStudentCourseDao orgStudentCourseDao;
    @Autowired
    private OrgSubAccountDao subAccountDao;
    @Autowired
    private OrgCourseDao orgCourseDao;
    @Autowired
    private OrgStudentCourseService courseService;

    private static final Semaphore updateTimeStuLessonStatusSemaphore = new Semaphore(1);
    private static final Semaphore startTimeStuLessonStatusSemaphore = new Semaphore(1);

    private static final BigDecimal TO_CHARGE_VALUE = new BigDecimal("0.2");

    /**
     * 同步排课状态
     *
     * @param beforeMinutes
     */
    @Transactional
    public void syncClassLessonStatus(int beforeMinutes, int type) {
        log.info("[ClassLessonStatus] Start===============");
        long begin = System.currentTimeMillis();
        if (type == UPDATE_TIME_TYPE) {
            boolean isUpdateTimeAllowed = updateTimeStuLessonStatusSemaphore.tryAcquire();
            if (!isUpdateTimeAllowed) {
                log.info("[StudentLesson] Other thread is execute.");
                return;
            }
        } else {
            boolean isStartTimeAllowed = startTimeStuLessonStatusSemaphore.tryAcquire();
            if (!isStartTimeAllowed) {
                log.info("[StudentLesson] Other thread is execute.");
                return;
            }
        }
        try {
            Calendar calendar = Calendar.getInstance();
            calendar.add(Calendar.MINUTE, -1 * beforeMinutes);
            PageDto pageDto = new PageDto();
            pageDto.setPageSize(PAGE_SIZE);
            Date now = new Date();
            List<OrgClassLesson> lessons = null;
            if (type == UPDATE_TIME_TYPE) {
                lessons = classLessonDao.getOrgClassLessonsByUpdateTime(calendar.getTime(), now, pageDto);// 按跟新时间同步数据
            } else {
                lessons = classLessonDao.getOrgClassLessonsByStartTime(calendar.getTime(), now, pageDto);// 按修改时间同步数据
            }
            while (lessons != null && lessons.size() > 0) {
                log.info("[ClassLessonStatus]Executing===============");
                updateClassLessonStatus(lessons);
                pageDto.setPageNum(pageDto.getPageNum() + 1);
                if (type == UPDATE_TIME_TYPE) {
                    lessons = classLessonDao.getOrgClassLessonsByUpdateTime(calendar.getTime(), now, pageDto);
                } else {
                    lessons = classLessonDao.getOrgClassLessonsByStartTime(calendar.getTime(), now, pageDto);
                }
            }
        } finally {
            if (type == UPDATE_TIME_TYPE) {
                updateTimeStuLessonStatusSemaphore.release();
            } else {
                startTimeStuLessonStatusSemaphore.release();
            }
        }
        log.info("[ClassLessonStatus] cost={}", (System.currentTimeMillis() - begin));
    }

    private void updateClassLessonStatus(List<OrgClassLesson> lessons) {
        Date now = new Date();
        Set<Long> unStartIds = new HashSet<>();
        Set<Long> finishedIds = new HashSet<>();
        for (OrgClassLesson lesson : lessons) {
            // 已到开课时间
            if (lesson.getStartTime().compareTo(now) <= 0) {
                finishedIds.add(lesson.getId());
            } else {
                unStartIds.add(lesson.getId());
            }
        }
        if (unStartIds.size() > 0) {
            studentLessonDao.batchUpdateStatus(unStartIds, LessonStatus.UN_START.getStatus());
        }
        if (finishedIds.size() > 0) {
            studentLessonDao.batchUpdateStatus(finishedIds, LessonStatus.FINISHED.getStatus());
        }
    }

    @Transactional
    public void syncStuLessonStatus(int beforeMinutes) {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.MINUTE, -1 * beforeMinutes);
        PageDto pageDto = new PageDto();
        pageDto.setPageSize(PAGE_SIZE);
        Date now = new Date();
        List<OrgStudentLesson> stuLessons =
            studentLessonDao.getStuLessonsByUpdateTime(calendar.getTime(), now, pageDto);
        while (stuLessons != null && stuLessons.size() > 0) {
            log.info("[StudentLesson] Sync by student lesson changed.size={}", stuLessons.size());
            updateLessonStartStatusByUpdateTime(stuLessons);
            pageDto.setPageNum(pageDto.getPageNum() + 1);
            stuLessons = studentLessonDao.getStuLessonsByUpdateTime(calendar.getTime(), now, pageDto);
        }
    }

    private void updateLessonStartStatusByUpdateTime(List<OrgStudentLesson> stuLessons) {
        Set<Long> lessonIds = new HashSet<>();
        for (OrgStudentLesson stuLesson : stuLessons) {
            lessonIds.add(stuLesson.getLessonId());
        }
        List<OrgClassLesson> classLessons = classLessonDao.getByIds(lessonIds);
        Map<Long, OrgClassLesson> classLessonMap = CollectionHelper.toIdMap(classLessons);
        Set<Long> finishedIds = Sets.newHashSet();
        Set<Long> unStartedIds = Sets.newHashSet();
        Date now = new Date();
        for (OrgStudentLesson stuLesson : stuLessons) {
            OrgClassLesson classLesson = classLessonMap.get(stuLesson.getLessonId());
            if (classLesson.getStartTime().compareTo(now) <= 0) {
                finishedIds.add(stuLesson.getId());
            } else {
                unStartedIds.add(stuLesson.getId());
            }
        }
        log.info("[StudentLesson] finishedIds={},unStartedIds={}", finishedIds, unStartedIds);
        if (!finishedIds.isEmpty()) {
            studentLessonDao.batchUpdateStartStatusById(finishedIds, LessonStatus.FINISHED.getStatus());
        }

        if (!unStartedIds.isEmpty()) {
            studentLessonDao.batchUpdateStartStatusById(unStartedIds, LessonStatus.UN_START.getStatus());
        }

    }

    @Override
    @Transactional
    public void syncStudentStatus(long orgId) {
        log.info("[SyncStudentStatus] Start===============,orgId={}", orgId);
        long begin = System.currentTimeMillis();
        PageDto pageDto = new PageDto();
        pageDto.setPageSize(PAGE_SIZE);

        List<StudentLessonStatistics> stuStatisticsList = studentLessonDao.getStudentLessonCounterList(orgId, pageDto);
        List<Long> allCourseIds = ListUtil.toKeyList(stuStatisticsList, "courseId", StudentLessonStatistics.class);
        Map<CourseTypeEnum,Set<Long>> courseIdsMap = courseService.splitCourseIdsByCourseType(allCourseIds);

        Long lastId = null;
        StudentLessonCount lastStuLessonCount = null;
        boolean isToCharge = false;// 是否是待续费学员
        Set<Long> lessonUserIds = new HashSet<>();// 所有有排课记录的用户ID
        while (stuStatisticsList.size() > 0) {
            log.info("[StudentLesson] sync data size={},orgId={}", stuStatisticsList.size(), orgId);
            Set<Long> toChargeIds = new HashSet<>();// 待续费
            Set<Long> pastIds = new HashSet<>();// 往期
            Set<Long> studyingIds = new HashSet<>();// 在读

            Set<Long> unLessonIds = new HashSet<>();// 未排课
            Map<Long, StudentLessonCount> countMap = new HashMap<>();
            if (lastId != null) {
                countMap.put(lastId, lastStuLessonCount);
            }

            Set<Long> userIds = Sets.newHashSet();
            for (StudentLessonStatistics statistics : stuStatisticsList) {
                long userId = statistics.getUserId();
                userIds.add(userId);
                lessonUserIds.add(userId);
            }

            List<StudentClassHourStatusDocument> searchStudentClassHourStatus =
                this.orgStudentCourseDao.searchStudentClassHourStatus(userIds, orgId);
            Map<String, StudentClass> statusMap = this.buildStudentClassStatusInfo(searchStudentClassHourStatus);

            for (StudentLessonStatistics statistics : stuStatisticsList) {
                Long userId = statistics.getUserId();
                Long courseId = statistics.getCourseId();
                String key = String.format("%s_%s", userId, courseId);
                StudentClass sc = statusMap.get(key);
                if (sc != null) {
                    statistics.setStudentLessonStatus(sc.getStatus());
                    if (ChargeUnit.isByTime(sc.getChargeUnit())) {
                        statistics.setTotalCount(Math.max(sc.getContractCount(), statistics.getTotalArrangeTime()));
                        statistics.setFinishedCount(statistics.getKexiaoTime());
                    } else {
                        statistics.setTotalCount(Math.max(sc.getContractCount(), statistics.getTotalCount()));
                    }
                } else {
                    statistics.setStudentLessonStatus(StudentCourseStatus.NORMAL.getCode());
                    statistics.setTotalCount(statistics.getFinishedCount());
                }
            }

            // 统计学员的总课次和已上课次
            for (StudentLessonStatistics counter : stuStatisticsList) {
                long userId = counter.getUserId();
                StudentLessonCount count = countMap.get(userId);
                if (count == null) {
                    count = new StudentLessonCount();
                    countMap.put(counter.getUserId(), count);
                }
                int courseStatus = count.add(counter);
                // 计算待续费
                if (courseStatus == TO_CHARGE_COURSE) {// 待续费
                    toChargeIds.add(counter.getUserId());
                    isToCharge = true;
                } else {
                    isToCharge = false;
                }
                lastId = counter.getUserId();
                lastStuLessonCount = count;
            }

            // 计算往期学员
            for (Long uerId : countMap.keySet()) {
                StudentLessonCount count = countMap.get(uerId);
                if (count.isPast()) {
                    pastIds.add(uerId);
                }
            }

            studyingIds = countMap.keySet();

            Map<Long, List<Long>> stuLessonMap = getStuCourseId(userIds, orgId);
            log.info("[StudentLesson] pass studentId enroll map={}", stuLessonMap);
            // 修订往期学员，即移除以上课时==总课时，但存在未排课的报名记录
            for (Long uerId : pastIds) {
                Set<Long> lessonCourseIds = countMap.get(uerId).getCourseIds();// 所有排课的课程Id
                List<Long> enrollCourseId = stuLessonMap.get(uerId);// 所有报名的课程ID
                if (enrollCourseId == null) {
                    continue;
                }
                enrollCourseId.removeAll(lessonCourseIds);
                if (enrollCourseId.size() > 0) {
                    List<OrgCourse> courses = orgCourseDao.getNormalCourseList(enrollCourseId);
                    if (courses != null && courses.size() > 0) {
                        unLessonIds.add(uerId);
                    }
                }
            }
            log.info("[StudentLesson] Before: toChargeIds={},pastIds={},studyingIds={},lastId={}", toChargeIds, pastIds,
                    studyingIds, lastId);
            // 每页的最后一条数据与下一页数据一起处理
            toChargeIds.remove(lastId);
            pastIds.remove(lastId);
            studyingIds.remove(lastId);

            // 删除往期学员中有未排课课程的学员
            pastIds.removeAll(unLessonIds);
            // 在读等于所有减往期
            studyingIds.removeAll(pastIds);
            // 移除待续费
            studyingIds.removeAll(toChargeIds);
            log.info("[StudentLesson] After: toChargeIds={},pastIds={},studyingIds={},lastId={}", toChargeIds, pastIds,
                studyingIds, lastId);
            if (toChargeIds.size() > 0) {
                studentDao.batchUpdateStudentStatus(orgId, toChargeIds, StudentLessonStatus.TO_CHARGE.getStatus());
            }
            if (pastIds.size() > 0) {
                studentDao.batchUpdateStudentStatus(orgId, pastIds, StudentLessonStatus.PAST.getStatus());
            }
            if (studyingIds.size() > 0) {
                studentDao.batchUpdateStudentStatus(orgId, studyingIds, StudentLessonStatus.STUDYING.getStatus());
            }

            pageDto.setPageNum(pageDto.getPageNum() + 1);
            stuStatisticsList = studentLessonDao.getStudentLessonCounterList(orgId, pageDto);
        }

        // 处理最后一条数据
        if (lastId != null) {
            StudentLessonStatus status = null;
            if (isToCharge) {
                status = StudentLessonStatus.TO_CHARGE;
            } else {
                if (lastStuLessonCount.isHasInStudying()) {
                    status = StudentLessonStatus.STUDYING;
                } else if (lastStuLessonCount.isPast()) {
                    status = StudentLessonStatus.PAST;
                }
            }
            if (status != null) {
                studentDao.batchUpdateStudentStatus(orgId, Arrays.asList(lastId), status.getStatus());
            }
        }

        // 将所有有报名记录，单未标记的用户设置成为在读
        List<OrgStudentCourse> allStuCourses =
            orgStudentCourseDao.getOrgStudentCourseByUserIds(orgId, null, "userId", "status");
        Set<Long> studyingIds = new HashSet<>();
        Set<Long> pastIds = new HashSet<>();
        for (OrgStudentCourse studentCourse : allStuCourses) {// 如果所有班级都为退班，则为往期
            if (lessonUserIds.contains(studentCourse.getUserId())) {// 已有排课记录，不需要特殊处理
                continue;
            }
            if (studentCourse.getStatus() != 0 && !studyingIds.contains(studentCourse.getUserId())) {
                pastIds.add(studentCourse.getUserId());
            } else {
                pastIds.remove(studentCourse.getUserId());
                studyingIds.add(studentCourse.getUserId());
            }
        }
        if (studyingIds.size() > 0) {
            log.info("[StudentLesson] Reset studyingIds={}", studyingIds);
            studentDao.batchUpdateStudentStatus(orgId, studyingIds, StudentLessonStatus.STUDYING.getStatus());
        }
        if (pastIds.size() > 0) {
            log.info("[StudentLesson] Reset pastIds={}", pastIds);
            studentDao.batchUpdateStudentStatus(orgId, pastIds, StudentLessonStatus.PAST.getStatus());
        }

        log.info("[ClassLessonStatus] Org({}) Update finished.cost={}", orgId, (System.currentTimeMillis() - begin));
    }

    private Map<String, StudentClass> buildStudentClassStatusInfo(
        List<StudentClassHourStatusDocument> studentClassHourStatusDocumentLists) {
        Map<String, StudentClass> statusMap = Maps.newHashMap();
        Iterator<StudentClassHourStatusDocument> iterator = studentClassHourStatusDocumentLists.iterator();
        while (iterator.hasNext()) {
            StudentClassHourStatusDocument next = iterator.next();
            Long userId = next.getUserId();
            Long courseId = next.getCourseId();
            Integer lessonCount = next.getLessonCount();
            StudentClass studentClass = new StudentClass();
            String key = userId + "_" + courseId;
            Integer status = next.getStatus();
            studentClass.setStatus(status);
            studentClass.setContractCount(lessonCount == null ? 0 : lessonCount);
            studentClass.setUserId(userId);
            studentClass.setCourseId(courseId);
            studentClass.setChargeUnit(next.getChargeUnit());
            statusMap.put(key, studentClass);
        }
        log.info("statusMap is :{}", statusMap);
        return statusMap;
    }

    private Map<Long, List<Long>> getStuCourseId(Collection<Long> stuIds, Long orgId) {
        if (stuIds == null || stuIds.size() < 1) {
            return Collections.EMPTY_MAP;
        }
        return orgStudentCourseDao.getCourseIdMapByStuIds(stuIds, orgId);
    }

    public static final int DEFAULT_COURSE = 0;
    public static final int IN_STUDYING_COURSE = 1;
    public static final int TO_CHARGE_COURSE = 2;

    @Data
    private class StudentLessonCount {
        private Set<Long> courseIds = new HashSet<>();
        private boolean hasInStudying;
        private boolean hasToCharge;
        private boolean isPast = true;
        private boolean hasCourse = false;

        public int add(StudentLessonStatistics statistics) {
            int retStatus = DEFAULT_COURSE;
            boolean classIsNormal = statistics.isNormalCourse();
            long maxCount = statistics.getTotalCount();
            long finishCount = statistics.getFinishedCount();
            if (classIsNormal && finishCount < maxCount) {
                isPast = false;
                hasInStudying = true;
                retStatus = IN_STUDYING_COURSE;
                if (this.isToCharge(maxCount, maxCount - finishCount)) {
                    this.hasToCharge = true;
                    retStatus = TO_CHARGE_COURSE;
                }
            }
            hasCourse = true;
            courseIds.add(statistics.getCourseId());
            return retStatus;
        }

        public boolean isPast() {
            return this.isPast && this.hasCourse;
        }

        private boolean isToCharge(long totalCount, long leftCount) {
            return this.getLeftCountRatio(totalCount, leftCount).compareTo(TO_CHARGE_VALUE) < 0;
        }

        public BigDecimal getLeftCountRatio(long totalCount, long leftCount) {
            if (leftCount <= 0 || totalCount == 0) {
                return BigDecimal.ZERO;
            }
            BigDecimal total = new BigDecimal(totalCount);
            BigDecimal left = new BigDecimal(leftCount);
            return left.divide(total, 2, BigDecimal.ROUND_HALF_EVEN);
        }

    }

    @Override
    public List<Integer> getAllTxOrgIds() {
        List<OrgSubAccount> accounts = subAccountDao.getAll("orgId");
        List<Integer> list = ListUtil.toKeyList(accounts, "orgId", OrgSubAccount.class);
        return list;
    }
}