package com.baijia.tianxiao.sal.signup.service.Impl;

import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.dal.enums.CourseTypeEnum;
import com.baijia.tianxiao.enums.SignupErrorCode;
import com.baijia.tianxiao.sal.common.api.AccountApiService;
import com.baijia.tianxiao.sal.signup.service.TransferClassOptionService;
import com.baijia.tianxiao.constant.LessonStatus;
import com.baijia.tianxiao.constant.SignStatus;
import com.baijia.tianxiao.constants.UserRoleEnum;
import com.baijia.tianxiao.dal.org.dao.*;
import com.baijia.tianxiao.dal.org.po.*;
import com.baijia.tianxiao.dal.signup.constant.SignupCourseStatus;
import com.baijia.tianxiao.dal.signup.dao.OrgSignupCourseDao;
import com.baijia.tianxiao.dal.signup.dao.OrgSignupCourseLessonDao;
import com.baijia.tianxiao.dal.signup.po.OrgSignupCourse;
import com.baijia.tianxiao.dal.signup.po.OrgSignupCourseLesson;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.exception.ParameterException;
import com.baijia.tianxiao.sal.common.api.CourseApiService;
import com.baijia.tianxiao.sal.common.api.KexiaoApiService;
import com.baijia.tianxiao.sal.common.dto.kexiao.KexiaoSignupCourseStat;
import com.baijia.tianxiao.sal.common.dto.kexiao.KexiaoStatistics;
import com.baijia.tianxiao.sal.signup.dto.TransferClassOption;
import com.baijia.tianxiao.sal.signup.dto.request.TransferClassRequest;
import com.baijia.tianxiao.sal.signup.dto.request.TransferLessonInfoDto;
import com.baijia.tianxiao.util.CollectionHelper;
import com.baijia.tianxiao.util.ListUtil;
import com.baijia.tianxiao.util.NumberUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;

/**
 * Created by lxp on 2017/4/26.
 */
@Service
@Slf4j
public class TransferClassOptionServiceImpl implements TransferClassOptionService {

    @Autowired
    private OrgStudentLessonDao studentLessonDao;
    @Autowired
    private OrgClassLessonDao classLessonDao;
    @Autowired
    private OrgStudentDao studentDao;
    @Autowired
    private OrgSignupCourseLessonDao signupCourseLessonDao;
    @Autowired
    private KexiaoApiService kexiaoApiService;
    @Autowired
    private OrgSignupCourseDao signupCourseDao;
    @Autowired
    private OrgCourseDao courseDao;
    @Autowired
    private OrgLessonSignDao orgLessonSignDao;
    @Autowired
    private CourseApiService courseApiService;
    @Autowired
    private AccountApiService accountApi;
    @Autowired
    private OrgLessonSignDao lessonSignDao;

    @Override
    public TransferClassOption getTransferOption(long orgId, long studentId, long classId) {
        TransferClassOption option = new TransferClassOption();
        OrgStudent student = studentDao.getById(studentId, "userId", "id");
        if (student == null) {
            throw new ParameterException(CommonErrorCode.PARAM_ERROR, "学员不存在或");
        }
        //1.获取统计信息
        TransferClassOption.CourseStat stat = getStat(orgId, classId);
        KexiaoStatistics statistics = kexiaoApiService.queryKexiaoStatByStudentClass(orgId, student.getUserId(), classId);

        //总统计数据
        stat.setTotalNumber(statistics.getBuyNumber());
        stat.setFinishNumber(statistics.getKexiaoNumber());
        stat.setTotalLeftNumber(statistics.getLeftNumber());
        stat.setLeftAmount(statistics.getLeftAmount());

        //总订单剩余可用数据
        stat.setLeftNumber(statistics.getBuyNumber() - statistics.getArrangeNormalNumber());

        option.setCourseStat(stat);
        //2.获取所有可转班的订单
        fillSignupOrders(option, orgId, student.getUserId(), stat.getCourseId());
        //3.获取所有可转班的课节
        fillStudentLessons(option, orgId, student.getUserId(), classId);

        //课节数据
        stat.setLessonNumber(option.getLessons().size());
        return option;
    }

    private TransferClassOption.CourseStat getStat(long orgId, long classId) {
        OrgCourse orgCourse = this.courseDao.getById(classId);
        TransferClassOption.CourseStat stat = new TransferClassOption.CourseStat();
        stat.setCourseType(orgCourse.getCourseType().intValue());
        stat.setCourseName(orgCourse.getName());
        if (CourseTypeEnum.isOneToOne(orgCourse.getCourseType())) {
            stat.setCourseId(orgCourse.getParentId());
        } else {
            stat.setCourseId(orgCourse.getId());
        }
        stat.setClassId(orgCourse.getId());
        stat.setChargeUnit(orgCourse.getChargeUnit());
        stat.setChargeMode(orgCourse.getChargeUnit());
        String name = null;
        if (orgCourse.getCascadeId() != null && orgCourse.getCascadeId() >= 0) {
            name = accountApi.getAccountName(orgId, orgCourse.getCascadeId().longValue());
        }
        if (StringUtils.isBlank(name)) {
            name = "未分配";
        }
        stat.setHeadTeacher(name);
        return stat;
    }

    private void fillSignupOrders(TransferClassOption option, long orgId, long userId, long courseId) {

        List<OrgSignupCourse> signupCourseList = signupCourseDao.getByCourseIdAndStudentId(orgId, userId, courseId, SignupCourseStatus.PAY_SUCCESS);
        Map<Long, OrgSignupCourse> signupCourseMap = CollectionHelper.toIdMap(signupCourseList);

        Map<Long, KexiaoSignupCourseStat> courseStatMap = kexiaoApiService.queryKexiaoStatBySignUpCourseIds(signupCourseList);

        Set<Long> keySet = courseStatMap.keySet();
        for (Long id : keySet) {
            KexiaoSignupCourseStat stat = courseStatMap.get(id);
            long orderLeftNumber = stat.getBuyNumber() - stat.getArrangeNormalNumber();
            long orderLeftAmount = stat.getBuyAmount() - stat.getArrangeNormalAmount();
            if (orderLeftNumber > 0) {
                OrgSignupCourse signupCourse = signupCourseMap.get(id);
                option.addSignupCourse(stat.getPurchaseId(), orderLeftNumber, signupCourse.getCreateTime(), orderLeftAmount);
            }
        }
    }

    private void fillStudentLessons(TransferClassOption option, long orgId, long userId, long classId) {
        List<OrgStudentLesson> lessons = studentLessonDao.getByUserIdAndClassId(orgId, userId, classId, LessonStatus.UN_START.getStatus());
        if (lessons == null || lessons.isEmpty()) {
            return;
        }
        List<Long> lessonIds = ListUtil.toKeyList(lessons, "lessonId", OrgStudentLesson.class);

        Map<String, OrgSignupCourseLesson> signupCourseLessonMap = signupCourseLessonDao.selectLessonInfoMap(orgId, lessonIds, Arrays.asList(userId));

        List<OrgClassLesson> classLessonList = classLessonDao.getByIds(lessonIds);
        Map<Long, OrgClassLesson> classLessonMap = CollectionHelper.toIdMap(classLessonList);
        List<OrgLessonSign> lessonSigns = lessonSignDao.getStudentSign(orgId, userId, UserRole.STUDENT.getRole(), lessonIds);
        Map<Long, OrgLessonSign> signMap = CollectionHelper.toKeyMap(lessonSigns, "lessonId");

        Map<Long, Integer> sortMap = new HashMap<>();
        for (OrgStudentLesson stuLesson : lessons) {
            OrgClassLesson classLesson = classLessonMap.get(stuLesson.getLessonId());
            if (classLesson == null) {
                throw new ParameterException(CommonErrorCode.PARAM_ERROR, "课节不存在");
            }
            OrgLessonSign sign = signMap.get(stuLesson.getLessonId());
            if (sign != null && sign.getStatus() != 0) {
                log.info("[TransferClass] OrgLessonSign={}", sign);
                continue;
            }
            sortMap.put(stuLesson.getLessonId(), classLesson.getNumber());
            OrgSignupCourseLesson signupCourseLesson = signupCourseLessonMap.get(stuLesson.getUserId() + "_" + stuLesson.getLessonId());
            if (signupCourseLesson != null) {
                option.addStudentLesson(stuLesson.getId(), signupCourseLesson.getAmount(), stuLesson.getLessonType(), classLesson);
            }
        }
        if (option.getLessons() != null && option.getLessons().size() > 0) {
            Collections.sort(option.getLessons(), new Comparator<TransferClassOption.StudentLesson>() {
                @Override
                public int compare(TransferClassOption.StudentLesson o1, TransferClassOption.StudentLesson o2) {
                    return sortMap.get(o1.getLessonId()) - sortMap.get(o2.getLessonId());
                }
            });
        }
    }

    @Override
    public boolean checkTransferParam(TransferClassRequest request) {
        if (request.getTransferLessons() == null || request.getTransferLessons().size() < 1) {
            return true;
        }
        OrgStudent student = studentDao.getById(request.getStudentId(), "userId", "id");
        if (student == null) {
            throw new ParameterException(CommonErrorCode.PARAM_ERROR, "学员不存在或");
        }
        long userId = student.getUserId();

        Set<Long> classLessonIds = new HashSet<>();
        Set<Long> purchaseIds = new HashSet<>();

        for (TransferLessonInfoDto req : request.getTransferLessons()) {
            if (req.getLessonId() != null && req.getLessonId() > 0) {
                classLessonIds.add(req.getLessonId());
            }
            if (req.getSignupPurchaseId() != null && req.getSignupPurchaseId() > 0) {
                purchaseIds.add(req.getSignupPurchaseId());
            }
        }

        if (!purchaseIds.isEmpty()) {
            List<OrgSignupCourse> signupCourseList = signupCourseDao.listByPurchaseIds(purchaseIds, request.getClassId());
            Map<Long, OrgSignupCourse> purchaseIdMap = CollectionHelper.toKeyMap(signupCourseList, "signupPurchaseId");
            Map<Long, KexiaoSignupCourseStat> courseStatMap = kexiaoApiService.queryKexiaoStatBySignUpCourseIds(signupCourseList);
            for (TransferLessonInfoDto req : request.getTransferLessons()) {
                OrgSignupCourse signupCourse = purchaseIdMap.get(req.getSignupPurchaseId());
                if (signupCourse != null) {
                    KexiaoSignupCourseStat stat = courseStatMap.get(signupCourse.getId());
                    checkSignupCourse(req, stat);
                }
            }
        }

        if (!classLessonIds.isEmpty()) {
            List<OrgStudentLesson> studentLessons = studentLessonDao.getByLessonIdsAndUserIds(classLessonIds, Arrays.asList(userId), null);
            for (OrgStudentLesson stuLesson : studentLessons) {
                checkStudentLesson(stuLesson);//校验课节的删除状态
            }
            checkLessonSignIn(request.getOrgId(), userId, classLessonIds);//校验签到状态
            checkoutKexiaoStatus(studentLessons);//校验课消状态


        }

        return true;
    }

    @Override
    public void delStudentLessons(long orgId, Collection<Long> lessonIds, Long urserId) {
        if (lessonIds != null && lessonIds.size() > 0) {
            studentLessonDao.delStudentFromLesson(orgId, lessonIds, urserId);
            signupCourseLessonDao.batchDelStuLesson(orgId, Arrays.asList(urserId), lessonIds);
        }
    }

    private void checkSignupCourse(TransferLessonInfoDto req, KexiaoSignupCourseStat stat) {
        if (!SignupCourseStatus.VALIDATE_STATS.contains(stat.getStatus())) {
            throw new BussinessException(SignupErrorCode.TRANSFER_ORDER_INVALIDATE);
        }
        if (stat.getLeftNumber() < req.getLessonCount()) {
            throw new BussinessException(SignupErrorCode.TRANSFER_LEFT_LESSON_NOT_ENOUGH);
        }
        if (stat.getLeftAmount() < NumberUtil.multiply(req.getLessonMoney(), NumberUtil.HUNDRED, 0).longValue()) {
            throw new BussinessException(SignupErrorCode.TRANSFER_LEFT_AMOUNT_NOT_ENOUGH);
        }
    }

    private void checkStudentLesson(OrgStudentLesson studentLesson) {
        if (studentLesson == null || studentLesson.getDelStatus() == 1) {
            throw new BussinessException(SignupErrorCode.TRANSFER_LESSON_DELETED);
        }
    }

    /**
     * 校验签到状态
     */
    private void checkLessonSignIn(long orgId, Long userId, Collection<Long> lessonIds) {
        List<OrgLessonSign> signedLesson =
                this.orgLessonSignDao.getStudentSign(orgId, userId, UserRoleEnum.STUDENT.getCode(), lessonIds);
        for (OrgLessonSign sign : signedLesson) {
            if (SignStatus.OPERATED_STATUS.contains(sign.getStatus())) {
                log.info("[TransferClass] check fail.OrgLessonSign={}", sign);
                throw new BussinessException(SignupErrorCode.TRANSFER_LESSON_INVALIDATE);
            }
        }
    }

    /**
     * 校验课消状态
     */
    private void checkoutKexiaoStatus(List<OrgStudentLesson> lessons) {

        Set<Long> classIds = new HashSet<>();
        for (OrgStudentLesson lesson : lessons) {
            classIds.add(lesson.getCourseId());
        }

        Map<Long, OrgCourseConsumeRule> consumeRuleMap = courseApiService.getClassRule(classIds);

        Set<Long> classLessonIds = new HashSet<>();
        for (OrgStudentLesson lesson : lessons) {
            OrgCourseConsumeRule rule = consumeRuleMap.get(lesson.getCourseId());
            if (rule == null || rule.getRuleValue() == 0) {
                classLessonIds.add(lesson.getLessonId());
            }
        }
        if (!classLessonIds.isEmpty()) {
            List<OrgClassLesson> classLessons = classLessonDao.getByIds(classLessonIds);

            for (OrgClassLesson classLesson : classLessons) {
                if (classLesson.getStartTime().compareTo(new Date()) < 0) {
                    throw new BussinessException(SignupErrorCode.TRANSFER_LESSON_KEXIAO);
                }
            }
        }
    }
}
