
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.room.service.impl;

import com.baijia.tianxiao.constant.InClassStatus;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgClassRoomDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseRoomDao;
import com.baijia.tianxiao.dal.org.dao.OrgLessonConflictDao;
import com.baijia.tianxiao.dal.org.dao.OrgTeacherLessonDao;
import com.baijia.tianxiao.dal.org.po.OrgClassLesson;
import com.baijia.tianxiao.dal.org.po.OrgClassRoom;
import com.baijia.tianxiao.dal.org.po.OrgCourseRoom;
import com.baijia.tianxiao.dal.org.po.OrgLessonConflict;
import com.baijia.tianxiao.dal.org.po.OrgTeacherLesson;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.room.dto.AppClassRoomDto;
import com.baijia.tianxiao.sal.room.dto.ClassRoomDto;
import com.baijia.tianxiao.sal.room.dto.RoomCountDto;
import com.baijia.tianxiao.sal.room.dto.RoomDto;
import com.baijia.tianxiao.sal.room.service.ClassRoomService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.BaseUtils;
import com.baijia.tianxiao.util.HanZiPinYinUtils;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.text.Collator;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

/**
 * @author cxm
 * @version 1.0
 * @title ClassRoomServiceImpl
 * @desc TODO
 * @date 2015年12月24日
 */
@Service
@Slf4j
public class ClassRoomServiceImpl implements ClassRoomService {

    @Resource
    private OrgClassRoomDao orgClassRoomDao;

    @Resource
    private OrgCourseRoomDao orgCourseRoomDao;

    @Resource
    private OrgClassLessonDao orgClassLessonDao;

    @Resource
    private OrgTeacherLessonDao orgTeacherLessonDao;

    @Resource
    private OrgLessonConflictDao orgLessonConflictDao;

    @Override
    public ClassRoomDto saveOrUpdateClassRoom(Long orgId, ClassRoomDto params) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        params.setOrgId(orgId);
        params.validate();
        OrgClassRoom classRoom = ClassRoomDto.converPo(params);
        if (params.getRoomId() != null && params.getRoomId() > 0) {
            orgClassRoomDao.update(classRoom);
        } else {
            if (classRoom.getBranchId() == null) {
                classRoom.setBranchId(0l);
            }
            orgClassRoomDao.save(classRoom);
            params.setRoomId(classRoom.getId());
        }
        OrgClassRoom temp = orgClassRoomDao.getById(params.getRoomId());
        params.setRecycleStatus(temp.getRecycleStatus());
        params.setBranchId(temp.getBranchId());
        params.setCreateTime(temp.getCreateTime());
        params.setUpdateTime(temp.getUpdateTime());
        List<OrgClassLesson> lessons =
                orgClassLessonDao.getStartLessons(orgId, params.getRoomId(), new Date());
        params.setClassRecord(lessons.size()>0?1:0);
        return params;
    }

    @Override
    public void delClassRoom(Long orgId, Long roomId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(roomId != null && roomId > 0, "roomId is illegal");
        List<OrgClassLesson> lessons =
       //     orgClassLessonDao.getLeftRoomIdLessons(orgId, roomId, null, DeleteStatus.NORMAL.getValue());
                orgClassLessonDao.getStartLessons(orgId, roomId, new Date());
        if (CollectionUtils.isNotEmpty(lessons)) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "有上课记录的教室无法清除");
        }

        orgCourseRoomDao.delCourseRoom(orgId, null, Lists.newArrayList(roomId));
        orgClassRoomDao.delLogicallyById(roomId);
    }

    @Override
    public List<ClassRoomDto> listOrgClassRooms(Long orgId, Long branchId, Long courseId, PageDto page) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        List<OrgClassRoom> classRooms = orgClassRoomDao.getClassRoom(orgId, null, branchId, false, DeleteStatus.NORMAL.getValue(),page);
        if (CollectionUtils.isNotEmpty(classRooms)) {
            List<ClassRoomDto> result = Lists.newArrayList();
            Set<Long> courseRoomIds = Sets.newHashSet();
            if (courseId != null && courseId > 0) {
                courseRoomIds.addAll(orgCourseRoomDao.getCourseRoomIds(orgId, courseId));
            }
            for (OrgClassRoom classRoom : classRooms) {
                if (courseRoomIds.contains(classRoom.getId())) {
                    continue;
                }
                result.add(ClassRoomDto.covert2Dto(classRoom));
            }
            return result;
        }
        return Collections.emptyList();
    }

    @Override
    public List<AppClassRoomDto> listOrgClassRoomsWithCourse(String key, Long orgId, Long courseId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        List<OrgClassRoom> classRooms = orgClassRoomDao.getClassRoom(orgId, key, null, false, DeleteStatus.NORMAL.getValue(),null);
        if (CollectionUtils.isNotEmpty(classRooms)) {
            List<AppClassRoomDto> result = Lists.newArrayList();
            Set<Long> courseRoomIds = Sets.newHashSet();
            if (courseId != null && courseId > 0) {
                courseRoomIds.addAll(orgCourseRoomDao.getCourseRoomIds(orgId, courseId));
            }
            for (OrgClassRoom classRoom : classRooms) {

                AppClassRoomDto dto = new AppClassRoomDto();
                if (courseRoomIds.contains(classRoom.getId())) {
                    dto.setChosen(InClassStatus.YES.isCode());
                }
                dto.setInitial(HanZiPinYinUtils.getInitialStr(classRoom.getRoomName()));
                dto.setRoomId(classRoom.getId());
                dto.setRoomName(classRoom.getRoomName());
                dto.setRoomSize(classRoom.getRoomSize());
                result.add(dto);
            }
            return result;
        }
        return Collections.emptyList();
    }

    @Override
    public void addClassRoomCourse(Long orgId, Long courseId, Collection<Long> roomIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        if (CollectionUtils.isEmpty(roomIds)) {
            log.warn("room ids is empty.");
            return;
        }
        log.info("add roomids:{} into course :{},orgId:{}", roomIds, courseId, orgId);
        List<OrgCourseRoom> courseRooms = orgCourseRoomDao.getCourseRoomList(orgId, courseId);
        boolean hasDefault = false;
        if (CollectionUtils.isNotEmpty(courseRooms)) {
            for (OrgCourseRoom courseRoom : courseRooms) {
                roomIds.remove(courseRoom.getRoomId());
                hasDefault |= courseRoom.isDefault();
            }
        }
        hasDefault = !hasDefault;
        if (CollectionUtils.isNotEmpty(roomIds)) {
            List<OrgCourseRoom> saveCourseRooms = Lists.newArrayList();

            List<OrgClassRoom> classRooms = orgClassRoomDao.getByIds(roomIds, "orgId", "id", "delStatus");
            OrgCourseRoom courseRoom = null;
            for (OrgClassRoom room : classRooms) {
                if (!orgId.equals(room.getOrgId()) || room.isDelStatus()) {
                    log.warn("room id :{} is not in current orgId:{}", room.getId(), room.getOrgId());
                    continue;
                }
                courseRoom = new OrgCourseRoom();
                courseRoom.setCourseId(courseId);
                courseRoom.setIsDefault(hasDefault);
                courseRoom.setOrgId(orgId);
                courseRoom.setRoomId(room.getId());
                hasDefault = false;
                saveCourseRooms.add(courseRoom);
            }
            log.debug("save course room :{}", saveCourseRooms);
            orgCourseRoomDao.saveAll(saveCourseRooms, "courseId", "isDefault", "orgId", "roomId");
        }
    }

    @Override
    public void delClassRoomFromCourse(Long orgId, Long courseId, Collection<Long> roomIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        if (CollectionUtils.isEmpty(roomIds)) {
            log.warn("room ids is empty.");
            return;
        }
        orgCourseRoomDao.delCourseRoom(orgId, courseId, roomIds);

        List<Long> lessonIds = orgClassLessonDao.getLeftLessonIds(orgId, courseId);
        if (CollectionUtils.isNotEmpty(lessonIds)) {
            int result = orgClassLessonDao.updateLessonClassRoom(orgId, lessonIds, null);
            log.info("delete class room from lessonIds:{},result:{}", lessonIds, result);
        }
    }

    @Override
    public List<ClassRoomDto> listOrgCourseRooms(Long orgId, Long courseId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        List<OrgCourseRoom> courseRooms = orgCourseRoomDao.getCourseRoomList(orgId, courseId);
        if (CollectionUtils.isEmpty(courseRooms)) {
            return Collections.emptyList();
        }
        Set<Long> roomIds = Sets.newHashSet();
        Long defaultRoomId = null;
        for (OrgCourseRoom courseRoom : courseRooms) {
            roomIds.add(courseRoom.getRoomId());
            if (courseRoom.isDefault()) {
                defaultRoomId = courseRoom.getRoomId();
            }
        }
        List<OrgClassRoom> classRooms = orgClassRoomDao.getByIds(roomIds);

        if (CollectionUtils.isNotEmpty(classRooms)) {
            List<ClassRoomDto> result = Lists.newArrayList();
            for (OrgClassRoom classRoom : classRooms) {
                ClassRoomDto dto = ClassRoomDto.covert2Dto(classRoom);
                if (classRoom.getId().equals(defaultRoomId)) {
                    dto.setIsDefault(1);
                }
                if(classRoom.getRecycleStatus() == DeleteStatus.NORMAL.getValue()){
                    result.add(dto);
                }
            }
            return result;
        }
        return Collections.emptyList();
    }

    @Override
    public void courseRoomSeting(Long orgId, Long courseId, Long roomId, boolean enable) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        Preconditions.checkArgument(roomId != null && roomId > 0, "roomId is illegal");
        log.info("set orgId:{} course:{} default room:{} : {}", orgId, courseId, roomId, enable);
        orgCourseRoomDao.courseDefaultRoomSet(orgId, courseId, roomId, enable);
    }

    @Override
    public ClassRoomDto info(Long orgId, Long roomId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(roomId != null && roomId > 0, "roomId is illegal");
        OrgClassRoom classRoom = orgClassRoomDao.getById(roomId);
        Preconditions.checkArgument(classRoom != null && classRoom.getOrgId().equals(orgId), "教室ID不正确");
        ClassRoomDto dto = ClassRoomDto.covert2Dto(classRoom);
        List<OrgClassLesson> lessons =
                orgClassLessonDao.getStartLessons(orgId, dto.getRoomId(), new Date());
        log.debug("[class record size ]={}",lessons.size());
        dto.setClassRecord(lessons.size()>0?1:0);
        return dto;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveCourseRooms(Long orgId, Long courseId, Collection<Long> roomIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        if (roomIds == null) {
            roomIds = Lists.newArrayList();
        }
        // 当前班级的教室
        List<Long> existRooms = this.orgCourseRoomDao.getCourseRoomIds(orgId, courseId);
        List<Long> tempRooms = Lists.newArrayList(existRooms);
        List<Long> toDelRooms = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(roomIds)) {
            tempRooms.removeAll(roomIds);
        }
        toDelRooms.addAll(tempRooms);

        // 已排课节占用教室
        Map<Long, OrgClassLesson> mapClassLesson = orgClassLessonDao.getClassLessonMap(orgId, courseId, 0, "roomId");
        Set<Long> usedRooms = BaseUtils.getPropertiesList(Lists.newArrayList(mapClassLesson.values()), "roomId");
        for (Long roomId : toDelRooms) {
            if (usedRooms.contains(roomId)) {
                throw new BussinessException(CommonErrorCode.PARAM_ERROR, "已经排课的教室无法删除");
            }
        }
        if (CollectionUtils.isNotEmpty(toDelRooms)) {
            this.orgCourseRoomDao.delCourseRoom(orgId, courseId, toDelRooms);
        }

        roomIds.removeAll(existRooms);
        if (CollectionUtils.isNotEmpty(roomIds)) {
            this.addClassRoomCourse(orgId, courseId, roomIds);
        }
    }

    @Override
    public List<OrgClassRoom> fuzzyQuery(Long orgId, String roomName) {
        List<OrgClassRoom> rooms = this.orgClassRoomDao.getOrgClassRoomByNameAndIds(orgId, null, roomName);
        return rooms;
    }

    @Override
    public List<RoomDto> listPcOrgClassRooms(Long orgId, String key, Long courseId, PageDto pageDto) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "机构id非法");
        // 查询课程已选的教室
        Set<Long> orgCourseRoomIds = Sets.newHashSet();
        if (courseId != null && courseId > 0) {
            List<OrgCourseRoom> orgCourseRooms = orgCourseRoomDao.getCourseRoomList(orgId, courseId);
            orgCourseRoomIds = BaseUtils.getPropertiesList(orgCourseRooms, "roomId");
        }
        List<OrgClassRoom> orgClassRooms = orgClassRoomDao.queryListByParam(orgId, key, orgCourseRoomIds, pageDto);
        List<RoomDto> roomDtos = Lists.newArrayList();
        for (OrgClassRoom orgClassRoom : orgClassRooms) {
            RoomDto roomDto = RoomDto.convertToDto(orgClassRoom);
            roomDtos.add(roomDto);
        }
        Collections.sort(roomDtos, new Comparator() {

            @Override
            public int compare(Object object1, Object object2) {
                RoomDto info1 = (RoomDto) object1;
                RoomDto info2 = (RoomDto) object2;
                // 按bean的名字排序
                return Collator.getInstance(Locale.CHINESE).compare(info1.getName(), info2.getName());
            }

        });
        return roomDtos;
    }

    @Override
    public List<RoomDto> selectClassRoom(Long orgId, List<Long> teacherIds, List<Long> courseIds, Integer type) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "机构id非法");
        // 通过老师id集合查询相关课节
        List<Long> teacherLessonIds = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(teacherIds)) {
            List<OrgTeacherLesson> orgTeacherLessons = orgTeacherLessonDao.queryTeacherLessons(orgId, teacherIds);
            Set<Long> lessonIds = BaseUtils.getPropertiesList(orgTeacherLessons, "lessonId");
            teacherLessonIds.addAll(lessonIds);
        }
        // 查询有冲突的课节
        List<Long> confictLessonIds = Lists.newArrayList();
        if (type == 1) {
            List<OrgLessonConflict> confictLessons = orgLessonConflictDao.queryConfilctByTeacherIds(orgId, teacherIds);
            Set<Long> conflictSet = BaseUtils.getPropertiesList(confictLessons, "lessonId");
            confictLessonIds.addAll(conflictSet);
        }
        // 满足条件的课节信息
        List<OrgClassLesson> orgClassLesson =
            orgClassLessonDao.queryByParams(orgId, teacherLessonIds, confictLessonIds, courseIds);
        Set<Long> roomIds = BaseUtils.getPropertiesList(orgClassLesson, "roomId");
        Map<Long, String> names = orgClassRoomDao.getRoomName(orgId, roomIds);
        return RoomDto.convertToDto(names);
    }

    @Override
    public void deleteClassRoom(Long orgId, Long roomId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(roomId != null && roomId > 0, "roomId is illegal");
        orgClassRoomDao.moveToRecycleBin(roomId);
    }

    @Override
    public void reuseClassRoom(Long orgId, Long roomId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(roomId != null && roomId > 0, "roomId is illegal");
        orgClassRoomDao.removeFromRecycleBin(roomId);
    }

    @Override
    public RoomCountDto getClassRecordCount(Long orgId, Long branchId) {
        List<OrgClassRoom> classRooms = orgClassRoomDao.getOrgClassRoomsList(orgId, branchId, DeleteStatus.DELETED.getValue(), null, null, null);
        log.debug("[org_id]={},[classroom size]={}",orgId,classRooms.size());
        List<Long> roomIds = getAllRoomId(classRooms);
        if(roomIds.size() == 0){
            RoomCountDto dto = new RoomCountDto();
            dto.setAllDeletedCount(0);
            dto.setClassRecordCount(0);
            return dto;
        }
        Map<Long, Integer> map = orgClassLessonDao.getRoomClassRecordCount(orgId,roomIds);
        int count = 0;
        for(Integer i : map.values()){
            if(i > 0){
                count++;
            }
        }
        RoomCountDto dto = new RoomCountDto();
        dto.setAllDeletedCount(classRooms.size());
        dto.setClassRecordCount(count);
        return dto;
    }

    private List<Long> getAllRoomId(List<OrgClassRoom> classRooms) {
        List<Long> roomIds = Lists.newArrayList();
        if(CollectionUtils.isEmpty(classRooms)){
            return roomIds;
        }
        for(OrgClassRoom room : classRooms){
            roomIds.add(room.getId());
        }
        return roomIds;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void clearAllClassRoom(Long orgId, Long branchId) {
        List<OrgClassRoom> classRooms = orgClassRoomDao.getOrgClassRoomsList(orgId, branchId, DeleteStatus.DELETED.getValue(), null, null, null);
        if(CollectionUtils.isEmpty(classRooms)){
            return;
        }
        List<Long> roomIds = getAllRoomId(classRooms);
        Map<Long, Integer> map = orgClassLessonDao.getRoomClassRecordCount(orgId,roomIds);
        //有上课记录的教室不清除
        for(Iterator<Long> it = roomIds.iterator();it.hasNext(); ){
            Long roomId = it.next();
            Integer lessonNum = map.get(roomId);
            if(lessonNum !=null && lessonNum.intValue() > 0){
                it.remove();
            }
        }
        orgCourseRoomDao.delCourseRoom(orgId, null, Lists.newArrayList(roomIds));
        for(Long roomId : roomIds){
            orgClassRoomDao.delLogicallyById(roomId);
        }
    }

    @Override
    public List<Long> getRecycleRoomIdsByCourseId(Long orgId, Long courseId) {
        List<OrgCourseRoom> courseRooms = orgCourseRoomDao.getCourseRoomList(orgId, courseId);
        if (CollectionUtils.isEmpty(courseRooms)) {
            return Collections.EMPTY_LIST;
        }
        Set<Long> roomIds = Sets.newHashSet();
        for (OrgCourseRoom courseRoom : courseRooms) {
            roomIds.add(courseRoom.getRoomId());
        }
        //删除了的教室
        List<Long> recycleClassRoomIds = orgClassRoomDao.getRecycleRoomId(orgId,roomIds);
        if (CollectionUtils.isEmpty(recycleClassRoomIds)) {
            return Collections.EMPTY_LIST;
        }
        return recycleClassRoomIds;
    }
}
