package com.baijia.tianxiao.sal.push.service.impl;

import com.baijia.tianxiao.constants.TianXiaoConstant;
import com.baijia.tianxiao.dal.org.dao.OrgSubAccountDao;
import com.baijia.tianxiao.dal.org.dao.TXCascadeAccountDao;
import com.baijia.tianxiao.dal.org.po.OrgSubAccount;
import com.baijia.tianxiao.dal.org.po.TXCascadeAccount;
import com.baijia.tianxiao.dal.pcAuthority.dao.TxAccountPermissionDao;
import com.baijia.tianxiao.dal.pcAuthority.po.TxAccountPermission;
import com.baijia.tianxiao.dal.push.constant.MsgSettingStatus;
import com.baijia.tianxiao.dal.push.constant.MsgUserRole;
import com.baijia.tianxiao.dal.push.dao.AccountMsgSettingDao;
import com.baijia.tianxiao.dal.push.po.AccountMsgSetting;
import com.baijia.tianxiao.enums.CrmErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.redis.AbstractBaseRedisDao;
import com.baijia.tianxiao.sal.push.dto.ConsultAvatarUrlAndNameDto;
import com.baijia.tianxiao.sal.push.service.ConsultAvatarUrlService;
import com.baijia.tianxiao.sal.push.service.PushRedisService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by liuxp on 16/5/29.
 */
@Slf4j
@Service
public class PushRedisServiceImpl extends AbstractBaseRedisDao<String, Object> implements PushRedisService {

    private static final String LAST_PUSH_NO_HASH = "TX#LAST#PUSH#NO#HASH";
    private static final String CONSULT_USER_AVATAR_HASH = "TX#CONSULT#USER#AVATAR#HASH";
    private static final String ORG_ACCOUNT_TYPE = "TX#ORG#ACCOUNT#TYPE#HASH";

    @Autowired
    private TXCascadeAccountDao cascadeAccountDao;
    @Autowired
    private OrgSubAccountDao orgSubAccountDao;
    @Autowired
    private AccountMsgSettingDao accountMsgSettingDao;
    @Autowired
    private TxAccountPermissionDao permissionDao;
    @Autowired
    private ConsultAvatarUrlService avatarUrlService;

    @Override
    public Integer getConsultUserCount(final Long orgId) {
        return redisTemplate.execute(new RedisCallback<Integer>() {
            @Override
            public Integer doInRedis(RedisConnection connection) throws DataAccessException {
                log.info("[Redis] Query start.");
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] key = serializer.serialize(LAST_PUSH_NO_HASH);
                byte[] field = serializer.serialize(orgId.toString());

                byte[] value = connection.hGet(key, field);
                log.info("[Redis] Query success===");
                if (value == null) {
                    return 0;
                } else {
                    int result = Integer.parseInt(serializer.deserialize(value));
                    log.info("[Redis] Query Result =" + result);
                    return result;
                }
            }
        });
    }

    @Override
    public void addConsultUserCount(final Long orgId) {
        redisTemplate.execute(new RedisCallback<Object>() {
            @Override
            public Object doInRedis(RedisConnection connection) throws DataAccessException {
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] key = serializer.serialize(LAST_PUSH_NO_HASH);
                byte[] field = serializer.serialize(orgId.toString());
                connection.hIncrBy(key, field, 1);
                log.info("[Redis] Update success. orgId=" + orgId);
                return null;
            }
        });
    }

    @Override
    public List<Integer> getCascadeIds(Long orgId) {
        List<Integer> ids = new ArrayList<>();
        List<TXCascadeAccount> cascadeAccounts = cascadeAccountDao.getTXCascadeAccountListByOrgId(orgId);

        if ((cascadeAccounts != null) && (cascadeAccounts.size() > 0)) {
            for (TXCascadeAccount account : cascadeAccounts) {
                ids.add(account.getId());
            }

            // 105代表咨询权限，参考TXPermissionConst
            List<TxAccountPermission> permissions = permissionDao.selectPermission(ids, 105L);
            log.info("[Switch] permissions cascadeIds=" + permissions);
            if (permissions != null) {
                for (TxAccountPermission tp : permissions) {
                    if (tp.getPType() == 1) {
                        log.info("[Switch] No permission,cascadeId=" + tp.getUId());
                        ids.remove(tp.getUId());
                    }
                }
            }

            ids.add(0, 0);

            List<AccountMsgSetting> settings = accountMsgSettingDao.getAccountMsgSettingList(orgId, ids);
            List<Integer> closeIds = new ArrayList<>();

            for (AccountMsgSetting setting : settings) {
                if (setting.getMsgSwitch() == MsgSettingStatus.CLOSE.getValue()) {
                    closeIds.add(setting.getCascadeId());
                }
            }

            log.info("[Switch] all cascadeIds=" + ids);
            log.info("[Switch] close cascadeIds=" + closeIds);
            ids.removeAll(closeIds);
            return ids;
        }
        return ids;
    }

    @Override
    public void setOrgAccountType(final Long orgId, final Integer cascadeId, final int type) {
        redisTemplate.execute(new RedisCallback<Object>() {
            @Override
            public Object doInRedis(RedisConnection connection) throws DataAccessException {
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] key = serializer.serialize(ORG_ACCOUNT_TYPE);
                byte[] field = serializer.serialize(getKey(orgId, cascadeId));
                connection.hIncrBy(key, field, type);
                log.info("[Redis] Update success. orgId={},cascadeId={},type={}", orgId, cascadeId, type);
                return null;
            }
        });
    }

    @Override
    public int getOrgAccountType(final Long orgId, final Integer cascadeId) {

        Integer accountType = redisTemplate.execute(new RedisCallback<Integer>() {
            @Override
            public Integer doInRedis(RedisConnection connection) throws DataAccessException {
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] key = serializer.serialize(ORG_ACCOUNT_TYPE);
                byte[] field = serializer.serialize(getKey(orgId, cascadeId));
                RedisSerializer redisSerializer = redisTemplate.getDefaultSerializer();
                byte[] value = connection.hGet(key, field);
                log.info("[Redis] Query success.");
                if (value != null) {
                    return null;
                } else {
                    return (Integer) redisSerializer.deserialize(value);
                }
            }
        });

        if (accountType == null) {
            accountType = getAccountTypeFromDb(orgId, cascadeId);
            setOrgAccountType(orgId, cascadeId, accountType);
        }

        return accountType;
    }

    private int getAccountTypeFromDb(Long orgId, Integer cascadeId) {
        if ((cascadeId == null) || (cascadeId == 0)) {
            OrgSubAccount subAccount = orgSubAccountDao.getMasterByOrgId(orgId.intValue());
            log.debug("OrgId=" + orgId + ";OrgSubAccount=" + subAccount);
            if (subAccount != null) {
                return MsgUserRole.HEADER.getValue();
            } else {
                return MsgUserRole.BRANCH_HEADER.getValue();
            }
        } else {
            TXCascadeAccount cascadeAccount = cascadeAccountDao.getById(cascadeId);
            if (!MsgUserRole.isSubOrg(cascadeAccount.getAccountType())) {
                throw new BussinessException(CrmErrorCode.ORG_ACCOUNT_TYPE_ERROR);
            }
            return cascadeAccount.getAccountType();
        }
    }

    private String getKey(Long orgId, Integer cascadeId) {
        if (cascadeId == null) {
            cascadeId = 0;
        }
        return orgId + "_" + cascadeId;
    }

    @Override
    public void setAvatar(final long consultUserId, final String url) {
        try {
            redisTemplate.execute(new RedisCallback<Object>() {
                @Override
                public Object doInRedis(RedisConnection connection) throws DataAccessException {
                    connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                    RedisSerializer<String> serializer = getRedisSerializer();
                    byte[] key = serializer.serialize(CONSULT_USER_AVATAR_HASH);
                    byte[] field = serializer.serialize(String.valueOf(consultUserId));
                    byte[] value = serializer.serialize(url);
                    connection.hSet(key, field, value);
                    return null;
                }
            });
        } catch (Exception e) {
            log.error("[Redis] Exception {}", e);
        }
    }

    @Override
    public void setAvatar(final long consultUserId) {
        try {
            ConsultAvatarUrlAndNameDto dto = avatarUrlService.getConsultAvatarUrl(consultUserId);
            setAvatar(consultUserId, dto.getAvatarUrl());
        } catch (Exception e) {
            log.error("[Redis] Exception {}", e);
        }

    }

    @Override
    public String getAvatar(final long consultUserId) {
        String avatar = null;
        try {
            avatar = redisTemplate.execute(new RedisCallback<String>() {
                @Override
                public String doInRedis(RedisConnection connection) throws DataAccessException {
                    connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                    RedisSerializer<String> serializer = getRedisSerializer();
                    byte[] key = serializer.serialize(ORG_ACCOUNT_TYPE);
                    byte[] field = serializer.serialize(String.valueOf(consultUserId));
                    byte[] value = connection.hGet(key, field);
                    if (value != null) {
                        return serializer.deserialize(value);
                    } else {
                        return null;
                    }
                }
            });
        } catch (Exception e) {
            log.error("[Redis] Exception {}", e);
        }

        return avatar;
    }
}
