package com.baijia.tianxiao.sal.organization.finance.service;

import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.excel.dto.ExportField;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.organization.finance.dto.response.TxFinanceExcelDto;
import com.baijia.tianxiao.sal.organization.finance.dto.response.TxFinanceInfoDto;
import com.baijia.tianxiao.sal.organization.finance.dto.response.TxFinanceRecordDto;
import com.baijia.tianxiao.sal.organization.finance.dto.response.TxTypeRecordDto;
import com.baijia.tianxiao.util.ArithUtil;
import com.baijia.tianxiao.util.date.DateUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.beanutils.PropertyUtilsBean;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.ss.util.RegionUtil;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * Created with IntelliJ IDEA.
 * User: Victor Weng
 * Date: 16/7/14
 * Time: 下午5:46
 * To change this template use File | Settings | File Templates.
 */
@Slf4j
public class ExcelFinanceExportService {

    private static String HEADER_FORMAT = "@";

    private static String LEFT_FORMAT = "@ ";

    public static final String EXPORT_DATE_FORMAT = "yyyy-MM-dd HH:mm:ss";

    public static final String UNIT_LESSON_MONEY_FORMULA = "IF(E%d>0,E%d*D%d/60,0)";

    public static final String LESSON_MONEY_FORMULA = "SUM(F%d:F%d)";

    public static PropertyUtilsBean propertyUtilsBean = new PropertyUtilsBean();

    private static CellStyle createCellStyle(Workbook workBook, String format) {
        CellStyle cellStyle = workBook.createCellStyle();
        DataFormat dataFormat = workBook.createDataFormat();
        cellStyle.setAlignment(CellStyle.ALIGN_CENTER);
        cellStyle.setVerticalAlignment(CellStyle.VERTICAL_CENTER);
        cellStyle.setDataFormat(dataFormat.getFormat(format));

        cellStyle.setBorderBottom(CellStyle.BORDER_THIN);
        cellStyle.setBorderLeft(CellStyle.BORDER_THIN);
        cellStyle.setBorderRight(CellStyle.BORDER_THIN);
        cellStyle.setBorderTop(CellStyle.BORDER_THIN);

        return cellStyle;
    }

    private static CellStyle createLeftStyle(Workbook workBook, String format) {
        CellStyle cellStyle = workBook.createCellStyle();
        DataFormat dataFormat = workBook.createDataFormat();
        cellStyle.setAlignment(CellStyle.ALIGN_LEFT);
        cellStyle.setVerticalAlignment(CellStyle.VERTICAL_CENTER);
        cellStyle.setDataFormat(dataFormat.getFormat(format));

        cellStyle.setBorderBottom(CellStyle.BORDER_THIN);
        cellStyle.setBorderLeft(CellStyle.BORDER_THIN);
        cellStyle.setBorderRight(CellStyle.BORDER_THIN);
        cellStyle.setBorderTop(CellStyle.BORDER_THIN);

        return cellStyle;
    }


    public static Sheet createFinanceRecordSheet(Workbook workbook, String sheetName, List<String> titles,
                                                 Map<String, CellStyle> cellStyleMap, List<ExportField> exportFields, List<TxFinanceRecordDto> recordDtos) {

        Sheet sheet = workbook.createSheet(sheetName);
        int columnSize = 0;//总列数
        for (ExportField field : exportFields) {
            columnSize += field.getExportFieldSize();
        }
        createLeftRow(sheet, cellStyleMap, titles, columnSize);
        int row = titles != null ? titles.size() : 0;
        createHeaderRow(sheet, cellStyleMap, exportFields, row++);
        Map<Integer, TxFinanceExcelDto> map = Maps.newHashMap();

        for (TxFinanceRecordDto dto : recordDtos) {
            for (TxTypeRecordDto incomeDto : dto.getIncomeList()) {
                TxFinanceExcelDto financeExcelDto = map.get(incomeDto.getType().intValue());
                if (financeExcelDto == null) {
                    financeExcelDto = new TxFinanceExcelDto();
                }

                financeExcelDto.setType(incomeDto.getType());
                financeExcelDto.setTypeStr(incomeDto.getTypeStr());
                financeExcelDto.setIncomeSum(financeExcelDto.getIncomeSum() + incomeDto.getTypeSum());
                financeExcelDto.setIncomeNum(financeExcelDto.getIncomeNum() + incomeDto.getTypeNum());
                financeExcelDto.setAllSum(financeExcelDto.getAllSum() + incomeDto.getTypeSum());
                map.put(financeExcelDto.getType().intValue(), financeExcelDto);
            }
            for (TxTypeRecordDto expendDto : dto.getExpendList()) {
                TxFinanceExcelDto financeExcelDto = map.get(expendDto.getType().intValue());
                if (financeExcelDto == null) {
                    financeExcelDto = new TxFinanceExcelDto();
                }

                financeExcelDto.setType(expendDto.getType());
                financeExcelDto.setTypeStr(expendDto.getTypeStr());
                financeExcelDto.setExpendSum(financeExcelDto.getExpendSum() - expendDto.getTypeSum());
                financeExcelDto.setExpendNum(financeExcelDto.getExpendNum() + expendDto.getTypeNum());
                financeExcelDto.setAllSum(financeExcelDto.getAllSum() - expendDto.getTypeSum());
                map.put(financeExcelDto.getType().intValue(), financeExcelDto);
            }
        }
        int allIncomeNum = 0;
        double allIncomeSum = 0.00;
        int allExpendNum = 0;
        double allExpendSum = 0.00;
        double aaaSum = 0.00;

        for (TxFinanceExcelDto dto : map.values()) {
            Row dataRow = sheet.createRow(row++);
            createCell(dataRow, 0, dto.getTypeStr(), cellStyleMap, exportFields.get(0));
            createCell(dataRow, 1, dto.getIncomeNum(), cellStyleMap, exportFields.get(1));
            createCell(dataRow, 2, ArithUtil.round(dto.getIncomeSum(), 2), cellStyleMap, exportFields.get(2));
            createCell(dataRow, 3, dto.getExpendNum(), cellStyleMap, exportFields.get(3));
            createCell(dataRow, 4, ArithUtil.round(dto.getExpendSum(), 2), cellStyleMap, exportFields.get(4));
            createCell(dataRow, 5, ArithUtil.round(dto.getAllSum(), 2), cellStyleMap, exportFields.get(5));
            allIncomeNum += dto.getIncomeNum();
            allIncomeSum += dto.getIncomeSum();
            allExpendNum += dto.getExpendNum();
            allExpendSum += dto.getExpendSum();
            aaaSum += dto.getAllSum();
        }

        Row dataRow = sheet.createRow(row++);
        createCell(dataRow, 0, "合计", cellStyleMap, exportFields.get(0));
        createCell(dataRow, 1, allIncomeNum, cellStyleMap, exportFields.get(1));
        createCell(dataRow, 2, ArithUtil.round(allIncomeSum, 2), cellStyleMap, exportFields.get(2));
        createCell(dataRow, 3, allExpendNum, cellStyleMap, exportFields.get(3));
        createCell(dataRow, 4, ArithUtil.round(allExpendSum, 2), cellStyleMap, exportFields.get(4));
        createCell(dataRow, 5, ArithUtil.round(aaaSum, 2), cellStyleMap, exportFields.get(5));


        List<String> tail = Lists.newArrayList();
        tail.add("#----------------------------------------账务汇总列表结束-------------------------------------");
        tail.add("#导出时间：[" + DateUtil.getAllDayStr(new Date()) + "]");

        createTailRow(row, sheet, cellStyleMap, tail, columnSize);
        return sheet;
    }


    public static Sheet createFinanceInfoSheet(Workbook workbook, String sheetName, List<String> titles,
                                               Map<String, CellStyle> cellStyleMap, List<ExportField> exportFields, List<TxFinanceInfoDto> infoDtos) {
        Sheet sheet = workbook.createSheet(sheetName);
        int columnSize = 0;//总列数
        for (ExportField field : exportFields) {
            columnSize += field.getExportFieldSize();
        }
        createLeftRow(sheet, cellStyleMap, titles, columnSize);

        int row = titles != null ? titles.size() : 0;
        createHeaderRow(sheet, cellStyleMap, exportFields, row++);
        int incomeNum = 0;
        double incomeSum = 0.00;
        int expendNum = 0;
        double expendSum = 0.00;
        for (int n = infoDtos.size() - 1; n >= 0; n--) {
            TxFinanceInfoDto dto = infoDtos.get(n);
            Row infoRow = sheet.createRow(row);
            createCell(infoRow, 0, dto.getPurchaseId(), cellStyleMap, exportFields.get(0));
            createCell(infoRow, 1, dto.getOpTypeStr(), cellStyleMap, exportFields.get(1));
            createCell(infoRow, 2, dto.getOpInfo(), cellStyleMap, exportFields.get(2));
            createCell(infoRow, 3, DateUtil.getAllDayStr(dto.getCreateTime()), cellStyleMap, exportFields.get(3));
            createCell(infoRow, 4, dto.getOpTo() != 2 ? ArithUtil.round(dto.getOpMoney(), 2) : "", cellStyleMap, exportFields.get(4));
            createCell(infoRow, 5, dto.getOpTo() == 2 ? ArithUtil.round(dto.getOpMoney(), 2) : "", cellStyleMap, exportFields.get(5));
            createCell(infoRow, 6, dto.getPayTypeStr(), cellStyleMap, exportFields.get(6));
            createCell(infoRow, 7, ArithUtil.round(dto.getBalance(), 2), cellStyleMap, exportFields.get(7));
            if (dto.getOpTo() == 2) {
                expendNum++;
                expendSum += dto.getOpMoney();
            } else {
                incomeNum++;
                incomeSum += dto.getOpMoney();
            }
            row++;
        }

        List<String> tail = Lists.newArrayList();
        tail.add("#-----------------------------------------业务明细列表结束------------------------------------");
        tail.add("#收入合计：" + incomeNum + "笔，收入金额共" + ArithUtil.round(incomeSum, 2) + "元");
        tail.add("#支出合计：" + expendNum + "笔，支出金额共" + ArithUtil.round(expendSum, 2) + "元");
        tail.add("#导出时间：[" + DateUtil.getAllDayStr(new Date()) + "]");

        createTailRow(row, sheet, cellStyleMap, tail, columnSize);

        return sheet;
    }

    private static void mergeCell(Sheet sheet, int firstRow, int lastRow, int firstColumn, int lastColumn) {
        CellRangeAddress rangeAddress = new CellRangeAddress(firstRow, lastRow, firstColumn, lastColumn);
        sheet.addMergedRegion(rangeAddress);
        RegionUtil.setBorderBottom(CellStyle.BORDER_THIN, rangeAddress, sheet, sheet.getWorkbook());
        RegionUtil.setBorderTop(CellStyle.BORDER_THIN, rangeAddress, sheet, sheet.getWorkbook());
        RegionUtil.setBorderLeft(CellStyle.BORDER_THIN, rangeAddress, sheet, sheet.getWorkbook());
        RegionUtil.setBorderRight(CellStyle.BORDER_THIN, rangeAddress, sheet, sheet.getWorkbook());
    }


    private static Cell createCell(Row row, int column, Object value, Map<String, CellStyle> cellStyleMap,
                                   ExportField exportField) {
        Cell cell = row.createCell(column);
        String localFormat = null;
        value = value != null ? value : exportField.getDefaultVal();

        if (value != null) {
            if (value instanceof Date) {
                row.getSheet().setColumnWidth(cell.getColumnIndex(), 20 * 256);
                localFormat = EXPORT_DATE_FORMAT;
                cell.setCellValue((Date) value);
            } else {
                if (value instanceof Number && (((Number) value).longValue() < Integer.MAX_VALUE)) {
                    cell.setCellType(Cell.CELL_TYPE_NUMERIC);
                    if (value instanceof Double) {
                        localFormat = "0.00";
                        cell.setCellValue(((Number) value).doubleValue());
                    } else {
                        localFormat = "0";
                        cell.setCellValue(((Number) value).longValue());
                    }
                } else {
                    cell.setCellType(Cell.CELL_TYPE_STRING);
                    cell.setCellValue(value.toString());
                    localFormat = "@";
                }
            }
        } else {
            cell.setCellValue("");
        }
        cell.setCellStyle(getCellStyle(row.getSheet().getWorkbook(), cellStyleMap,
                exportField.getFormat() != null ? exportField.getFormat() : localFormat));
        return cell;
    }

    private static CellStyle getCellStyle(Workbook workbook, Map<String, CellStyle> cellStyleMap, String format) {
        if (!cellStyleMap.containsKey(format)) {
            cellStyleMap.put(format, createCellStyle(workbook, format));
        }
        return cellStyleMap.get(format);
    }

    private static CellStyle getLeftCellStyle(Workbook workbook, Map<String, CellStyle> cellStyleMap, String format) {
        if (!cellStyleMap.containsKey(format)) {
            cellStyleMap.put(format, createLeftStyle(workbook, format));
        }
        return cellStyleMap.get(format);
    }

    /**
     * 递归创建出行头
     *
     * @param sheet
     * @param cellStyleMap
     * @param exportFields
     * @param row
     */
    private static void createHeaderRow(Sheet sheet, Map<String, CellStyle> cellStyleMap,
                                        List<ExportField> exportFields, int row) {
        Row headerRow = sheet.createRow(row);
        int column = 0;
        CellStyle style = getCellStyle(sheet.getWorkbook(), cellStyleMap, HEADER_FORMAT);
        for (ExportField field : exportFields) {
            column = createHeaderCell(headerRow, style, field, column);
        }
    }

    private static int createHeaderCell(Row row, CellStyle style, ExportField field, int column) {
        if (CollectionUtils.isEmpty(field.getChildren())) {
            row.getSheet().setColumnWidth(column, field.getWidth());
            Cell cell = row.createCell(column++);
            cell.setCellStyle(style);
            cell.setCellValue(field.getName());
        } else {
            for (ExportField child : field.getChildren()) {
                column = createHeaderCell(row, style, child, column);
            }
        }
        return column;

    }

    private static void createLeftRow(Sheet sheet, Map<String, CellStyle> cellStyleMap, List<String> titles,
                                      int headerSize) {
        int row = 0;
        // 补充头
        if (CollectionUtils.isNotEmpty(titles)) {
            for (String title : titles) {
                Row titleRow = sheet.createRow(row);
                Cell cell = titleRow.createCell(0);
                cell.setCellValue(title);
                cell.setCellStyle(getLeftCellStyle(sheet.getWorkbook(), cellStyleMap, LEFT_FORMAT));
                mergeCell(sheet, row, row, 0, headerSize - 1);
                row++;
            }
        }
    }

    private static void createTailRow(int row, Sheet sheet, Map<String, CellStyle> cellStyleMap, List<String> titles,
                                      int headerSize) {
        // 补充头
        if (CollectionUtils.isNotEmpty(titles)) {
            for (String title : titles) {
                Row titleRow = sheet.createRow(row);
                Cell cell = titleRow.createCell(0);
                cell.setCellValue(title);
                cell.setCellStyle(getLeftCellStyle(sheet.getWorkbook(), cellStyleMap, LEFT_FORMAT));
                mergeCell(sheet, row, row, 0, headerSize - 1);
                row++;
            }
        }
    }

    public static void exportExcel(HttpServletResponse response, Workbook workBook, String excelFileName) {
        if (StringUtils.isBlank(excelFileName)) {
            excelFileName = System.currentTimeMillis() + ".xlsx";
        }
        response.setContentType("application/vnd.ms-excel");
        try {
            response.setHeader("Content-Disposition",
                    "attachment; filename=" + URLEncoder.encode(excelFileName, "utf-8"));
        } catch (UnsupportedEncodingException e1) {
            response.setHeader("Content-Disposition", "attachment; filename=" + excelFileName);
        }

        try {
            workBook.write(response.getOutputStream());
            response.getOutputStream().flush();
        } catch (IOException e) {
            log.error("export data catch error:", e);
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "导出数据处理失败");
        }
    }

}
