
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.sal.organization.todo.service.impl;

import com.baijia.tianxiao.constant.BackLogType;
import com.baijia.tianxiao.constant.Flag;
import com.baijia.tianxiao.constants.org.BizConf;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgCrontabTaskDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgCrontabTaskLog;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.push.constant.NoticeType;
import com.baijia.tianxiao.dal.push.dto.content.NoticeMsgContent;
import com.baijia.tianxiao.dal.push.utils.ActionUtil;
import com.baijia.tianxiao.dal.roster.dao.TxConsultUserDao;
import com.baijia.tianxiao.dal.roster.po.TxConsultUser;
import com.baijia.tianxiao.dal.todo.constant.BacklogOperation;
import com.baijia.tianxiao.dal.todo.constant.BacklogStatus;
import com.baijia.tianxiao.dal.todo.dao.TxBacklogDao;
import com.baijia.tianxiao.dal.todo.dao.TxBacklogOperationLogDao;
import com.baijia.tianxiao.dal.todo.dao.TxBacklogParticipantDao;
import com.baijia.tianxiao.dal.todo.po.TxBacklog;
import com.baijia.tianxiao.dal.todo.po.TxBacklogOperationLog;
import com.baijia.tianxiao.dal.todo.po.TxbacklogParticipant;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.exception.ParameterException;
import com.baijia.tianxiao.exception.PermissionException;
import com.baijia.tianxiao.filter.TianxiaoMContext;
import com.baijia.tianxiao.sal.common.api.AccountApiService;
import com.baijia.tianxiao.sal.common.api.TXStudentCommentAPIService;
import com.baijia.tianxiao.sal.organization.org.service.impl.RequestSourceDesc;
import com.baijia.tianxiao.sal.organization.todo.constant.GroupType;
import com.baijia.tianxiao.sal.organization.todo.constant.RelatedStudentType;
import com.baijia.tianxiao.sal.organization.todo.constant.RemindType;
import com.baijia.tianxiao.sal.organization.todo.constant.TaskStatus;
import com.baijia.tianxiao.sal.organization.todo.constant.TodoErrorCode;
import com.baijia.tianxiao.sal.organization.todo.constant.param.HomepageParamExpired;
import com.baijia.tianxiao.sal.organization.todo.constant.param.HomepageParamUnexpired;
import com.baijia.tianxiao.sal.organization.todo.dto.BackLogHomePageDto;
import com.baijia.tianxiao.sal.organization.todo.dto.BackLogHomePageListDto;
import com.baijia.tianxiao.sal.organization.todo.dto.BacklogAccountDto;
import com.baijia.tianxiao.sal.organization.todo.dto.BacklogDto;
import com.baijia.tianxiao.sal.organization.todo.dto.BacklogListResponseDto;
import com.baijia.tianxiao.sal.organization.todo.dto.RelatedStudent;
import com.baijia.tianxiao.sal.organization.todo.dto.SaveBacklogDto;
import com.baijia.tianxiao.sal.organization.todo.service.TxBacklogService;
import com.baijia.tianxiao.sal.organization.utils.DataAuthority;
import com.baijia.tianxiao.sal.push.dto.ConsultAvatarUrlAndNameDto;
import com.baijia.tianxiao.sal.push.dto.PushConfig;
import com.baijia.tianxiao.sal.push.service.ConsultAvatarUrlService;
import com.baijia.tianxiao.sal.push.service.ConsultMessageService;
import com.baijia.tianxiao.sal.push.utils.NativeUrlUtil;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.date.DateUtil;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import lombok.extern.slf4j.Slf4j;

/**
 * @title TxBacklogServiceImpl
 * @desc TODO
 * @author zhangbing
 * @date 2016年2月25日
 * @version 1.0
 */

@Service
@Slf4j
public class TxBacklogServiceImpl implements TxBacklogService {

    // 待办事项
    private static final String BACKLOG_ACTION = "backlog_info";

    @Autowired
    private OrgAccountDao orgAccountDao;
    @Autowired
    private TxBacklogDao txBacklogDao;
    @Autowired
    private TxConsultUserDao txConsultUserDao;
    @Autowired
    private OrgStudentDao orgStudentDao;
    @Autowired
    private OrgCrontabTaskDao orgCrontabTaskDao;
    @Autowired
    private TxBacklogParticipantDao txBacklogParticipantDao;
    @Autowired
    private TxBacklogOperationLogDao txBacklogOperationLogDao;
    
    @Autowired
    private TxConsultUserDao consultUserDao;
    @Autowired
    private ConsultMessageService messageService;
    @Autowired
    private ConsultAvatarUrlService consultAvatarUrlService;
    @Autowired
    private AccountApiService accountApiService;
    @Autowired
    private TXStudentCommentAPIService txStudentCommentAPIService;
    
    @Override
    @Transactional(rollbackFor = { Exception.class, BussinessException.class })
    public Long saveOrUpdateBacklog(Long orgId, Long cascadeId, SaveBacklogDto saveBacklogDto) throws CloneNotSupportedException {
    	
    	Long backlogId = saveBacklogDto.getBacklogId();
        if (backlogId==null || backlogId<=0) {
        	//添加
        	log.info("addBacklog------saveBacklogDto={}", saveBacklogDto);
            TxBacklog txBacklog = new TxBacklog();
            this.backlogDto2Po(saveBacklogDto, txBacklog, orgId, cascadeId);
            this.txBacklogDao.save(txBacklog);
            
            List<TxbacklogParticipant> participants = getParticipants(orgId, cascadeId, txBacklog.getId(), saveBacklogDto.getParticipants());
            txBacklogParticipantDao.saveAll(participants, false);
            
            //添加跟进记录
            if(txBacklog.getType().intValue() == BackLogType.FOLLOW_UP.getValue()){
            	
            	if(txBacklog.getStudentId()!=null && txBacklog.getStudentId()>0){
            		OrgStudent student = orgStudentDao.getById(txBacklog.getStudentId());
            		txStudentCommentAPIService.saveByBacklogAdd(student.getUserId(), txBacklog, participants);
            		
            	}else if(txBacklog.getConsultUserId()!=null && txBacklog.getConsultUserId()>0){
            		txStudentCommentAPIService.saveByBacklogAdd(null, txBacklog, participants);
            	}
            }
            
            return txBacklog.getId();
            
        } else {
        	//修改
        	log.info("updateBacklog------saveBacklogDto={}",saveBacklogDto);
            TxBacklog txBacklogInDB = txBacklogDao.getBacklogByIdAndOrgId(backlogId, orgId);
            TxBacklog txBacklogNew = txBacklogInDB.clone();
            
            if (null != txBacklogNew) {
                this.verifyParamAndStatus(txBacklogNew, saveBacklogDto);
                this.syncStudentInfo(txBacklogNew, saveBacklogDto, orgId);
                this.backlogDto2Po(saveBacklogDto, txBacklogNew, orgId, cascadeId);
                this.txBacklogDao.update(txBacklogNew, true);
                
                //判断参与人是否发生变化
                boolean participantsChange = false;
                List<TxbacklogParticipant> participantsNew = getParticipants(orgId, cascadeId, txBacklogNew.getId(), saveBacklogDto.getParticipants());
                List<TxbacklogParticipant> participantsInDB = txBacklogParticipantDao.listByBackLogId(backlogId);
                HashSet<Long> set = new HashSet<Long>();
                
                if(participantsNew.size() != participantsInDB.size()){
                	participantsChange = true;
                }else{
                	int countDB = participantsInDB.size();
                	for(TxbacklogParticipant obj:participantsNew){
                		set.add(obj.getCascadeId());
                	}
                	for(TxbacklogParticipant obj:participantsInDB){
                		set.add(obj.getCascadeId());
                	}
                	if(set.size() != participantsInDB.size()){
                		participantsChange = true;
                	}
                }
                
                if(participantsChange){
                	txBacklogParticipantDao.delByBackLogId(backlogId);
                	txBacklogParticipantDao.saveAll(participantsNew, false);
                }
                
                //添加跟进记录
                Long userId = null;
                if(txBacklogNew.getStudentId()!=null && txBacklogNew.getStudentId()>0){
            		OrgStudent student = orgStudentDao.getById(txBacklogNew.getStudentId());
            		userId = student.getUserId();
            	}
                txStudentCommentAPIService.saveByBacklogUpdate(userId, txBacklogInDB, txBacklogNew, participantsInDB,participantsNew);
                
            } else {
                throw new BussinessException(TodoErrorCode.BACKLOG_NOT_EXIST);
            }
            return txBacklogNew.getId();
        }
    }
    
    List<TxbacklogParticipant> getParticipants(Long orgId, Long cascadeId, Long txBacklogId, String participantsStr){
    	Set<Long> participantIds = new HashSet<Long>();
        participantIds.add(cascadeId);
        if(StringUtils.isNotBlank(participantsStr)){
        	for(String obj : participantsStr.split(",")){
        		if(StringUtils.isNotBlank(obj)){
        			participantIds.add(Long.parseLong(obj));
        		}
        	}
        }
        
        List<TxbacklogParticipant> participants = new ArrayList<TxbacklogParticipant>();
        TxbacklogParticipant txbacklogParticipan = null;
        for(Long participant:participantIds){
        	txbacklogParticipan = new TxbacklogParticipant(txBacklogId, orgId, participant);
        	participants.add(txbacklogParticipan);
        }
        
        return participants;
    }
    
    
    
    @Override
    @Transactional(rollbackFor = { Exception.class, BussinessException.class })
    public void delBacklog(Long orgId, Long cascadeId, Long backlogId) {
        Preconditions.checkArgument(orgId != null, "orgId can not be null");
        Preconditions.checkArgument(backlogId != null && backlogId > 0, "backlogId is illegal");

        TxBacklog txBacklog = this.txBacklogDao.getById(backlogId, "id", "orgId", "delStatus");
        List<TxbacklogParticipant> participants = txBacklogParticipantDao.listByBackLogId(backlogId);
        log.info("delBacklog---------backlogId={},orgId={},txBacklog={}", backlogId, orgId, txBacklog);

        if (null != txBacklog && DeleteStatus.NORMAL.getValue() == txBacklog.getDelStatus()) {
            if (null != txBacklog.getOrgId() && txBacklog.getOrgId().longValue() == orgId.longValue()) {
            	
            	boolean realDelBackLog = false;
            	if(txBacklog.getStatus() == BacklogStatus.UNEXPIRED.getValue()){
            		if(!txBacklog.getCascadeId().equals(cascadeId)){
            			throw new PermissionException("仅创建人可以删除未过期的待办事项。");
            		}
            		txBacklogParticipantDao.delByBackLogId(backlogId);
            		realDelBackLog = true;
            		txBacklog.setDelStatus(DeleteStatus.DELETED.getValue());
                    txBacklog.setUpdateTime(new Date());
                    txBacklogDao.update(txBacklog, false, "delStatus", "updateTime");
                    txBacklogParticipantDao.delByBackLogId(backlogId);
            		
            	}else if(txBacklog.getStatus() == BacklogStatus.EXPIRED.getValue()){
            		if(participants.size()==1 && participants.get(0).getCascadeId().equals(cascadeId)){
            			realDelBackLog = true;
            		}
            		txBacklogParticipantDao.delBy(backlogId, cascadeId);
            	}
                
                if(realDelBackLog){
                	txBacklog.setDelStatus(DeleteStatus.DELETED.getValue());
                    txBacklog.setUpdateTime(new Date());
                    txBacklogDao.update(txBacklog, false, "delStatus", "updateTime");
                    
                    //添加跟进记录
                    if(txBacklog.getType().intValue() == BackLogType.FOLLOW_UP.getValue()){
                	
	                	if(txBacklog.getStudentId()!=null && txBacklog.getStudentId()>0){
	                		OrgStudent student = orgStudentDao.getById(txBacklog.getStudentId());
	                		txStudentCommentAPIService.saveByBacklogAdd(student.getUserId(), txBacklog, participants);
	                		
	                	}else if(txBacklog.getConsultUserId()!=null && txBacklog.getConsultUserId()>0){
	                		txStudentCommentAPIService.saveByBacklogAdd(null, txBacklog, participants);
	                	}
	                }
                }
                
            } else {
                throw new BussinessException(TodoErrorCode.STUDENT_NOT_IN_ORG);
            }
        } else {
            throw new BussinessException(TodoErrorCode.BACKLOG_NOT_EXIST);
        }
    }

    
    
    
    @Override
    @DataAuthority(resourceTypes = { RequestSourceDesc.BACK_LOG_LIST })
    public BacklogListResponseDto getBacklogList(Long orgId, Long cascadeId, int groupType, PageDto page) {
        Preconditions.checkArgument(orgId != null, "orgId can not be null");
        Preconditions.checkArgument(groupType >= 0 && groupType <= 3, "groupType is illegal");
        
        OrgAccount account = this.orgAccountDao.getById(orgId);
        if (account == null) {
            throw new BussinessException(TodoErrorCode.ORG_NOT_EXIST);
        }

        BacklogListResponseDto responseDto = new BacklogListResponseDto();
        List<BacklogDto> backlogListDto = Lists.newArrayList();
        List<TxBacklog> list = Lists.newArrayList();
        responseDto.setGroupType(groupType);
        responseDto.setList(backlogListDto);

        // 如果判断当前的登录角色不是员工，不对数据获取进行限制
        boolean canAccess = RequestSourceDesc.BACK_LOG_LIST.canAccess("getBacklogList", this.getClass(), new Class<?>[] { Long.class, int.class, PageDto.class });
        if (canAccess) {
        	cascadeId = null;
        }

        list = this.txBacklogDao.getBacklogListByGroup(orgId, cascadeId.intValue(), groupType, page);
        log.info("getBacklogList-------orgId={},groupType={},page={},list={}", orgId, GroupType.getNoteWithValue(groupType), page, list);
        
        List<Long> consultUserIds = new ArrayList<>();
        List<Long> studentIds = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(list)) {
            for (TxBacklog po : list) {
                BacklogDto dto = new BacklogDto();
                backlogPo2Dto(po, dto);
                if (po.getStudentId() != null && po.getStudentId() > 0) {
                    studentIds.add(po.getStudentId());
                } else {
                    consultUserIds.add(po.getConsultUserId());
                }
                backlogListDto.add(dto);
            }

            Map<Long, RelatedStudent> studentMap = buildStudents(studentIds, orgId);
            Map<Long, RelatedStudent> consultUserMap = buildConsultUsers(consultUserIds, orgId);

            for (BacklogDto dto : backlogListDto) {
                RelatedStudent relatedStudent = null;
                if (dto.getRelatedStudent() == null) {
                    log.warn("RelatedStudent is null!");
                    continue;
                }
                if (dto.getRelatedStudent().getType() == RelatedStudentType.STUDENT.getValue()) {
                    relatedStudent = studentMap.get(dto.getRelatedStudent().getStudentId());
                } else {
                    relatedStudent = consultUserMap.get(dto.getRelatedStudent().getStudentId());
                }
                if (relatedStudent != null) {
                    dto.setRelatedStudent(relatedStudent);
                } else {
                    log.warn("[Backlog] student is not exist.{}", dto.getRelatedStudent());
                }
            }

        }
        return responseDto;
    }

    

    @Override
    @Transactional(rollbackFor = { Exception.class, BussinessException.class })
    public void setBacklogStatus(Long orgId, Long cascadeId, Long backlogId, int finish) {
        Preconditions.checkArgument(orgId != null, "orgId can not be null");
        Preconditions.checkArgument(backlogId != null && backlogId > 0, "backlogId is illegal");

        TxBacklog txBacklog = txBacklogDao.getBacklogByIdAndOrgId(backlogId, orgId, "id", "finish");
        log.info("setBacklogStatus------txBacklog={}", txBacklog);
        if (null != txBacklog) {
            if (finish != txBacklog.getFinish()) {
                txBacklog.setFinish(finish);
                txBacklog.setUpdateTime(new Date());
                this.txBacklogDao.update(txBacklog, "finish", "updateTime");
                
                TxBacklogOperationLog operationLog = new TxBacklogOperationLog();
                operationLog.setBacklogId(txBacklog.getId());
                operationLog.setOrgId(orgId);
                operationLog.setCascadeId(cascadeId.longValue());
                if (finish == Flag.FALSE.getInt()) {
                	operationLog.setOperation( BacklogOperation.UPDATE_TO_UNFINISH.getValue() );
                }else{
                	operationLog.setOperation( BacklogOperation.UPDATE_TO_FINISH.getValue() );
                }
                txBacklogOperationLogDao.save(operationLog);
                
            } else {
                throw new BussinessException(
                    0 == finish ? TodoErrorCode.BACKLOG_HAS_UNFINISH : TodoErrorCode.BACKLOG_HAS_FINISH);
            }
        } else {
            throw new BussinessException(TodoErrorCode.BACKLOG_NOT_EXIST);
        }
    }

    private void backlogPo2Dto(TxBacklog po, BacklogDto dto) {
        dto.setIsSys(po.getIsSys());
        dto.setBacklogId(po.getId());
        dto.setContent(StringUtils.isNotBlank(po.getContent()) ? po.getContent() : "");
        dto.setFinish(po.getFinish());
        dto.setFinishString(TaskStatus.getNoteByValue(po.getFinish()));
        dto.setRelatedStudent(buildStudent(po.getConsultUserId(), po.getStudentId()));
        dto.setBacklogType(po.getType());
        
        if (null != po.getEndTime()) {
            dto.setEndTime(po.getEndTime().getTime());
            if (null != po.getRemindTime()) {
                dto.setRemindType(RemindType.getByDiff(po.getRemindTime().getTime() - po.getEndTime().getTime()).getValue());
            } else {
                dto.setRemindType(RemindType.NO_REMIND.getValue());
            }
        } else {
            dto.setEndTime(System.currentTimeMillis());
            dto.setRemindType(RemindType.NO_REMIND.getValue());
        }
        
    }

    private RelatedStudent buildStudent(Long consultUserId, Long studentId) {
        RelatedStudent relatedStudent = null;
        if (null != studentId && studentId > 0) {
            relatedStudent = new RelatedStudent();
            relatedStudent.setType(RelatedStudentType.STUDENT.getValue());
            relatedStudent.setStudentId(studentId);
        } else if (null != consultUserId && consultUserId > 0) {
            relatedStudent = new RelatedStudent();
            relatedStudent.setType(0);
            relatedStudent.setStudentId(consultUserId);
        }
        log.info("buildStudent------relatedStudent={}", relatedStudent);
        return relatedStudent;
    }

    private Map<Long, RelatedStudent> buildStudents(Collection<Long> studentIds, Long orgId) {
        Map<Long, RelatedStudent> retMap = new HashMap<>();
        if (null != studentIds && studentIds.size() > 0) {

            List<OrgStudent> students = orgStudentDao.getStudentByIds(orgId, studentIds, "id", "userId");

            Map<Long, Long> map = new HashMap<>();

            if (students != null) {
                for (OrgStudent stu : students) {
                    map.put(stu.getUserId(), stu.getId());
                }
            }

            Map<Long, ConsultAvatarUrlAndNameDto> avatarMap =
                consultAvatarUrlService.batchStudentAvatarUrlAndNameDMap(map.keySet(), orgId);
            Set<Long> keySet = avatarMap.keySet();
            for (Long id : keySet) {
                ConsultAvatarUrlAndNameDto dto = avatarMap.get(id);
                RelatedStudent relatedStudent = new RelatedStudent();
                relatedStudent.setType(RelatedStudentType.STUDENT.getValue());
                relatedStudent.setStudentId(map.get(id));
                relatedStudent.setName(dto.getName());
                relatedStudent.setAvatar(dto.getAvatarUrl());
                retMap.put(map.get(id), relatedStudent);
            }
        }
        return retMap;
    }

    private Map<Long, RelatedStudent> buildConsultUsers(Collection<Long> consultUserIds, Long orgId) {
        Map<Long, RelatedStudent> retMap = new HashMap<>();
        if (null != consultUserIds && consultUserIds.size() > 0) {
            List<TxConsultUser> consultUsers =
                this.txConsultUserDao.batchTxConsultUserByIds(consultUserIds, null, null);
            if (null != consultUsers) {
                Map<Long, ConsultAvatarUrlAndNameDto> avatarMap =
                    consultAvatarUrlService.batchConsultAvatarUrlAndNameDtoMap(consultUserIds, orgId);
                Set<Long> keySet = avatarMap.keySet();
                for (Long id : keySet) {
                    ConsultAvatarUrlAndNameDto dto = avatarMap.get(id);
                    RelatedStudent relatedStudent = new RelatedStudent();
                    relatedStudent.setType(0);
                    relatedStudent.setStudentId(id);
                    relatedStudent.setName(dto.getName());
                    relatedStudent.setAvatar(dto.getAvatarUrl());
                    retMap.put(id, relatedStudent);
                }
            }
        }
        return retMap;
    }


    private void backlogDto2Po(SaveBacklogDto dto, TxBacklog po, Long orgId, Long cascadeId) {
    	Date now = new Date();
    	
        po.setStudentId(0l);
        po.setConsultUserId(0l);
        po.setOrgId(orgId);
        po.setContent(StringUtils.isNotEmpty(dto.getContent()) ? dto.getContent() : "");
        po.setEndTime(new Date(dto.getEndTime()));
        po.setCreateTime(null == po.getCreateTime() ?now : po.getCreateTime());
        po.setUpdateTime(now);
        if( !(po.getType()!=null && dto.getBacklogType()==null) ){
        	po.setType(dto.getBacklogType());
        }
        
        //FIXME
        if(po.getCascadeId() == null){
        	if (dto.getCreatorCascadeId() != null) {
                po.setCascadeId(dto.getCreatorCascadeId().intValue());
            }else{
            	po.setCascadeId(cascadeId.intValue());
            }
        }

        if (dto.getRemindType() >= RemindType.ON_TIME.getValue() && dto.getRemindType() <= RemindType.TWO_DAY_AHEAD.getValue()) {
            po.setRemindTime(new Date((dto.getEndTime() - RemindType.getDiffByValue(dto.getRemindType()))));
        } else {
            po.setRemindTime(null);
        }

        if (null != dto.getType() && null != dto.getStudentId()) {
            if (RelatedStudentType.CONSULT.getValue() == dto.getType().intValue()) {
                Long consultUserId = dto.getStudentId();
                TxConsultUser consultUser = this.txConsultUserDao.getById(consultUserId, "id", "orgId", "delStatus", "studentId");
                if (null != consultUser && DeleteStatus.NORMAL.getValue() == consultUser.getDelStatus()) {
                    if (orgId.longValue() == consultUser.getOrgId().longValue()) {
                        po.setConsultUserId(consultUserId);
                    } else {
                        throw new BussinessException(TodoErrorCode.STUDENT_NOT_IN_ORG);
                    }
                } else {
                    throw new BussinessException(TodoErrorCode.STUDENT_NOT_EXIST);
                }
            } else if (RelatedStudentType.STUDENT.getValue() == dto.getType().intValue()) {
                Long studentId = dto.getStudentId();
                OrgStudent student = this.orgStudentDao.getById(studentId, "org_id", "delStatus");
                if (null != student && DeleteStatus.NORMAL.getValue() == student.getDelStatus()) {
                    if (orgId.longValue() == student.getOrgId().longValue()) {
                        po.setStudentId(studentId);
                        List<TxConsultUser> list = this.txConsultUserDao.lookByStudentId(orgId, studentId, "id", "studentId");
                        if (CollectionUtils.isNotEmpty(list)) {
                            po.setConsultUserId(list.get(0).getId());
                        }
                    } else {
                        throw new BussinessException(TodoErrorCode.STUDENT_NOT_IN_ORG);
                    }
                } else {
                    throw new BussinessException(TodoErrorCode.STUDENT_NOT_EXIST);
                }
            }
        }
    }

    /**
     * 编辑待办事项之前进行参数及状态验证
     *
     * @param txBacklog 待办事项记录
     * @param backlogDto 编辑传入参数
     */
    private void verifyParamAndStatus(TxBacklog txBacklog, SaveBacklogDto saveBacklogDto) {
        if (txBacklog.getEndTime().before(DateUtil.getCurrentDate())) {
            throw new BussinessException(TodoErrorCode.CAN_NOT_EDIT_EXPIRED_BACKLOG);
        }

        if (BizConf.TRUE.intValue() == txBacklog.getFinish()) {
            throw new BussinessException(TodoErrorCode.CAN_NOT_EDIT_FINISH_BACKLOG);
        }

        if (BizConf.TRUE.intValue() == txBacklog.getIsSys()) {
            if (saveBacklogDto.getType()!=null && saveBacklogDto.getStudentId()!=null) {
                if (0 == saveBacklogDto.getType().intValue()) {
                    if (null != txBacklog.getConsultUserId() 
                        && txBacklog.getConsultUserId().longValue() != saveBacklogDto.getStudentId().longValue()) {
                        throw new BussinessException(TodoErrorCode.CAN_NOT_EDIT_SYSTEM_BACKLOG_WITH_STUDENT);
                    }
                } else {
                    if (null != txBacklog.getStudentId() 
                    	&& txBacklog.getStudentId().longValue() != saveBacklogDto.getStudentId().longValue()) {
                        throw new BussinessException(TodoErrorCode.CAN_NOT_EDIT_SYSTEM_BACKLOG_WITH_STUDENT);
                    }
                }
            }
        }
    }

    /**
     * 编辑系统待办事项时若截止时间改变，则需要同步咨询学员(正式学员)的下次跟进时间信息
     *
     * @param txBacklog 待办事项记录
     * @param backlogDto 编辑传入参数
     * @param orgId 机构id
     */
    private void syncStudentInfo(TxBacklog txBacklog, SaveBacklogDto saveBacklogDto, Long orgId) {
        if (BizConf.TRUE.intValue() == txBacklog.getIsSys()) {
            if (saveBacklogDto.getEndTime().longValue() != txBacklog.getEndTime().getTime()  && saveBacklogDto.getType()!=null) {
                if (RelatedStudentType.CONSULT.getValue() == saveBacklogDto.getType().intValue()) {
                    if (saveBacklogDto.getStudentId().longValue() == txBacklog.getConsultUserId().longValue()) {
                        TxConsultUser consulterUser =
                            this.consultUserDao.getOrgConsultUser(orgId, txBacklog.getConsultUserId());
                        if (null != consulterUser) {
                            consulterUser.setNextRemindTime(new Date(saveBacklogDto.getEndTime()));
                            consulterUser.setUpdateTime(new Date());
                            this.txConsultUserDao.update(consulterUser, "nextRemindTime", "updateTime");
                        } else {
                            throw new BussinessException(TodoErrorCode.STUDENT_NOT_EXIST);
                        }
                    } else {
                        throw new BussinessException(TodoErrorCode.BACKLOG_NOT_EXIST);
                    }
                } else if (RelatedStudentType.STUDENT.getValue() == saveBacklogDto.getType().intValue()) {
                    if (saveBacklogDto.getStudentId().longValue() == txBacklog.getStudentId().longValue()) {
                        OrgStudent orgStudent = this.orgStudentDao.getById(txBacklog.getStudentId());
                        if (null != orgStudent
                            && orgStudent.getDelStatus().intValue() == DeleteStatus.NORMAL.getValue()) {
                            orgStudent.setNextRemindTime(new Date(saveBacklogDto.getEndTime()));
                            orgStudent.setUpdateTime(new Date());
                            this.orgStudentDao.update(orgStudent, "nextRemindTime", "updateTime");
                        } else {
                            throw new BussinessException(TodoErrorCode.STUDENT_NOT_EXIST);
                        }
                    } else {
                        throw new BussinessException(TodoErrorCode.BACKLOG_NOT_EXIST);
                    }
                }
            }
        }
    }

    /**
     * 待办事项提醒
     *
     * @param startTime 系统第一次启动,数据库org_schedule_task_log中没有执行记录,则从第一条开始处理 如果org_schedule_task_log中有记录，从上次处理的截止时间开始处理
     * @param
     */
    @Override
    public void remind(Date startTime, Date endTime, OrgCrontabTaskLog taskLog, boolean isNew) {
        List<TxBacklog> result = txBacklogDao.getBacklogListByRemindTime(startTime, endTime);

        if (result != null && !result.isEmpty()) {
            prepared(result, taskLog, isNew);
        } else {
            taskLog.setUpdateTime(new Timestamp(System.currentTimeMillis()));
            if (isNew) {
                orgCrontabTaskDao.save(taskLog);
            } else {
                orgCrontabTaskDao.update(taskLog, "executeTime", "updateTime");
            }
        }
    }

    private void prepared(List<TxBacklog> result, OrgCrontabTaskLog taskLog, boolean isNew) {
        for (TxBacklog backlog : result) {
            pushBacklog(backlog);
        }

        taskLog.setUpdateTime(new Timestamp(System.currentTimeMillis()));
        taskLog.setValue(result.get(result.size() - 1).getId());
        if (isNew) {
            orgCrontabTaskDao.save(taskLog);
        } else {
            orgCrontabTaskDao.update(taskLog, "executeTime", "updateTime", "value");
        }
    }

    private void pushBacklog(TxBacklog backlog) {
        Map<String, Object> param = new HashMap<>();
        param.put("bid", backlog.getId());
        log.info("[Notice] Todo notice");
        try {
            String content = NoticeType.TO_DO.getContent() + ":" + backlog.getContent();
            messageService.sendNotice(backlog.getOrgId(), backlog.getCascadeId(), NoticeMsgContent.createNoticeContent(
                NoticeType.TO_DO, ActionUtil.getAction(ActionUtil.ACTION_TO_CRM_TODO_TASK, param), content));
        } catch (Exception e) {
            log.error("[Notice] exception ", e);
        }

    }

    private PushConfig getPushConfig(TxBacklog backlog) {
        PushConfig config = new PushConfig();
        config.setTip("您有一个待办事项需要处理");
        String url = NativeUrlUtil.getUrl(BACKLOG_ACTION);
        url = NativeUrlUtil.addParam(url, "bid", backlog.getId());
        config.setJumpUrl(url);
        return config;
    }

    /**
     * 查询待办事项详情
     */
    @Override
    public BacklogDto getBacklogDetail(Long orgId, Long cascadeId, Long bid) throws BussinessException {
        TxBacklog backlog = this.txBacklogDao.getById(bid);
        BacklogDto dto = null;
        if (backlog != null && backlog.getOrgId() == orgId.longValue()) {
            dto = new BacklogDto();
            this.backlogPo2Dto(backlog, dto);
            RelatedStudent relatedStudent = null;
            if (dto.getRelatedStudent() != null) {
                if (dto.getRelatedStudent().getType() == RelatedStudentType.STUDENT.getValue()) {
                    Map<Long, RelatedStudent> studentMap = buildStudents(Arrays.asList(dto.getRelatedStudent().getStudentId()), orgId);
                    relatedStudent = studentMap.get(dto.getRelatedStudent().getStudentId());
                } else {
                    Map<Long, RelatedStudent> consultUserMap = buildConsultUsers(Arrays.asList(dto.getRelatedStudent().getStudentId()), orgId);
                    relatedStudent = consultUserMap.get(dto.getRelatedStudent().getStudentId());
                }
                dto.setRelatedStudent(relatedStudent);
            }
            
            
            String creatorName = accountApiService.getAccountName(backlog.getOrgId(), backlog.getCascadeId().longValue());
            BacklogAccountDto creator = new BacklogAccountDto(creatorName, backlog.getCascadeId().longValue());
            dto.setCreator(creator);
            if(creator.getCascadeId().longValue() == cascadeId.longValue()){
            	dto.setIsCreator( Flag.TRUE.getInt() );
            }
            
            List<TxbacklogParticipant> participants = txBacklogParticipantDao.listByBackLogId(backlog.getId());
            List<BacklogAccountDto> participantDtos = new ArrayList<BacklogAccountDto>();
            BacklogAccountDto participantDto = null;
            String participantName = null;
            for(TxbacklogParticipant obj:participants){
            	participantName = accountApiService.getAccountName(obj.getOrgId(), obj.getCascadeId().longValue());
            	participantDto = new BacklogAccountDto(participantName, obj.getCascadeId().longValue());
            	participantDtos.add(participantDto);
            }
            dto.setParticipants(participantDtos);
            
            
            List<TxBacklogOperationLog> operationLogs = txBacklogOperationLogDao.listByBackLogId(backlog.getId());
            List<String> operationStrList = new ArrayList<String>();
            dto.setOperations(operationStrList);
            
            if(CollectionUtils.isNotEmpty(operationLogs)){
            	String fmt = "%s 于%s将事项标记为 [%s]";
            	SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd hh:mm");
            	String accountName = null;
            	BacklogOperation opeationEnum = null;
            	for(TxBacklogOperationLog operationLog:operationLogs){
            		accountName = accountApiService.getAccountName(operationLog.getOrgId(), operationLog.getCascadeId());
            		opeationEnum = BacklogOperation.getByValue(operationLog.getOperation().intValue());
            		if(BacklogOperation.UPDATE_TO_FINISH == opeationEnum){
            			operationStrList.add( String.format(fmt, accountName, sdf.format(operationLog.getCreateTime()), "完成") );
            		}else if(BacklogOperation.UPDATE_TO_UNFINISH == opeationEnum){
            			operationStrList.add( String.format(fmt, accountName, sdf.format(operationLog.getCreateTime()), "未完成") );
            		}
            	}
            }
            
        } else {
            throw new BussinessException(TodoErrorCode.BACKLOG_NOT_EXIST);
        }

        return dto;
    }

    
    
	@Override
	public void batchCompleteExpired(Long orgId, Long cascadeId) {
		Date expiredTime = DateUtil.getCurrentDate();
		txBacklogDao.batchCompleteExpired(orgId, cascadeId, expiredTime);
	}

	
	
	@Override
	public void batchDelExpired(Long orgId, Long cascadeId) {
		Date expiredTime = DateUtil.getCurrentDate();
		txBacklogDao.batchDelExpired(orgId, cascadeId, expiredTime);
	}

	
	
	@Override
	public BackLogHomePageDto getBackLogHomePageDto(Long orgId, Long cascadeId,
													HomepageParamUnexpired unexpired,
													HomepageParamExpired expired,
													Long expiredListLastId,
													Integer expiredListLimit,
													Integer searchFinish,
    										    	Integer searchStatus) {
		
		BackLogHomePageDto result = new BackLogHomePageDto();
		List<BackLogHomePageListDto> unexpiredList = new ArrayList<BackLogHomePageListDto>();
		List<BackLogHomePageListDto> expiredList = new ArrayList<BackLogHomePageListDto>();
		result.setUnexpired(unexpiredList);
		result.setExpired(expiredList);
		
		List<Long> idList = txBacklogParticipantDao.listBacklogId(orgId, cascadeId);
		
		if(CollectionUtils.isNotEmpty(idList)){
			Date today = DateUtil.getCurrentDate();
			Date tomorrow = DateUtil.getDiffDateTime(today, 1);
			
			List<TxBacklog> unExpiredTxBacklogs = null;
			List<TxBacklog> expiredTxBacklogs = null;
			Integer finish = null;
			Date minEndTime = null;
			Date maxEndTime = null;
			
			if(searchStatus!=null){
				if(searchStatus==0){//待办
					minEndTime = DateUtil.getCurrentDate();
				}else if(searchStatus==1){//过期
					maxEndTime = DateUtil.getCurrentDate();
				}
			}
			finish = searchFinish;
			
			//数据查询
			switch(unexpired){
				case NULL:
					break;
					
				case TODAY:
					if(minEndTime==null && maxEndTime==null){
						minEndTime = today;
						maxEndTime = tomorrow;
					}
					unExpiredTxBacklogs = txBacklogDao.list(idList, null, null, finish, minEndTime, maxEndTime);
					break;
					
				case TODAY_AND_FUTURE:
					if(searchStatus==null){
						minEndTime = today;
						maxEndTime = null;
					}
					unExpiredTxBacklogs = txBacklogDao.list(idList, null, null, finish, minEndTime, maxEndTime);
			    	break;
			}
			
			
			switch(expired){
				case NULL:
					break;
					
				case UNFINISH_AND_EXPIRED:
					if(searchFinish==null){
						finish = Flag.FALSE.getInt();
					}
					if(searchStatus==null){
						minEndTime = null;
						maxEndTime = today;
					}
					expiredTxBacklogs = txBacklogDao.list(idList, expiredListLastId, expiredListLimit, finish, minEndTime, maxEndTime);
					break;
					
				case ALL_EXPIRED:
					if(searchStatus==null){
						minEndTime = null;
						maxEndTime = today;
					}
					expiredTxBacklogs = txBacklogDao.list(idList, expiredListLastId, expiredListLimit, finish, minEndTime, maxEndTime);
					break;
			}
			
			
			//查询关联学生名称
			Map<Long,String> studentNameMap = new HashMap<Long,String>();
			Map<Long,String> consultUserNameMap = new HashMap<Long,String>();
			
			if(CollectionUtils.isNotEmpty(unExpiredTxBacklogs)){
				for(TxBacklog obj:unExpiredTxBacklogs){
					if(obj.getStudentId()!=null && obj.getStudentId()>0){
						studentNameMap.put(obj.getStudentId(), null);
					}else if(obj.getConsultUserId()!=null && obj.getConsultUserId()>0){
						consultUserNameMap.put(obj.getConsultUserId(), null);
					}
				}
			}
			
			if(CollectionUtils.isNotEmpty(expiredTxBacklogs)){
				for(TxBacklog obj:expiredTxBacklogs){
					if(obj.getStudentId()!=null && obj.getStudentId()>0){
						studentNameMap.put(obj.getStudentId(), null);
					}else if(obj.getConsultUserId()!=null && obj.getConsultUserId()>0){
						consultUserNameMap.put(obj.getConsultUserId(), null);
					}
				}
			}
			
			if(studentNameMap.size()>0){
				studentNameMap = orgStudentDao.mapIdVsName(studentNameMap.keySet());
			}
			if(consultUserNameMap.size()>0){
				consultUserNameMap = txConsultUserDao.mapIdVsName(consultUserNameMap.keySet());
			}
			
			//填充数据
			fillBackLogHomePageListDto(unexpiredList, unExpiredTxBacklogs, studentNameMap, consultUserNameMap);
			fillBackLogHomePageListDto(expiredList, expiredTxBacklogs, studentNameMap, consultUserNameMap);
			
			if(CollectionUtils.isNotEmpty(expiredTxBacklogs)){
				result.setExpiredListLastId( expiredTxBacklogs.get(expiredTxBacklogs.size()-1).getId() );
			}
		}
		
		return result;
	}
	
	
	void fillBackLogHomePageListDto(List<BackLogHomePageListDto> resultData, List<TxBacklog> sourceData, Map<Long,String> studentNameMap, Map<Long,String> consultUserNameMap){
		BackLogHomePageListDto listDto = null;
		Long studentId = null;
		Long consultUserId = null;
		
		if(CollectionUtils.isNotEmpty(sourceData)){
			for(TxBacklog obj:sourceData){
				listDto = new BackLogHomePageListDto();
				listDto.setBacklogId( obj.getId() );
				listDto.setBacklogType( obj.getType() );
				listDto.setContent( obj.getContent() );
				listDto.setEndTime( obj.getEndTime().getTime() );
				listDto.setFinish( obj.getFinish() );
				
				studentId = obj.getStudentId();
				consultUserId = obj.getConsultUserId();
				if(studentId!=null && studentId>0){
					listDto.setRelatedStudent( new RelatedStudent(RelatedStudentType.STUDENT.getValue(), studentId, MapUtils.getString(studentNameMap, studentId, "查询失败")) );
				}else if(consultUserId!=null && consultUserId>0){
					listDto.setRelatedStudent( new RelatedStudent(RelatedStudentType.CONSULT.getValue(), consultUserId, MapUtils.getString(consultUserNameMap, consultUserId, "查询失败")) );
				}
			    
				resultData.add(listDto);
			}
		}
	}
	
	

	
}
