
/**
 * Baijiahulian.com Inc.
 * Copyright (c) 2014-2015 All Rights Reserved.
 */

package com.baijia.tianxiao.sal.organization.org.service.impl;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.baijia.tianxiao.constants.org.BizConf;
import com.baijia.tianxiao.sal.organization.constant.WxTemplateType;
import com.baijia.tianxiao.util.collection.CollectorUtil;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONArray;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.dal.org.constant.TemplateStatus;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgHomepageConfigDao;
import com.baijia.tianxiao.dal.org.dao.OrgHomepageTemplateDao;
import com.baijia.tianxiao.dal.org.po.OrgHomepageConfig;
import com.baijia.tianxiao.dal.org.po.OrgHomepageTemplate;
import com.baijia.tianxiao.enums.OrgErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.organization.org.service.OrgHomepageConfigService;

import net.sf.json.JSONObject;


/**
 * @author zhangbing
 * @version 1.0
 * @title OrgHomepageTemplateImpl
 * @desc TODO
 * @date 2015年12月4日
 */
@Service
@Slf4j
public class OrgHomepageConfigServiceImpl implements OrgHomepageConfigService {

    @Autowired
    private OrgAccountDao orgAccountDao;

    @Autowired
    private OrgHomepageTemplateDao orgHomepageTemplateDao;

    @Autowired
    private OrgHomepageConfigDao orgHomepageConfigDao;

    /**
     * 添加模板
     *
     * @param orgId      : 机构ID
     * @param templateId : yunying.tx_homepage_template.id
     * @param content    : 机构自定义模板配置信息
     * @see com.baijia.tianxiao.sal.organization.org.service.OrgHomepageTemplateService#(java.lang.Integer, java.lang.Integer, java.lang.String)
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addTemplate(Integer orgId, Integer templateId, String content) {
        if (doBefore(orgId)) {
            OrgHomepageConfig conf = new OrgHomepageConfig();
            conf.setContent(content);
            conf.setOrgId(orgId);
            conf.setTemplateId(templateId);
            conf.setCreateTime(new Timestamp(System.currentTimeMillis()));
            this.orgHomepageConfigDao.save(conf, false, new String[]{});
        } else {
            throw new BussinessException(OrgErrorCode.USER_NOT_FOUND);
        }
    }

    /**
     * @param orgId      : 机构ID
     * @param templateId : yunying.tx_homepage_conf.id
     * @param content    : 机构自定义模板的内容
     * @see com.baijia.tianxiao.sal.organization.org.service.OrgHomepageTemplateService#(java.lang.Integer, java.lang.Integer, java.lang.String)
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void editTemplate(Integer orgId, Integer templateId, String content, String type) throws BussinessException {
        if (doBefore(orgId)) {
            OrgHomepageTemplate template = this.orgHomepageTemplateDao.getById(templateId, new String[]{});
            if (template != null) {
                if ("scheme".equals(type)) {
                    setScheme(orgId, template);
                } else if ("theme".equals(type)) {
                    setTheme(orgId, template, content);
                } else if ("config".equals(type)) {
                    setConfig(orgId, template, content);
                }
            }
        } else {
            throw new BussinessException(OrgErrorCode.USER_NOT_FOUND);
        }
    }

    /**
     * 设置模板
     */
    void setScheme(Integer orgId, OrgHomepageTemplate template) {
        OrgHomepageConfig conf = this.orgHomepageConfigDao.getTemplateConf(orgId, template.getId());

        OrgHomepageConfig usingTemplate = this.orgHomepageConfigDao.getTemplateConfByStatus(orgId, TemplateStatus.USING.getCode());
        if (usingTemplate != null) {
            usingTemplate.setUpdateTime(new Timestamp(System.currentTimeMillis()));
            usingTemplate.setStatus(TemplateStatus.UNUSED.getCode());

            this.orgHomepageConfigDao.update(usingTemplate);
        }
        if (conf != null) {
            conf.setUpdateTime(new Timestamp(System.currentTimeMillis()));
//            conf.setContent(JSONObject.fromObject(template.getContent()).getJSONObject("config").toString());
            conf.setStatus(TemplateStatus.USING.getCode());
            this.orgHomepageConfigDao.update(conf, false, new String[]{"content", "status", "updateTime"});
        } else {
            conf = new OrgHomepageConfig();
            conf.setCreateTime(new Timestamp(System.currentTimeMillis()));
            conf.setOrgId(orgId);
            conf.setTemplateId(template.getId());
            conf.setContent(JSONObject.fromObject(template.getContent()).getJSONObject("config").toString());
            conf.setStatus(TemplateStatus.USING.getCode());
            this.orgHomepageConfigDao.save(conf, false, new String[]{"createTime", "orgId", "templateId", "content", "status"});
        }
    }

    /**
     * 设置模板主题颜色
     *
     * @param orgId
     * @param template
     * @param theme
     */
    void setTheme(Integer orgId, OrgHomepageTemplate template, String theme) {
        OrgHomepageConfig conf = this.orgHomepageConfigDao.getTemplateConf(orgId, template.getId());
        if (conf != null && conf.getStatus() == TemplateStatus.USING.getCode()) {
            String content = conf.getContent();
            JSONObject json = JSONObject.fromObject(content);
            json.put("theme", theme);

            conf.setUpdateTime(new Timestamp(System.currentTimeMillis()));
            conf.setContent(json.toString());
            this.orgHomepageConfigDao.update(conf, false, new String[]{"updateTime", "content"});
        }
    }

    /**
     * 更新模板内容
     *
     * @param orgId
     * @param template
     * @param config
     */
    void setConfig(Integer orgId, OrgHomepageTemplate template, String config) {
        this.setScheme(orgId, template);
        OrgHomepageConfig conf = this.orgHomepageConfigDao.getTemplateConf(orgId, template.getId());
        if (conf != null && conf.getStatus() == TemplateStatus.USING.getCode()) {
            conf.setUpdateTime(new Timestamp(System.currentTimeMillis()));
            conf.setContent(config);
            this.orgHomepageConfigDao.update(conf, false, new String[]{"updateTime", "content"});
        }
    }

    /**
     * 获取机构使用中的模板配置
     */
    @Override
    public OrgHomepageConfig getUsingTemplate(Integer orgId) {
        return orgHomepageConfigDao.getTemplateConfByStatus(orgId, TemplateStatus.USING.getCode());
    }

    /**
     * check orgId是否有对应的机构
     *
     * @param orgId 机构Id
     * @return
     */
    private boolean doBefore(Integer orgId) {
        return this.orgAccountDao.getById(orgId, new String[]{}) != null;
    }

    /**
     * 恢复默认设置
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void clearTemplateConfig(Integer orgId) {
        OrgHomepageConfig usingTemplate = this.orgHomepageConfigDao.getTemplateConfByStatus(orgId, TemplateStatus.USING.getCode());
        if (usingTemplate != null) {
            usingTemplate.setUpdateTime(new Timestamp(System.currentTimeMillis()));
            usingTemplate.setStatus(TemplateStatus.UNUSED.getCode());

            this.orgHomepageConfigDao.update(usingTemplate, false, new String[]{"status", "updateTime"});
        }
    }

    @Override
    public String defaultHeadImg(Integer orgId) {
        OrgHomepageConfig config = getUsingTemplate(orgId);

        String imgUrl = "";
        if (config == null) {
            return BizConf.DEFAULT_HEAD_URL;
        }
        OrgHomepageTemplate template = orgHomepageTemplateDao.getById(config.getTemplateId());

        log.debug("template content =={}", config.getContent());
        JSONObject json = JSONObject.fromObject(config.getContent());
        JSONArray jsonArray = JSONArray.fromObject(json.get("widgets"));
        JSONObject arrayObject = JSONObject.fromObject(jsonArray.get(1));
        JSONObject imgObject = JSONObject.fromObject(arrayObject.get("config"));

        log.debug("template name =={}", template.getTemplateName());
        if (template.getTemplateName().equals(WxTemplateType.ABROAD.getName()) ||
                template.getTemplateName().equals(WxTemplateType.ZAOJIAO.getName()) ||
                template.getTemplateName().equals(WxTemplateType.KAIXUE.getName()) ||
                template.getTemplateName().equals(WxTemplateType.LANGUAGE.getName())) {
            imgUrl = imgObject.getString("img");
        } else {
            imgUrl = imgObject.getString("src");
        }
        return imgUrl.equals("") ? null : imgUrl;
    }


    @Override
    public List<Integer> getNumbersAllUsingConf() {
        List<Long> orgIds = orgHomepageConfigDao.getAllUsingConf();
        if (orgIds == null || orgIds.size() == 0) {
            return Collections.EMPTY_LIST;
        }
        Map<Long, Integer> orgNumMap = orgAccountDao.getorgNumbersByOrgIds(orgIds);
        if (orgNumMap == null || orgNumMap.size() == 0) {
            return Collections.EMPTY_LIST;
        }
        return new ArrayList<Integer>(orgNumMap.values());
    }

    public static void main(String[] args) {
        String str = "{\"theme\":\"default\",\"widgets\":[{\"type\":\"titleBar\",\"config\":{}},{\"type\":\"header\",\"config\":{\"src\":\"http://img.gsxservice.com/7116056_bxwg5djr.png\"}},{\"type\":\"timeLine\",\"config\":{\"legend\":[{\"text\":\"Brand Story\",\"maxlength\":16},{\"text\":\"品牌故事\",\"maxlength\":8}],\"list\":[{\"text\":\"匠心挚诚 追求极致\",\"maxlength\":10},{\"img\":\"http://img.gsxservice.com/7284841_vm6g3ldy.png\",\"text\":\"2015年我们从北京三里屯VILLAGE起航\",\"maxlength\":24},{\"img\":\"http://img.gsxservice.com/7284845_z6f0fij7.png\",\"text\":\"只为健康、安全、美味\",\"maxlength\":18},{\"img\":\"http://img.gsxservice.com/7284847_nb4impwl.png\",\"text\":\"用心成就卓越\",\"maxlength\":18},{\"img\":\"http://img.gsxservice.com/7284849_fb4h1byi.png\",\"text\":\"每位学员享受追梦的过程\",\"maxlength\":18}]}},{\"type\":\"video\",\"config\":{\"img\":\"http://img.gsxservice.com/7284850_bmd89lh6.png\",\"text\":\"传承欧洲百年工艺与现代艺术的结合\",\"url\":\"\",\"maxlength\":20}},{\"type\":\"imageText\",\"config\":{\"skin\":\"skin10\",\"img\":\"http://img.gsxservice.com/7285127_qt3qmux3.jpg\",\"txtContainer\":true,\"texts\":[{\"text\":\"ibaker像一架飞机，\",\"maxlength\":12},{\"text\":\"踏上航班，离梦想更进一步\",\"maxlength\":14}],\"submit\":{\"text\":\"马上咨询GO>\",\"action\":\"makePhoneCall\",\"editable\":true,\"maxlength\":8}}},{\"type\":\"definationTable\",\"config\":{\"legend\":[{\"text\":\"CLASSES\",\"maxlength\":16},{\"text\":\"课程介绍\",\"maxlength\":8}],\"list\":[{\"title\":{\"text\":\"教学目的\",\"maxlength\":8,\"clazz\":\"icon-list-os\"},\"content\":{\"text\":\"课程涵盖面广，便于学员随市场与消费客户的变化而调节和变化，通过学习您将了解世界不同国家的传统和流行甜品风格和独特艺术创意，开阔您的视野。使学员在短时间内学到真正对自己有帮助的课程，具备扎实的功底与举一反三、随心变换品种的能力。\",\"maxlength\":200}},{\"title\":{\"text\":\"适合人群\",\"maxlength\":8,\"clazz\":\"icon-person-o\"},\"content\":{\"text\":\"不仅真的烘培爱好者（系统学习欧式西点）、现有咖啡店甜点品种提升（提高技术及管理运营水平，感受纯正欧式教育）、以及希望开创西点生意的学员（渴望创业的城市白领、职场人士、小资群体和大学生），零基础且希望进入烘培行业或者学习专业西点的在内的广大群体。\",\"maxlength\":200}},{\"title\":{\"text\":\"课程特点\",\"maxlength\":8,\"clazz\":\"icon-hat-o\"},\"content\":{\"text\":\"课程紧凑，款款精品。欧式西点培训课程，理论结合实操在兴趣中学习，在成就感中强化记忆。我们的课程从易到难，无论是有基础还是没有基础的学员都适合学习。\",\"maxlength\":200}},{\"title\":{\"text\":\"课程内容\",\"maxlength\":8,\"clazz\":\"icon-doc-o\"},\"content\":{\"text\":\"咖啡甜品Coffee pastries，欧式蛋糕Cakes，手工巧克力Chocolate, 慕斯Mousse，西餐厅甜品Desserts，西点装饰技巧Gamishes,法式西点系列，面包、蛋挞、派系列，世界经典蛋糕系列，芝士蛋糕系列，西式酱汁系列，西点基础理论，西点艺术设计、西点技巧。\",\"maxlength\":200}}],\"submit\":{\"text\":\"预约试听课程&gt;\",\"action\":\"makePhoneCall\",\"editable\":true,\"maxlength\":10}}},{\"type\":\"materialTable\",\"config\":{\"legend\":[{\"text\":\"CLASSES\",\"maxlength\":16},{\"text\":\"课堂作品\",\"maxlength\":8}],\"list\":[{\"skin\":\"skin11\",\"img\":\"http://img.gsxservice.com/7284854_1ivvni6g.png\",\"imgType\":\"ad\",\"texts\":[{\"text\":\"蓝色地中海\",\"maxlength\":10}],\"submit\":{\"text\":\"马上咨询\",\"action\":\"makePhoneCall\"}},{\"skin\":\"skin11\",\"img\":\"http://img.gsxservice.com/7284856_meevdjku.png\",\"imgType\":\"ad\",\"texts\":[{\"text\":\"百象果巧克力慕斯蛋糕\",\"maxlength\":10}],\"submit\":{\"text\":\"马上咨询\",\"action\":\"makePhoneCall\"}},{\"skin\":\"skin11\",\"img\":\"http://img.gsxservice.com/7284860_xde50syb.png\",\"imgType\":\"ad\",\"texts\":[{\"text\":\"苹果派\",\"maxlength\":10}],\"submit\":{\"text\":\"马上咨询\",\"action\":\"makePhoneCall\"}},{\"skin\":\"skin11\",\"img\":\"http://img.gsxservice.com/7284866_1k92andu.png\",\"imgType\":\"ad\",\"texts\":[{\"text\":\"草莓蛋糕\",\"maxlength\":10}],\"submit\":{\"text\":\"马上咨询\",\"action\":\"makePhoneCall\"}}],\"submit\":{\"tplDataHref\":\"album\",\"text\":\"更多作品>\"}}},{\"type\":\"imageText\",\"config\":{\"skin\":\"skin12\",\"img\":\"http://img.gsxservice.com/7285661_45sdf56q.jpg\",\"txtContainer\":true,\"texts\":[{\"text\":\"Ibaker会员\",\"maxlength\":10},{\"text\":\"在这里尽享厨艺社交\",\"maxlength\":14},{\"text\":\"还可享受课程会员价优惠\",\"maxlength\":14}],\"submit\":{\"text\":\"我想加入>\",\"maxlength\":5,\"editable\":true,\"action\":\"makeOrder\"}}},{\"type\":\"classifyCourseList\",\"config\":{\"skin\":\"skin2\",\"legend\":[{\"text\":\"REGISTRATION\",\"maxlength\":16},{\"text\":\"在线报名\",\"maxlength\":8}],\"button\":{\"action\":\"makePhoneCall\",\"text\":\"在线咨询\"}}},{\"type\":\"imageText\",\"config\":{\"skin\":\"skin13\",\"img\":\"http://img.gsxservice.com/7287105_h4984q45.jpg\",\"txtContainer\":true,\"texts\":[{\"text\":\"加盟ibaker\",\"maxlength\":10},{\"text\":\"与我们一起倡导健康、安全、美味\",\"maxlength\":16}],\"submit\":{\"text\":\"联系我们\",\"editable\":true,\"action\":\"makePhoneCall\",\"maxlength\":4}}},{\"type\":\"lifeContactUs\",\"config\":{\"legend\":[{\"text\":\"CONTACT US\",\"maxlength\":16},{\"text\":\"联系我们\",\"maxlength\":8}],\"contacts\":{\"weixin\":{\"text\":\"ibaker1209\",\"maxlength\":20},\"qq\":{\"text\":\"43403928109\",\"maxlength\":20}},\"branches\":[{\"name\":{\"text\":\"ibaker深圳万达店\",\"maxlength\":20},\"address\":{\"text\":\"广东省深圳市南湖万达F12309\",\"maxlength\":20},\"tel\":{\"text\":\"0755-88888888\",\"maxlength\":20}},{\"name\":{\"text\":\"ibaker上海浦东店\",\"maxlength\":20},\"address\":{\"text\":\"上海市浦东新区\",\"maxlength\":20},\"tel\":{\"text\":\"021-88888888\",\"maxlength\":20}}],\"submit\":{\"text\":\"查看全部加盟店\",\"maxlength\":10,\"tplDataHref\":\"branch\",\"editable\":true}}}]}";
        JSONObject json = JSONObject.fromObject(str);
        JSONArray jsonArray = JSONArray.fromObject(json.get("widgets"));
        JSONObject arrayObject = JSONObject.fromObject(jsonArray.get(1));
        JSONObject imgObject = JSONObject.fromObject(arrayObject.get("config"));

        System.out.println("jsonObject ===" + imgObject.getString("src"));

    }

}

    