
package com.baijia.tianxiao.sal.course.util;

import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.excel.dto.ExportField;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.course.dto.response.OrgTeacherLessonInfoDto;
import com.baijia.tianxiao.sal.course.dto.response.TeacherClassHourDto;
import com.baijia.tianxiao.sal.course.dto.response.TeacherCourseDto;

import org.apache.commons.beanutils.PropertyUtilsBean;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.DataFormat;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.ss.util.RegionUtil;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletResponse;

import lombok.extern.slf4j.Slf4j;

/**
 * 导出excel的服务
 *
 * @author cxm
 * @version 1.0
 * @title ExcelExportService
 * @desc TODO
 * @date 2016年3月30日
 */
@Slf4j
public class ExcelExportService {

    private static String HEADER_FORMAT = "@";

    public static final String EXPORT_DATE_FORMAT = "yyyy-MM-dd HH:mm:ss";
    
    public static final String UNIT_LESSON_MONEY_FORMULA = "IF(I%d>0,I%d*C%d,0)";
    
    public static final String UNIT_HOUR_MONEY_FORMULA = "IF(K%d>0,K%d*D%d/60,0)";

    public static final String UNIT_STUDENTSIGN_MONEY_FORMULA = "IF(M%d>0,M%d*F%d,0)";

    public static final String LESSON_MONEY_FORMULA = "SUM(J%d:J%d,L%d:L%d,N%d:N%d)";

    public static PropertyUtilsBean propertyUtilsBean = new PropertyUtilsBean();

    private static CellStyle createCellStyle(Workbook workBook, String format) {
        CellStyle cellStyle = workBook.createCellStyle();
        DataFormat dataFormat = workBook.createDataFormat();
        cellStyle.setAlignment(CellStyle.ALIGN_CENTER);
        cellStyle.setVerticalAlignment(CellStyle.VERTICAL_CENTER);
        cellStyle.setDataFormat(dataFormat.getFormat(format));

        cellStyle.setBorderBottom(CellStyle.BORDER_THIN);
        cellStyle.setBorderLeft(CellStyle.BORDER_THIN);
        cellStyle.setBorderRight(CellStyle.BORDER_THIN);
        cellStyle.setBorderTop(CellStyle.BORDER_THIN);

        return cellStyle;
    }

    public static Sheet createSheet(Workbook workbook, String sheetName, List<Map<String,Object>> titles,
                                    Map<String, CellStyle> cellStyleMap, List<ExportField> exportFields, List<TeacherClassHourDto> datas,
                                    boolean showLesson) {

        Sheet sheet = workbook.createSheet(sheetName);
        int columnSize = 0;
        for (ExportField field : exportFields) {
            columnSize += field.getExportFieldSize();
        }
        createTitleRow(sheet, cellStyleMap, titles, columnSize);
       // int row = titles != null ? titles.size() : 0;
        int row = 0;
        if(!CollectionUtils.isEmpty(titles)){
            row = (Integer)titles.get(titles.size()-1).get("row");
        }
        row++;
        createHeaderRow(sheet, cellStyleMap, exportFields, row);
        row++;
        for (TeacherClassHourDto tch : datas) {
            // int preLessonSize = 0;
            Row dataRow = sheet.createRow(row);
            createCell(dataRow, 0, tch.getTeacherName(), cellStyleMap, exportFields.get(0));
            int totalLessonCount = 0;
            for (int j = 0; j < tch.getCourses().size(); j++) {
                TeacherCourseDto course = tch.getCourses().get(j);
                int index = 1;
                int rowNum = row;
                if (totalLessonCount > 0) {
                    rowNum += totalLessonCount;
                } else {
                    rowNum += j;
                }
                Row courseRow = null;
                if (j == 0) {
                    courseRow = dataRow;
                } else {
                    courseRow = sheet.createRow(rowNum);
                }
                createCell(courseRow, index, course.getCourseName(), cellStyleMap, exportFields.get(index));
                if (!showLesson) {
                    index++;
                    createCell(courseRow, index, course.getLessonCount(), cellStyleMap, exportFields.get(index));
                    index++;
                    createCell(courseRow, index, course.getMinutes(), cellStyleMap, exportFields.get(index));
                    index++;
                    createCell(courseRow, index, course.getStudentCount(), cellStyleMap, exportFields.get(index));
                    index++;
                    createCell(courseRow, index, course.getSignCount(), cellStyleMap, exportFields.get(index));
                    index++;
                    createCell(courseRow, index, course.getLeaveCount(), cellStyleMap, exportFields.get(index));
                    index++;
                    createCell(courseRow, index, course.getAbsentCount(), cellStyleMap, exportFields.get(index));
                    Cell cell = null;
                    while (index < exportFields.size() - 1) {
                        index++;
                        cell = createCell(courseRow, index, null, cellStyleMap, exportFields.get(index));
                        if (index == 9) {
                            cell.setCellFormula(
                                    String.format(UNIT_LESSON_MONEY_FORMULA, rowNum + 1, rowNum + 1, rowNum + 1));
                        }
                        if (index == 11) {
                            cell.setCellFormula(
                                    String.format(UNIT_HOUR_MONEY_FORMULA, rowNum + 1, rowNum + 1, rowNum + 1));
                        }
                        if (index == 13) {
                            cell.setCellFormula(
                                    String.format(UNIT_STUDENTSIGN_MONEY_FORMULA, rowNum + 1, rowNum + 1, rowNum + 1));
                        }
                    }
                } else {
                    /*if (course.getLessons().size() > 1) {
                        mergeCell(sheet, rowNum, rowNum + course.getLessons().size() - 1, index, index);
                    }*/
                    for (int k = 0; k < course.getLessons().size(); k++) {
                        OrgTeacherLessonInfoDto lessonInfo = course.getLessons().get(k);
                        Row lessonRow = null;
                        if (k == 0) {
                            lessonRow = courseRow;
                        } else {
                            lessonRow = sheet.createRow(rowNum + k);
                        }
                        createLessonRow(lessonRow, index, lessonInfo, exportFields, cellStyleMap);
                    }
                    if (course.getLessons().size() > 1) {
                        mergeCell(sheet, rowNum, rowNum + course.getLessons().size() - 1, index, index);
                    }
                    totalLessonCount += course.getLessons().size();
                    // preLessonSize = course.getLessons().size() - 1;
                }
            }
            if (!showLesson) {
                mergeCell(sheet, row, row + tch.getCourses().size() - 1, 0, 0);

                Cell cell = sheet.getRow(row).getCell(exportFields.size() - 1);
                cell.setCellFormula(String.format(LESSON_MONEY_FORMULA, row + 1, row + tch.getCourses().size(), row + 1, row + tch.getCourses().size(), row + 1, row + tch.getCourses().size()));
                mergeCell(sheet, row, row + tch.getCourses().size() - 1, exportFields.size() - 1,
                        exportFields.size() - 1);
                row += tch.getCourses().size();
            } else {
                if (totalLessonCount <= 1) {
                    totalLessonCount = 1;
                }
                log.debug("merge teacher cell:{},{}", row, row + totalLessonCount - 1);
                mergeCell(sheet, row, row + totalLessonCount - 1, 0, 0);
                row += totalLessonCount;
            }
        }

        return sheet;
    }


    private static void mergeCell(Sheet sheet, int firstRow, int lastRow, int firstColumn, int lastColumn) {
        if(firstRow == lastRow &&firstColumn == lastColumn ){
            return;
        }
        CellRangeAddress rangeAddress = new CellRangeAddress(firstRow, lastRow, firstColumn, lastColumn);
        sheet.addMergedRegion(rangeAddress);
        RegionUtil.setBorderBottom(CellStyle.BORDER_THIN, rangeAddress, sheet, sheet.getWorkbook());
        try {
            RegionUtil.setBorderTop(CellStyle.BORDER_THIN, rangeAddress, sheet, sheet.getWorkbook());
            RegionUtil.setBorderLeft(CellStyle.BORDER_THIN, rangeAddress, sheet, sheet.getWorkbook());
            RegionUtil.setBorderRight(CellStyle.BORDER_THIN, rangeAddress, sheet, sheet.getWorkbook());
        } catch (Exception e) {
            log.warn("excel mergeCell error: firstRow={},lastRow={},firstColumn={},lastColumn={}",firstRow,lastRow,firstColumn,lastColumn);
        }
    }

    private static void createLessonRow(Row lessonRow, int baseColumn, OrgTeacherLessonInfoDto lessonInfo,
                                        List<ExportField> exportFields, Map<String, CellStyle> cellStyleMap) {
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getLessonNumberInfo(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getRoomName(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getStartTime(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getLessonTime(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getMinutes(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getStudentCount(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getSignCount(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getLeaveCount(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getAbsentCount(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getUnsignCount(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getSignRate(), cellStyleMap, exportFields.get(baseColumn));

    }

    private static Cell createCell(Row row, int column, Object value, Map<String, CellStyle> cellStyleMap,
                                   ExportField exportField) {
        Cell cell = row.createCell(column);
        String localFormat = null;
        value = value != null ? value : exportField.getDefaultVal();

        if (value != null) {
            if (value instanceof Date) {
                row.getSheet().setColumnWidth(cell.getColumnIndex(), 20 * 256);
                localFormat = EXPORT_DATE_FORMAT;
                cell.setCellValue((Date) value);
            } else {
                if (value instanceof Number && (((Number) value).longValue() < Integer.MAX_VALUE)) {
                    cell.setCellType(Cell.CELL_TYPE_NUMERIC);
                    if (value instanceof Double) {
                        localFormat = "0.00";
                        cell.setCellValue(((Number) value).doubleValue());
                    } else {
                        localFormat = "0";
                        cell.setCellValue(((Number) value).longValue());
                    }
                } else {
                    cell.setCellType(Cell.CELL_TYPE_STRING);
                    cell.setCellValue(value.toString());
                    localFormat = "@";
                }
            }
        } else {
            cell.setCellValue("");
        }
        cell.setCellStyle(getCellStyle(row.getSheet().getWorkbook(), cellStyleMap,
                exportField.getFormat() != null ? exportField.getFormat() : localFormat));
        return cell;
    }

    private static CellStyle getCellStyle(Workbook workbook, Map<String, CellStyle> cellStyleMap, String format) {
        if (!cellStyleMap.containsKey(format)) {
            cellStyleMap.put(format, createCellStyle(workbook, format));
        }
        return cellStyleMap.get(format);
    }

    /**
     * 递归创建出行头
     *
     * @param sheet
     * @param cellStyleMap
     * @param exportFields
     * @param row
     */
    private static void createHeaderRow(Sheet sheet, Map<String, CellStyle> cellStyleMap,
                                        List<ExportField> exportFields, int row) {
        Row headerRow = sheet.createRow(row);
        int column = 0;
        CellStyle style = getCellStyle(sheet.getWorkbook(), cellStyleMap, HEADER_FORMAT);
        for (ExportField field : exportFields) {
            column = createHeaderCell(headerRow, style, field, column);
        }
    }

    private static int createHeaderCell(Row row, CellStyle style, ExportField field, int column) {
        if (CollectionUtils.isEmpty(field.getChildren())) {
            row.getSheet().setColumnWidth(column, field.getWidth());
            Cell cell = row.createCell(column++);
            cell.setCellStyle(style);
            cell.setCellValue(field.getName());
        } else {
            for (ExportField child : field.getChildren()) {
                column = createHeaderCell(row, style, child, column);
            }
        }
        return column;

    }

    private static void createTitleRow(Sheet sheet, Map<String, CellStyle> cellStyleMap, List<Map<String,Object>> titles,
                                       int headerSize) {
        //int row = 0;
        // 补充头
        Integer tempRow = null;
        Row titleRow = null;
        if (CollectionUtils.isNotEmpty(titles)) {
            for (Map<String,Object> title : titles) {
                Integer row = (Integer)title.get("row");
                String titleName = (String)title.get("titleName");
                int startCell = (Integer)title.get("startCell");
                int endCell = (Integer)title.get("endCell");
                if(tempRow == null ||tempRow.intValue() != row.intValue() ){
                    titleRow = sheet.createRow(row);
                    tempRow = row.intValue();
                }
                
                Cell cell = titleRow.createCell(startCell);
                cell.setCellValue(titleName);
                cell.setCellStyle(getCellStyle(sheet.getWorkbook(), cellStyleMap, HEADER_FORMAT));
                mergeCell(sheet, row, row, startCell, endCell);
            }
        }
    }

    public static void exportExcel(HttpServletResponse response, Workbook workBook, String excelFileName) throws IOException {
        
        if (StringUtils.isBlank(excelFileName)) {
            excelFileName = System.currentTimeMillis() + ".xlsx";
        }
        response.setContentType("application/vnd.ms-excel");
        try {
            response.setHeader("Content-Disposition",
                    "attachment; filename=" + URLEncoder.encode(excelFileName, "utf-8"));
        } catch (UnsupportedEncodingException e1) {
            response.setHeader("Content-Disposition", "attachment; filename=" + excelFileName);
        }
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        try {
            workBook.write(os);
            response.setContentLength(os.size());
            os.writeTo(response.getOutputStream());
            response.getOutputStream().flush();
        } catch (IOException e) {
            log.error("export data catch error:", e);
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "导出数据处理失败");
        }
        
       
    }

}
