
 /**
 * Baijiahulian.com Inc.
 * Copyright (c) 2014-2015 All Rights Reserved.
 */
    
package com.baijia.tianxiao.sal.course.service.impl;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.constants.CourseType;
import com.baijia.tianxiao.dal.course.dao.OrgCourseGroupDao;
import com.baijia.tianxiao.dal.course.dao.OrgCourseGroupRelateDao;
import com.baijia.tianxiao.dal.course.dao.TeacherClassCourseDao;
import com.baijia.tianxiao.dal.course.po.OrgCourseGroup;
import com.baijia.tianxiao.dal.course.po.OrgCourseGroupRelate;
import com.baijia.tianxiao.dal.course.po.TeacherClassCourse;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.enums.ModifyStatus;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.course.dto.request.OrgHotCourseRequestDto;
import com.baijia.tianxiao.sal.course.dto.request.OrgRecommendCourseRequestDto;
import com.baijia.tianxiao.sal.course.service.OrgCourseGroupRelateService;
import com.baijia.tianxiao.util.BaseUtils;
import com.google.common.collect.Sets;

/**
 * @title OrgCourseGroupRelateServiceImpl
 * @desc TODO 
 * @author zhangbing
 * @date 2015年12月28日
 * @version 1.0
 */
@Service
public class OrgCourseGroupRelateServiceImpl implements OrgCourseGroupRelateService {
    
    private static final Logger logger = LoggerFactory.getLogger(OrgCourseGroupRelateServiceImpl.class);

    @Autowired
    private OrgCourseGroupRelateDao orgCourseGroupRelateDao;
    
    @Autowired
    private OrgCourseDao orgCourseDao;
    
    @Autowired
    private TeacherClassCourseDao teacherClassCourseDao;
    
    @Autowired
    private OrgAccountDao orgAccountDao;
    
    @Autowired
    private OrgCourseGroupDao orgCourseGroupDao;
    
    /**
     * 删除对应分类下的课程
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void removeHotCourse(Long orgId, Long courseId, Integer courseGroupId, Integer courseType) {
        OrgAccount account = orgAccountDao.getAccountById(orgId.intValue());
        if (account != null) {
            if (courseType == CourseType.CLASS.getCode()) {
                TeacherClassCourse classCourse = teacherClassCourseDao.getById(courseId);
                if (classCourse != null && account.getNumber() == classCourse.getOrganizationNumber()) {
                    this.update(orgId, courseId, courseGroupId, courseType);
                }
            } else if (courseType == CourseType.ORG_COURSE.getCode()) {
                OrgCourse orgCourse = this.orgCourseDao.getById(courseId);
                if (orgCourse != null && orgCourse.getOrgNumber().intValue() == account.getNumber()) {
                    this.update(orgId, courseId, courseGroupId, courseType);
                }
            }
        }
    }
    
    /**
     * 将课程从对应分类下移除
     * @param orgId 机构ID
     * @param courseId 课程ID
     * @param courseGroupId 课程分组ID
     * @param courseType 课程类型
     */
    private void update(Long orgId, Long courseId, Integer courseGroupId, Integer courseType) {
        OrgCourseGroupRelate orgCourseGroupRelate = this.orgCourseGroupRelateDao.getCourseGroupByCourseId(courseGroupId, courseType, courseId);
        if (orgCourseGroupRelate != null) {
            orgCourseGroupRelate.setIsDeleted(DeleteStatus.DELETED.getValue());
            orgCourseGroupRelate.setUpdateTime(new Date());
            this.orgCourseGroupRelateDao.update(orgCourseGroupRelate, new String[]{"isDeleted","updateTime"});
        }
    }

    /**
     * 批量编辑分类课程排序
     */
    @Override
    public void batchEdit(List<OrgHotCourseRequestDto> params, Integer orgId) {
        logger.info("batchEdit.params:{}, orgId:{}", params, orgId);
        for (OrgHotCourseRequestDto dto : params) {
            Integer groupId = dto.getGroupId();
            if (validateParams(orgId, groupId)) {
                List<OrgRecommendCourseRequestDto> hotCourses = dto.getList();
                declare(hotCourses, orgId, groupId);
            }
        }
    }
    
    /**
     * 更新groupId分类下课程排序:
     * @param groupId
     * @param hotCourses
     */
    private void declare(List<OrgRecommendCourseRequestDto> hotCourses, Integer orgId, Integer courseGroupId) throws BussinessException{
        if (hotCourses != null && !hotCourses.isEmpty()) {
            
            int minDisplayOrder = this.getMaxDisplayOrder(hotCourses, courseGroupId);
            Set<Long> includeCourseIds = Sets.newHashSet();
            for (OrgRecommendCourseRequestDto hotCourse : hotCourses) {
                OrgCourseGroupRelate courseGroupRelate = this.orgCourseGroupRelateDao.getCourseGroupByCourseId(courseGroupId, 
                    hotCourse.getCourseType(), hotCourse.getCourseId());
                int modifyStatus = hotCourse.getModifyStatus();
                
                logger.info("hotCourse:{}, courseGroupRelate:{}", hotCourse, courseGroupRelate);
                if (ModifyStatus.containsKey(modifyStatus)) {
                    ModifyStatus enumVal = ModifyStatus.get(modifyStatus);
                    switch (enumVal) {
                        case NORMAL:
                            if (courseGroupRelate != null) {
                                includeCourseIds.add(hotCourse.getCourseId());
                                updateOrDonothingForNormalCourse(hotCourse, courseGroupRelate, minDisplayOrder--);
                            }
                            break;
                        case UPDATE:
                            if (courseGroupRelate != null && !includeCourseIds.contains(hotCourse.getCourseId())) {
                                updateCourseDisplayOrderToZero(courseGroupRelate);
                            }
                            break;
                        case ADDED:
                            includeCourseIds.add(hotCourse.getCourseId());
                            if (courseGroupRelate != null) {
                                update(courseGroupRelate, minDisplayOrder--);
                            } else {
                                addCourse(courseGroupId, hotCourse, minDisplayOrder--);
                            }
                            break;
                        case DELETE:
                            if (courseGroupRelate != null) {
                                deleteCourse(courseGroupRelate);
                            }
                            break;
                        default:
                            break;
                    }
                }
            }
        }
    }
    
    @SuppressWarnings("unchecked")
    private int getMaxDisplayOrder(List<OrgRecommendCourseRequestDto> hotCourses, Integer courseGroupId) {
        List<Long> excludeIds = BaseUtils.getListFilter(hotCourses, "courseId");
        int maxDisplayOrder = this.orgCourseGroupRelateDao.getMaxDisplayOrderByExcludeIds(courseGroupId, excludeIds);
        if (maxDisplayOrder == 0) {
            maxDisplayOrder = hotCourses.size() + 1;
        } else {
            maxDisplayOrder += hotCourses.size() + 1;
        }
        return maxDisplayOrder;
    }
    
    private void deleteCourse(OrgCourseGroupRelate courseGroupRelate) {
        courseGroupRelate.setDisplayOrder(0);
        courseGroupRelate.setIsDeleted(DeleteStatus.DELETED.getValue());
        courseGroupRelate.setUpdateTime(new Date());
        
        logger.info("deleteCourse.DELETE:{}", courseGroupRelate);
        this.orgCourseGroupRelateDao.update(courseGroupRelate, new String[]{"displayOrder","isDeleted","updateTime"});
    }

    /**
     * 添加课程到分类
     * @param courseGroupId
     * @param hotCourse
     * @param minDisplayOrder
     */
    @Override
    public void addCourse(int courseGroupId, OrgRecommendCourseRequestDto hotCourse, int minDisplayOrder) {
        if(null == orgCourseGroupRelateDao.getCourseGroupByCourseId(courseGroupId, hotCourse.getCourseType(), hotCourse.getCourseId())){
            OrgCourseGroupRelate courseGroupRelate = new OrgCourseGroupRelate();
            courseGroupRelate.setCourseGroupId(courseGroupId);
            courseGroupRelate.setCourseId(hotCourse.getCourseId());
            courseGroupRelate.setCourseType(hotCourse.getCourseType());
            courseGroupRelate.setCreateTime(new Date());
            courseGroupRelate.setDisplayOrder(minDisplayOrder);
            courseGroupRelate.setIsDeleted(DeleteStatus.NORMAL.getValue());
            logger.info("addCourse.courseGroupRelate.ADD:{}", courseGroupRelate);
            this.orgCourseGroupRelateDao.save(courseGroupRelate);
        }
    }
    @Override
    public void addCourse(Long groupId, Collection<Long> courseIds){
        orgCourseGroupRelateDao.batchAddWeike(groupId, courseIds);
    }
    
    /**
     * 将被替换的课程displayOrder设置为0
     * @param courseGroupRelate
     */
    private void updateCourseDisplayOrderToZero(OrgCourseGroupRelate courseGroupRelate) {
        courseGroupRelate.setUpdateTime(new Date());
        courseGroupRelate.setDisplayOrder(0);
        logger.info("updateCourseDisplayOrderToZero.UPDATE.courseGroupRelate:{}", courseGroupRelate);
        this.orgCourseGroupRelateDao.update(courseGroupRelate, new String[]{"displayOrder", "updateTime"});
    }
    
    /**
     * 如果这门课程没有做任何修改，当这门课程displayOrder<minDisplayOrder时候更新顺序
     * @param hotCourse
     * @param courseGroupRelate
     * @param minDisplayOrder
     */
    private void updateOrDonothingForNormalCourse(OrgRecommendCourseRequestDto hotCourse, OrgCourseGroupRelate courseGroupRelate, int minDisplayOrder) {
        int displayOrder = courseGroupRelate.getDisplayOrder();
        if (displayOrder < minDisplayOrder) {
            courseGroupRelate.setUpdateTime(new Date());
            courseGroupRelate.setDisplayOrder(minDisplayOrder);
            logger.info("updateOrDonothingForNormalCourse.NORMAL.courseGroupRelate:{}", courseGroupRelate);
            this.orgCourseGroupRelateDao.update(courseGroupRelate, new String[]{"displayOrder", "updateTime"});
        }
    }
    
    private void update(OrgCourseGroupRelate courseGroupRelate, int minDisplayOrder) {
        //一旦课程被替换，原始课程displayOrder=0
        courseGroupRelate.setDisplayOrder(minDisplayOrder);
        courseGroupRelate.setUpdateTime(new Date());
        this.orgCourseGroupRelateDao.update(courseGroupRelate, new String[]{"displayOrder", "updateTime"});
    }
    
    /**
     * 
     * @param orgNumber  机构number
     * @param courseId   课程Id
     * @param courseType 课程类型
     * @return
     */
    private boolean validateCourse(Integer orgNumber, Long courseId, Integer courseType) {
        if (courseType == CourseType.CLASS.getCode()) {
            TeacherClassCourse classCourse = this.teacherClassCourseDao.getById(courseId);
            return classCourse != null && classCourse.getOrganizationNumber() == orgNumber;
        } else if (courseType == CourseType.ORG_COURSE.getCode()) {
            OrgCourse orgCourse = this.orgCourseDao.getById(courseId);
            return orgCourse != null && orgCourse.getOrgNumber() == orgNumber.longValue();
        }
        return false;
    }
    
    /**
     * 校验groupId是否是orgId所创建 
     * @param orgId 机构ID
     * @param groupId 自定义分类ID
     * @return
     */
    private boolean validateParams(Integer orgId, Integer groupId) {
        OrgCourseGroup courseGroup = this.orgCourseGroupDao.getById(groupId);
        boolean ret = (courseGroup != null) && (orgId.intValue() == courseGroup.getOrgId().intValue() && courseGroup.getIsDeleted().intValue() == DeleteStatus.NORMAL.getValue());
        logger.info("validateParams.courseGroup:{}, ret:{}", courseGroup, ret);
        return ret;
    }
    
    
}

    