package com.baijia.tianxiao.sal.course.service.impl;

import com.baijia.tianxiao.dal.org.dao.OrgLessonConflictDao;
import com.baijia.tianxiao.dal.org.po.OrgLessonConflict;
import com.baijia.tianxiao.sal.course.dto.request.LessonConflictCheckDto;
import com.baijia.tianxiao.sal.course.enums.LessonConflictEnum;
import com.baijia.tianxiao.sal.course.service.OrgLessonConflictService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.beust.jcommander.internal.Sets;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by zhangrui on 16/8/24.
 */
@Slf4j
@Service
public class OrgLessonConflictServiceImpl implements OrgLessonConflictService {

    @Resource
    private OrgLessonConflictDao orgLessonConflictDao;

    @Override
    public List<OrgLessonConflict> checkTeacherConflict(Long orgId,Long teacherId, Date startTime, Date endTime,Long lessonId,PageDto pageDto) {
        validata(orgId,teacherId,null,startTime,endTime);
        return orgLessonConflictDao.queryTeacherLessonConflict(orgId,teacherId,startTime,endTime,lessonId,pageDto);
    }

    @Override
    public List<OrgLessonConflict> checkRoomConflict(Long orgId,Long roomId, Date startTime, Date endTime,Long lessonId,PageDto pageDto) {
        validata(orgId,null,roomId,startTime,endTime);
        return orgLessonConflictDao.queryRoomLessonConflict(orgId,roomId, startTime, endTime,lessonId, pageDto);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveOrgLessonConflicts(List<OrgLessonConflict> lists) {
        orgLessonConflictDao.saveOrgLessonConflicts(lists);
    }

    @Override
    public int queryTeacherConflictNum(Long orgId,Long teacherId, Date startTime, Date endTime,Long lessonId) {
        validata(orgId,teacherId,null,startTime,endTime);
        return orgLessonConflictDao.queryTeacherConflictNum(orgId,teacherId,startTime,endTime,lessonId);
    }

    @Override
    public int queryRoomConflictNum(Long orgId,Long roomId, Date startTime, Date endTime,Long lessonId) {
        validata(orgId,null,roomId,startTime,endTime);
        return orgLessonConflictDao.queryRoomConflictNum(orgId,roomId,startTime,endTime,lessonId);
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateOrgLessonConflictForTeacherChange(OrgLessonConflict conflict) {
        //teacher可能存在一对多的情况,所以需要删除再添加,场景:一个课节对应2个老师,此时把两个老师换成了另外一个老师
        Map<String,Object> condition = Maps.newHashMap();
        condition.put("org_id",conflict.getOrgId());
        condition.put("lesson_id",conflict.getLessonId());

        PageDto pageDto = new PageDto();
        pageDto.setPageNum(1);
        pageDto.setPageSize(1);
        List<OrgLessonConflict> resultList = orgLessonConflictDao.queryByCondition(condition,pageDto);

        if(resultList.size() > 0){
            OrgLessonConflict conflict1 = resultList.get(0);
            this.orgLessonConflictDao.delByCondition(condition);
            conflict.setRoomId(conflict1.getRoomId());
            conflict.setStartTime(conflict1.getStartTime());
            conflict.setEndTime(conflict1.getEndTime());
            orgLessonConflictDao.save(conflict);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateOrgLessonConflictForTeacherChange(List<OrgLessonConflict> conflictList) {

        Set<String> set = Sets.newHashSet();
        PageDto pageDto = new PageDto();
        pageDto.setPageNum(1);
        pageDto.setPageSize(1);
        for(OrgLessonConflict conflict:conflictList){

            Map<String,Object> condition = Maps.newHashMap();
            condition.put("org_id",conflict.getOrgId());
            condition.put("lesson_id",conflict.getLessonId());

            List<OrgLessonConflict> resultList = orgLessonConflictDao.queryByCondition(condition,pageDto);
            if(resultList.size() > 0){

                if(!set.contains(conflict.getOrgId() + "_" + conflict.getLessonId())){
                    set.add(conflict.getOrgId() + "_" + conflict.getLessonId());
                    this.orgLessonConflictDao.delByCondition(condition);
                }

                OrgLessonConflict conflict1 = resultList.get(0);
                conflict.setRoomId(conflict1.getRoomId());
                conflict.setStartTime(conflict1.getStartTime());
                conflict.setEndTime(conflict1.getEndTime());
                orgLessonConflictDao.save(conflict);
            }


        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateOrgLessonConflictForRoomOrTimeChange(OrgLessonConflict conflict) {
        Map<String,Object> fields = Maps.newHashMap();
        if(conflict.getRoomId() != null && conflict.getRoomId() > 0){
            fields.put("room_id",conflict.getRoomId());
        }
        if(conflict.getStartTime() != null){
            fields.put("start_time",conflict.getStartTime());
        }
        if(conflict.getEndTime() != null){
            fields.put("end_time",conflict.getEndTime());
        }
        if(conflict.getDelStatus() != null){
            fields.put("del_status",conflict.getDelStatus());
        }

        Map<String,Object> condition = Maps.newHashMap();
        condition.put("org_id",conflict.getOrgId());
        condition.put("lesson_id",conflict.getLessonId());
        this.orgLessonConflictDao.updateLessonConflictByCondition(conflict.getOrgId(),fields,condition);
    }


    @Override
    public LessonConflictEnum lessonConflictFlagCheck(Long orgId, LessonConflictCheckDto lessonConflictCheckDto) {

        LessonConflictEnum result = LessonConflictEnum.NO_CONFLICT;

        Long teacherId = lessonConflictCheckDto.getTeacherId();
        Long roomId = lessonConflictCheckDto.getRoomId();
        Date startTime = lessonConflictCheckDto.getStartTimeDate();
        Date endTime = lessonConflictCheckDto.getEndTimeDate();
        Long lessonId = lessonConflictCheckDto.getLessonId();

        int teacherConflictNum = 0;
        int roomConflictNum = 0;
        if(teacherId != null && teacherId > 0){
            teacherConflictNum = this.queryTeacherConflictNum(orgId,teacherId,startTime,endTime,lessonId);
        }

        if(roomId != null && roomId > 0){
            roomConflictNum = this.queryRoomConflictNum(orgId,roomId,startTime,endTime,lessonId);
        }

        if(teacherConflictNum > 0 && roomConflictNum > 0){
            result = LessonConflictEnum.TEACHER_AND_ROOM;
        }else if(teacherConflictNum > 0){
            result = LessonConflictEnum.TEACHER;
        }else if(roomConflictNum > 0){
            result = LessonConflictEnum.ROOM;
        }

        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delByLessonIds(Long orgId,List<Long> lessonIds) {
        if(CollectionUtils.isEmpty(lessonIds)){
            return;
        }
        Map<String,Object> conditions = Maps.newHashMap();
        conditions.put("org_id",orgId);
        conditions.put("lesson_id",lessonIds);
        orgLessonConflictDao.delByCondition(conditions);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateOrgLessonConflictsForce(OrgLessonConflict conflict) {
        Map<String,Object> conditions = Maps.newHashMap();
        conditions.put("org_id",conflict.getOrgId());
        conditions.put("lesson_id",conflict.getLessonId());
        orgLessonConflictDao.delByCondition(conditions);
        orgLessonConflictDao.save(conflict);
    }

    private void validata(Long orgId,Long teacherId, Long roomId, Date startTime, Date endTime){
        Preconditions.checkArgument(orgId != null && orgId > 0 ,"机构id不合法");
        Preconditions.checkArgument((teacherId != null && teacherId > 0) || (roomId != null && roomId > 0) ,"老师id/教室id不合法");
        Preconditions.checkArgument(startTime != null, "开始时间不合法");
        Preconditions.checkArgument(endTime != null, "结束时间不合法");
    }
}
