/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */

package com.baijia.tianxiao.sal.course.service.impl;

import com.baijia.tianxiao.constants.CourseType;
import com.baijia.tianxiao.dal.course.dao.OrgCourseGroupDao;
import com.baijia.tianxiao.dal.course.dao.OrgCourseGroupRelateDao;
import com.baijia.tianxiao.dal.course.po.OrgCourseGroup;
import com.baijia.tianxiao.dal.enums.CourseTypeEnum;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.enums.OrgErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.course.dto.response.OrgCourseGroupDto;
import com.baijia.tianxiao.sal.course.service.OrgCourseGroupService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * @title OrgCourseGroupServiceImpl
 * @desc TODO
 * @author zhangbing
 * @date 2015年12月22日
 * @version 1.0
 */
@Slf4j
@Service
public class OrgCourseGroupServiceImpl implements OrgCourseGroupService {

    private static final Logger logger = LoggerFactory.getLogger(OrgCourseGroupServiceImpl.class);

    @Autowired
    private OrgCourseGroupDao orgCourseGroupDao;

    @Autowired
    private OrgCourseGroupRelateDao orgCourseGroupRelateDao;

    @Resource
    private OrgAccountDao orgAccountDao;

    @Autowired
    private OrgCourseDao orgCourseDao;

    /**
     * 更新机构自定义分类名称
     * 
     * @param orgId:机构ID
     * @param courseGroupId:yunying.org_course_group.id
     * @param groupName:yunying.org_course_group.name 自定义分类名称
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void modify(Integer orgId, Integer courseGroupId, String groupName) throws BussinessException {
        logger.info("modify.orgId:{}, courseGroupId:{}", orgId, courseGroupId);
        OrgCourseGroup orgCourseGroup = orgCourseGroupDao.getById(courseGroupId);

        if (orgCourseGroup == null) {
            throw new BussinessException(OrgErrorCode.NO_SUBJECT_FOR_REQ);
        }
        if (orgCourseGroup.getOrgId().intValue() != orgId.intValue()) {
            throw new BussinessException(CommonErrorCode.PERMISSION_DENY);
        }
        if (!groupName.equals(orgCourseGroup.getName())) {
            orgCourseGroup.setName(groupName);
            orgCourseGroup.setUpdateTime(new Date());
            this.orgCourseGroupDao.update(orgCourseGroup, new String[] { "name", "updateTime" });
        }
    }

    @Override
    public Integer add(Integer orgId, String groupName) {
        if (orgCourseGroupDao.countByName(orgId, groupName) > 0) {
            throw new BussinessException(OrgErrorCode.COURSE_GROUP_NAME_EXISTS);
        }
        OrgCourseGroup orgCourseGroup = new OrgCourseGroup();
        orgCourseGroup.setName(groupName);
        orgCourseGroup.setOrgId(orgId);
        orgCourseGroupDao.save(orgCourseGroup, "name", "orgId");
        return orgCourseGroup.getId();
    }

    @Override
    public Integer addLast(Integer orgId, String groupName) {
        if (orgCourseGroupDao.countByName(orgId, groupName) > 0) {
            throw new BussinessException(OrgErrorCode.COURSE_GROUP_NAME_EXISTS);
        }

        List<OrgCourseGroup> orgCourseGroups = this.orgCourseGroupDao.getOrgCourseGroupList(orgId);

        Collection<Integer> groupIds = group2Ids(orgCourseGroups);

        OrgCourseGroup orgCourseGroup = new OrgCourseGroup();
        orgCourseGroup.setName(groupName);
        orgCourseGroup.setOrgId(orgId);
        orgCourseGroupDao.save(orgCourseGroup, "name", "orgId");

        groupIds.add(orgCourseGroup.getId());
        order(groupIds);
        return orgCourseGroup.getId();
    }

    private Collection<Integer> group2Ids(Collection<OrgCourseGroup> orgCourseGroups) {
        List<Integer> result = Lists.newArrayList();
        for (OrgCourseGroup group : orgCourseGroups) {
            result.add(group.getId());
        }
        return result;
    }

    private Long getOrgNumber(Long orgId) {
        OrgAccount orgAccount = this.orgAccountDao.getAccountById(orgId.intValue(), "number");
        if (orgAccount == null) {
            log.error("orgId={} not exist in orgAccount.", orgId);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "机构帐号错误");
        }
        return orgAccount.getNumber().longValue();
    }

    @Override
    public List<OrgCourseGroupDto> getByCourseId(Long orgId, Long courseId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "illegal orgId!");
        Preconditions.checkArgument(courseId != null && courseId > 0, "illegal courseId!");
        List<OrgCourseGroupDto> list = Lists.newArrayList();
        List<Integer> groupIds =
            this.orgCourseGroupRelateDao.getGroupIdByCourseId(courseId, CourseType.ORG_COURSE.getCode());
        if (CollectionUtils.isEmpty(groupIds)) {
            return Collections.emptyList();
        }
        Long orgNumber = this.getOrgNumber(orgId);
        Collection<Long> courseIds = orgCourseDao.getCourseIds(orgNumber, null, CourseTypeEnum.IS_COURSE_TRUE.getCode(),null,null);
        Map<Integer, Integer> cntMap = this.orgCourseGroupDao.getCourseCountMap(orgId.intValue(), courseIds, groupIds);
        List<OrgCourseGroup> orgCourseGroups = orgCourseGroupDao.getByIds(groupIds);
        for (OrgCourseGroup group : orgCourseGroups) {
            if (group.getIsDeleted() != DeleteStatus.DELETED.getValue()) {
                OrgCourseGroupDto orgCourseGroupDto = new OrgCourseGroupDto();
                orgCourseGroupDto.setGroupId(group.getId());
                orgCourseGroupDto.setGroupName(group.getName());
                Integer cnt = cntMap.get(group.getId());
                orgCourseGroupDto.setCourseNum(cnt == null ? 0 : cnt);
                list.add(orgCourseGroupDto);
            }
        }
        return list;
    }

    @Override
    public List<OrgCourseGroupDto> list(Integer orgId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "illegal orgId!");
        List<OrgCourseGroupDto> list = Lists.newArrayList();
        List<OrgCourseGroup> orgCourseGroups = this.orgCourseGroupDao.getOrgCourseGroupList(orgId);
        log.debug("orgCourseGroups={}", orgCourseGroups);
        if (CollectionUtils.isEmpty(orgCourseGroups)) {
            return Collections.emptyList();
        }
        Collection<Integer> groupIds = group2Ids(orgCourseGroups);
        Long orgNumber = this.getOrgNumber(orgId.longValue());
        Collection<Long> courseIds = orgCourseDao.getCourseIds(orgNumber, null,CourseTypeEnum.IS_COURSE_TRUE.getCode(),null,null);
        Map<Integer, Integer> cntMap = this.orgCourseGroupDao.getCourseCountMap(orgId, courseIds, groupIds);
        for (OrgCourseGroup group : orgCourseGroups) {
            OrgCourseGroupDto orgCourseGroupDto = new OrgCourseGroupDto();
            orgCourseGroupDto.setGroupId(group.getId());
            orgCourseGroupDto.setGroupName(group.getName());
            Integer cnt = cntMap.get(group.getId());
            orgCourseGroupDto.setCourseNum(cnt == null ? 0 : cnt);
            list.add(orgCourseGroupDto);
        }
        return list;
    }

    @Override
    public List<OrgCourseGroupDto> listNoCount(Integer orgId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "illegal orgId!");
        List<OrgCourseGroupDto> list = Lists.newArrayList();
        List<OrgCourseGroup> orgCourseGroups = this.orgCourseGroupDao.getOrgCourseGroupList(orgId);
        if (CollectionUtils.isEmpty(orgCourseGroups)) {
            return Collections.emptyList();
        }
        for (OrgCourseGroup group : orgCourseGroups) {
            OrgCourseGroupDto orgCourseGroupDto = new OrgCourseGroupDto();
            orgCourseGroupDto.setGroupId(group.getId());
            orgCourseGroupDto.setGroupName(group.getName());
            list.add(orgCourseGroupDto);
        }
        return list;
    }

    @Override
    public Integer delete(Integer groupId) {

        OrgCourseGroup orgCourseGroup = new OrgCourseGroup();
        orgCourseGroup.setId(groupId);
        orgCourseGroup.setIsDeleted(1);
        return this.orgCourseGroupDao.update(orgCourseGroup, "isDeleted");
    }

    @Override
    public void order(Collection<Integer> groupIds) {

        if (CollectionUtils.isEmpty(groupIds)) {
            return;
        }
        int order = groupIds.size();
        for (Integer id : groupIds) {
            OrgCourseGroup orgCourseGroup = orgCourseGroupDao.getById(id);
            orgCourseGroup.setDisplayOrder(order--);
            orgCourseGroup.setUpdateTime(new Date());
            orgCourseGroupDao.saveOrUpdate(orgCourseGroup);
        }
    }

    @Override
    public List<OrgCourseGroupDto> list(Integer orgId, String key, PageDto pageDto) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "illegal orgId!");
        List<OrgCourseGroupDto> list = Lists.newArrayList();
        List<OrgCourseGroup> orgCourseGroups = this.orgCourseGroupDao.getOrgCourseGroupList(orgId,key,pageDto);
        log.debug("orgCourseGroups={}", orgCourseGroups);
        if (CollectionUtils.isEmpty(orgCourseGroups)) {
            return Collections.emptyList();
        }
        Collection<Integer> groupIds = group2Ids(orgCourseGroups);
        Long orgNumber = this.getOrgNumber(orgId.longValue());
        Collection<Long> courseIds = orgCourseDao.getCourseIds(orgNumber, null,CourseTypeEnum.IS_COURSE_TRUE.getCode(),null,null);
        Map<Integer, Integer> cntMap = this.orgCourseGroupDao.getCourseCountMap(orgId, courseIds, groupIds);
        for (OrgCourseGroup group : orgCourseGroups) {
            OrgCourseGroupDto orgCourseGroupDto = new OrgCourseGroupDto();
            orgCourseGroupDto.setGroupId(group.getId());
            orgCourseGroupDto.setGroupName(group.getName());
            Integer cnt = cntMap.get(group.getId());
            orgCourseGroupDto.setCourseNum(cnt == null ? 0 : cnt);
            list.add(orgCourseGroupDto);
        }
        return list;
    }

}
