
 /**
 * Baijiahulian.com Inc.
 * Copyright (c) 2014-2015 All Rights Reserved.
 */
    
package com.baijia.tianxiao.sal.course.service.impl;

import java.sql.Timestamp;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.constants.CourseType;
import com.baijia.tianxiao.constants.org.BizConf;
import com.baijia.tianxiao.dal.course.dao.OrgRecommendCourseDao;
import com.baijia.tianxiao.dal.course.dao.TeacherClassCourseDao;
import com.baijia.tianxiao.dal.course.po.OrgRecommendCourse;
import com.baijia.tianxiao.dal.course.po.TeacherClassCourse;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.enums.ModifyStatus;
import com.baijia.tianxiao.sal.course.dto.request.OrgRecommendCourseRequestDto;
import com.baijia.tianxiao.sal.course.service.OrgRecommendCourseService;
import com.baijia.tianxiao.util.BaseUtils;
import com.google.common.collect.Lists;


/**
 * @title OrgRecommendCourseServiceImpl
 * @desc TODO 
 * @author zhangbing
 * @date 2015年12月8日
 * @version 1.0
 */
@Service("orgRecommendCourseService")
public class OrgRecommendCourseServiceImpl implements OrgRecommendCourseService {
    
    private static final Logger logger = LoggerFactory.getLogger(OrgRecommendCourseServiceImpl.class);

    @Autowired
    private OrgRecommendCourseDao orgRecommendCourseDao;
    
    @Autowired
    private OrgAccountDao orgAccountDao;
    
    @Autowired
    private OrgCourseDao orgCourseDao;
    
    @Autowired
    private TeacherClassCourseDao teacherClassCourseDao;

    @Override
    public List<OrgRecommendCourse> getOrgRecommentCourseList(Integer orgId) {
        return this.orgRecommendCourseDao.getRecommendCourseList(orgId);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void removeRecommendCourse(Integer orgId, Long courseId, Integer courseType) {
        OrgRecommendCourse recommendCourse = this.orgRecommendCourseDao.getOrgRecommendCourse(orgId, courseId, courseType);
        
        if (recommendCourse != null) {
            recommendCourse.setIsDeleted(DeleteStatus.DELETED.getValue());
            recommendCourse.setUpdateTime(new Timestamp(System.currentTimeMillis()));
            
            logger.debug("removeRecommendCourse.orgId:{}, courseId:{}, courseType:{}, recommendCourse:{}", orgId, courseId, courseType, recommendCourse);
            this.orgRecommendCourseDao.removeRecommendCourse(recommendCourse);
        }
    }

    /**
     * 神兽逻辑ca
     */
    @Override
    public void batchEdit(List<OrgRecommendCourseRequestDto> recommendCourses, Integer orgId) {
        try {
            int minDisPlayOrder = this.getMaxDisplayOrder(recommendCourses, orgId.longValue());
            List<OrgRecommendCourse> cache = Lists.newArrayList();
            logger.info("batchEdit.recommendCourses:{}", recommendCourses);
            for (OrgRecommendCourseRequestDto dto : recommendCourses) {
                Integer modifyStatus = dto.getModifyStatus();
                logger.info("batchEdit.dto:{},modifyStatus:{}", dto, modifyStatus);
                OrgRecommendCourse sourceRecommendCourse = this.validateParams(orgId, dto.getCourseId(), dto.getCourseType());
                    switch (ModifyStatus.values()[modifyStatus]) {
                        case NORMAL:
                            if (sourceRecommendCourse != null && minDisPlayOrder != sourceRecommendCourse.getDisplayOrder()) {
                                sourceRecommendCourse.setUpdateTime(new Timestamp(System.currentTimeMillis()));
                                sourceRecommendCourse.setDisplayOrder(minDisPlayOrder--);
                                this.orgRecommendCourseDao.update(sourceRecommendCourse, new String[]{"updateTime", "displayOrder"});
                                logger.info("NORMAL.sourceRecommendCourse:{}", sourceRecommendCourse);
                            }
                            break;
                        case UPDATE:
                            if (sourceRecommendCourse != null) {
                                logger.info("UPDATE.sourceRecommendCourse:{}", sourceRecommendCourse);
                                cache.add(sourceRecommendCourse);
                            }
                            break;
                        case ADDED:
                            if (sourceRecommendCourse != null) {
                                sourceRecommendCourse.setUpdateTime(new Timestamp(System.currentTimeMillis()));
                                sourceRecommendCourse.setDisplayOrder(minDisPlayOrder--);
                                this.orgRecommendCourseDao.update(sourceRecommendCourse, new String[]{"updateTime", "displayOrder"});
                                logger.info("ADDED.sourceRecommendCourse:{}", sourceRecommendCourse);
                            } else {
                                save(dto, minDisPlayOrder--, orgId);
                            }
                            break;
                        case DELETE:
                            if (sourceRecommendCourse != null) {
                                sourceRecommendCourse.setUpdateTime(new Timestamp(System.currentTimeMillis()));
                                sourceRecommendCourse.setDisplayOrder(0);
                                sourceRecommendCourse.setIsDeleted(DeleteStatus.DELETED.getValue());
                                this.orgRecommendCourseDao.update(sourceRecommendCourse, new String[]{"updateTime", "displayOrder","isDeleted"});
                                logger.info("DELETE.sourceRecommendCourse:{}", sourceRecommendCourse);
                            }
                            break;
                        default:
                            break;
                    }
                
                doAfter(cache, orgId, minDisPlayOrder);
                logger.info("batchEdit.orgId:{}, dto:{}, sourceRecommendCourse:{}", orgId, dto, sourceRecommendCourse);
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        
    }
    
    @SuppressWarnings("unchecked")
    private int getMaxDisplayOrder(List<OrgRecommendCourseRequestDto> recommendCourses, Long orgId) {
        List<Long> excludeIds = BaseUtils.getListFilter(recommendCourses, "courseId");
        int maxDisplayOrder = this.orgRecommendCourseDao.getMaxDisplayOrderByExcludeIds(orgId, excludeIds);
        if (maxDisplayOrder == 0) {
            maxDisplayOrder = recommendCourses.size() + 1;
        } else {
            maxDisplayOrder += recommendCourses.size() + 1;
        }
        return maxDisplayOrder;
    }
    
    /**
     * 所有添加、未做修改、删除的课程操作完成后,再计算推荐课程的总数是否达到上限，如果达到上限
     * cache中多余的(totalCount - BizConf.MAX_RECOMMEND_COURSE_COUNT)课程都会从推荐课程中移除
     * 如果totalCount未达到上限，cache中的数据从
     * 
     * @param cache
     * @param orgId
     */
    private void doAfter(List<OrgRecommendCourse> cache, int orgId, int threadhold) {
        if (cache != null && !cache.isEmpty()) {
            int totalCount = this.orgRecommendCourseDao.getOrgRecommendCourseCount(orgId);
            
            int size = cache.size();
            if (totalCount > BizConf.MAX_RECOMMEND_COURSE_COUNT) {
                int edgeVal = totalCount - BizConf.MAX_RECOMMEND_COURSE_COUNT;
                if(size >= edgeVal) {
                    List<OrgRecommendCourse> deleteCourse = cache.subList((size - edgeVal), size);
                    deleteRecommendCourse(deleteCourse);

                    if (edgeVal > 0) {
                        List<OrgRecommendCourse> updateCourse = cache.subList(0, (size - edgeVal));
                        updateCourse(updateCourse, threadhold);
                    }
                }
            } else {
                updateCourse(cache, threadhold);
            }
        }
    }
    
    /**
     * 待更新的课程:被替换的课程有可能会从推荐列表中删除,也有可能保留，displayOrder有可能保留也有可能更新
     * 当未更新、添加、删除的课程操作后，数据库中推荐课程总数totalCount大于10，需要将cache中删除(totalCount - 10)个课程
     * 当未更新、添加、删除的课程操作后，数据库中推荐课程总数totalCount小于等于10，需要将cache中课程重新排序，排序原则：
     * 1.最先被替换的排在cache中的第一位,其他依次递减
     * @param updateCourse
     */
    private void updateCourse(List<OrgRecommendCourse> updateCourse, int threadhold) {
        if (updateCourse != null && !updateCourse.isEmpty()) {
            for (OrgRecommendCourse course : updateCourse) {
                if (course.getDisplayOrder() != (threadhold--)) {
                    course.setUpdateTime(new Timestamp(System.currentTimeMillis()));
                    course.setDisplayOrder(threadhold--);
                    this.orgRecommendCourseDao.update(course, new String[]{"updateTime", "displayOrder"});
                }
            }
        }
    }
    
    /**
     * 将超出总数限制的课程从推荐列表移除
     * @param cache 待处理的课程
     * @param degeVal 需要删除的课程数
     * @param size cache.size()
     */
    private void deleteRecommendCourse(List<OrgRecommendCourse> deleteCourse) {
        if (deleteCourse != null && !deleteCourse.isEmpty()) {
            for (OrgRecommendCourse item : deleteCourse) {
                item.setUpdateTime(new Timestamp(System.currentTimeMillis()));
                item.setDisplayOrder(0);
                item.setIsDeleted(DeleteStatus.DELETED.getValue());
                logger.info("UPDATE.sourceRecommendCourse:{}", item);
                this.orgRecommendCourseDao.update(item, new String[]{"updateTime", "displayOrder","isDeleted"});
            }
        }
    }
    
    /**
     * 新添加的课程如果不属于推荐课程save it
     * @param dto
     * @param minDisPlayOrder
     * @param orgId
     */
    private void save(OrgRecommendCourseRequestDto dto, int minDisPlayOrder, int orgId) {
          OrgRecommendCourse destRecommendCourse = new OrgRecommendCourse();
          destRecommendCourse.setCourseid(dto.getCourseId().intValue());
          destRecommendCourse.setCourseType(dto.getCourseType());
          destRecommendCourse.setCreateTime(new Timestamp(System.currentTimeMillis()));
          destRecommendCourse.setDisplayOrder(minDisPlayOrder);
          destRecommendCourse.setIsDeleted(DeleteStatus.NORMAL.getValue());
          destRecommendCourse.setOrgId(orgId);
          
          logger.info("save.destRecommendCourse:{}, dto:{}", destRecommendCourse, dto);
          this.orgRecommendCourseDao.save(destRecommendCourse);
    }
    
//    /**
//     * 新增或者更新推荐课程
//     * @param orgId
//     * @param totalCount
//     * @param displayOrder
//     * @param dto
//     * @param account
//     */
//    private void saveOrUpdate(Integer orgId, int totalCount, int displayOrder, OrgRecommendCourseRequestDto dto, OrgAccount account) {
//        OrgRecommendCourse destRecommendCourse = this.validateParams(orgId, dto.getDestCourseId(), dto.getDestCourseType());
//        if (destRecommendCourse != null) {
//            if (totalCount < BizConf.MAX_RECOMMEND_COURSE_COUNT) {
//                destRecommendCourse.setDisplayOrder(displayOrder);
//                destRecommendCourse.setUpdateTime(new Timestamp(System.currentTimeMillis()));
//                this.orgRecommendCourseDao.update(destRecommendCourse, new String[]{"displayOrder","updateTime"});
//            } else {
//                if (validateCourse(account.getNumber(), dto.getDestCourseId(), dto.getDestCourseType())) {
//                    destRecommendCourse = new OrgRecommendCourse();
//                    destRecommendCourse.setCourseid(dto.getDestCourseId().intValue());
//                    destRecommendCourse.setCourseType(dto.getDestCourseType());
//                    destRecommendCourse.setCreateTime(new Timestamp(System.currentTimeMillis()));
//                    destRecommendCourse.setDisplayOrder(displayOrder);
//                    destRecommendCourse.setIsDeleted(DeleteStatus.NORMAL.getValue());
//                    destRecommendCourse.setOrgId(orgId);
//                    this.orgRecommendCourseDao.save(destRecommendCourse);
//                }
//            }
//        }
//    }
    
    /**
     * 
     * @param orgNumber  机构number
     * @param courseId   课程Id
     * @param courseType 课程类型
     * @return
     */
    private boolean validateCourse(Integer orgNumber, Long courseId, Integer courseType) {
        if (courseType == CourseType.CLASS.getCode()) {
            TeacherClassCourse classCourse = this.teacherClassCourseDao.getById(courseId);
            return classCourse != null && classCourse.getOrganizationNumber() == orgNumber;
        } else if (courseType == CourseType.ORG_COURSE.getCode()) {
            OrgCourse orgCourse = this.orgCourseDao.getById(courseId);
            return orgCourse != null && orgCourse.getOrgNumber() == orgNumber.longValue();
        }
        return false;
    }
    
    /**
     * 参数的验证
     * @param orgId
     * @param courseId
     * @param courseType
     * @return
     */
    private OrgRecommendCourse validateParams(Integer orgId, Long courseId, Integer courseType) {
        OrgRecommendCourse result = this.orgRecommendCourseDao.getOrgRecommendCourse(orgId, courseId, courseType);
        logger.info("validateParams.result:{}", result);
        return result;
    }
}

    