/**
 * Baijiahulian.com Inc. Copyright (c) 2015-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.common.impl;

import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import com.baijia.tianxiao.dal.signup.constant.TransferClassOrder;
import com.baijia.tianxiao.sal.common.constant.FollowType;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.baijia.tianxiao.constant.Flag;
import com.baijia.tianxiao.constant.StudentFiannceOpType;
import com.baijia.tianxiao.constants.TianXiaoConstant;
import com.baijia.tianxiao.dal.callservice.constant.OrgCallStatus;
import com.baijia.tianxiao.dal.callservice.po.OrgPushCallInfo;
import com.baijia.tianxiao.dal.constant.ChargeType;
import com.baijia.tianxiao.dal.constant.ChargeUnit;
import com.baijia.tianxiao.dal.org.constant.StudentType;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.roster.constant.AddType;
import com.baijia.tianxiao.dal.roster.constant.DownLoadStatus;
import com.baijia.tianxiao.dal.roster.constant.MobileStatus;
import com.baijia.tianxiao.dal.roster.dao.TxConsultUserDao;
import com.baijia.tianxiao.dal.roster.dao.TxStudentCommentDao;
import com.baijia.tianxiao.dal.roster.po.TxConsultUser;
import com.baijia.tianxiao.dal.roster.po.TxStudentComment;
import com.baijia.tianxiao.dal.signup.po.OrgSignupCourse;
import com.baijia.tianxiao.dal.signup.po.OrgSignupInfo;
import com.baijia.tianxiao.dal.todo.po.TxBacklog;
import com.baijia.tianxiao.dal.todo.po.TxbacklogParticipant;
import com.baijia.tianxiao.sal.common.api.AccountApiService;
import com.baijia.tianxiao.sal.common.api.TXStudentCommentAPIService;
import com.baijia.tianxiao.util.GenericsUtils;

import lombok.extern.slf4j.Slf4j;

@Slf4j
@Service
public class TXStudentCommentAPIServiceImpl implements TXStudentCommentAPIService {

    @Autowired
    private TxStudentCommentDao txStudentCommentDao;
    @Autowired
    private TxConsultUserDao consultUserDao;
    @Autowired
    private OrgStudentDao orgStudentDao;
    @Autowired
    private AccountApiService accountApiService;

    @Override
    public TxStudentComment saveTxStudentComment(TxStudentComment comment) {
        txStudentCommentDao.save(comment, false);
        if (comment.getUserId() != null) {
            orgStudentDao.updateLastRemindTime(comment.getOrgId(), comment.getUserId());
        }
        return comment;
    }

    TxStudentComment buildTxStudentComment(TxConsultUser consultUser) {
        TxStudentComment comment = new TxStudentComment();
        comment.setConsultUserId(consultUser.getId());
        comment.setUserId(consultUser.getUserId());
        comment.setOrgId(consultUser.getOrgId());
        comment.setOrigin(consultUser.getConsultSource());
        comment.setIsSystem(AddType.SYSTEM.getCode());
        comment.setType(0);
        comment.setFollowType(0);
        return comment;

    }

    @Override
    public TxStudentComment saveByOrgActivitySignUp(TxConsultUser consultUser, String activityName, String userName,
                                                    String mobile, String award) {
        TxStudentComment comment = buildTxStudentComment(consultUser);

        StringBuffer content = new StringBuffer("活动名称：");
        content.append(activityName).append("\n");
        content.append("姓名：").append(userName).append("\n");
        content.append("电话号码：").append(mobile);
        if (StringUtils.isNotBlank(award)) {// 存在award 说明来自抽奖活动
            content.append("\n").append("中奖信息：").append(award);
        }
        comment.setContent(content.toString());
        txStudentCommentDao.save(comment, false);
        updateLastRemindTime(comment);
        return comment;
    }

    @Override
    public TxStudentComment saveByOrgCall400(TxConsultUser consultUser, OrgPushCallInfo callInfo,
                                             boolean isNewConsultUser) {
        TxStudentComment comment = buildTxStudentComment(consultUser);

        if (isNewConsultUser) {
            comment.setContent("系统添加了客户: " + (StringUtils.isNotBlank(consultUser.getName()) ? consultUser.getName()
                    : TianXiaoConstant.ANONYMOUS_CONSULT_USER) + "\n");
        } else {
            comment.setContent("");
        }

        if (callInfo.getStatus().intValue() == OrgCallStatus.CONNECTED.getCode()) {
            if (null != callInfo.getStorageId() && callInfo.getStorageId() > 0) {
                comment.setSoundId(callInfo.getStorageId());
                comment.setDownStatus(DownLoadStatus.FINISH.getCode());
            } else {
                comment.setSoundId(callInfo.getId());
                comment.setDownStatus(DownLoadStatus.UNFINISH.getCode());
            }
            comment.setSeconds(callInfo.getDuringTime());
        }
        comment.setCallStatus(callInfo.getStatus());
        comment.setIsMobile(MobileStatus.IS_400_CALL.getCode());

        txStudentCommentDao.save(comment, false);
        updateLastRemindTime(comment);
        return comment;
    }

    private void updateLastRemindTime(TxStudentComment comment) {
        if (comment.getUserId() != null && comment.getUserId() > 0) {
            orgStudentDao.updateLastRemindTime(comment.getOrgId(), comment.getUserId());
        } else {
            consultUserDao.updateLastRemindTime(comment.getOrgId(), comment.getConsultUserId());
        }
    }

    @Override
    public TxStudentComment saveByAdvisory(TxConsultUser consultUser, Long userId, String advisoryWords,
                                           boolean isNewConsultUser) {
        TxStudentComment comment = buildTxStudentComment(consultUser);

        StringBuffer content = new StringBuffer();
        if (isNewConsultUser) {
            content.append("系统添加了客户: " + (StringUtils.isNotBlank(consultUser.getName()) ? consultUser.getName()
                    : TianXiaoConstant.APPOINTMENT_STUDENT_NAME) + "\n");
        }
        content.append("留单内容: " + (StringUtils.isNotBlank(advisoryWords) ? advisoryWords : "无"));
        comment.setContent(content.toString());
        comment.setUserId(userId);

        txStudentCommentDao.save(comment, false);
        updateLastRemindTime(comment);
        return comment;
    }

    @Override
    public TxStudentComment saveByConsultUserManualAdd(TxConsultUser consultUser) {
        TxStudentComment comment = buildTxStudentComment(consultUser);

        String contentFmt = "添加了线索 %s";
        contentFmt = String.format(contentFmt, consultUser.getName());
        comment.setContent(contentFmt);

        txStudentCommentDao.save(comment, false);
        updateLastRemindTime(comment);
        return comment;
    }

    @Override
    public TxStudentComment saveByConsultUserDel(TxConsultUser consultUser) {
        TxStudentComment comment = buildTxStudentComment(consultUser);

        String contentFmt = "删除线索 %s";
        contentFmt = String.format(contentFmt, consultUser.getName());
        comment.setContent(contentFmt);

        txStudentCommentDao.save(comment, false);
        updateLastRemindTime(comment);
        return comment;
    }

    @Override
    public TxStudentComment saveByConsultUserUpdate(TxConsultUser consultUser) {
        TxStudentComment comment = buildTxStudentComment(consultUser);

        String contentFmt = "修改线索 %s";
        contentFmt = String.format(contentFmt, consultUser.getName());
        comment.setContent(contentFmt);

        txStudentCommentDao.save(comment, false);
        updateLastRemindTime(comment);
        return comment;
    }

    @Override
    public TxStudentComment saveByConsultUserAssign(TxConsultUser consultUser, String ownerName) {
        TxStudentComment comment = buildTxStudentComment(consultUser);

        String contentFmt = "%s 被分配了 %s 这条线索.";
        contentFmt = String.format(contentFmt, ownerName, consultUser.getName());
        comment.setContent(contentFmt);

        txStudentCommentDao.save(comment, false);
        updateLastRemindTime(comment);
        return comment;
    }

    @Override
    public TxStudentComment saveByConsultUserPull(TxConsultUser consultUser, String operatorName) {
        TxStudentComment comment = buildTxStudentComment(consultUser);

        String contentFmt = "%s 领取了 %s 这条线索.";
        contentFmt = String.format(contentFmt, operatorName, consultUser.getName());
        comment.setContent(contentFmt);

        txStudentCommentDao.save(comment, false);
        updateLastRemindTime(comment);
        return comment;
    }

    @Override
    public TxStudentComment saveByConsultUserPush(TxConsultUser consultUser, String operatorName) {
        TxStudentComment comment = buildTxStudentComment(consultUser);

        String contentFmt = "%s 把线索释放回公海.";
        contentFmt = String.format(contentFmt, operatorName, consultUser.getName());
        comment.setContent(contentFmt);

        txStudentCommentDao.save(comment, false);
        updateLastRemindTime(comment);
        return comment;
    }

    @Override
    public void batchSaveByRelease(List<Long> consulterIds) {

        List<TxStudentComment> list = new ArrayList<TxStudentComment>();
        String content = "线索到期 自动释放回公海.";

        for (int i = 0; i < consulterIds.size(); i++) {
            TxConsultUser consultUser = consultUserDao.getById(consulterIds.get(i));
            TxStudentComment comment = buildTxStudentComment(consultUser);
            comment.setContent(content);
            comment.setFollowType(FollowType.RELEASE.getType());
            list.add(comment);

            // 每100批量插入
            if ((i != 0 && i % 100 == 0) || i == consulterIds.size() - 1) {
                txStudentCommentDao.saveAll(list, false);
                list = new ArrayList<TxStudentComment>();
            }
            updateLastRemindTime(comment);
        }
    }

    @Override
    public TxStudentComment saveByConsultUserChangeStatus(TxConsultUser consultUser, String operatorName,
                                                          String reason) {
        TxStudentComment comment = buildTxStudentComment(consultUser);

        String contentFmt = "";
        if (consultUser.getIsInvalid().intValue() == Flag.FALSE.getInt()) {
            contentFmt = "%s 将线索 %s 的状态修改为有效.";
            contentFmt = String.format(contentFmt, operatorName, consultUser.getName());
        } else {
            contentFmt = "%s 将线索 %s 的状态修改为无效.\n标记原因:%s";
            contentFmt = String.format(contentFmt, operatorName, consultUser.getName(), reason);
        }
        comment.setContent(contentFmt);
        comment.setFollowType(FollowType.CHANGE_STATUS.getType());

        txStudentCommentDao.save(comment, false);
        updateLastRemindTime(comment);
        return comment;
    }

    @Override
    public TxStudentComment saveByConsultUserPassTo(TxConsultUser consultUser, String from, String to) {
        TxStudentComment comment = buildTxStudentComment(consultUser);

        String contentFmt = "%s 把线索转交给 %s";
        contentFmt = String.format(contentFmt, from, to);
        comment.setContent(contentFmt);

        txStudentCommentDao.save(comment, false);
        updateLastRemindTime(comment);
        return comment;
    }

    @Override
    public TxStudentComment saveByMobileChange(TxConsultUser consultUser, String oldMobile, String newMobile) {
        if (!oldMobile.equals(newMobile)) {
            TxStudentComment comment = buildTxStudentComment(consultUser);

            String contentFmt = " 将手机号【%s】修改为【%s】";
            contentFmt = String.format(contentFmt, oldMobile, newMobile);
            comment.setContent(contentFmt);
            txStudentCommentDao.save(comment, false);
            updateLastRemindTime(comment);
            return comment;
        }
        return null;
    }

    @Override
    public TxStudentComment saveByNameChange(TxConsultUser consultUser, String oldName, String newName) {
        if (!oldName.equals(newName)) {
            TxStudentComment comment = buildTxStudentComment(consultUser);

            String contentFmt = " 将姓名【%s】修改为【%s】";
            contentFmt = String.format(contentFmt, oldName, newName);
            comment.setContent(contentFmt);
            txStudentCommentDao.save(comment, false);
            updateLastRemindTime(comment);
            return comment;
        }
        return null;
    }

    @Override
    public TxStudentComment saveByStudentMerge(TxConsultUser sourceConsultUser, TxConsultUser destConsultUser,
                                               Long orgId) {
        List<TxStudentComment> comments =
                txStudentCommentDao.getComments(sourceConsultUser.getId(), orgId, StudentType.CONSULT_USER.getCode());

        // 修改源跟进记录所属人
        if (comments != null && !comments.isEmpty()) {
            for (TxStudentComment comment : comments) {
                comment.setConsultUserId(destConsultUser.getId());
                comment.setUserId(destConsultUser.getUserId());
                comment.setUpdateTime(new Date());
                this.txStudentCommentDao.update(comment, new String[]{"consultUserId", "userId", "updateTime"});
            }
        }

        // 添加合并跟进记录
        String sourceName = sourceConsultUser.getName();
        String sourceNickName = sourceConsultUser.getNickName();
        String destName = destConsultUser.getName();
        String destNickName = destConsultUser.getNickName();

        String sourceMobile = sourceConsultUser.getMobile();
        String destMobile = destConsultUser.getMobile();
        StringBuffer sb = new StringBuffer();
        sb.append("您将学员【" + getConsulterName(sourceName, sourceNickName) + "】");
        if (StringUtils.isNotBlank(sourceMobile)) {
            sb.append("【" + sourceMobile + "】");
        }
        sb.append("和学员【" + getConsulterName(destName, destNickName) + "】");
        if (StringUtils.isNotBlank(destMobile)) {
            sb.append("【" + destMobile + "】");
        }
        sb.append("合并了");

        TxStudentComment addComment = new TxStudentComment();
        addComment.setCallStatus(0);
        addComment.setConsultUserId(destConsultUser.getId());
        addComment.setUserId(destConsultUser.getUserId());
        addComment.setOrgId(orgId);
        addComment.setCreateTime(new Date());
        addComment.setIsSystem(AddType.SYSTEM.getCode());
        addComment.setContent(sb.toString());

        this.txStudentCommentDao.save(addComment);
        updateLastRemindTime(addComment);
        log.info("orgId:{}, addComment:{}", orgId, addComment);

        return addComment;
    }

    private String getConsulterName(String source, String dest) {
        return StringUtils.isNotBlank(source) ? source : StringUtils.isNotBlank(dest) ? dest : "匿名用户";
    }

    public static void main(String args[]) {
        System.out.println(new DecimalFormat("0.00").format(0.01));
        System.out.println(new DecimalFormat("0.00").format((double) 0 / 100));
        System.out.println(new DecimalFormat("0.00").format((double) 12300 / 100));
        System.out.println(new DecimalFormat("0.00").format((double) 1234 / 100));
    }

    @Override
    public TxStudentComment saveBySignUpPurchase(OrgSignupInfo orgSignupInfo, String courseNames) {
        StringBuffer sb = new StringBuffer();
        sb.append("报班:[").append(courseNames).append("]总价:¥");
        if (orgSignupInfo.getTotalPrices().equals(0)) {
            sb.append("0.00").append("元");
        } else {
            sb.append(new DecimalFormat("#.00").format(orgSignupInfo.getTotalPrices().doubleValue() / 100)).append("元");
        }

        TxStudentComment comment = new TxStudentComment();
        comment.setOrgId(orgSignupInfo.getOrgId());
        comment.setUserId(orgSignupInfo.getUserId());
        comment.setCreateTime(orgSignupInfo.getUpdateTime());
        comment.setUpdateTime(orgSignupInfo.getUpdateTime());
        comment.setContent(sb.toString());
        comment.setIsSystem(AddType.SYSTEM.getCode());
        comment.setFollowType(FollowType.SIGNUP_COURSE.getType());

        this.txStudentCommentDao.save(comment);
        updateLastRemindTime(comment);
        return comment;
    }

    @Override
    public TxStudentComment saveByCoursePurchase(Long orgId, Long userId, Date createTime, String courseName,
                                                 Double price) {
        String content = "报班:[%s]总价:¥%s元";
        content = String.format(content, courseName, price);

        TxStudentComment comment = new TxStudentComment();
        comment.setOrgId(orgId);
        comment.setUserId(userId);
        comment.setCreateTime(createTime);
        comment.setUpdateTime(createTime);
        comment.setContent(content);
        comment.setIsSystem(AddType.SYSTEM.getCode());
        comment.setFollowType(FollowType.SIGNUP_COURSE.getType());

        this.txStudentCommentDao.save(comment);
        updateLastRemindTime(comment);
        return comment;
    }

    @Override
    public TxStudentComment saveByAddStudent(OrgStudent student) {
        TxStudentComment comment = new TxStudentComment();
        if (StringUtils.isNotEmpty(student.getName())) {
            comment.setContent("添加学员 " + student.getName());
        } else {
            comment.setContent("添加学员 " + student.getNickName());
        }
        comment.setOrgId(student.getOrgId());
        comment.setUserId(student.getUserId());
        comment.setCreateTime(student.getCreateTime());
        comment.setIsSystem(AddType.SYSTEM.getCode());

        this.txStudentCommentDao.save(comment);
        updateLastRemindTime(comment);
        return comment;
    }

    @Override
    public TxStudentComment saveByStudentPassTo(OrgStudent student, String from, String to) {
        TxStudentComment comment = new TxStudentComment();
        String contentFmt = "%s 把学员转交给 %s";
        contentFmt = String.format(contentFmt, from, to);
        comment.setContent(contentFmt);
        comment.setOrgId(student.getOrgId());
        comment.setUserId(student.getUserId());
        comment.setCreateTime(new Date());
        comment.setIsSystem(AddType.SYSTEM.getCode());
        txStudentCommentDao.save(comment, false);
        updateLastRemindTime(comment);
        return comment;
    }

    private static final String chargeByPeriods = "[%s]实交：￥%s, 总课次：%s次";
    private static final String chargeByClassHour = "[%s]实交：￥%s，总课时：%s小时";
    private static final String chargeByTimes = "[%s]实交：￥%s，总课次：%s次";
    private static final String charegeByOthers = "[%s]总价:¥%s";
    private static final String transferClassByPeriods = "[%s]总价：￥%s，总课次：%s次。";
    private static final String transferClassByHour = "[%s]总价：￥%s，总课时：%s小时。";
    private static final String transferClassByTimes = "[%s]总价：￥%s，总课次：%s次。";


    @Override
    public TxStudentComment saveBySignUpPurchase(OrgSignupInfo orgSignupInfo,
                                                 Map<Long, OrgSignupCourse> orgSignupCourseMap, Map<Long, OrgCourse> orgCourseMap) {
        if (GenericsUtils.isNullOrEmpty(orgSignupCourseMap)) {
            log.info("can not find any signupCourse info :{}", orgSignupInfo);
            return null;
        }

        log.info("orgSignupCourseMap is:{} and orgCourseMap is:{} ", orgSignupCourseMap, orgCourseMap);
        StringBuilder sb = new StringBuilder("");
        if (orgSignupInfo.getOrderType() == TransferClassOrder.TRANSFER_ORDER.getCode()) {
            sb.append("转班报名: ");
        } else {
            sb.append("报班: ");
        }
        for (Map.Entry<Long, OrgCourse> entry : orgCourseMap.entrySet()) {
            Long courseId = entry.getKey();
            OrgSignupCourse osc = orgSignupCourseMap.get(courseId);
            if (osc == null) {
                continue;
            }
            OrgCourse oc = entry.getValue();
            int chargeType = oc.getChargeType();
            int chargeUnit = osc.getChargeUnit();
            String className = oc.getName();
            double balance = (osc.getTotalPayPrice()) / 100D;
            String balanceStr = new DecimalFormat("0.00").format(balance);
            int count = osc.getLessonCount() == 0 ? 1 : osc.getLessonCount();
            if (orgSignupInfo.getOrderType() == TransferClassOrder.TRANSFER_ORDER.getCode()) {
                if (ChargeType.chargeByPeriods(chargeType, chargeUnit)) { // 按期
                    int times = osc.getLessonCount();
                    sb.append(String.format(transferClassByPeriods, className, balanceStr, times));
                } else if (ChargeType.chargeByClassHour(chargeType, chargeUnit)) {
                    int totalCount = countClassHour(count, chargeUnit);
                    sb.append(String.format(transferClassByHour, className, balanceStr, totalCount));
                } else {
                    sb.append(String.format(transferClassByTimes, className, balanceStr, count));
                }
                sb.append("原班转出金额￥" + new DecimalFormat("0.00").format(orgSignupInfo.getTransferClassMoney() / 100d));
                if (orgSignupInfo.getTransferRefundMoney().longValue() > 0) {
                    if (orgSignupInfo.getTransferRefundType().intValue() == 1) {
                        sb.append("，现金退款￥").append(new DecimalFormat("0.00").format(orgSignupInfo.getTransferRefundMoney() / 100d));
                    } else {
                        sb.append("，退款至学员账户：￥").append(new DecimalFormat("0.00").format(orgSignupInfo.getTransferRefundMoney() / 100d));
                    }
                } else if (orgSignupInfo.getTotalPrices().longValue() + orgSignupInfo.getStudentPayPrice().longValue() > 0) {
                    sb.append("，补款实交：￥").append(new DecimalFormat("0.00").format((orgSignupInfo.getTotalPrices().doubleValue() + orgSignupInfo.getStudentPayPrice().doubleValue()) / 100d));
                } else {
                }
                sb.append("  ");
            } else {
                if (ChargeType.chargeByPeriods(chargeType, chargeUnit)) { // 按期
                    int times = osc.getLessonCount();
                    sb.append(String.format(chargeByPeriods, className, balanceStr, times)).append(",");
                } else if (ChargeType.chargeByClassHour(chargeType, chargeUnit)) {
                    int totalCount = countClassHour(count, chargeUnit);
                    sb.append(String.format(chargeByClassHour, className, balanceStr, totalCount)).append(",");
                } else if (ChargeType.chargeByTimes(chargeType, chargeUnit)) {
                    sb.append(String.format(chargeByTimes, className, balanceStr, count)).append(",");
                } else {
                    sb.append(String.format(charegeByOthers, className, balanceStr)).append(",");
                }
            }
        }
        String commentInfo = GenericsUtils.deleteLastCharToString(sb);

        TxStudentComment comment = new TxStudentComment();
        comment.setOrgId(orgSignupInfo.getOrgId());
        comment.setUserId(orgSignupInfo.getUserId());
        comment.setCreateTime(orgSignupInfo.getUpdateTime());
        comment.setUpdateTime(orgSignupInfo.getUpdateTime());
        comment.setContent(commentInfo);
        comment.setType(1);
        comment.setIsSystem(AddType.SYSTEM.getCode());

        if (orgSignupInfo.getOrderType() == TransferClassOrder.TRANSFER_ORDER.getCode()) {
            comment.setFollowType(FollowType.TRANSFER_CLASS.getType());
        } else {
            comment.setFollowType(FollowType.SIGNUP_COURSE.getType());
        }

        updateLastRemindTime(comment);

        this.txStudentCommentDao.save(comment);
        return comment;

    }

    /**
     * @param count
     * @param chargeUnit
     * @return
     */
    private int countClassHour(int count, int chargeUnit) {
        if (chargeUnit == ChargeUnit.BY_HALF_HOUR.getCode()) {
            return count * 5 / 10;
        } else if (chargeUnit == ChargeUnit.BY_MINUTE.getCode()) {
            return count / 60;
        }
        return count;
    }

    public static final String QUIT_CLASS_TO_ACCOUNT_FORMAT = "退班：[%s]退学费￥%s，学费退至学员账户";
    public static final String QUIT_CLASS_TO_MONEY_FORMAT = "退班：[%s]退学费￥%s，学费以现金方式退回";
    public static final String REFUND_CASH_FORMAT = "学员账户支出￥%s用于退款给学员";
    public static final String SIGNUP_PAY_FORMAT = "学员账户支出￥%s用于报名[%s]";
    public static final String RECHARGE_FORMAT = "充值：学员账户充值￥%s，充值后，账号余额￥%s";

    @Override
    public void saveByStudentFinanceRecord(Long studentId, Long opMoney, Long currBalance, String opInfo, Integer opType, Integer opTo) {

        log.info("studentId is:{} ,opMoney :{} ,opInfo:{},opType:{},opTo:{}", studentId, opMoney, opInfo, opType, opTo);
        OrgStudent os = this.orgStudentDao.getById(studentId);
        if (os == null) {
            return;
        }
        StringBuilder sb = new StringBuilder();
        Double money = opMoney == null ? 0.0 : opMoney / 100.00D;
        String balanceStr = new DecimalFormat("0.00").format(money);

        Double currBalanceMoney = currBalance == null ? 0.0 : currBalance / 100.00d;
        String currBalanceMoneStr = new DecimalFormat("0.00").format(currBalanceMoney);
        FollowType type = null;
        if (opType == StudentFiannceOpType.QUIT_CLASS.getCode()) {
            if (opTo == 1) { // 1:收入 2:支出
                sb.append(String.format(QUIT_CLASS_TO_ACCOUNT_FORMAT, opInfo, balanceStr));
            } else if (opTo == 2) {
                sb.append(String.format(QUIT_CLASS_TO_MONEY_FORMAT, opInfo, balanceStr));
            }
            type = FollowType.QUIT_CLASS;
        } else if (opType == StudentFiannceOpType.REFUND_CASH.getCode()) {
            sb.append(String.format(REFUND_CASH_FORMAT, balanceStr));
            type = FollowType.QUIT_CLASS;
        } else if (opType == StudentFiannceOpType.RECHARGE.getCode()) {
            sb.append(String.format(RECHARGE_FORMAT, balanceStr, currBalanceMoneStr));
            type = FollowType.RECHARGE;
        } else if (opType == StudentFiannceOpType.PAY_SUCCESS.getCode()) {
            sb.append(String.format(SIGNUP_PAY_FORMAT, balanceStr, opInfo));
            type = FollowType.SIGNUP_COURSE;
        } else {
            log.info("with orgStudent:{} ,will return ", os);
            return;
        }
        String commentInfo = sb.toString();

        Date commentTime = new Date();
        TxStudentComment comment = new TxStudentComment();
        comment.setOrgId(os.getOrgId());
        comment.setUserId(os.getUserId());
        comment.setCreateTime(commentTime);
        comment.setUpdateTime(commentTime);
        comment.setContent(commentInfo);
        comment.setType(1);
        comment.setFollowType(type.getType());
        comment.setIsSystem(AddType.SYSTEM.getCode());
        this.txStudentCommentDao.save(comment);
        updateLastRemindTime(comment);
        log.info("create comment :{} for student:{} ", commentInfo, os);
    }

    String getParticipantsNameStr(List<TxbacklogParticipant> list) {
        StringBuilder builder = new StringBuilder();
        String accountName = "";
        TxbacklogParticipant obj;

        if (CollectionUtils.isNotEmpty(list)) {
            for (int i = 0; i < list.size(); i++) {
                obj = list.get(i);
                accountName = accountApiService.getAccountName(obj.getOrgId(), obj.getCascadeId());
                builder.append(accountName);
                if (i != list.size() - 1) {
                    builder.append(", ");
                }
            }
        }

        return builder.toString();
    }

    @Override
    public TxStudentComment saveByBacklogAdd(Long userId, TxBacklog txBacklog,
                                             List<TxbacklogParticipant> participants) {

        Long cascadeId = (txBacklog.getCascadeId() != null ? txBacklog.getCascadeId().longValue() : 0L);
        String accountName = accountApiService.getAccountName(txBacklog.getOrgId(), cascadeId);
        String participantsNameStr = getParticipantsNameStr(participants);
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm");

        StringBuilder builder = new StringBuilder();
        builder.append(accountName).append(" 创建了学员跟进待办事项").append("\n");
        builder.append("跟进时间: ").append(sdf.format(txBacklog.getEndTime())).append("\n");
        builder.append("参与人 : ").append(participantsNameStr).append("\n");
        builder.append("待办事项: ").append(txBacklog.getContent());

        TxStudentComment comment = new TxStudentComment();
        comment.setUserId(userId);
        comment.setConsultUserId(txBacklog.getConsultUserId());
        comment.setOrgId(txBacklog.getOrgId());
        comment.setCreateTime(new Date());
        comment.setIsSystem(AddType.SYSTEM.getCode());
        comment.setContent(builder.toString());

        txStudentCommentDao.save(comment, false);
        updateLastRemindTime(comment);
        return comment;
    }

    @Override
    public TxStudentComment saveByBacklogDel(Long userId, TxBacklog txBacklog,
                                             List<TxbacklogParticipant> participants) {

        Long cascadeId = (txBacklog.getCascadeId() != null ? txBacklog.getCascadeId().longValue() : 0L);
        String accountName = accountApiService.getAccountName(txBacklog.getOrgId(), cascadeId);
        String participantsNameStr = getParticipantsNameStr(participants);
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm");

        StringBuilder builder = new StringBuilder();
        builder.append(accountName).append(" 删除了学员跟进待办事项").append("\n");
        builder.append("跟进时间: ").append(sdf.format(txBacklog.getEndTime())).append("\n");
        builder.append("参与人 : ").append(participantsNameStr).append("\n");
        builder.append("待办事项: ").append(txBacklog.getContent());

        TxStudentComment comment = new TxStudentComment();
        comment.setUserId(userId);
        comment.setConsultUserId(txBacklog.getConsultUserId());
        comment.setOrgId(txBacklog.getOrgId());
        comment.setCreateTime(new Date());
        comment.setIsSystem(AddType.SYSTEM.getCode());
        comment.setContent(builder.toString());
        txStudentCommentDao.save(comment, false);
        updateLastRemindTime(comment);

        return comment;
    }

    @Override
    public TxStudentComment saveByBacklogUpdate(Long userId, TxBacklog txBacklogOld, TxBacklog txBacklogNew,
                                                List<TxbacklogParticipant> participantsOld, List<TxbacklogParticipant> participantsNew) {

        Long cascadeId = (txBacklogNew.getCascadeId() != null ? txBacklogNew.getCascadeId().longValue() : 0L);
        String accountName = accountApiService.getAccountName(txBacklogNew.getOrgId(), cascadeId);
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm");

        boolean dataChange = true;
        StringBuilder builder = new StringBuilder();
        builder.append(accountName).append(" 修改了学员跟进待办事项").append("\n");

        if (!txBacklogOld.getEndTime().equals(txBacklogNew.getEndTime())) {
            dataChange = true;
            builder.append("原跟进时间: ").append(sdf.format(txBacklogOld.getEndTime())).append("\n");
            builder.append("现跟进时间: ").append(sdf.format(txBacklogNew.getEndTime())).append("\n");
        }

        if (participantsOld != null && participantsNew != null) {
            boolean participantsChange = false;
            if (participantsOld.size() != participantsNew.size()) {
                participantsChange = true;
            } else {
                HashSet<Long> set = new HashSet<Long>();
                for (TxbacklogParticipant obj : participantsOld) {
                    set.add(obj.getCascadeId());
                }
                for (TxbacklogParticipant obj : participantsNew) {
                    set.add(obj.getCascadeId());
                }
                if (participantsOld.size() != set.size()) {
                    participantsChange = true;
                }
            }

            if (participantsChange) {
                dataChange = true;
                builder.append("原参与人 : ").append(getParticipantsNameStr(participantsOld)).append("\n");
                builder.append("现参与人 : ").append(getParticipantsNameStr(participantsNew)).append("\n");
            }
        }

        if (!txBacklogOld.getEndTime().equals(txBacklogNew.getEndTime())) {
            dataChange = true;
            builder.append("原待办事项: ").append(txBacklogOld.getContent()).append("\n");
            builder.append("现待办事项: ").append(txBacklogNew.getContent());
        }

        if (dataChange) {
            TxStudentComment comment = new TxStudentComment();
            comment.setUserId(userId);
            comment.setConsultUserId(txBacklogNew.getConsultUserId());
            comment.setOrgId(txBacklogNew.getOrgId());
            comment.setCreateTime(new Date());
            comment.setIsSystem(AddType.SYSTEM.getCode());

            comment.setContent(builder.toString());
            txStudentCommentDao.save(comment, false);
            updateLastRemindTime(comment);
            return comment;
        } else {
            return null;
        }
    }


    @Override
    public Map<Long, Integer> getCommentTotal(Date startTime, Date endTime, List<Long> tianxiaoOrgIds, Integer notifyParents, Integer isSystemType, Boolean isStudentComment) {
        Map<Long, Integer> result = Maps.newConcurrentMap();
        List<Long> tempOrgIds = Lists.newArrayList();
        for (int i = 0; i < tianxiaoOrgIds.size(); i++) {
            if ((i > 0 && (i % 50 == 0)) || (i == tianxiaoOrgIds.size() - 1)) {
                if (CollectionUtils.isNotEmpty(tempOrgIds)) {
                    result.putAll(txStudentCommentDao.getCommentTotal(startTime, endTime, tempOrgIds, notifyParents, isSystemType, isStudentComment));
                }
                tempOrgIds.clear();
            } else {
                tempOrgIds.add(tianxiaoOrgIds.get(i));
            }
        }
        return result;
    }
}
