/**
 * Baijiahulian.com Inc. Copyright (c) 2015-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.common.impl;

import java.util.*;

import com.baijia.tianxiao.dal.wechat.dao.FansDao;
import com.baijia.tianxiao.dal.wechat.po.Fans;
import com.baijia.tianxiao.image.AvatarUtil;
import com.baijia.tianxiao.sal.common.api.OrgStudentApiService;

import com.baijia.tianxiao.util.GenericsUtils;
import lombok.extern.slf4j.Slf4j;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import com.baijia.tianxiao.consants.DataStatus;
import com.baijia.tianxiao.constant.Flag;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.roster.dao.TxConsultUserDao;
import com.baijia.tianxiao.dal.roster.po.TxConsultUser;
import com.baijia.tianxiao.sal.common.api.ConsulterAPIService;
import com.baijia.tianxiao.sal.common.constant.DBOperation;
import com.baijia.tianxiao.sal.common.dto.DBResultDto;

import org.springframework.stereotype.Service;

@Slf4j
@Service
public class ConsulterAPIServiceImpl implements ConsulterAPIService {

    @Autowired
    private TxConsultUserDao consultUserDao;
    @Autowired
    private OrgStudentDao orgStudentDao;
    @Autowired
    private FansDao fansDao;
    @Autowired
    private OrgStudentApiService studentApiService;

    @Override
    public void saveConsultUser(TxConsultUser consultUser) {
        consultUserDao.save(consultUser);
    }

    @Override
    public void updateConsultUser(TxConsultUser consultUser, boolean updateNullValue) {
        Date now = new Date();
        consultUser.setUpdateTime(now);
        consultUserDao.update(consultUser, updateNullValue);
    }

    @Override
    public DBResultDto<TxConsultUser> saveOrUpdateByMobile(TxConsultUser baseData) {
        // 入参检查
        if (baseData == null || StringUtils.isBlank(baseData.getMobile())) {
            log.error("baseData:{}", baseData);
            throw new RuntimeException("参数错误");
        }

        // 按手机号 查找 线索、学员
        List<TxConsultUser> consultUsers = consultUserDao.lookByMobile(baseData.getOrgId(), baseData.getMobile());
        OrgStudent student =
            orgStudentDao.getStudentByMobileAndOrgId(baseData.getOrgId(), baseData.getMobile(), "id", "name",
                "delStatus", "weixin");
        if (CollectionUtils.isEmpty(consultUsers) && student != null && StringUtils.isNotEmpty(student.getWeixin())) {
            consultUsers = consultUserDao.lookByParams(baseData.getOrgId(), student.getWeixin());
        }

        TxConsultUser consultUser = null;
        DBOperation operation = null;
        if (CollectionUtils.isEmpty(consultUsers)) {
            // 线索不存在
            consultUser = baseData;
            if (student != null) {
                consultUser.setStudentId(student.getId());
                consultUser.setWeixinOpenId(student.getWeixin());
            }
            consultUserDao.save(consultUser);
            operation = DBOperation.INSERT;

        } else {
            // 线索已存在
            consultUser = consultUsers.get(0);
            boolean isNeedToUpdate = false;
            if (student != null && student.getDelStatus().intValue() == DataStatus.NORMAL.getValue()) {
                if (!student.getName().equals(consultUser.getName())) {
                    isNeedToUpdate = true;
                    consultUser.setName(student.getName());
                }
                if (StringUtils.isNotEmpty(student.getWeixin())
                    && !student.getWeixin().equals(consultUser.getWeixinOpenId())) {
                    isNeedToUpdate = true;
                    consultUser.setWeixinOpenId(student.getWeixin());
                }
                if (student.getId().equals(consultUser.getStudentId())) {
                    isNeedToUpdate = true;
                    consultUser.setStudentId(consultUser.getStudentId());
                }
            }

            // 如果该咨询之前不是线索，就更新为线索
            if (Flag.getBoolean(baseData.getIsConsulter()) && !Flag.getBoolean(consultUser.getIsConsulter())) {
                isNeedToUpdate = true;
                consultUser.setIsConsulter(baseData.getIsConsulter());
            }
            if (isNeedToUpdate) {
                consultUser.setUpdateTime(new Date());
                consultUserDao.update(consultUser);
            }
            operation = DBOperation.UPDATE;
        }

        // 退出检查
        if (consultUser == null || operation == null) {
            log.error("baseData:{}", baseData);
            throw new RuntimeException("返回结果错误");
        }

        return new DBResultDto<TxConsultUser>(consultUser, operation);
    }

    @Override
    public DBResultDto<TxConsultUser> saveOrUpdateByWechatOpenId(TxConsultUser baseData) {
        // 入参检查
        if (baseData == null || StringUtils.isBlank(baseData.getWeixinOpenId())) {
            log.error("baseData:{}", baseData);
            throw new RuntimeException("参数错误");
        }

        log.info("[Wechat Message] BaseData is :{} ", baseData);

        // 按openId 查找 线索、学员
        List<TxConsultUser> consultUsers =
            consultUserDao.lookByWeixinOpenId(baseData.getOrgId(), baseData.getWeixinOpenId());
        List<OrgStudent> orgStudents =
            orgStudentDao.getStudentByOpenIdAndOrgId(baseData.getOrgId(), baseData.getWeixinOpenId());

        boolean consultNeedUpdate = false;
        boolean stuNeedUpdate = false;

        TxConsultUser consultUser = null;
        OrgStudent student = null;
        DBOperation operation = null;
        if (CollectionUtils.isEmpty(consultUsers)) {
            // 线索不存在
            consultUser = baseData;
            if (orgStudents != null && orgStudents.size() > 0) {
                student = orgStudents.get(0);
                log.info("baseData :{} consultUser not exists but find a student :{} ", baseData, student);
                consultUser.setStudentId(student.getId());
                consultUser.setMobile(student.getMobile());
                stuNeedUpdate = true;
                consultNeedUpdate = true;
            }
            consultUserDao.save(consultUser);
            operation = DBOperation.INSERT;

        } else {
            // 线索已存在
            consultUser = consultUsers.get(0);
            log.info("find an exists consultUser :{} ", consultUser);
            if (consultUser.getStudentId() != null && consultUser.getStudentId() > 0) {
                student = this.orgStudentDao.getById(consultUser.getStudentId());
                if (student != null && student.getDelStatus().intValue() == DataStatus.NORMAL.getValue()) {
                    consultUser.setName(student.getName());
                    consultUser.setMobile(student.getMobile());
                    consultNeedUpdate = true;
                }
            } else {
                if (orgStudents != null && orgStudents.size() > 0) {
                    student = orgStudents.get(0);
                    consultUser.setStudentId(student.getId());
                    consultUser.setMobile(student.getMobile());
                    stuNeedUpdate = true;
                    consultNeedUpdate = true;
                }
            }

            if (!StringUtils.equals(consultUser.getWeixinNickName(), baseData.getWeixinNickName())) {
                consultUser.setWeixinNickName(baseData.getWeixinNickName());
                consultNeedUpdate = true;
            }

            log.info("[Wechat Message] consultNeedUpdate={},studentId={},stuNeedUpdate={}", consultNeedUpdate,
                stuNeedUpdate, consultUser.getStudentId());

            if (consultNeedUpdate) {
                if (student != null) {
                    consultUser.setName(student.getName());
                    consultUser.setNickName(student.getName());
                }
                consultUser.setUpdateTime(new Date());
                consultUserDao.update(consultUser);
            }

            if (stuNeedUpdate) {
                student.setWeixin(consultUser.getWeixinOpenId());
                orgStudentDao.saveOrUpdate(student);
            }
            operation = DBOperation.UPDATE;
        }

        // 退出检查
        if (consultUser == null || operation == null) {
            log.error("baseData:{}", baseData);
            throw new RuntimeException("返回结果错误");
        }

        return new DBResultDto<TxConsultUser>(consultUser, operation);
    }

    @Override
    public DBResultDto<TxConsultUser> saveOrUpdateByUserId(TxConsultUser consultUser) {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public void updateLastRemindTime(Long consultUserId, Date lastRemindTime) {
        Date now = new Date();
        TxConsultUser consultUser = consultUserDao.getById(consultUserId);
        consultUser.setLastRemindTime(lastRemindTime);
        consultUser.setUpdateTime(now);
        consultUserDao.update(consultUser, false);
    }

    @Override
    public void updateKefuId(TxConsultUser consultUser, Long kefuId) {
        if (consultUser == null || consultUser.getId() == null) {
            log.error("consultUser1 is null - consultUser:{}", consultUser);
            return;
        }
        Date now = new Date();
        consultUser.setKefuId(kefuId);
        consultUser.setUpdateTime(now);
        consultUserDao.update(consultUser, false);
    }

    /**
     * 根据学员查找咨询用户 （1）根据studentId （2）如果不存在，根据手机号 （3）如果还不存在，再根据微信查询
     * 
     * @param student
     * @return
     */
    @Override
    public TxConsultUser getConsultUserByStudent(OrgStudent student) {
        List<TxConsultUser> users = consultUserDao.lookByStudentId(student.getOrgId(), student.getId());
        if (users != null && users.size() > 0) {
            return users.get(0);
        }

        users = consultUserDao.lookByMobile(student.getOrgId(), student.getMobile());
        if (users != null && users.size() > 0) {
            return users.get(0);
        }

        if (StringUtils.isNotBlank(student.getWeixin())) {
            users = consultUserDao.lookByWeixinOpenId(student.getOrgId(), student.getWeixin());
            if (users != null && users.size() > 0) {
                return users.get(0);
            }
        }
        return null;
    }

    @Override
    public void batchSetConsultAvatarUrl(Collection<TxConsultUser> txConsultUsers) {

        log.info("txConsultUsers  parm:{}, size param:{}", txConsultUsers, txConsultUsers.size());

        List<String> openIds = new ArrayList<>();

        List<Long> studentIds = new ArrayList<>();
        for (TxConsultUser txConsultUser : txConsultUsers) {
            Long studentId = txConsultUser.getStudentId();
            if (studentId != null && studentId > 0) {
                studentIds.add(studentId);
            }
            String openId = txConsultUser.getWeixinOpenId();
            if (openId!=null && !openId.equals("") && !openId.equals("0")) {
                openIds.add(openId);
            }
        }
        log.info("获取学员 studentIds param:{}", studentIds);

        // 批量查询用户微信详情
        Map<String, Fans> fansMap = new HashMap<>();
        if (GenericsUtils.notNullAndEmpty(openIds)) {
            fansMap = this.fansDao.mapKeyOpenIdValueFans(openIds);
        }

        // 批量获取学员头像
        Map<Long, String> stuAvatarMap = new HashMap<>();
        if (GenericsUtils.notNullAndEmpty(studentIds)) {
            List<OrgStudent> orgStudents = this.orgStudentDao.getByIds(studentIds);
            stuAvatarMap = studentApiService.batchGetStudentAvatarUrl(orgStudents);
        }

        for (TxConsultUser txConsultUser : txConsultUsers) {
            Long studentId = txConsultUser.getStudentId();
            String avatar = null;

            // 是学员
            if (studentId != null && studentId > 0) {
                avatar = stuAvatarMap.get(studentId);
            } else {
                // 咨询学员
                avatar = txConsultUser.getPortrait();
                String openId = txConsultUser.getWeixinOpenId();

                // 获取微信头像
                if (GenericsUtils.isNullOrEmpty(avatar) && GenericsUtils.notNullAndEmpty(openId)) {
                    Fans fan = fansMap.get(openId);
                    if (GenericsUtils.notNullAndEmpty(fan)) {
                        avatar = fan.getHeadImgUrl();
                        if (GenericsUtils.isNullOrEmpty(txConsultUser.getName())) {
                            txConsultUser.setName(fan.getNick());
                        }
                    }
                }

                // 如果头像为空，给予默认头像
                if (GenericsUtils.isNullOrEmpty(avatar)) {
                    avatar = AvatarUtil.getUserAvatar(txConsultUser.getId());
                }
            }

            txConsultUser.setPortrait(avatar);
        }
    }

    @Override
    public String getConsultAvatarUrl(Long consultUserId) {
        TxConsultUser user = consultUserDao.getById(consultUserId, "id", "studentId", "weixinOpenId", "name", "portrait");
        batchSetConsultAvatarUrl(Arrays.asList(user));
        return user.getPortrait();
    }
}
