package com.baijia.tianxiao.sal.comment.service.impl;

import com.baijia.tianxiao.constant.Flag;
import com.baijia.tianxiao.dal.comment.dao.TxCustomFastCommentDao;
import com.baijia.tianxiao.dal.comment.po.TxCustomFastComment;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.comment.dto.TxCustomFastCommentDto;
import com.baijia.tianxiao.sal.comment.service.TxCustomFastCommentService;
import com.baijia.tianxiao.sal.organization.constant.ClientType;
import com.baijia.tianxiao.sal.organization.org.service.TxAccountModuleAuthService;
import com.baijia.tianxiao.util.TwoTuple;
import com.baijia.tianxiao.util.exception.BussinessPreconditions;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.List;
import java.util.Map;

import lombok.extern.slf4j.Slf4j;

/**
 * @author weihongyan
 * @implNote <(▰˘◡˘▰)>
 * @since 01/08/2017 11:22 AM
 */
@Slf4j
@Service
@Transactional
public class TxCustomFastCommentServiceImpl implements TxCustomFastCommentService {
    @Autowired
    private TxCustomFastCommentDao txCustomFastCommentDao;
    @Autowired
    private TxAccountModuleAuthService txAccountModuleAuthService;

    @Override
    public List<TxCustomFastCommentDto> listComments(Long orgId, Integer disable) {
        List<TxCustomFastCommentDto> result = Lists.newArrayList();
        if (!txAccountModuleAuthService.hasClientType(orgId, ClientType.CUSTOM_COMMENT)) {
            return result;
        }
        List<TxCustomFastComment> comments = txCustomFastCommentDao.getCommentsByOrgId(orgId, disable);
        for (TxCustomFastComment comment : comments) {
            result.add(transform(comment));
        }
        return result;
    }

    private TxCustomFastCommentDto transform(TxCustomFastComment comment) {
        TxCustomFastCommentDto dto = new TxCustomFastCommentDto();
        dto.setId(comment.getId());
        dto.setComment(comment.getComment());
        dto.setDisable(comment.getDisable());
        dto.setSort(comment.getSort());
        return dto;
    }

    @Override
    public TxCustomFastCommentDto addComment(Long orgId, String comment) {
        BussinessPreconditions.checkArgument(txAccountModuleAuthService.hasClientType(orgId, ClientType.CUSTOM_COMMENT),
            "无自定义评价项权限");
        BussinessPreconditions.checkArgument(StringUtils.isNotBlank(comment) && comment.length() <= 10,
            CommonErrorCode.PARAM_ERROR, "自定义评价不能为空或者超过10个字符");
        BussinessPreconditions.checkArgument(txCustomFastCommentDao.countByOrgId(orgId) < 50,
            "评价项不能超过50个, 请删除一些评价项后再添加");
        BussinessPreconditions.checkArgument(txCustomFastCommentDao.getByOrgIdAndComment(orgId, comment) == null,
            "该评价已经存在, 请勿重复创建");

        TxCustomFastComment fastComment = new TxCustomFastComment();
        fastComment.setComment(comment);
        fastComment.setOrgId(orgId);
        fastComment.setSort(txCustomFastCommentDao.getMaxSortByOrgId(orgId) + 1);
        fastComment.setDisable(Flag.FALSE.getInt());
        fastComment.setCreateTime(new Date());
        fastComment.setUpdateTime(fastComment.getCreateTime());
        try {
            txCustomFastCommentDao.save(fastComment);
        } catch (DuplicateKeyException exception) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "该评价已经存在, 请勿重复创建");
        }

        return transform(fastComment);
    }

    @Override
    public TxCustomFastCommentDto editComment(Long orgId, Long id, String comment, Integer disable) {
        BussinessPreconditions.checkArgument(txAccountModuleAuthService.hasClientType(orgId, ClientType.CUSTOM_COMMENT),
            "无自定义评价项权限");
        TxCustomFastComment fastComment = txCustomFastCommentDao.getById(id);
        BussinessPreconditions.checkArgument(null != fastComment && orgId.longValue() == fastComment.getOrgId(),
            "can not find comment!");

        if (null != comment) {
            BussinessPreconditions.checkArgument(StringUtils.isNotBlank(comment) && comment.length() <= 10,
                CommonErrorCode.PARAM_ERROR, "自定义评价不能为空或者超过10个字符");
            BussinessPreconditions.checkArgument(txCustomFastCommentDao.getByOrgIdAndComment(orgId, comment) == null,
                "该评价已经存在, 请勿重复创建");
            fastComment.setComment(comment);
        }

        if (null != disable) {
            BussinessPreconditions.checkArgument(disable == Flag.FALSE.getInt() || disable == Flag.TRUE.getInt(),
                "disable value error!");
            fastComment.setDisable(disable);
            fastComment.setSort(txCustomFastCommentDao.getMaxSortByOrgId(orgId) + 1);
        }
        fastComment.setUpdateTime(new Date());
        try {
            txCustomFastCommentDao.update(fastComment, "comment", "updateTime", "disable", "sort");

        } catch (DuplicateKeyException e) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "该评价已经存在, 请勿重复创建");
        }

        return this.transform(fastComment);
    }

    @Override
    public int deleteByCommentId(Long orgId, Long id) {
        BussinessPreconditions.checkArgument(txAccountModuleAuthService.hasClientType(orgId, ClientType.CUSTOM_COMMENT),
            "无自定义评价项权限");
        TxCustomFastComment fastComment = txCustomFastCommentDao.getById(id);
        BussinessPreconditions.checkArgument(null != fastComment && orgId.longValue() == fastComment.getOrgId(),
            "can not find comment!");
        return this.txCustomFastCommentDao.delById(fastComment.getId());
    }

    @Override
    public TwoTuple<TxCustomFastCommentDto, TxCustomFastCommentDto> exchangeSort(Long orgId, Long id1, Long id2) {
        BussinessPreconditions.checkArgument(txAccountModuleAuthService.hasClientType(orgId, ClientType.CUSTOM_COMMENT),
            "无自定义评价项权限");
        BussinessPreconditions.checkArgument(id1.longValue() != id2, "can not be equal");
        TxCustomFastComment a = txCustomFastCommentDao.getById(id1);
        BussinessPreconditions.checkArgument(null != a && orgId.longValue() == a.getOrgId(),
            "can not find comment by id1!");
        TxCustomFastComment b = txCustomFastCommentDao.getById(id2);
        BussinessPreconditions.checkArgument(null != b && orgId.longValue() == b.getOrgId(),
            "can not find comment by id2!");

        Integer sort = a.getSort();
        a.setSort(b.getSort());
        a.setUpdateTime(new Date());
        b.setSort(sort);
        b.setUpdateTime(a.getUpdateTime());
        txCustomFastCommentDao.update(a, "sort", "updateTime");
        txCustomFastCommentDao.update(b, "sort", "updateTime");
        return new TwoTuple<>(transform(a), transform(b));
    }

    @Override
    public void batchSetSort(Long orgId, Map<Long, Integer> map) {
        BussinessPreconditions.checkArgument(txAccountModuleAuthService.hasClientType(orgId, ClientType.CUSTOM_COMMENT),
            "无自定义评价项权限");
        if (MapUtils.isEmpty(map)) {
            return;
        } else {
            BussinessPreconditions.checkArgument(map.size() == Sets.newHashSet(map.values()).size(),
                "map contains duplicated values!");
            List<TxCustomFastComment> list = txCustomFastCommentDao.getByIds(map.keySet(), "id", "sort");
            Date updateTime = new Date();
            for (TxCustomFastComment comment : list) {
                Integer sort = map.get(comment.getId());
                BussinessPreconditions.checkArgument(null != sort, CommonErrorCode.PARAM_ERROR,
                    "sort value can not be null!");
                if (sort.intValue() == comment.getSort()) {
                    continue;
                }
                comment.setSort(sort);
                comment.setUpdateTime(updateTime);
                txCustomFastCommentDao.update(comment, "sort", "updateTime");
            }
        }
    }

}
