/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.sal.comment.service.impl;

import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Resource;

import com.baijia.tianxiao.constants.sms.TxSmsCodeType;
import com.baijia.tianxiao.sal.common.api.CommonMsgService;
import com.baijia.tianxiao.sal.common.dto.msg.SendMsgRequest;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.constants.UserRoleEnum;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgInfoDao;
import com.baijia.tianxiao.dal.comment.dao.OrgLessonCommentDao;
import com.baijia.tianxiao.dal.org.dao.OrgStorageDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgTeacherLessonDao;
import com.baijia.tianxiao.dal.org.po.OrgClassLesson;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.dal.org.po.OrgInfo;
import com.baijia.tianxiao.dal.comment.po.OrgLessonComment;
import com.baijia.tianxiao.dal.org.po.OrgStorage;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.org.po.OrgTeacherLesson;
import com.baijia.tianxiao.dal.org.po.TXSaleClueRule;
import com.baijia.tianxiao.dal.sync.constant.MsgSyncType;
import com.baijia.tianxiao.dal.sync.dao.TxMsgSyncTimestampDao;
import com.baijia.tianxiao.dal.sync.po.TxMsgSyncTimestamp;
import com.baijia.tianxiao.dal.user.dao.TeacherDao;
import com.baijia.tianxiao.dal.user.po.Teacher;
import com.baijia.tianxiao.dal.wechat.constant.WechateTemplateMsgType;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.comment.constant.Biz;
import com.baijia.tianxiao.sal.comment.constant.CommentErrorCode;
import com.baijia.tianxiao.sal.comment.constant.CommentReqType;
import com.baijia.tianxiao.sal.comment.dto.AddCommentReponse;
import com.baijia.tianxiao.sal.comment.dto.CommentInfoDto;
import com.baijia.tianxiao.sal.comment.dto.request.CommentAddRequestDto;
import com.baijia.tianxiao.sal.comment.service.LessonCommentService;
import com.baijia.tianxiao.sal.common.dto.wechatMsgRequest.AfterCourseCommentNotifyMsg;
import com.baijia.tianxiao.sal.common.utils.WechatTemplateMsgHelper;
import com.baijia.tianxiao.sal.organization.org.service.TXSaleClueRuleService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.ArithUtil;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.CourseSmsTokenUtil;
import com.baijia.tianxiao.util.ShortUrlUtil;
import com.baijia.tianxiao.util.SmsContentHelper;
import com.baijia.tianxiao.util.date.DateUtil;
import com.baijia.tianxiao.util.storage.StorageUtil;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * @title LessonCommentServiceImpl
 * @desc TODO
 * @author shanyu
 * @date 2016年1月3日
 * @version 1.0
 */
@Slf4j
@Service
public class LessonCommentServiceImpl implements LessonCommentService {

    // 短信里面已经不用了
    private static String content = "%s,你好：%s点评了你在%s%s的表现，快去看看吧。";

    private static String url = "地址：%s";

    private static String defaultName = "-";

    private static String remark = "点击查看完整的评语内容";

    @Resource
    private OrgLessonCommentDao orgLessonCommentDao;

    @Resource
    private OrgStudentLessonDao orgStudentLessonDao;

    @Resource
    private OrgTeacherLessonDao orgTeacherLessonDao;

    @Resource
    private OrgStorageDao orgStorageDao;

    @Resource
    private OrgClassLessonDao orgClassLessonDao;

    @Resource
    private OrgStudentDao orgStudentDao;

    @Resource
    private OrgCourseDao orgCourseDao;

    @Resource
    private TeacherDao teacherDao;

    @Resource
    private OrgInfoDao orgInfoDao;

    @Resource
    private TxMsgSyncTimestampDao txMsgSyncTimestampDao;

    @Resource
    private Environment environment;

    @Autowired(required = false)
    private CommonMsgService commonMsgService;

    @Resource
    private TXSaleClueRuleService tXSaleClueRuleService;

    private boolean checkOrgPermissionEvaluateMsg(Long orgId) {
        if (orgId == null) {
            return false;
        }
        TXSaleClueRule tXSaleClueRule = tXSaleClueRuleService.getByOrgId(orgId.intValue());
        if (tXSaleClueRule != null && tXSaleClueRule.getEvaluateMsg() == 0) {
            return true;
        } else {
            return false;
        }

    }

    @Override
    @Transactional(readOnly = true)
    public List<CommentInfoDto> getLessonComments(Long lessonId, Long orgId, Integer userRole) {
        List<OrgLessonComment> comments = null;
        if (userRole.intValue() == UserRole.STUDENT.getRole()) {
            log.info("get lesson student comments, lessonId={}, orgId={}", lessonId, orgId);
            comments = this.orgLessonCommentDao.getLessonStudentComments(lessonId, false);
        } else {
            log.info("get lesson teacher comments, lessonId={}, orgId={}", lessonId, orgId);
            comments = this.orgLessonCommentDao.getLessonTeacherComments(lessonId, false);
        }
        List<CommentInfoDto> responseDtos = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(comments)) {
            Map<Integer, String> urlMap = getUrlMap(comments);
            for (OrgLessonComment comment : comments) {
                CommentInfoDto dto = buildCommentInfoDto(comment, urlMap);
                responseDtos.add(dto);
            }
        }
        return responseDtos;
    }

    @Override
    @Transactional(readOnly = true)
    public List<CommentInfoDto> getStudentComments(Integer userRole, Long userId, Long orgId,
        Collection<Long> lessonIds, PageDto pageDto) {
        List<OrgLessonComment> comments = null;
        if (userRole.intValue() == UserRole.STUDENT.getRole()) {
            log.info("get student comments,  userId={},orgId={}", orgId);
            comments =
                this.orgLessonCommentDao.getStudentComments(orgId, userId, userRole, true, false, lessonIds, pageDto);
        } else {
            log.info("get teacher comments of student, userId={}, orgId={}", userId, orgId);
            comments =
                this.orgLessonCommentDao.getStudentComments(orgId, userId, userRole, false, false, lessonIds, pageDto);
        }
        return buildCommentInfoDtos(comments);
    }

    private List<CommentInfoDto> buildCommentInfoDtos(List<OrgLessonComment> comments) {
        List<CommentInfoDto> responseDtos = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(comments)) {
            Map<Integer, String> urlMap = getUrlMap(comments);
            for (OrgLessonComment comment : comments) {
                CommentInfoDto dto = buildCommentInfoDto(comment, urlMap);
                responseDtos.add(dto);
            }
        }
        return responseDtos;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public AddCommentReponse addComment(CommentAddRequestDto dto, Long orgId, String studentCommentDetailUrl) {
        Preconditions.checkArgument(dto.getScore() != null, "score is not null!");
        if (StringUtils.isNotBlank(dto.getContent()) && dto.getContent().length() > 2000) {
            throw new BussinessException(CommentErrorCode.CONTENT_TOO_LONG);
        }
        OrgClassLesson lesson = this.orgClassLessonDao.getById(dto.getLessonId());
        if (lesson.getEndTime().after(new Date())) {
            throw new BussinessException(CommentErrorCode.LESSON_NOT_END);
        }
        // 移除7天后不能评论的限制
        /*
         * if (DateUtil.getDiffDateTime(lesson.getEndTime(), 7).before(new Date())) { throw new
         * BussinessException(CommentErrorCode.COMMENT_TIME_OUT); }
         */
        OrgCourse course = orgCourseDao.getByCourseId(lesson.getCourseId());
        OrgTeacherLesson teacherLesson = this.orgTeacherLessonDao.getByLessonId(orgId, lesson.getId());
        OrgInfo orgInfo = this.orgInfoDao.getOrgInfo(orgId.intValue());
        Teacher teacher = null;
        if (teacherLesson != null) {
            teacher = this.teacherDao.getByUserId(teacherLesson.getTeacherId());
        }
        OrgLessonComment orgLessonComment = new OrgLessonComment();
        orgLessonComment.setLessonId(dto.getLessonId());
        orgLessonComment.setCourseId(lesson.getCourseId());
        orgLessonComment.setUserRole(dto.getUserRole());
        orgLessonComment.setContent(dto.getContent());
        orgLessonComment.setCustomFastComment(dto.getCustomFastComment());
        orgLessonComment.setOrgId(orgId);
        orgLessonComment.setStorageIds(dto.getStorageIds());
        orgLessonComment.setScore((int) (dto.getScore() * 100));
        if (dto.getUserRole().intValue() == UserRole.STUDENT.getRole()) {
            OrgStudent student = orgStudentDao.getById(dto.getFromId());
            orgLessonComment.setFromId(student.getUserId());
            orgLessonComment.setToId(dto.getToId());
            if (!check(student.getUserId(), UserRole.STUDENT.getRole(), dto.getLessonId())
                || !check(dto.getToId(), UserRole.TEACHER.getRole(), dto.getLessonId())) {
                throw new BussinessException(CommentErrorCode.NOT_IN_LESSON);
            }
            OrgLessonComment oldCommont = this.orgLessonCommentDao.getLessonCommentDetail(dto.getLessonId(),
                student.getUserId(), UserRole.STUDENT.getRole(), null);
            if (oldCommont != null) {
                // 增加容错逻辑，如果已经系统评价，则先删除系统评价，避免系统默认评价后，用户再评价报错
                if (oldCommont.getIsSystem() == true) {
                    this.orgLessonCommentDao.delById(oldCommont.getId());
                } else {
                    throw new BussinessException(CommentErrorCode.COMMENT_REPEAT);
                }

            }

        } else if (dto.getUserRole().intValue() == UserRole.TEACHER.getRole()) {
            OrgStudent student = orgStudentDao.getById(dto.getToId());
            orgLessonComment.setFromId(dto.getFromId());
            orgLessonComment.setToId(student.getUserId());
            if (!check(student.getUserId(), UserRole.STUDENT.getRole(), dto.getLessonId())
                || !check(dto.getFromId(), UserRole.TEACHER.getRole(), dto.getLessonId())) {
                throw new BussinessException(CommentErrorCode.NOT_IN_LESSON);
            }
            if (this.orgLessonCommentDao.getLessonCommentDetail(dto.getLessonId(), student.getUserId(),
                UserRole.TEACHER.getRole(), false) != null) {
                throw new BussinessException(CommentErrorCode.COMMENT_REPEAT);
            }

            // 给学员发送通知
            try {
                if (checkOrgPermissionEvaluateMsg(orgId)) {
                    String url = getStudentCommentDetailUrl(student.getId(), course.getId(), student.getOrgId(),
                        lesson.getId(), studentCommentDetailUrl);
                    String teacherName = teacher == null ? null : teacher.getRealName();
                    // 构建短信内容
                    String smsContent = SmsContentHelper.createCommentNotifySmsMsg(teacherName,
                        lesson.getStartTime().getTime(), orgInfo.getShortName(), url);
                    // 构建微信模板消息内容
                    AfterCourseCommentNotifyMsg msg = AfterCourseCommentNotifyMsg.newInstance(student.getName(),
                            teacherName, course.getName(), lesson.getStartTime().getTime(), url);
                    SendMsgRequest createSendMsgRequestToStu =
                            WechatTemplateMsgHelper.createSendMsgRequestToStu(orgId, student.getMobile(), student.getId(),
                                    student.getWeixin(), smsContent, WechateTemplateMsgType.TEACHER_EVALUATION_TO_STU, msg);

                    createSendMsgRequestToStu.setSmsCodeType(TxSmsCodeType.COMMENT_NOTIFY);
                    commonMsgService.sendMsg(createSendMsgRequestToStu);
                }

            } catch (Exception e) {
                log.error("error={}", e);
            }
        } else {
            throw new BussinessException(CommentErrorCode.USER_ROLE_ERRER);
        }
        try {
            this.orgLessonCommentDao.save(orgLessonComment);
        } catch (Exception e) {
            throw new BussinessException(CommentErrorCode.COMMENT_REPEAT);
        }

        AddCommentReponse response = new AddCommentReponse();
        response.setCommentId(orgLessonComment.getId());
        return response;
    }

    /**
     * 构造学生评价提醒信息
     * 
     * @param teacherName
     * @param
     * @param lesson
     * @param course
     * @param orgInfo
     * @param student
     * @return
     */
    private SendMsgRequest getStudentCommentDetail(String teacherName, OrgClassLesson lesson, OrgCourse course,
        OrgInfo orgInfo, OrgStudent student, String studentCommentDetailUrl) {
        SendMsgRequest request = new SendMsgRequest();
        request.setOrgId(orgInfo.getOrgId().longValue());
        request.setMobile(student.getMobile());
        request.setCountSms(false);
        request.setReceiverId(student.getId());
        request.setReceiverRole(UserRoleEnum.STUDENT);
        request.setSenderId(orgInfo.getOrgId().longValue());
        request.setSenderRole(UserRoleEnum.ORG);
        request.setWechatTemplateId(WechateTemplateMsgType.TEACHER_EVALUATION_TO_STU.getValue());
        request.setWeixinOpenId(student.getWeixin());
        String url = getStudentCommentDetailUrl(student.getId(), course.getId(), student.getOrgId(), lesson.getId(),
            studentCommentDetailUrl);
        // 构建微信模板消息内容
        AfterCourseCommentNotifyMsg msg = AfterCourseCommentNotifyMsg.newInstance(student.getName(), teacherName,
            course.getName(), lesson.getStartTime().getTime(), url);
        Map<String, Object> createAfterCourseNotifyWechatMsg =
            WechatTemplateMsgHelper.createAfterCourseNotifyWechatMsg(msg);
        request.setWechatParams(createAfterCourseNotifyWechatMsg);
        // 构建短信内容
        request.setSmsContent(SmsContentHelper.createCommentNotifySmsMsg(teacherName, lesson.getStartTime().getTime(),
            orgInfo.getShortName(), url));

        // String weixinFirstContent =
        // String.format(content, getStringValue(student.getName()), getStringValue(teacherName),
        // getStringValue(course.getName()), NotifyMessageUtils.generateLessonName(lesson.getName(), false));
        // Map<String, Object> params = Maps.newHashMap();
        // params.put("first", weixinFirstContent);
        // params.put("remark", remark);
        // params.put("keyword1", getStringValue(student.getName()));
        // params.put("keyword2",
        // getStringValue(course.getName() + NotifyMessageUtils.generateLessonName(lesson.getName(), true)));
        // params.put("keyword3", getStringValue(teacherName));
        // params.put("url", url);
        // request.setWechatParams(params);
        return request;
    }

    /**
     * 没有名称返回“－”
     * 
     * @param message
     * @return
     */
    private String getStringValue(String message) {
        if (StringUtils.isBlank(message)) {
            return defaultName;
        }
        return message;
    }

    private String getStudentCommentDetailUrl(Long studentId, Long courseId, Long orgId, Long lessonId,
        String studentCommentDetailUrl) {
        Map<String, Object> params = Maps.newHashMap();
        params.put("studentId", studentId);
        params.put("courseId", courseId);
        String token;
        try {
            token = CourseSmsTokenUtil.encodeExpireToken(params, orgId, null);
            String detailUrl = String.format(studentCommentDetailUrl, token, lessonId, UserRole.TEACHER.getRole());
            return ShortUrlUtil.getShortUrl(detailUrl);
        } catch (Exception e) {
            log.warn("get lesson student comment error", e);
            return null;
        }
    }

    /**
     * 老师或者学员是否属于该课节
     * 
     * @param userId
     * @param userRole
     * @param lessonId
     * @return
     */
    private Boolean check(Long userId, Integer userRole, Long lessonId) {
        if (userRole == UserRole.STUDENT.getRole()) {
            return this.orgStudentLessonDao.isStudentInlesson(lessonId, userId);
        } else {
            return this.orgTeacherLessonDao.isTeacherInLesson(lessonId, userId);
        }

    }

    private Map<Integer, String> getUrlMap(List<OrgLessonComment> comments) {
        Set<Integer> storageIds = Sets.newHashSet();
        if (CollectionUtils.isNotEmpty(comments)) {
            for (OrgLessonComment comment : comments) {
                if (StringUtils.isNotBlank(comment.getStorageIds())) {
                    String[] storageArr = StringUtils.split(comment.getStorageIds(), ",");
                    for (String str : storageArr) {
                        Integer storageId = Integer.parseInt(str);
                        storageIds.add(storageId);
                    }
                }
            }
        }
        return getUrlMap(storageIds);
    }

    private Map<Integer, String> getUrlMap(OrgLessonComment comment) {
        Set<Integer> storageIds = Sets.newHashSet();
        if (StringUtils.isNotBlank(comment.getStorageIds())) {
            String[] storageArr = StringUtils.split(comment.getStorageIds(), ",");
            for (String str : storageArr) {
                Integer storageId = Integer.parseInt(str);
                storageIds.add(storageId);
            }
        }
        return getUrlMap(storageIds);
    }

    private Map<Integer, String> getUrlMap(Set<Integer> storageIds) {
        if (CollectionUtils.isNotEmpty(storageIds)) {
            List<OrgStorage> storages = this.orgStorageDao.getByIds(storageIds);
            return CollectorUtil.collectMap(storages, new Function<OrgStorage, Integer>() {
                @Override
                public Integer apply(OrgStorage arg0) {
                    return arg0.getId();
                }
            }, new Function<OrgStorage, String>() {
                @Override
                public String apply(OrgStorage arg0) {
                    return StorageUtil.constructUrl(arg0.getFid(), arg0.getSn(), arg0.getMimeType());
                }
            });
        }
        return Maps.newHashMap();
    }

    private CommentInfoDto buildCommentInfoDto(OrgLessonComment comment, Map<Integer, String> urlMap) {
        CommentInfoDto dto = new CommentInfoDto();
        dto.setCommentId(comment.getId());
        dto.setContent(comment.getContent());
        dto.setCustomFastComment(comment.getCustomFastComment());
        dto.setScore(ArithUtil.round(((double) comment.getScore()) / 100, 1));
        dto.setLessonId(comment.getLessonId());
        dto.setCourseId(comment.getCourseId());
        if (StringUtils.isNotBlank(comment.getStorageIds())) {
            String[] storageArr = StringUtils.split(comment.getStorageIds(), ",");
            List<String> urlList = Lists.newArrayList();
            for (String str : storageArr) {
                Integer storageId = Integer.parseInt(str);
                if (storageId != null) {
                    String url = urlMap.get(storageId);
                    if (url != null) {
                        urlList.add(url);
                    }
                }
            }
            if (CollectionUtils.isNotEmpty(urlList)) {
                dto.setUrls(StringUtils.join(urlList.toArray(), ","));
            }
        }
        dto.setUserRole(comment.getUserRole());
        dto.setFromId(comment.getFromId());
        dto.setToId(comment.getToId());
        dto.setCreateTime(comment.getCreateTime());
        return dto;
    }

    @Override
    @Transactional(readOnly = true)
    public CommentInfoDto getCommentDetail(Long commentId, Long orgId, Integer userRole) {
        OrgLessonComment comment = this.orgLessonCommentDao.getById(commentId);
        if (comment == null || comment.getOrgId().longValue() != orgId.longValue()) {
            throw new BussinessException(CommentErrorCode.COMMENT_NOT_EXSIST);
        }
        Map<Integer, String> urlMap = getUrlMap(comment);
        return buildCommentInfoDto(comment, urlMap);
    }

    @Override
    @Transactional(readOnly = true)
    public CommentInfoDto getCommentDetail(Long lessonId, Long userId, Long orgId, Integer userRole) {
        OrgLessonComment comment = this.orgLessonCommentDao.getLessonCommentDetail(lessonId, userId, userRole, null);
        log.debug("comment = {}", comment);
        if (comment == null || comment.getOrgId().longValue() != orgId.longValue()) {
            throw new BussinessException(CommentErrorCode.COMMENT_NOT_EXSIST);
        }
        Map<Integer, String> urlMap = getUrlMap(comment);
        OrgClassLesson orgClassLesson = this.orgClassLessonDao.getById(lessonId);
        Preconditions.checkArgument(orgClassLesson != null, "lesson not exist!");
        CommentInfoDto dto = buildCommentInfoDto(comment, urlMap);
        dto.setStartTime(orgClassLesson.getStartTime());
        dto.setEndTime(orgClassLesson.getEndTime());
        dto.setIndex(orgClassLesson.getNumber());

        OrgCourse course = orgCourseDao.getByCourseId(orgClassLesson.getCourseId(), "name");
        if (course != null) {
            dto.setCourseName(course.getName());
        }

        return dto;
    }

    @Override
    @Transactional(readOnly = true)
    public List<CommentInfoDto> getLessonCommentsWithLessonInfo(@NonNull Long orgId, @NonNull Long studentId,
        @NonNull Integer userRole, PageDto pageDto) {

        Long userId = this.orgStudentDao.getUserId(studentId);
        Preconditions.checkArgument(userId != null && userId > 0, "student not exits!");
        List<Long> orgLessonIds = this.orgClassLessonDao.getOrgLessonIds(orgId);
        List<CommentInfoDto> commentInfoDtoList =
            this.getStudentComments(userRole, userId, orgId, orgLessonIds, pageDto);
        log.debug("commentInfoDtoList = {}", commentInfoDtoList);
        getAndSetCourseAndLesson(orgId, commentInfoDtoList);
        return commentInfoDtoList;
    }

    /**
     * 设置课程课节信息
     * 
     * @param orgId
     * @param commentInfoDtoList
     */
    private void getAndSetCourseAndLesson(Long orgId, List<CommentInfoDto> commentInfoDtoList) {
        List<Long> courseIds = Lists.newArrayList();
        List<Long> lessonIds = Lists.newArrayList();
        for (CommentInfoDto commentInfoDto : commentInfoDtoList) {
            courseIds.add(commentInfoDto.getCourseId());
            lessonIds.add(commentInfoDto.getLessonId());
        }
        log.debug("courseIds = {}", courseIds);
        log.debug("lessonIds = {}", lessonIds);
        Map<Long, OrgClassLesson> userLessonMap = this.orgClassLessonDao.getClassLessonMap(orgId, lessonIds, 0);
        Map<Long, OrgCourse> courseIdCourseMap = this.orgCourseDao.getOrgCourseMap(courseIds);
        log.debug("userLessonMap = {}", userLessonMap);
        log.debug("courseIdCourseMap = {}", courseIdCourseMap);
        for (CommentInfoDto commentInfoDto : commentInfoDtoList) {
            OrgClassLesson lesson = userLessonMap.get(commentInfoDto.getLessonId());
            OrgCourse course = courseIdCourseMap.get(commentInfoDto.getCourseId());
            if (lesson != null) {
                commentInfoDto.setIndex(lesson.getNumber());
                commentInfoDto.setStartTime(lesson.getStartTime());
                commentInfoDto.setEndTime(lesson.getEndTime());
            }
            if (course != null) {
                commentInfoDto.setCourseName(course.getName());
            }
        }

    }

    @Override
    @Transactional(readOnly = true)
    public Map<Long, Boolean> getStudentLessonCommentMap(Long userId, Collection<Long> lessonIds) {

        List<Long> list = this.orgLessonCommentDao.getCommentLessonIdsOfStudent(userId);
        log.debug("list = {}", list);
        Set<Long> set = new HashSet<Long>(list);
        Map<Long, Boolean> map = Maps.newHashMap();
        for (Long lessonId : lessonIds) {
            if (set.contains(lessonId)) {
                map.put(lessonId, true);
            } else {
                map.put(lessonId, false);
            }
        }
        log.debug("lessonIds = {}", lessonIds);
        return map;
    }

    @Override
    @Transactional(readOnly = true)
    public CommentInfoDto getCommentDetail(Long commentId, Long orgId) {
        Preconditions.checkArgument(commentId != null, "commentId is null!");
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        OrgLessonComment comment = this.orgLessonCommentDao.getById(commentId);
        if (comment == null || comment.getOrgId().longValue() != orgId.longValue()) {
            throw new BussinessException(CommentErrorCode.COMMENT_NOT_EXSIST);
        }
        Map<Integer, String> urlMap = getUrlMap(comment);
        return buildCommentInfoDto(comment, urlMap);
    }

    @Override
    public void addLessonDefaultComment() {
        TxMsgSyncTimestamp syncInfo =
            this.txMsgSyncTimestampDao.getSyncTimestampByType(MsgSyncType.LESSON_STUDENT_DEFAULT_COMMENT.getSyncType());
        log.debug("syncInfo={}", syncInfo);
        Date beginTime = null;
        if (syncInfo == null) {
            syncInfo = new TxMsgSyncTimestamp();
            syncInfo.setSyncType(MsgSyncType.LESSON_STUDENT_DEFAULT_COMMENT.getSyncType());
            syncInfo.setSyncTime(new Date());
            this.txMsgSyncTimestampDao.save(syncInfo, false);
        } else {
            beginTime = syncInfo.getSyncTime();
        }
        // 七天前
        Date endTime = DateUtil.getDayDiff(-7);
        PageDto pageDto = new PageDto();
        pageDto.setPageNum(1);
        pageDto.setPageSize(200);
        List<OrgClassLesson> lessons =
            this.orgClassLessonDao.querylessonsByEndTime(beginTime, endTime, DeleteStatus.NORMAL.getValue(), pageDto);
        log.debug("query lessons beginTIme={},endTime={} lessons.size={}", beginTime, endTime, lessons.size());
        Map<Long, OrgClassLesson> lessonMap = CollectorUtil.collectMap(lessons, new Function<OrgClassLesson, Long>() {
            @Override
            public Long apply(OrgClassLesson arg0) {
                return arg0.getId();
            }
        });
        int count = 0;
        while (CollectionUtils.isNotEmpty(lessons) && count < 10) {
            log.debug("pageDto={},next sync time ={},lessons ={}", pageDto, lessons.get(0).getEndTime(),
                lessonMap.keySet());
            syncInfo.setSyncTime(lessons.get(0).getEndTime());
            Map<Long, Long> lessonTeacherMap =
                this.orgTeacherLessonDao.queryLessonTeacherIdMap(null, lessonMap.keySet());
            addDefaultComment(lessonMap.keySet(), lessonMap, lessonTeacherMap);
            this.txMsgSyncTimestampDao.update(syncInfo, "syncTime");
            pageDto.setPageNum(pageDto.getPageNum() + 1);
            lessons = this.orgClassLessonDao.querylessonsByEndTime(beginTime, endTime, DeleteStatus.NORMAL.getValue(),
                pageDto);
            lessonMap = CollectorUtil.collectMap(lessons, new Function<OrgClassLesson, Long>() {
                @Override
                public Long apply(OrgClassLesson arg0) {
                    return arg0.getId();
                }
            });
            count++;
        }
    }

    /**
     * 
     * @param lessonIds
     */
    private void addDefaultComment(Collection<Long> lessonIds, Map<Long, OrgClassLesson> lessonMap,
        Map<Long, Long> lessonTeacherMap) {
        Map<Long, List<Long>> lessonStudentMap = this.orgStudentLessonDao.getLessonStudentList(lessonIds);
        Map<Long, List<Long>> lessonStudentCommentMap =
            this.orgLessonCommentDao.getLessonCommentStudentMap(lessonIds, null, UserRole.STUDENT.getRole());
        Map<Long, List<Long>> lessonTeacherCommentMap =
            this.orgLessonCommentDao.getLessonCommentStudentMap(lessonIds, null, UserRole.TEACHER.getRole());
        log.debug("lessonStudentCommentMap={}", lessonStudentCommentMap);
        log.debug("lessonTeacherCommentMap={}", lessonTeacherCommentMap);
        List<OrgLessonComment> addComments = Lists.newArrayList();
        Iterator<Map.Entry<Long, List<Long>>> iterator = lessonStudentMap.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<Long, List<Long>> entry = iterator.next();
            Long lessonId = entry.getKey();
            OrgClassLesson lesson = lessonMap.get(lessonId);
            if (lesson != null) {
                List<Long> studentIds = entry.getValue(); // 课节学生id
                List<Long> teacherStudentIds = Lists.newArrayList();
                teacherStudentIds.addAll(studentIds);

                if (CollectionUtils.isNotEmpty(studentIds)) {
                    List<Long> hasStudentCommentIds = lessonStudentCommentMap.get(lessonId);// 已评价课节学生id
                    List<Long> hasTeacherCommentIds = lessonTeacherCommentMap.get(lessonId);// 已评价课节学生id
                    if (hasStudentCommentIds != null && CollectionUtils.isNotEmpty(hasStudentCommentIds)) {
                        studentIds.removeAll(hasStudentCommentIds);
                    }
                    if (hasTeacherCommentIds != null && CollectionUtils.isNotEmpty(hasTeacherCommentIds)) {
                        teacherStudentIds.removeAll(hasTeacherCommentIds);
                    }
                    for (Long userId : studentIds) {
                        OrgLessonComment comment = new OrgLessonComment();
                        comment.setOrgId(lesson.getOrgId());
                        comment.setCourseId(lesson.getCourseId());
                        comment.setLessonId(lessonId);
                        comment.setFromId(userId);
                        comment.setUserRole(UserRole.STUDENT.getRole());
                        if (lessonTeacherMap.get(lessonId) != null) {
                            comment.setToId(lessonTeacherMap.get(lessonId));
                        } else {
                            continue;
                        }
                        comment.setContent("");
                        comment.setCustomFastComment("");
                        comment.setScore(500);
                        comment.setIsSystem(true);
                        addComments.add(comment);
                    }
                    for (Long userId : teacherStudentIds) {// 老师默认评价
                        OrgLessonComment comment = new OrgLessonComment();
                        comment.setOrgId(lesson.getOrgId());
                        comment.setCourseId(lesson.getCourseId());
                        comment.setLessonId(lessonId);
                        comment.setFromId(lessonTeacherMap.get(lessonId));
                        comment.setUserRole(UserRole.TEACHER.getRole());
                        if (lessonTeacherMap.get(lessonId) != null) {
                            comment.setToId(userId);
                        } else {
                            continue;
                        }
                        comment.setContent("");
                        comment.setCustomFastComment("");
                        comment.setScore(500);
                        comment.setIsSystem(true);
                        addComments.add(comment);
                    }
                }
            }
        }
        if (CollectionUtils.isNotEmpty(addComments)) {
            this.orgLessonCommentDao.saveAll(addComments, "orgId", "courseId", "lessonId", "fromId", "userRole", "toId",
                "content", "score", "isSystem");
        }
    }

    @Override
    public List<CommentInfoDto> queryComments(Long orgId, Collection<Long> courseIds, Collection<Long> teacherIds,
        Integer commentReqType, Boolean isSystem, Integer userRole, PageDto pageDto) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        log.info(
            "get comments of students with param: courseIds={},teacherIds={},commentReqType={}, isSystem={},pageDto={}",
            courseIds, teacherIds, commentReqType, isSystem, pageDto);
        List<Long> lessonIds = null;
        Integer lowScore = (commentReqType != null && commentReqType.intValue() == CommentReqType.LOW_SCORE.getCode())
            ? Biz.LOW_SCORE * 100 : null;
        Integer highScore = (commentReqType != null && commentReqType.intValue() == CommentReqType.HIGN_SCORE.getCode())
            ? Biz.HIGH_SCORE * 100 : null;
        Boolean hasPic =
            (commentReqType != null && commentReqType.intValue() == CommentReqType.WITH_PIC.getCode()) ? true : null;
        log.info("low_score={},hasPic={},hign_score={}", lowScore, hasPic, highScore);
        if (CollectionUtils.isNotEmpty(teacherIds) || CollectionUtils.isNotEmpty(courseIds)) {// 老师，班级筛选
            lessonIds = this.orgClassLessonDao.filterLessonList(orgId, courseIds, teacherIds);
        }
        List<OrgLessonComment> comments = this.orgLessonCommentDao.getComments(orgId, lessonIds, userRole, lowScore,
            hasPic, highScore, pageDto, isSystem);
        List<CommentInfoDto> dtos = buildCommentInfoDtos(comments);
        getAndSetCourseAndLesson(orgId, dtos);
        log.info("get comments result={}", dtos.size());
        return dtos;
    }
}
