package com.baijia.tianxiao.sal.card.service.impl;

import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.constant.Flag;
import com.baijia.tianxiao.constant.LessonType;
import com.baijia.tianxiao.constant.OrgLessonSignSourceEnum;
import com.baijia.tianxiao.constant.SignStatus;
import com.baijia.tianxiao.constant.finance.PurchaseTimescardStatus;
import com.baijia.tianxiao.constants.sms.TxSmsCodeType;
import com.baijia.tianxiao.dal.card.dao.TxCardSignDao;
import com.baijia.tianxiao.dal.card.enums.TxCardSignStatus;
import com.baijia.tianxiao.dal.card.po.TxCardSign;
import com.baijia.tianxiao.dal.enums.CourseTypeEnum;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseTeacherDao;
import com.baijia.tianxiao.dal.org.dao.OrgInfoDao;
import com.baijia.tianxiao.dal.org.dao.OrgLessonSignDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentLessonDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgClassLesson;
import com.baijia.tianxiao.dal.org.po.OrgInfo;
import com.baijia.tianxiao.dal.org.po.OrgLessonSign;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.org.po.OrgStudentLesson;
import com.baijia.tianxiao.dal.signup.dao.OrgSignupCourseLessonDao;
import com.baijia.tianxiao.dal.signup.dao.TxPurchaseTimescardDao;
import com.baijia.tianxiao.dal.signup.po.TxPurchaseTimescard;
import com.baijia.tianxiao.dal.wechat.constant.WechateTemplateMsgType;
import com.baijia.tianxiao.enums.ErpErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.card.dto.TimesCardDto;
import com.baijia.tianxiao.sal.card.dto.TxCardSignDto;
import com.baijia.tianxiao.sal.card.service.TxCardSignService;
import com.baijia.tianxiao.sal.common.api.CommonMsgService;
import com.baijia.tianxiao.sal.common.api.OrgStudentApiService;
import com.baijia.tianxiao.sal.common.dto.msg.SendMsgRequest;
import com.baijia.tianxiao.sal.common.dto.wechatMsgRequest.SignWechatTemplateMsg;
import com.baijia.tianxiao.sal.common.utils.WechatTemplateMsgHelper;
import com.baijia.tianxiao.sal.course.dto.response.TeacherResponseDto;
import com.baijia.tianxiao.sal.course.service.CourseTeacherService;
import com.baijia.tianxiao.sal.course.util.ErpUtils;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.SmsContentHelper;
import com.baijia.tianxiao.util.collection.CollectorUtil;
import com.baijia.tianxiao.util.date.DateUtil;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * @author weihongyan
 * @implNote <(▰˘◡˘▰)>
 * @since 13/07/2017 11:12 AM
 */
@Slf4j
@Service
public class TxCardSignServiceImpl implements TxCardSignService {

    @Autowired
    private OrgInfoDao orgInfoDao;

    @Autowired
    private TxCardSignDao txCardSignDao;

    @Autowired
    private OrgCourseDao orgCourseDao;

    @Autowired
    private OrgAccountDao orgAccountDao;

    @Autowired
    private OrgClassLessonDao orgClassLessonDao;

    @Autowired
    private OrgCourseTeacherDao orgCourseTeacherDao;

    @Autowired
    private CourseTeacherService courseTeacherService;

    @Autowired
    private TxPurchaseTimescardDao txPurchaseTimescardDao;

    @Autowired
    private OrgStudentDao orgStudentDao;

    @Autowired
    private OrgStudentLessonDao orgStudentLessonDao;

    @Autowired
    private OrgLessonSignDao orgLessonSignDao;

    @Autowired
    private CommonMsgService commonMsgService;

    @Autowired
    private OrgStudentApiService studentApiService;

    @Autowired
    private OrgSignupCourseLessonDao signupCourseLessonDao;

    @Autowired
    private OrgSignupCourseLessonDao courseLessonDao;

    @Override
    @Transactional(readOnly = true)
    public List<TxCardSignDto> list(@NonNull Long orgId, List<Long> courseIds, String courseName, String studentName,
        String studentMobile, Date start, Date end, boolean noLesson, @NonNull PageDto pageDto) {

        Collection<Long> classIds = Sets.newHashSet();
        if (StringUtils.isNotEmpty(courseName)) {
            Integer number = orgAccountDao.getById(orgId, "number").getNumber();
            classIds.addAll(orgCourseDao.getCourseIdsByOrgNumberAndCourseName(number.longValue(), courseName, null,
                CourseTypeEnum.IS_COURSE_TRUE.getCode(), CourseTypeEnum.IS_CLASS_TRUE.getCode(),
                CourseTypeEnum.COURSE_TYPE_CLASS.getCode()));
            if (CollectionUtils.isEmpty(classIds)) {
                return Collections.EMPTY_LIST;
            }
        }

        Collection<Long> studentIds = null;
        if (StringUtils.isNotEmpty(studentName) || StringUtils.isNotEmpty(studentMobile)) {
            studentIds = orgStudentDao
                .searchStudentByLike(orgId, studentMobile, DeleteStatus.NORMAL.getValue(), studentName, "id").stream()
                .map(value -> value.getId()).collect(Collectors.toList());
            if (CollectionUtils.isEmpty(studentIds)) {
                return Collections.EMPTY_LIST;
            }
        }

        if (CollectionUtils.isNotEmpty(courseIds)) {
            classIds.addAll(courseIds);
        }

        List<TxCardSign> cardSigns = txCardSignDao.list(orgId, studentIds, classIds, start, end, noLesson, pageDto);
        return fillDetail(orgId, cardSigns);
    }

    private List<TxCardSignDto> fillDetail(Long orgId, List<TxCardSign> cardSigns) {
        List<TxCardSignDto> result = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(cardSigns)) {
            List<Long> classIds = Lists.newArrayList();
            List<Long> studentIds = Lists.newArrayList();
            List<Long> lessonIds = Lists.newArrayList();
            cardSigns.stream().forEach(txCardSign -> {
                studentIds.add(txCardSign.getStudentId());
                if (txCardSign.getStudentLessonId() > 0) {
                    classIds.add(txCardSign.getClassId());
                    lessonIds.add(txCardSign.getLessonId());
                }
            });

            List<OrgStudent> studentList =
                orgStudentDao.getByIds(studentIds, "id", "name", "nickName", "mobile", "gender");
            log.debug("query students by:{}, result:{}", studentIds, studentList);
            Map<Long, String> classNameMap = orgCourseDao.getCourseNameMap(classIds);
            Map<Long, String> studentNameMap =
                CollectorUtil.collectMap(studentList, input -> input.getId(), input -> input.getNotEmptyName());
            Map<Long, String> studentMobileMap =
                CollectorUtil.collectMap(studentList, input -> input.getId(), input -> input.getMobile());
            Map<Long, Integer> studentGenderMap =
                CollectorUtil.collectMap(studentList, input -> input.getId(), input -> input.getGender());
            Map<Long, OrgClassLesson> lessonMap = CollectorUtil.collectMap(
                orgClassLessonDao.getByIds(lessonIds, "id", "number", "startTime", "endTime"), input -> input.getId());
            Map<Long, String> teacherNameMap = getTeachersOfCourses(orgId, classIds);

            cardSigns.stream().forEach(txCardSign -> {
                TxCardSignDto dto = new TxCardSignDto();
                dto.setId(txCardSign.getId());
                dto.setStudentId(txCardSign.getStudentId());
                dto.setUserId(txCardSign.getUserId());
                dto.setStudentLessonId(txCardSign.getStudentLessonId());
                dto.setStudentName(studentNameMap.get(txCardSign.getStudentId()));
                dto.setStudentGender(studentGenderMap.get(txCardSign.getStudentId()));
                dto.setStudentMobile(studentMobileMap.get(txCardSign.getStudentId()));
                OrgClassLesson lesson = lessonMap.get(txCardSign.getLessonId());
                dto.setLessonName(null == lesson ? "无课程信息"
                    : classNameMap.get(txCardSign.getClassId()) + String.format("-第%s节", lesson.getNumber()));
                dto.setStartTime(null == lesson ? null : lesson.getStartTime());
                dto.setEndTime(null == lesson ? null : lesson.getEndTime());
                dto.setTeacherName(teacherNameMap.getOrDefault(txCardSign.getClassId(), "无"));
                dto.setSignTime(txCardSign.getCreateTime());
                dto.setRemark(txCardSign.getRemark());
                result.add(dto);
            });
        }
        return result;
    }

    private Map<Long, String> getTeachersOfCourses(Long orgId, Collection<Long> courseIds) {
        log.info("orgId={},courseIds={}", orgId, courseIds);
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        Map<Long, String> resultMap = Maps.newHashMap();
        Map<Long, List<Long>> courseTeachersMap = orgCourseTeacherDao.getTeacherMap(courseIds);

        Set<Long> teacherIdsSet = Sets.newHashSet();
        for (Long key : courseTeachersMap.keySet()) {
            List<Long> list = courseTeachersMap.get(key);
            teacherIdsSet.addAll(list);
        }

        List<TeacherResponseDto> teacherDtoList = this.courseTeacherService.getTeachers(teacherIdsSet, orgId);
        Map<Long, String> teacherMap =
            CollectorUtil.collectMap(teacherDtoList, from -> from.getTeacherId(), from -> from.getTeacherName());

        for (Long courseId : courseIds) {
            List<Long> teacherIds = courseTeachersMap.get(courseId);
            List<String> teacherList = Lists.newArrayList();
            if (CollectionUtils.isNotEmpty(teacherIds)) {
                for (Long tid : teacherIds) {
                    teacherList.add(teacherMap.get(tid));
                }
                resultMap.put(courseId,
                    teacherList.stream().filter(s -> s != null).reduce((s, s2) -> s + "," + s2).orElse(""));
            }
        }
        return resultMap;
    }

    @Override
    @Transactional(readOnly = true)
    public TimesCardDto getCardInfo(Long orgId, Long cardId) {
        TxPurchaseTimescard timescard = txPurchaseTimescardDao.getByCardId(orgId, cardId, null);
        Preconditions.checkArgument(null != timescard, "您扫描的次卡不存在");
        Preconditions.checkArgument(orgId.longValue() == timescard.getOrgId(), "您扫描的次卡不属于您的机构");
        Preconditions.checkArgument(timescard.getStatus() != PurchaseTimescardStatus.INIT.getCode()
            && timescard.getStatus() != PurchaseTimescardStatus.CANCEL.getCode(), "您扫描的次卡未支付");
        OrgStudent orgStudent = orgStudentDao.getById(timescard.getStudentId());
        Preconditions.checkArgument(null != orgStudent, "学生不存在");
        Preconditions.checkArgument(
            timescard.getLessonCount() < 0 || timescard.getLessonCount() > txCardSignDao.getSignTimes(cardId),
            "次卡使用次数已经达到上限!");
        TimesCardDto timesCardDto = new TimesCardDto();
        timesCardDto.setId(timescard.getId());
        timesCardDto.setSignupPurchaseId(timescard.getSignupPurchaseId());
        timesCardDto.setCardNumber(timescard.getCardNumber());
        timesCardDto.setSignInTimes(txCardSignDao.getSignTimes(cardId));
        timesCardDto.setLessonCount(timescard.getLessonCount());
        timesCardDto.setCardRemark(timescard.getRemark());
        timesCardDto.setStartTime(timescard.getStartTime());
        timesCardDto.setEndTime(timescard.getEndTime());
        timesCardDto.setStudentId(timescard.getStudentId());
        timesCardDto.setUserId(timescard.getUserId());
        timesCardDto.setStudentName(orgStudent.getName());
        timesCardDto.setStudentAvatar(
            studentApiService.batchGetStudentAvatarUrl(Arrays.asList(orgStudent)).get(timescard.getStudentId()));

        return timesCardDto;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<OrgLessonSign> qrSignBindStudentLessonSign(@NonNull Long orgId, Long cascadeId, List<Long> cardSignIds,
        @NonNull Long lessonId, boolean commit) {
        List<TxCardSign> cardSigns = txCardSignDao.getByIds(cardSignIds);
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(cardSigns), "TxCardSign list is empty!");
        Preconditions.checkArgument(
            cardSigns.stream().allMatch(txCardSign -> txCardSign.getIsdel() == DeleteStatus.NORMAL.getValue()),
            "TxCardSign list can not contains deleted ones");
        Preconditions.checkArgument(cardSigns.stream()
            .allMatch(txCardSign -> txCardSign.getStatus() == TxCardSignStatus.INIT.getCode()
                && txCardSign.getClassId() == Flag.FALSE.getInt() && txCardSign.getLessonId() == Flag.FALSE.getInt()
                && txCardSign.getStudentLessonId() == Flag.FALSE.getInt()),
            "数据不一致! 列表包含已经补充过的扫码签到!");
        return this.bindStudentLessonSign(orgId, cascadeId, cardSigns, lessonId, commit);
    }

    /**
     * @apiNote 扫码签到绑定课节
     * @param commit 是否正式提交
     * @return 成功绑定签到的userIds
     */
    @Transactional(rollbackFor = Exception.class)
    private List<OrgLessonSign> bindStudentLessonSign(@NonNull Long orgId, Long cascadeId,
        @NonNull List<TxCardSign> txCardSigns, @NonNull Long lessonId, boolean commit) {
        // 校验数据
        if (CollectionUtils.isEmpty(txCardSigns)) {
            return Collections.EMPTY_LIST;
        }
        OrgClassLesson classLesson = orgClassLessonDao.getById(lessonId);
        Preconditions.checkArgument(null != classLesson && classLesson.getDelStatus() == DeleteStatus.NORMAL.getValue(),
            "课节被删除了!");
        Preconditions
            .checkArgument(txCardSigns.stream().allMatch(txCardSign -> orgId.longValue() == txCardSign.getOrgId())
                && orgId.longValue() == classLesson.getOrgId(), "orgId不一致!");
        // 准备数据
        Map<Long, TxCardSign> txCardSignMap = CollectorUtil.collectMap(txCardSigns, input -> input.getUserId());
        Set<Long> userIds = txCardSignMap.keySet();
        Preconditions.checkArgument(userIds.size() == txCardSigns.size(), "一次操作中不能有重复学员");
        Map<Long, OrgStudentLesson> studentLessonMap = CollectorUtil.collectMap(
            orgStudentLessonDao.getByLessonIdsStudentIds(Lists.newArrayList(lessonId), userIds),
            input -> input.getUserId());
        Map<Long, OrgLessonSign> studentSignMap =
            CollectorUtil.collectMap(orgLessonSignDao.getByOrgIdCourseIdLessonIdsStudentIds(orgId,
                classLesson.getCourseId(), Lists.newArrayList(lessonId), userIds), input -> input.getUserId());
        // 找出不能签到的数据
        Set<Long> alreadyHasNormalLessonUserIds = Sets.newHashSet();
        Set<Long> alreadyHasLessonSignUserIds = Sets.newHashSet();
        for (Long userId : userIds) {
            OrgStudentLesson orgStudentLesson = studentLessonMap.get(userId);
            OrgLessonSign orgLessonSign = studentSignMap.get(userId);
            // 已经排了正价课节
            if (null != orgStudentLesson && orgStudentLesson.getLessonType() != LessonType.FREE.getCode()) {
                alreadyHasNormalLessonUserIds.add(userId);
                log.info("[bindStudentLessonSign] userId:{}, 正价课:{}", userId, orgStudentLesson);
                continue;
            }
            // 已经有了签到
            if (null != orgLessonSign && orgLessonSign.getStatus() != SignStatus.UNSIGN.getCode()) {
                alreadyHasLessonSignUserIds.add(userId);
                log.info("[bindStudentLessonSign] userId:{}, 签到:{}", userId, orgLessonSign);
                continue;
            }
        }

        // 如果不是提交操作,有签到不上的数据就给前端提示. 抛出异常中断事务.
        StringBuilder exceptionStr = new StringBuilder();
        boolean throwErr = false;
        if (CollectionUtils.isNotEmpty(alreadyHasNormalLessonUserIds)) {
            exceptionStr.append(String.format("已经有%s个学生在该课节绑定正价课 ", alreadyHasNormalLessonUserIds.size()));
            userIds.removeAll(alreadyHasNormalLessonUserIds);
            throwErr = true;
        }
        if (CollectionUtils.isNotEmpty(alreadyHasLessonSignUserIds)) {
            exceptionStr.append(String.format("已经有%s个学生在该课节有签到状态", alreadyHasLessonSignUserIds.size()));
            userIds.removeAll(alreadyHasLessonSignUserIds);
            throwErr = true;
        }
        if (!commit && throwErr) {
            throw new BussinessException(ErpErrorCode.QR_SIGN_CANNOT_BIND_LESSON, exceptionStr.toString());
        }

        List<OrgLessonSign> result = Lists.newArrayList();

        // 正式准备提交的数据
        for (Long userId : userIds) {
            OrgStudentLesson orgStudentLesson = studentLessonMap.get(userId);
            OrgLessonSign orgLessonSign = studentSignMap.get(userId);
            TxCardSign cardSign = txCardSignMap.get(userId);

            if (null == orgStudentLesson) { // 未排课 -> 排个正价课,并且给签到. 签到来源是扫码签到
                orgStudentLesson = new OrgStudentLesson();
                orgStudentLesson.setOrgId(orgId);
                orgStudentLesson.setLessonId(classLesson.getId());
                orgStudentLesson.setUserId(userId);
                orgStudentLesson.setCourseId(classLesson.getCourseId());
                orgStudentLesson.setCreateTime(new Date());
                orgStudentLesson.setUpdateTime(new Date());
                orgStudentLesson.setDelStatus(DeleteStatus.NORMAL.getValue());
                orgStudentLesson.setLessonType(LessonType.NORMAL.getCode());
                orgStudentLesson
                    .setLessonDuration(DateUtil.getMinuteDiff(classLesson.getStartTime(), classLesson.getEndTime()));
                orgStudentLessonDao.save(orgStudentLesson, "orgId", "lessonId", "userId", "courseId", "createTime",
                    "updateTime", "delStatus", "lessonType", "lessonDuration");
                // 回填扫码签到
                cardSign.setClassId(classLesson.getCourseId());
                cardSign.setLessonId(classLesson.getId());
                cardSign.setStudentLessonId(orgStudentLesson.getId());
                cardSign.setUpdateTime(new Date());
                cardSign.setStatus(TxCardSignStatus.NEW_LESSON.getCode());
                txCardSignDao.update(cardSign, "classId", "lessonId", "studentLessonId", "updateTime", "status");
            } else if (orgStudentLesson.getLessonType() == LessonType.FREE.getCode() // 免费课升级正价课,并且给签到
                && (orgLessonSign == null || orgLessonSign.getStatus() == SignStatus.UNSIGN.getCode())) {
                orgStudentLesson.setLessonType(LessonType.NORMAL.getCode());
                orgStudentLesson.setUpdateTime(new Date());
                orgStudentLessonDao.update(orgStudentLesson, "lessonType", "updateTime");
                //清除原来赠送课课销数据
                signupCourseLessonDao.deleteSignupCourseLesson(orgStudentLesson.getUserId(),orgStudentLesson.getLessonId());
                // 回填扫码签到
                cardSign.setClassId(classLesson.getCourseId());
                cardSign.setLessonId(classLesson.getId());
                cardSign.setStudentLessonId(orgStudentLesson.getId());
                cardSign.setUpdateTime(new Date());
                cardSign.setStatus(TxCardSignStatus.UPGRADE_TO_PAY_LESSON.getCode());
                txCardSignDao.update(cardSign, "classId", "lessonId", "studentLessonId", "updateTime", "status");
                //删除赠送课对应的课销数据
                courseLessonDao.batchDelStuLesson(orgId,Arrays.asList(orgStudentLesson.getUserId()),
                        Arrays.asList(orgStudentLesson.getLessonId()));
            } else {
                continue;
            }

            // 统一给签到
            if (null != orgLessonSign) {
                orgLessonSignDao.orgLessonSignBatchEdit(orgId, lessonId, Lists.newArrayList(userId),
                    SignStatus.SIGNED.getCode(), cascadeId, OrgLessonSignSourceEnum.QR_SCAN.getValue());
                result.add(orgLessonSign);
            } else {
                OrgLessonSign lessonSign = new OrgLessonSign();
                lessonSign.setCourseId(classLesson.getCourseId());
                lessonSign.setCreateTime(new Date());
                lessonSign.setLessonId(lessonId);
                lessonSign.setOrgId(orgId);
                lessonSign.setStatus(SignStatus.SIGNED.getCode());
                lessonSign.setUserId(userId);
                lessonSign.setUpdateTime(new Date());
                lessonSign.setUserRole(UserRole.STUDENT.getRole());
                lessonSign.setCascadeId(cascadeId);
                lessonSign.setTeacherId(0);
                lessonSign.setSource(OrgLessonSignSourceEnum.QR_SCAN.getValue());
                orgLessonSignDao.save(lessonSign, false, "courseId", "createTime", "lessonId", "orgId", "status",
                    "userId", "updateTime", "userRole", "cascadeId", "source", "teacherId");
                result.add(lessonSign);
            }
        }
        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void qrSign(@NonNull Long orgId, Long cascadeId, @NonNull Long cardId, Long lessonId, String remark) {
        TxPurchaseTimescard timescard = txPurchaseTimescardDao.getByCardId(orgId, cardId, null);
        Preconditions.checkArgument(null != timescard && orgId.longValue() == timescard.getOrgId(), "您扫描的次卡不存在");
        if (timescard.getStatus() != PurchaseTimescardStatus.SUCCESS.getCode()) {
            Preconditions.checkArgument(timescard.getStatus() != PurchaseTimescardStatus.INIT.getCode()
                && timescard.getStatus() != PurchaseTimescardStatus.CANCEL.getCode(), "您扫描的次卡未支付");
            Preconditions.checkArgument(timescard.getStatus() != PurchaseTimescardStatus.REFUND.getCode(),
                "您扫描的次卡已经退卡，无法签到");
            Preconditions.checkArgument(timescard.getStatus() != PurchaseTimescardStatus.ECPIRE.getCode(),
                "您扫描的次卡已经过期失效，无法签到");
        }
        Preconditions.checkArgument(timescard.getStartTime().before(new Date()), "您的次卡未到开始使用时间，无法签到");
        Preconditions.checkArgument(
            null == timescard.getEndTime()
                || DateUtil.addDay(DateUtil.getStartOfDay(timescard.getEndTime()), 1).after(new Date()),
            "您扫描的次卡已经过期失效，无法签到");
        Integer totalSignTimes = txCardSignDao.getSignTimes(cardId);
        Preconditions.checkArgument(timescard.getLessonCount() < 0 || timescard.getLessonCount() > totalSignTimes,
            "次卡使用次数已经达到上限!");
        Preconditions.checkArgument(timescard.getStatus() == PurchaseTimescardStatus.SUCCESS.getCode());

        TxCardSign txCardSign = new TxCardSign();
        txCardSign.setOrgId(orgId);
        txCardSign.setStudentId(timescard.getStudentId());
        txCardSign.setUserId(timescard.getUserId());
        txCardSign.setCardId(cardId);
        txCardSign.setStatus(TxCardSignStatus.INIT.getCode());
        txCardSign.setRemark(StringUtils.defaultString(remark, ""));
        txCardSign.setClassId(0L);
        txCardSign.setLessonId(0L);
        txCardSign.setStudentLessonId(0L);
        txCardSign.setCreateTime(new Date());
        txCardSign.setUpdateTime(txCardSign.getCreateTime());
        txCardSign.setIsdel(DeleteStatus.NORMAL.getValue());
        this.txCardSignDao.save(txCardSign);
        List<OrgLessonSign> signs = null;
        if (null != lessonId) {
            try {
                signs = this.bindStudentLessonSign(orgId, cascadeId, Lists.newArrayList(txCardSign), lessonId, false);
            } catch (BussinessException e) {
                if (e.getErrorCode() == ErpErrorCode.QR_SIGN_CANNOT_BIND_LESSON) {
                    throw new BussinessException(ErpErrorCode.QR_SIGN_CANNOT_BIND_LESSON, "该学员在该课节上是正价课或已有签到状态");
                } else {
                    throw e;
                }
            }
        }
        txPurchaseTimescardDao.increaseSignInCount(cardId);

        // 发送签到通知
        OrgInfo orgInfo = orgInfoDao.getOrgInfo(orgId.intValue());
        OrgAccount orgAccount = orgAccountDao.getById(orgId);
        OrgStudent orgStudent = orgStudentDao.getById(timescard.getStudentId());
        String courseName = null;
        String teacherName = null;
        OrgClassLesson lesson = null;
        if (CollectionUtils.isNotEmpty(signs)) {
            OrgLessonSign sign = signs.get(0);
            courseName = orgCourseDao.getById(sign.getCourseId(), "name").getName();
            lesson = orgClassLessonDao.getById(sign.getLessonId());
            teacherName = getTeachersOfCourses(orgId, Lists.newArrayList(sign.getCascadeId()))
                .getOrDefault(sign.getCascadeId(), "-");
        }
        this.signSendMsg(orgAccount, orgInfo, orgStudent, teacherName, lesson, courseName, txCardSign.getCreateTime().getTime());

        // 次卡用尽, 发送短信提醒
        if (timescard.getLessonCount() > 0 && timescard.getLessonCount() <= totalSignTimes + 1) {
            commonMsgService.sendTxSms(orgId, orgId, UserRole.ORGANIZATION.getRole(), null, null,
                TxSmsCodeType.TIMECARD_EXHAUSTED, null, orgStudent.getMobile(),
                SmsContentHelper.createTimescardExhausted(orgInfo.getShortName()));
        }
    }

    private void signSendMsg(OrgAccount orgAccount, OrgInfo orgInfo, OrgStudent student, String teacherName, OrgClassLesson lesson,
        String courseName, Long signTime) {
        try {
            SendMsgRequest sendMsgRequest =
                getSendMsgRequest(orgAccount, orgInfo, student, teacherName, lesson, courseName, signTime);
            log.info("sendMsgRequest is:{}", sendMsgRequest);
            sendMsgRequest.setSmsCodeType(TxSmsCodeType.SIGN_NOTIFY_PRESENT);
            this.commonMsgService.sendMsg(sendMsgRequest);
        } catch (Exception e) {
            log.error("error : {} ", e);
            log.warn(e.getMessage());
        }
    }

    private SendMsgRequest getSendMsgRequest(OrgAccount orgAccount, OrgInfo orgInfo, OrgStudent student, String teacherName,
        OrgClassLesson lesson, String courseName, Long signTime) {
        String studentName = StringUtils.defaultIfBlank(student.getNotEmptyName(), "-");
        String orgName = orgInfo.getShortName();
        String extention = orgInfo.getExtension();
        teacherName = StringUtils.defaultIfBlank(teacherName, "-");
        // 构建短信内容
        String smsContent = SmsContentHelper.createSignRecordSmsMsg(1, studentName, signTime, courseName, orgName);
        String url = ErpUtils.createClassSchedule(orgAccount.getNumber(), orgInfo.getOrgId().longValue(), student.getId());
        // 构建微信内容
        SignWechatTemplateMsg msg = SignWechatTemplateMsg.newInstance(studentName, orgName, SignStatus.SIGNED.getCode(),
            StringUtils.defaultIfBlank(courseName, "-"), lesson == null ? null : lesson.getStartTime().getTime(),
            lesson == null ? null : lesson.getEndTime().getTime(), signTime, teacherName, extention, url);
        SendMsgRequest createSendMsgRequestToStu =
            WechatTemplateMsgHelper.createSendMsgRequestToStu(orgInfo.getOrgId().longValue(), student.getMobile(),
                student.getId(), student.getWeixin(), smsContent, WechateTemplateMsgType.COURSE_SIGNIN_TO_STU, msg);
        return createSendMsgRequestToStu;
    }
}
