package com.baijia.tianxiao.sal.card.service.impl;

import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.constant.finance.PurchaseTimescardStatus;
import com.baijia.tianxiao.constants.CommonConfig;
import com.baijia.tianxiao.constants.sms.TxSmsCodeType;
import com.baijia.tianxiao.dal.card.dao.TxCardSignDao;
import com.baijia.tianxiao.dal.card.dto.TimesCardSearchDto;
import com.baijia.tianxiao.dal.card.dto.TimesCardSignSearchDto;
import com.baijia.tianxiao.dal.org.dao.OrgInfoDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.po.OrgBaseInfoDto;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.signup.dao.TxPurchaseTimescardDao;
import com.baijia.tianxiao.dal.signup.po.TxPurchaseTimescard;
import com.baijia.tianxiao.exception.ParameterException;
import com.baijia.tianxiao.filter.TianxiaoPCContext;
import com.baijia.tianxiao.sal.card.dto.StudentTimesCardDto;
import com.baijia.tianxiao.sal.card.dto.TimesCardQrCodeDto;
import com.baijia.tianxiao.sal.card.dto.TimesCardStudentDto;
import com.baijia.tianxiao.sal.card.dto.TimescardRuleDto;
import com.baijia.tianxiao.sal.card.enums.CardStatus;
import com.baijia.tianxiao.sal.card.service.StudentTimesCardService;
import com.baijia.tianxiao.sal.card.service.TxTimescardRuleService;
import com.baijia.tianxiao.sal.common.api.CommonMsgService;
import com.baijia.tianxiao.sal.common.api.OrgStudentApiService;
import com.baijia.tianxiao.sal.organization.org.service.TxCascadeCredentialService;
import com.baijia.tianxiao.sal.student.dto.request.TimesCardStudentSearchDto;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.CollectionHelper;
import com.baijia.tianxiao.util.ListUtil;
import com.baijia.tianxiao.util.SmsContentHelper;
import com.baijia.tianxiao.util.date.DateUtil;
import com.baijia.tianxiao.util.mobile.MaskUtil;

import com.google.common.base.Preconditions;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import lombok.extern.slf4j.Slf4j;

/**
 * Created by bjhl on 17/7/17.
 */
@Service
@Slf4j
public class StudentTimesCardServiceImpl implements StudentTimesCardService {

    @Autowired
    private TxPurchaseTimescardDao purchaseTimescardDao;

    @Autowired
    private TxCardSignDao signDao;

    @Autowired
    private OrgStudentDao studentDao;

    @Autowired
    private OrgInfoDao orgInfoDao;

    @Autowired
    private OrgStudentApiService studentApiService;

    @Autowired
    private TxTimescardRuleService txTimescardRuleService;

    @Autowired
    private TxCascadeCredentialService txCascadeCredentialService;

    @Autowired
    private CommonMsgService commonMsgService;

    @Override
    public List<StudentTimesCardDto> getCardListByStudentId(long orgId, long studentId) {
        Set<Integer> status = new HashSet<>();
        status.add(PurchaseTimescardStatus.SUCCESS.getCode());
        status.add(PurchaseTimescardStatus.ECPIRE.getCode());
        status.add(PurchaseTimescardStatus.REFUND.getCode());
        List<TxPurchaseTimescard> cards = purchaseTimescardDao.getByStudentId(orgId, studentId, status);
        List<StudentTimesCardDto> ret = new ArrayList<>();
        if (cards == null || cards.size() < 1) {
            return ret;
        }
        List<Long> ids = ListUtil.toKeyList(cards, "id", TxPurchaseTimescard.class);
        Map<Long, Integer> signTimesMap = signDao.getSignTimes(ids);
        for (TxPurchaseTimescard card : cards) {
            StudentTimesCardDto dto = toStudentTimesCardDto(card);
            dto.setKexiaoCount(signTimesMap.get(card.getId()) == null ? 0 : signTimesMap.get(card.getId()));
            if (dto.getBuyCount()>0 && dto.getBuyCount() <= dto.getKexiaoCount()) {
                dto.setStatus(CardStatus.END.getCode());
                dto.setStatusStr(CardStatus.END.getDesc());
            }
            ret.add(dto);
        }
        return ret;
    }

    private StudentTimesCardDto toStudentTimesCardDto(TxPurchaseTimescard card) {
        StudentTimesCardDto dto = new StudentTimesCardDto();
        dto.setBuyCount(card.getLessonCount());
        dto.setContractEndTime(card.getEndTime());
        dto.setCardId(card.getId());
        dto.setKexiaoCount(0);
        dto.setRemark(card.getRemark());
        dto.setContractStartTime(card.getStartTime());
        dto.setDelayCount(card.getDelayCount());
        dto.setDelayDays(card.getDeferDays());
        dto.setAmount(card.getPayPrice());
        CardStatus status = getCardStatus(card);
        dto.setStatus(status.getCode());
        dto.setStatusStr(status.getDesc());
        dto.setPurcharseId(String.valueOf(card.getSignupPurchaseId()));
        return dto;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delay(long orgId, long cardId, Date delayDate) {
        TxPurchaseTimescard card = purchaseTimescardDao.getByCardId(orgId, cardId, null);
        if (card == null) {
            log.info("The cardId is not exist.cardId={},orgId={}", cardId, orgId);
            throw new ParameterException("卡号不存在");
        }
        if (card.getEndTime() == null) {
            throw new ParameterException("无限期的卡不能延期");
        }

        TimescardRuleDto ruleDto = txTimescardRuleService.getRuleByOrgId(orgId);
        Date oldEndTime = card.getEndTime();
        card.setUpdateTime(new Date());
        card.setEndTime(delayDate);
        card.setDelayCount(card.getDelayCount() + 1);
        card.setDeferDays(card.getDeferDays() + DateUtil.getDateDiff(oldEndTime, delayDate));
        Preconditions.checkArgument(card.getDelayCount() <= ruleDto.getMaxDeferTimes(),
            String.format("延期次数不能超过%s次", ruleDto.getMaxDeferTimes()));
        Preconditions.checkArgument(card.getDeferDays() <= ruleDto.getMaxTotalDeferDays(),
            String.format("总延期天数不能超过%s天", ruleDto.getMaxTotalDeferDays()));
        Preconditions.checkArgument(DateUtil.getDateDiff(oldEndTime, delayDate) >= ruleDto.getMinDeferDaysEachtime(),
            String.format("单次延期操作不能少于%s天", ruleDto.getMinDeferDaysEachtime()));
        purchaseTimescardDao.update(card);

        OrgBaseInfoDto orgInfo = orgInfoDao.getBaseInfo(card.getOrgId().intValue());
        OrgStudent orgStudent = studentDao.getById(card.getStudentId(), "mobile");
        commonMsgService.sendTxSms(orgId, orgId, UserRole.ORGANIZATION.getRole(), null, null,
            TxSmsCodeType.TIMECARD_DELAY, null, orgStudent.getMobile(),
            SmsContentHelper.createTimescardDelay(delayDate, orgInfo.getShortName()));
    }

    @Override
    @Transactional
    public void modifyRemark(long orgId, Collection<Long> cardIds, String remark) {
        List<TxPurchaseTimescard> cards = purchaseTimescardDao.getByIds(cardIds);
        for (TxPurchaseTimescard card : cards) {
            card.setRemark(remark);
            card.setUpdateTime(new Date());
            purchaseTimescardDao.update(card);
        }
    }

    @Override
    public List<TimesCardStudentDto> getTimesCardStudentList(long orgId, TimesCardStudentSearchDto searchDto,
        PageDto pageDto) {
        List<TimesCardStudentDto> retList = new ArrayList<>();
        TimesCardSearchDto timesCardSearchDto = searchDto.getTimesCardSearchDto();
        if (timesCardSearchDto == null) {
            timesCardSearchDto = new TimesCardSearchDto();
        }

        TimesCardSignSearchDto signSearchDto = searchDto.getSignSearchDto();
        if (signSearchDto == null) {
            signSearchDto = new TimesCardSignSearchDto();
        }

        // 姓名或手机号筛选
        Map<Long, OrgStudent> studentMap = null;
        if (StringUtils.isNotBlank(searchDto.getMobile()) || StringUtils.isNotBlank(searchDto.getName())) {
            List<OrgStudent> studentList = studentDao.getStudentsLikeMobileAndName(orgId, searchDto.getMobile(),
                searchDto.getName(), "id", "name", "mobile", "gender");
            if (studentList == null || studentList.size() <= 0) {
                log.info("No student.orgId={},name={},mobile={}", orgId, searchDto.getName(), searchDto.getMobile());
                return retList;
            }
            studentMap = CollectionHelper.toIdMap(studentList);
            timesCardSearchDto.setStudentIds(studentMap.keySet());
        }

        // 次卡签到筛选
        Map<Long, Integer> cardCountMap = null;
        if (signSearchDto.isFilter()) {
            cardCountMap = signDao.searchCardIdBySign(orgId, signSearchDto);
            if (cardCountMap == null || cardCountMap.isEmpty()) {
                log.info("No sign times card.orgId={},signSearchDto={}", orgId, signSearchDto);
                return retList;
            }
            timesCardSearchDto.setCardIds(cardCountMap.keySet());
        }

        // 次卡基本信息筛选
        List<TxPurchaseTimescard> cardList =
            purchaseTimescardDao.searchTimesCardStudentList(orgId, timesCardSearchDto, pageDto);
        if (cardList == null || cardList.size() <= 0) {
            log.info("No times card.orgId={},searchDto={}", orgId, timesCardSearchDto);
            return retList;
        }

        if (studentMap == null) {
            List<Long> studentIds = ListUtil.toKeyList(cardList, "studentId", TxPurchaseTimescard.class);
            List<OrgStudent> studentList = studentDao.getByIds(studentIds, "id", "name", "mobile", "gender");
            studentMap = CollectionHelper.toIdMap(studentList);
        }

        List<Long> cardIds = ListUtil.toKeyList(cardList, "id", TxPurchaseTimescard.class);
        if (cardCountMap == null) {
            signSearchDto.setCardIds(cardIds);
            cardCountMap = signDao.searchCardIdBySign(orgId, signSearchDto);
        }

        Map<Long, Date> signMap = signDao.getLastSignRecord(orgId, cardIds);
        Map<Long, Integer> signTimesMap = signDao.getSignTimes(cardIds);
        boolean isShow = txCascadeCredentialService.isShowMobile(orgId, TianxiaoPCContext.getTXCascadeId());
        for (TxPurchaseTimescard card : cardList) {
            TimesCardStudentDto dto =
                toTimesCardStudentDto(card, studentMap.get(card.getStudentId()), cardCountMap.get(card.getId()));
            dto.setLastSignTime(signMap.get(dto.getCardId()));
            dto.setSignCount(signTimesMap.get(card.getId()) == null ? 0 : signTimesMap.get(card.getId()));
            if (dto.getBuyCount()>0 && dto.getBuyCount() <= dto.getSignCount()) {
                dto.setStatus(CardStatus.END.getCode());
                dto.setStatusStr(CardStatus.END.getDesc());
            }
            if (!isShow) {
                dto.setMobile(MaskUtil.maskMobile(dto.getMobile()));
            }
            retList.add(dto);
        }
        return retList;
    }

    private TimesCardStudentDto toTimesCardStudentDto(TxPurchaseTimescard card, OrgStudent student, Integer signCount) {
        TimesCardStudentDto dto = new TimesCardStudentDto();
        dto.setBuyCount(card.getLessonCount());
        dto.setCardId(card.getId());
        dto.setContractStartTime(card.getStartTime());
        dto.setContractEndTime(card.getEndTime());
        dto.setDelayCount(card.getDelayCount());
        dto.setMobile(student.getMobile());
        dto.setName(student.getName());
        dto.setRemark(card.getRemark());
        dto.setStudentId(card.getStudentId());
        dto.setStatusStr(getCardStatus(card).getDesc());
        dto.setStatus(getCardStatus(card).getCode());
        dto.setDelayDays(card.getDeferDays());
        if (signCount == null) {
            dto.setSignCount(0);
        } else {
            dto.setSignCount(signCount);
        }
        dto.setGenderStr(student.getGender() == 0 ? "男" : "女");
        return dto;
    }

    @Override
    public TimesCardQrCodeDto getTimesCardQrCode(long orgId, long cardId) {
        TxPurchaseTimescard card = purchaseTimescardDao.getByCardId(orgId, cardId, null);
        TimesCardQrCodeDto dto = new TimesCardQrCodeDto();
        dto.setRemark(card.getRemark());
        dto.setContractEndTime(card.getEndTime());

        OrgStudent student = studentDao.getById(card.getStudentId());
        dto.setName(student.getName());

        dto.setAvatar(studentApiService.batchGetStudentAvatarUrl(Arrays.asList(student)).get(student.getId()));
        dto.setUrl(CommonConfig.KE_DOMAIN + TimesCardQrCodeDto.RESOURCE_NAME + cardId);
        return dto;
    }

    private CardStatus getCardStatus(TxPurchaseTimescard card) {
        if (card.getStartTime() != null && card.getStartTime().after(new Date())) {
            return CardStatus.UN_START;
        }

        if (card.getEndTime() != null) {
            if (card.getStatus() == PurchaseTimescardStatus.REFUND.getCode()) {
                return CardStatus.END;
            } else if (new Date().after(DateUtil.getStartOfDay(DateUtil.addDay(card.getEndTime(), 1)))) {
                return CardStatus.END;
            }
        }else {
            if (card.getStatus() == PurchaseTimescardStatus.REFUND.getCode()) {
                return CardStatus.END;
            }
        }
        return CardStatus.NORMAL;
    }

    public static void main(String[] args) {
        System.out.print(DateUtil.getStartOfDay(DateUtil.addDay(new Date(), 1)));
    }
}
