/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.callservice.service.impl;

import java.util.List;

import javax.annotation.Resource;

import com.baijia.tianxiao.dal.callservice.dao.CallServiceInfoDao;
import com.baijia.tianxiao.dal.callservice.po.CallServiceInfo;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.dal.advisory.dao.OrgCallRecorderDao;
import com.baijia.tianxiao.dal.advisory.po.OrgCallRecorder;
import com.baijia.tianxiao.dal.org.constant.AuditColumType;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgStorageDao;
import com.baijia.tianxiao.dal.org.dao.OrgTxtMsgDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgStorage;
import com.baijia.tianxiao.dal.org.po.OrgTxtMsg;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.callservice.dto.CallUrlAndDurationResponseDto;
import com.baijia.tianxiao.sal.callservice.exception.CallServiceRecordException;
import com.baijia.tianxiao.sal.callservice.service.OrgCallService;
import com.baijia.tianxiao.util.storage.StorageUtil;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;

import lombok.extern.slf4j.Slf4j;

/**
 * @title OrgCallServiceImpl
 * @desc TODO
 * @author cxm
 * @date 2015年12月10日
 * @version 1.0
 */
@Service
@Slf4j
public class OrgCallServiceImpl implements OrgCallService {

    @Resource
    private OrgCallRecorderDao orgCallRecordDao;
    @Resource
    private CallServiceInfoDao callServiceInfoDao;

    @Resource
    private OrgStorageDao orgStorageDao;

    @Resource
    private OrgTxtMsgDao orgTxtMsgDao;

    @Resource
    private OrgAccountDao orgAccountDao;

    /***
     * 查询400通话记录
     * @param callId
     * @return
     */
    @Override
    @Transactional(readOnly = true)
    public CallUrlAndDurationResponseDto getRecordFileUrl(Long callId) {
        Preconditions.checkArgument(callId != null && callId > 0, "call id : %s is illegal.", callId);

        OrgCallRecorder callRecord = orgCallRecordDao.getById(callId, "id", "status", "storageId", "duringTime");
        if (callRecord == null) {
            log.warn("can not get call record by callId:{}", callId);
            throw new CallServiceRecordException(CommonErrorCode.PARAM_ERROR, "无法找到对应的通话记录.");
        }
        if (callRecord.getStatus() != 1) {
            log.warn("call is not connected");
            return null;
        }

        if (callRecord.getStorageId() == null || callRecord.getStorageId() == 0) {
            log.warn("storage id is illegal:{}", callRecord.getStorageId());
            return null;
            //throw new CallServiceRecordException(CommonErrorCode.PARAM_ERROR, "");
        }
        OrgStorage storage = orgStorageDao.getById(callRecord.getStorageId());
        if (storage == null) {
            log.warn("can not found storage by storage id:{}", callRecord.getStorageId());
            throw new CallServiceRecordException(CommonErrorCode.PARAM_ERROR, "");
        }
        String url = StorageUtil.constructUrl(storage.getFid(), storage.getSn(), storage.getMimeType());
        log.info("get org call record:{},record file url:{}", callRecord, url);
        return new CallUrlAndDurationResponseDto(callRecord.getDuringTime(), url);
    }

    @Override
    public CallUrlAndDurationResponseDto getShuanghuFileUrl(Long callId) {
        Preconditions.checkArgument(callId != null && callId > 0, "call id : %s is illegal.", callId);

        CallServiceInfo info = callServiceInfoDao.getById(callId, "id", "status", "storageId", "duration");
        if (info == null) {
            log.warn("can not get CallServiceInfo by callId:{}", callId);
            throw new CallServiceRecordException(CommonErrorCode.PARAM_ERROR, "无法找到对应的通话记录.");
        }
        if (info.getStatus() != 1) {
            log.warn("call is not connected");
            return null;
        }

        if (info.getStorageId() == null || info.getStorageId() == 0) {
            log.warn("storage id is illegal:{}", info.getStorageId());
            return null;
        }
        OrgStorage storage = orgStorageDao.getById(info.getStorageId());
        if (storage == null) {
            log.warn("can not found storage by storage id:{}", info.getStorageId());
            throw new CallServiceRecordException(CommonErrorCode.PARAM_ERROR, "");
        }
        String url = StorageUtil.constructUrl(storage.getFid(), storage.getSn(), storage.getMimeType());
        log.info("get org call record:{},record file url:{}", info, url);
        return new CallUrlAndDurationResponseDto(info.getDuration(), url);
    }

    @Override
    @Transactional(readOnly = true)
    public List<String> getOrgContactTel(Long orgId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "org id is illegal");
        List<String> contactTels = Lists.newArrayList();
        OrgTxtMsg txtMsg = orgTxtMsgDao.getOrgTxtByIdAndType(orgId.intValue(), AuditColumType.PHONE.getValue());
        if (txtMsg == null || StringUtils.isBlank(txtMsg.getValue())) {
            OrgAccount acc = orgAccountDao.getAccountById(orgId.intValue(), "mobile");
            if (acc == null) {
                log.warn("can not found org account by orgId:{}", orgId);
                throw new BussinessException(CommonErrorCode.PERMISSION_DENY);
            }
            contactTels.add(acc.getMobile());
        } else {
            contactTels = Lists.newArrayList(StringUtils.split(txtMsg.getValue(), ','));
        }
        log.info("get contact tel by orgId:{},result:{}", orgId, contactTels);
        return contactTels;

    }

}
