/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.callservice.service.impl;

import com.baijia.tianxiao.dal.callservice.constant.PartyCallType;
import com.baijia.tianxiao.dal.callservice.constant.RLCallStatus;
import com.baijia.tianxiao.dal.callservice.dao.CallServiceCallbackRecordDao;
import com.baijia.tianxiao.dal.callservice.dao.CallServiceDialBackDao;
import com.baijia.tianxiao.dal.callservice.dao.CallServiceInfoDao;
import com.baijia.tianxiao.dal.callservice.po.CallServiceCallbackRecord;
import com.baijia.tianxiao.dal.callservice.po.CallServiceDialBack;
import com.baijia.tianxiao.dal.callservice.po.CallServiceInfo;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgTxtMsgDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgTxtMsg;
import com.baijia.tianxiao.dal.user.dao.UserDao;
import com.baijia.tianxiao.dal.user.po.User;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.callservice.constants.CallInfoStatus;
import com.baijia.tianxiao.sal.callservice.constants.CallServiceErrorCode;
import com.baijia.tianxiao.sal.callservice.dto.BidirectionalCallResponse;
import com.baijia.tianxiao.sal.callservice.dto.CallRecordDto;
import com.baijia.tianxiao.sal.callservice.dto.CallServiceResponse;
import com.baijia.tianxiao.sal.callservice.dto.CallbackRecordRequest;
import com.baijia.tianxiao.sal.callservice.dto.MakeCallDto;
import com.baijia.tianxiao.sal.callservice.dto.MnsMessageBodyDto;
import com.baijia.tianxiao.sal.callservice.dto.MnsMessageBodyDto.MnsCallType;
import com.baijia.tianxiao.sal.callservice.dto.RLBiCallResponse;
import com.baijia.tianxiao.sal.callservice.ronglian.RLCallHelper;
import com.baijia.tianxiao.sal.callservice.service.CallService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.date.DateUtil;
import com.baijia.tianxiao.util.httpclient.HttpClientUtils;
import com.baijia.tianxiao.util.json.JacksonUtil;
import com.baijia.tianxiao.util.properties.PropertiesReader;
import com.baijia.tianxiao.validation.ParamValidateUtils;
import com.aliyun.mns.client.CloudAccount;
import com.aliyun.mns.client.CloudQueue;
import com.aliyun.mns.common.ClientException;
import com.aliyun.mns.common.ServiceException;
import com.aliyun.mns.model.Message;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

/**
 * @author cxm
 * @version 1.0
 * @title CallServiceImpl
 * @desc TODO
 * @date 2015年12月5日
 */
@Slf4j
@Service
public class CallServiceImpl implements CallService, InitializingBean {

    @Resource
    private UserDao userDao;

    @Resource
    private OrgAccountDao orgAccountDao;

    @Autowired
    private OrgTxtMsgDao orgTxtMsgDao;

    @Autowired
    private CallServiceInfoDao callServiceInfoDao;

    @Autowired
    private CallServiceDialBackDao callServiceDialBackDao;

    @Autowired
    private CallServiceCallbackRecordDao callServiceCallbackRecordDao;

    @Autowired(required = false)
    private CloudAccount cloudAccount;
    private CloudQueue downloadFileQueue;
    private CloudQueue consulterFollowRecordQueue;

    @Override
    public boolean callParty(MakeCallDto callDto) {
        fillCallAndSubscriberId(callDto);
        Preconditions.checkArgument(callDto.getSubscriberUserId() != null,
                "subscriber user id or user number can not be null");
        Preconditions.checkArgument(callDto.getCalledUserId() != null, "called user id or user number can not be null");

        long timeStamp = System.currentTimeMillis();

        Properties callserviceProp = PropertiesReader.getProperties("callservice.properties");
        String secStr = callDto.getCalledUserId().toString() + callDto.getSubscriberUserId().toString() + timeStamp
                + callserviceProp.getProperty("callservice.secure.key", CALLSERVICE_VALIDATE_KEY);

        Map<String, String> params = Maps.newHashMap();
        if (StringUtils.isNoneBlank(callDto.getSubscriberMobile())) {
            secStr += callDto.getSubscriberMobile();
            params.put("subscriberNumber", callDto.getSubscriberMobile());
        }
        if (StringUtils.isNoneBlank(callDto.getCalledUserMobile())) {
            secStr += callDto.getCalledUserMobile();
            params.put("calledPartyMobile", callDto.getCalledUserMobile());
        }
        String sign = DigestUtils.md5Hex(secStr);
        params.put("sign", sign);
        params.put("timeStamp", timeStamp + "");
        params.put("callType", "4");
        params.put("calledParty", callDto.getCalledUserId().toString());
        params.put("callSubscriber", callDto.getSubscriberUserId().toString());
        String url = callserviceProp.getProperty("callservice.makecall.url");
        Preconditions.checkArgument(StringUtils.isNoneBlank(url), "url is illegal:" + url);

        try {
            log.info("callParty.url:{}, params:{}", url, params);
            String resp = HttpClientUtils.doGet(url, params);
            log.info("callParty.resp:{}", resp);
            if (StringUtils.isNoneBlank(resp)) {
                CallServiceResponse response = JacksonUtil.str2Obj(resp, CallServiceResponse.class);
                if (response.getStatus() != 200) {
                    log.warn("make call error:{}", response.getError());
                    return false;
                }
                log.info("success submit call party request1 to url:{},params:{},resp:{}", url, params, resp);
                return true;
            }
            return false;
        } catch (Exception e) {
            log.warn("call party catch error,params:{},url:{},error:{}", params, url, e);
            return false;
        }
    }

    private void fillCallAndSubscriberId(MakeCallDto callDto) {
        if ((callDto.getSubscriberUserId() == null) || (callDto.getSubscriberUserId() <= 0)) {
            Preconditions.checkArgument((callDto.getSubscriberNumber() != null) && (callDto.getSubscriberNumber() > 0),
                    "subscriber user id or user number can not both be null");
            if (PartyCallType.getOrgSubsriberType().contains(callDto.getCallType())) {
                OrgAccount orgAcc = orgAccountDao.getAccountByNumber(callDto.getSubscriberNumber().intValue());
                if (orgAcc != null) {
                    callDto.setSubscriberUserId(orgAcc.getId().longValue());
                }
            } else {
                User user = userDao.getByNumber(callDto.getSubscriberNumber(), "id");
                if (user != null) {
                    callDto.setSubscriberUserId(user.getId());
                }
            }
        }
        if ((callDto.getCalledUserId() == null)
                || (!callDto.getCalledUserId().equals(ANONYMOUS_USER_ID) && (callDto.getCalledUserId() <= 0))) {
            Preconditions.checkArgument((callDto.getCalledUserNumber() != null) && (callDto.getCalledUserNumber() > 0),
                    "subscriber user id or user number can not both be null");
            if (PartyCallType.getOrgCalledType().contains(callDto.getCallType())) {
                OrgAccount orgAcc = orgAccountDao.getAccountByNumber(callDto.getSubscriberNumber().intValue());
                if (orgAcc != null) {
                    callDto.setCalledUserId(orgAcc.getId().longValue());
                }
            } else {
                User user = userDao.getByNumber(callDto.getSubscriberNumber(), "id");
                if (user != null) {
                    callDto.setCalledUserId(user.getId());
                }
            }
        }
    }

    @Override
    public BidirectionalCallResponse bidirectionalCall(MakeCallDto callDto) {

        fillCallAndSubscriberId(callDto);
        fillSubcriberMobile(callDto);
        validDuration(callDto.getSubscriberUserId().intValue());

        // 存储基本拨号信息
        CallServiceInfo callServiceInfo = new CallServiceInfo();
        callServiceInfo.setCallSubscriber(callDto.getSubscriberUserId());
        callServiceInfo.setCalledParty(callDto.getCalledUserId());
        callServiceInfo.setCallSubscriberNum(callDto.getSubscriberMobile());
        callServiceInfo.setCalledPartyNum(callDto.getCalledUserMobile());
        callServiceInfo.setStatus(CallInfoStatus.STATUS_PENDING);
        callServiceInfo.setCreateTime(new Date());
        callServiceInfo.setMonth(DateUtil.getCurrentYM());
        callServiceInfo.setCallType(PartyCallType.O2S.getCode());
        callServiceInfoDao.save(callServiceInfo);

        BidirectionalCallResponse res = new BidirectionalCallResponse();
        res.setCallId(callServiceInfo.getId());
        try {
            // 拨号
            RLBiCallResponse rlRes = RLCallHelper.bidirectionalCall(callServiceInfo.getId().toString(),
                    callDto.getSubscriberMobile(), callDto.getCalledUserMobile());
            res.setCallres(true);
            res.setSubscriberDisplayMobile(PropertiesReader.getValue("callservice", RLCallHelper.FROM_DISPLAY));
            // 更新拨号记录
            callServiceInfo.setStatus(CallInfoStatus.STATUS_RESPONSED);
            callServiceInfo.setUniqueId(rlRes.getCallSid());
            callServiceInfoDao.update(callServiceInfo);
        } catch(BussinessException e){
        	log.info("bidirectionalCall fail! content:{}",e);
        	res.setCallres(false);
        } catch (Exception e) {
            log.error("bidirectionalCall error!callDto:{}", callDto, e);
            res.setCallres(false);
            callServiceInfo.setStatus(CallInfoStatus.STATUS_RETRUN_FAILED);
            callServiceInfoDao.update(callServiceInfo);
        }
        return res;
    }

    /**
     * 当月通话总时长校验
     *
     * @param orgId
     */
    private void validDuration(int orgId) {
        if (orgId == 62113) {//针对一个机构双呼时长特殊处理  by hongyan   20161020 //FIXME
            Integer currentDuration = callServiceInfoDao.getTotalDurationByOrgId(orgId);
            if (currentDuration > DEFAULT_HOLDING_SEC * 5) {
                throw new BussinessException(CallServiceErrorCode.TIME_EXHAUSTED);
            }
            return;
        }
        Integer currentDuration = callServiceInfoDao.getTotalDurationByOrgId(orgId);
        if (currentDuration > DEFAULT_HOLDING_SEC) {
            throw new BussinessException(CallServiceErrorCode.TIME_EXHAUSTED);
        }
    }

    /**
     * 如果未传入主叫号码，则取机构的联系方式
     *
     * @param callDto
     */
    private void fillSubcriberMobile(MakeCallDto callDto) {
        if (StringUtils.isBlank(callDto.getSubscriberMobile())) {
            String mobile = null;
            OrgTxtMsg orgTxtMsg = orgTxtMsgDao.getOrgTxtById(callDto.getSubscriberUserId().intValue());

            if ((orgTxtMsg != null) && (orgTxtMsg.getAuditstatus() != null)
                    && (orgTxtMsg.getAuditstatus().intValue() == 1)) {
                mobile = orgTxtMsg.getValue();
                mobile = ParamValidateUtils.filterInvalidateTel(mobile);
            }
            if (StringUtils.isBlank(mobile)) {
                OrgAccount account = orgAccountDao.getAccountById(callDto.getSubscriberUserId().intValue());
                mobile = account.getMobile();
            }
            callDto.setSubscriberMobile(mobile);
        }
    }

    @Override
    @Transactional(rollbackFor = Throwable.class)
    public String bidirectionalCallRecord(CallbackRecordRequest dto) {
        if ((dto != null) && NumberUtils.isNumber(dto.getCallerCdr().getUserData())) {
            // 查询存在的通话记录
            CallServiceInfo info = callServiceInfoDao.getById(Long.valueOf(dto.getCallerCdr().getUserData()));
            if (info != null) {
                log.info("update bidirectionalCall info:" + ToStringBuilder.reflectionToString(info));
                // 再次校验通话合法性
                if (info.getUniqueId().equals(dto.getCallerCdr().getCallSid())) {
                    SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
                    try {
                        if (StringUtils.isNoneBlank(dto.getCallerCdr().getBeginCallTime())) {
                            info.setCallStart(sdf.parse(dto.getCallerCdr().getBeginCallTime()));
                        }
                        if (StringUtils.isNotBlank(dto.getCallerCdr().getEndtime())) {
                            info.setCallEnd(sdf.parse(dto.getCallerCdr().getEndtime()));
                        }
                        info.setUpdateTime(new Date());
                    } catch (ParseException e) {
                        log.error("trans bidirectionalCall time error!", e);
                    }
                    if (StringUtils.isNotBlank(dto.getCallerCdr().getDuration())) {
                        info.setDuration(Integer.valueOf(dto.getCallerCdr().getDuration()));
                    }
                    if (StringUtils.isNotBlank(dto.getCallerCdr().getByetype())) {
                        info.setCdrStatus(RLCallStatus.getCdrStatus(dto.getCallerCdr().getByetype()));
                        info.setResult(Integer.valueOf(dto.getCallerCdr().getByetype()));
                    }
                    saveCallServiceCallbackRecord(dto, info.getCallSubscriber().intValue(), sdf);
                    callServiceInfoDao.update(info);
                }
                // 推咨询队列
                //consulterFollowRecordQueue.putMessage(new Message("1$" + info.getId()));
                //修改消息体为对象
                consulterFollowRecordQueue.putMessage(new Message("1$" + JacksonUtil.obj2Str(info)));

                CallServiceDialBack callServiceDialBack = new CallServiceDialBack();
                callServiceDialBack.setFromMobile(info.getCallSubscriberNum());
                callServiceDialBack.setToMobile(info.getCalledPartyNum());
                callServiceDialBack.setCallStart(info.getCallStart());
                callServiceDialBack.setCallEnd(info.getCallEnd());
                callServiceDialBack.setDuration(info.getDuration());
                callServiceDialBack.setStart(info.getCallStart());
                callServiceDialBack.setCallType(PartyCallType.O2S.getCode());
                callServiceDialBack.setUniqueId(info.getUniqueId());
                callServiceDialBack.setCdrStatus(info.getCdrStatus());
                callServiceDialBackDao.save(callServiceDialBack);

                // 向队列发下载消息
                if (downloadFileQueue != null) {
                    if (StringUtils.isNotBlank(dto.getRecordurl())) {
                        MnsMessageBodyDto messageBody = new MnsMessageBodyDto(MnsCallType.RL.getCode(),
                                info.getUniqueId(), dto.getRecordurl(), info.getCallType().toString());
                        try {
                            log.info("send download msg :{} to mns", messageBody);
                            downloadFileQueue.putMessage(new Message(JacksonUtil.obj2Str(messageBody)));
                        } catch (ServiceException | ClientException e) {
                            log.error("send download record msg catch error:{},callInfo:{}", e, info);
                            log.error("send download record msg catch error:{},request1:{}", e, dto);
                        } catch (Exception e) {
                            log.error("send download record msg catch error:{},callInfo:{}", e, info);
                            log.error("send download record msg catch error:{},request1:{}", e, dto);
                        }
                    } else {
                        log.info("recordurl is null! calledCdr:{},callServiceInfo:{}", dto, info);
                    }
                }

            }
        }
        return RLCallHelper.RL_SUCCESS;
    }

    public static void main(String[] args)
            throws JsonParseException, JsonMappingException, IOException, ParseException {
        String body =
                "{\"calledCdr\":{\"appId\":\"aaf98f8954383ba201544b7cc0090e78\",\"beginCallTime\":\"\",\"byetype\":\"\",\"callSid\":\"16090516000923970001016300009b4c\",\"called\":\"13104042882\",\"caller\":\"15046260225\",\"duration\":\"120\",\"endtime\":\"\",\"lineNumber\":\"423\",\"lostRate\":\"\",\"ringingBeginTime\":\"\",\"ringingEndTime\":\"\",\"starttime\":\"\",\"subId\":\"55d173d4128d11e6bb9bac853d9f54f2\",\"userData\":\"79807\"},\"callerCdr\":{\"appId\":\"aaf98f8954383ba201544b7cc0090e78\",\"beginCallTime\":\"20160905160009\",\"byetype\":\"-10\",\"callSid\":\"16090516000923970001016300009b4c\",\"called\":\"15046260225\",\"caller\":\"01058364406\",\"duration\":\"61\",\"endtime\":\"\",\"lineNumber\":\"423\",\"lostRate\":\"0.00\",\"ringingBeginTime\":\"20160905160015\",\"ringingEndTime\":\"20160905160038\",\"starttime\":\"\",\"subId\":\"55d173d4128d11e6bb9bac853d9f54f2\",\"userData\":\"79807\"},\"orderid\":\"CM1016320160905160009143114\",\"recordurl\":\"\"}";
        CallbackRecordRequest dto = JacksonUtil.str2Obj(body, CallbackRecordRequest.class);
        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddhhmmss");
        Integer orgId = 123;
        System.out.println(dto);
        CallServiceCallbackRecord cscr = new CallServiceCallbackRecord();
        cscr.setOrgId(orgId);
        cscr.setCalledAppId(dto.getCalledCdr().getAppId());
        cscr.setCalledBeginCallTime(StringUtils.isEmpty(dto.getCalledCdr().getBeginCallTime()) ? null
                : sdf.parse(dto.getCalledCdr().getBeginCallTime()));
        cscr.setCalledByetype(dto.getCalledCdr().getByetype());
        cscr.setCalledCallSid(dto.getCalledCdr().getCallSid());
        cscr.setCalledCalled(dto.getCalledCdr().getCalled());
        cscr.setCalledCaller(dto.getCalledCdr().getCaller());
        cscr.setCalledDuration(StringUtils.isEmpty(dto.getCallerCdr().getDuration()) ? 0
                : Integer.parseInt(dto.getCalledCdr().getDuration()));
        cscr.setCalledEndtime(
                StringUtils.isEmpty(dto.getCalledCdr().getEndtime()) ? null : sdf.parse(dto.getCalledCdr().getEndtime()));
        cscr.setCalledLineNumber(dto.getCalledCdr().getLineNumber());
        cscr.setCalledLostRate(dto.getCalledCdr().getLostRate());
        cscr.setCalledRingingBeginTime(StringUtils.isEmpty(dto.getCalledCdr().getRingingBeginTime()) ? null
                : sdf.parse(dto.getCalledCdr().getRingingBeginTime()));
        cscr.setCalledRingingEndTime(StringUtils.isEmpty(dto.getCalledCdr().getRingingEndTime()) ? null
                : sdf.parse(dto.getCalledCdr().getRingingEndTime()));
        cscr.setCalledStarttime(StringUtils.isEmpty(dto.getCalledCdr().getStarttime()) ? null
                : sdf.parse(dto.getCalledCdr().getStarttime()));
        cscr.setCalledSubId(dto.getCalledCdr().getSubId());
        cscr.setCalledUserData(StringUtils.isEmpty(dto.getCalledCdr().getUserData()) ? (-1)
                : Integer.parseInt(dto.getCalledCdr().getUserData()));
        cscr.setCallerAppId(dto.getCallerCdr().getAppId());
        cscr.setCallerBeginCallTime(StringUtils.isEmpty(dto.getCallerCdr().getBeginCallTime()) ? null
                : sdf.parse(dto.getCallerCdr().getBeginCallTime()));
        cscr.setCallerByetype(dto.getCallerCdr().getByetype());
        cscr.setCallerCallSid(dto.getCallerCdr().getCallSid());
        cscr.setCallerCalled(dto.getCallerCdr().getCalled());
        cscr.setCallerCaller(dto.getCallerCdr().getCaller());
        cscr.setCallerDuration(StringUtils.isEmpty(dto.getCallerCdr().getDuration()) ? 0
                : Integer.parseInt(dto.getCallerCdr().getDuration()));
        cscr.setCallerEndtime(
                StringUtils.isEmpty(dto.getCallerCdr().getEndtime()) ? null : sdf.parse(dto.getCallerCdr().getEndtime()));
        cscr.setCallerLineNumber(dto.getCallerCdr().getLineNumber());
        cscr.setCallerLostRate(dto.getCallerCdr().getLostRate());
        cscr.setCallerRingingBeginTime(StringUtils.isEmpty(dto.getCallerCdr().getRingingBeginTime()) ? null
                : sdf.parse(dto.getCallerCdr().getRingingBeginTime()));
        cscr.setCallerRingingEndTime(StringUtils.isEmpty(dto.getCallerCdr().getRingingEndTime()) ? null
                : sdf.parse(dto.getCallerCdr().getRingingEndTime()));
        cscr.setCallerStarttime(StringUtils.isEmpty(dto.getCallerCdr().getStarttime()) ? null
                : sdf.parse(dto.getCallerCdr().getStarttime()));
        cscr.setCallerSubId(dto.getCallerCdr().getSubId());
        cscr.setCallerUserData(StringUtils.isEmpty(dto.getCallerCdr().getUserData()) ? (-1)
                : Integer.parseInt(dto.getCallerCdr().getUserData()));
        cscr.setOrderId(dto.getOrderid());
        cscr.setRecordUrl(dto.getRecordurl());

        cscr.setCallStatus(RLCallStatus.getCdrStatus(dto.getCallerCdr().getByetype()));
        cscr.setConnectMinutes((cscr.getCalledDuration() == 0) ? 0 : ((cscr.getCalledDuration() - 1) / 60) + 1);
        cscr.setTotalMinutes((cscr.getCallerDuration() == 0) ? cscr.getConnectMinutes()
                : ((cscr.getCallerDuration() - 1) / 60) + 1 + cscr.getConnectMinutes());
        System.out.println(cscr.toString());
        System.out.println(61 / 60);
    }

    /**
     * @param dto      容联回调内容
     * @param sdf
     * @param orgId 机构id
     * @throws ParseException
     */
    private void saveCallServiceCallbackRecord(CallbackRecordRequest dto, int orgId, SimpleDateFormat sdf) {
        try {
            CallServiceCallbackRecord cscr = new CallServiceCallbackRecord();
            cscr.setOrgId(orgId);
            cscr.setCalledAppId(dto.getCalledCdr().getAppId());
            cscr.setCalledBeginCallTime(StringUtils.isEmpty(dto.getCalledCdr().getBeginCallTime()) ? null
                    : sdf.parse(dto.getCalledCdr().getBeginCallTime()));
            cscr.setCalledByetype(dto.getCalledCdr().getByetype());
            cscr.setCalledCallSid(dto.getCalledCdr().getCallSid());
            cscr.setCalledCalled(dto.getCalledCdr().getCalled());
            cscr.setCalledCaller(dto.getCalledCdr().getCaller());
            cscr.setCalledDuration(StringUtils.isEmpty(dto.getCallerCdr().getDuration()) ? 0
                    : Integer.parseInt(dto.getCalledCdr().getDuration()));
            cscr.setCalledEndtime(StringUtils.isEmpty(dto.getCalledCdr().getEndtime()) ? null
                    : sdf.parse(dto.getCalledCdr().getEndtime()));
            cscr.setCalledLineNumber(dto.getCalledCdr().getLineNumber());
            cscr.setCalledLostRate(dto.getCalledCdr().getLostRate());
            cscr.setCalledRingingBeginTime(StringUtils.isEmpty(dto.getCalledCdr().getRingingBeginTime()) ? null
                    : sdf.parse(dto.getCalledCdr().getRingingBeginTime()));
            cscr.setCalledRingingEndTime(StringUtils.isEmpty(dto.getCalledCdr().getRingingEndTime()) ? null
                    : sdf.parse(dto.getCalledCdr().getRingingEndTime()));
            cscr.setCalledStarttime(StringUtils.isEmpty(dto.getCalledCdr().getStarttime()) ? null
                    : sdf.parse(dto.getCalledCdr().getStarttime()));
            cscr.setCalledSubId(dto.getCalledCdr().getSubId());
            cscr.setCalledUserData(StringUtils.isEmpty(dto.getCalledCdr().getUserData()) ? (-1)
                    : Integer.parseInt(dto.getCalledCdr().getUserData()));
            cscr.setCallerAppId(dto.getCallerCdr().getAppId());
            cscr.setCallerBeginCallTime(StringUtils.isEmpty(dto.getCallerCdr().getBeginCallTime()) ? null
                    : sdf.parse(dto.getCallerCdr().getBeginCallTime()));
            cscr.setCallerByetype(dto.getCallerCdr().getByetype());
            cscr.setCallerCallSid(dto.getCallerCdr().getCallSid());
            cscr.setCallerCalled(dto.getCallerCdr().getCalled());
            cscr.setCallerCaller(dto.getCallerCdr().getCaller());
            cscr.setCallerDuration(StringUtils.isEmpty(dto.getCallerCdr().getDuration()) ? 0
                    : Integer.parseInt(dto.getCallerCdr().getDuration()));
            cscr.setCallerEndtime(StringUtils.isEmpty(dto.getCallerCdr().getEndtime()) ? null
                    : sdf.parse(dto.getCallerCdr().getEndtime()));
            cscr.setCallerLineNumber(dto.getCallerCdr().getLineNumber());
            cscr.setCallerLostRate(dto.getCallerCdr().getLostRate());
            cscr.setCallerRingingBeginTime(StringUtils.isEmpty(dto.getCallerCdr().getRingingBeginTime()) ? null
                    : sdf.parse(dto.getCallerCdr().getRingingBeginTime()));
            cscr.setCallerRingingEndTime(StringUtils.isEmpty(dto.getCallerCdr().getRingingEndTime()) ? null
                    : sdf.parse(dto.getCallerCdr().getRingingEndTime()));
            cscr.setCallerStarttime(StringUtils.isEmpty(dto.getCallerCdr().getStarttime()) ? null
                    : sdf.parse(dto.getCallerCdr().getStarttime()));
            cscr.setCallerSubId(dto.getCallerCdr().getSubId());
            cscr.setCallerUserData(StringUtils.isEmpty(dto.getCallerCdr().getUserData()) ? (-1)
                    : Integer.parseInt(dto.getCallerCdr().getUserData()));
            cscr.setOrderId(dto.getOrderid());
            cscr.setRecordUrl(dto.getRecordurl());

            cscr.setCallStatus(RLCallStatus.getCdrStatus(dto.getCallerCdr().getByetype()));
            cscr.setConnectMinutes((cscr.getCalledDuration() == 0) ? 0 : ((cscr.getCalledDuration() - 1) / 60) + 1);
            cscr.setTotalMinutes((cscr.getCallerDuration() == 0) ? 0 : ((cscr.getCallerDuration() - 1) / 60) + 1);
            callServiceCallbackRecordDao.save(cscr);
        } catch (NumberFormatException | ParseException e) {
            log.error("save CallServiceCallbackRecord error detail is:{}", e.getMessage());
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "save CallServiceCallbackRecord error");
        }
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        if (cloudAccount != null) {
            Properties mnsProperties = PropertiesReader.getProperties("mns.properties");
            downloadFileQueue =
                    cloudAccount.getMNSClient().getQueueRef(mnsProperties.getProperty("call.recordfile.download.queue"));
            consulterFollowRecordQueue =
                    cloudAccount.getMNSClient().getQueueRef(mnsProperties.getProperty("consulter.followrecord.queue"));

        } else {
            log.warn("init cloud account error.");
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.baijia.tianxiao.sal.callservice.service.CallService#cancelBidirectionalCall(java.lang.String)
     */

    @Override
    public void cancelBidirectionalCall(String callId) {
        CallServiceInfo callServiceInfo = callServiceInfoDao.getById(Long.valueOf(callId));
        if (callServiceInfo != null && StringUtils.isNotBlank(callServiceInfo.getUniqueId())) {
            RLCallHelper.cancelBidirectionalCall(callServiceInfo.getUniqueId());
        }
    }

    @Override
    public List<CallRecordDto> listCallRecord(Integer cascadeId, Integer userId, Integer userType, PageDto pageDto) {
        // List<CallRecordDto> dtoList = new ArrayList<CallRecordDto>();
        // List<CallServiceInfo> infos = callServiceInfoDao.listCallServiceInfoDesc(orgId, mobile, pageDto);
        // if(!CollectionUtils.isEmpty(infos)){
        // CallRecordDto dto = null;
        // List<Integer> storageIds = new ArrayList<Integer>();
        // for(CallServiceInfo info:infos){
        // dto = new CallRecordDto();
        // if( info.getCdrStatus().intValue() == CallServiceCdrStatus.CONNECTED.getCode().intValue()){
        // dto.setCallStatus(1);//接通
        // }else{
        // dto.setCallStatus(2);//未接通
        // }
        //
        // if(info.getCallSubscriberNum().equals("mobile")){
        // dto.setCallType(1);//呼出
        // }else{
        // dto.setCallType(2);//接听
        // }
        // dto.setCreateTime( info.getCreateTime().getTime() );
        // dto.setId( info.getId() );
        // dto.setSeconds( info.getDuration() );
        // dto.setSoundId( info.getStorageId() );
        // if(info.getStorageId()!=null && info.getStorageId()>0){
        // storageIds.add( info.getStorageId().intValue() );
        // }
        // dtoList.add(dto);
        // }
        //
        // //填充 通话语音url
        // List<OrgStorage> storages = this.orgStorageDao.getByIds(storageIds);
        // OrgStorage orgStorage = null;
        // Map<Integer, OrgStorage> storageMap = CollectorUtil.collectMap(storages, new Function<OrgStorage, Integer>()
        // {
        // @Override
        // public Integer apply(OrgStorage arg0) {
        // return arg0.getId();
        // }
        // });
        // for(CallRecordDto callRecordDto : dtoList){
        // orgStorage = storageMap.get(callRecordDto.getSoundId());
        // if(orgStorage!=null){
        // dto.setSoundUrl( StorageUtil.constructUrl(orgStorage.getFid(), orgStorage.getSn(),
        // orgStorage.getMimeType()));
        // }
        // }
        // }
        // return dtoList;
        return null;
    }

}
