/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.marketing.referral.service.impl;

import com.baijia.commons.lang.utils.collection.CollectionUtils;
import com.baijia.tianxiao.dal.activity.dao.ActivityAccessLogDao;
import com.baijia.tianxiao.dal.activity.dao.referral.ReferralInfoDao;
import com.baijia.tianxiao.dal.activity.dao.referral.ReferralRecordDao;
import com.baijia.tianxiao.dal.activity.po.ActivityAccessLog;
import com.baijia.tianxiao.dal.activity.po.DairyCountStatistics;
import com.baijia.tianxiao.dal.activity.po.referral.ReferralInfo;
import com.baijia.tianxiao.sal.marketing.commons.enums.TemplateTypeCategory;
import com.baijia.tianxiao.sal.marketing.referral.dto.ReferralBaseInfo;
import com.baijia.tianxiao.sal.marketing.referral.dto.ReferralStatisticData;
import com.baijia.tianxiao.sal.marketing.referral.service.ReferralStatisticsService;
import com.baijia.tianxiao.util.GenericsUtils;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import lombok.extern.slf4j.Slf4j;

/**
 * @author gaodan
 * @version 1.0
 * @title ReferralStatisticsServiceImpl
 * @desc TODO
 * @date 2016年3月3日
 */
@Slf4j
@Service
public class ReferralStatisticsServiceImpl implements ReferralStatisticsService {

    private static final int REFERRAL_ACTIVITY_CATEGORY = TemplateTypeCategory.REFERRAL_TYPE.getType();

    @Autowired
    private ReferralInfoDao referralInfoDao; // 可以访问DAO数据库中转介绍数据信息的类
    @Autowired
    private ActivityAccessLogDao activityAccessLogDao;
    @Autowired
    private ReferralRecordDao referralRecordDao;

    /**
     * 统计一定时间内转介绍访问的次数
     */
    @Override
    public List<DairyCountStatistics> accessDairyCountStatistics(Long activityId, Long orgId, List<String> dateList) {

        boolean check = validateActivity(activityId, orgId);
        if (!check) {
            return Collections.emptyList();
        }
        List<DairyCountStatistics> retResult = new ArrayList<>();
        List<ActivityAccessLog> logs =
            this.activityAccessLogDao.selectActivityAccessLogByDate(dateList, activityId, REFERRAL_ACTIVITY_CATEGORY);
        if (GenericsUtils.isNullOrEmpty(logs)) {
            logs = new ArrayList<>();
        }
        Map<String, ActivityAccessLog> logMaps =
            CollectionUtils.extractMap(logs, new CollectionUtils.Extracter<String, ActivityAccessLog>() {
                @Override
                public String extract(ActivityAccessLog arg0) {
                    return arg0.getAccessDate();
                }

            });
        Integer count = 0;
        for (String date : dateList) {
            DairyCountStatistics dcs = new DairyCountStatistics();
            ActivityAccessLog log = logMaps.get(date);
            if (log != null) {
                count = log.getAccessCount();
            }
            dcs.setCount(count);
            dcs.setDate(date);
            count = 0;
            retResult.add(dcs);
        }
        return retResult;
    }

    /**
     * 判断当前机构是否参加此活动
     *
     * @param activityId
     * @param orgId
     * @return
     */
    private boolean validateActivity(Long activityId, Long orgId) {
        // ReferralInfo为转介绍活动的PO对象，什么是PO对象？
        ReferralInfo referralInfo = referralInfoDao.getReferralInfo(activityId, orgId);
        if (referralInfo == null) {
            log.info("can not find any activity with activityId:{} and orgId:{}", activityId, orgId);
            return false;
        }
        return true;
    }

    @Override
    public void setStatistics(List<ReferralBaseInfo> rbis) {
        if (GenericsUtils.isNullOrEmpty(rbis)) {
            return;
        }
        List<Long> activityIds =
            CollectionUtils.extractList(rbis, new CollectionUtils.Extracter<Long, ReferralBaseInfo>() {
                @Override
                public Long extract(ReferralBaseInfo arg0) {
                    return arg0.getActivityId();
                }
            });
        Map<Long, Integer> accessLogs =
            this.activityAccessLogDao.selectTotalByActivityId(activityIds, REFERRAL_ACTIVITY_CATEGORY);
        Map<Long, Integer> customerCounts = this.referralRecordDao.listReferrals(activityIds);
        Map<Long, Integer> brokerCounts = this.referralRecordDao.listBrokerCount(activityIds);
        Map<Long, Integer> toPayAmounts = this.referralRecordDao.listToPayCount(activityIds);

        for (ReferralBaseInfo rbi : rbis) {
            Long activityId = rbi.getActivityId();
            Integer accessCount = accessLogs.get(activityId);
            rbi.setBrowseCount(accessCount == null ? 0 : accessCount);
            Integer customerCount = customerCounts.get(activityId);
            rbi.setCustomerNum(customerCount == null ? 0 : customerCount);
            Integer brokerCount = brokerCounts.get(activityId);
            rbi.setBrokerNum(brokerCount == null ? 0 : brokerCount);
            Integer toPayAmount = toPayAmounts.get(activityId);
            rbi.setToPayAmount(toPayAmount == null ? 0 : toPayAmount);
        }
        log.info("set statistics over :{}", rbis);
    }

    @Override
    public Map<Long, ReferralStatisticData> getPeriodReferralData(Date startTime, Date endTime, String startDate, String curDate,
        List<Long> TianxiaoOrgIds, int way) {
        log.info("param:{},{},{}", startTime, endTime, curDate);
        Map<Long, ReferralStatisticData> referralData = new HashMap<>();
        List<Long> orgIds = this.referralInfoDao.getExistOrgIds(TianxiaoOrgIds);
        if (GenericsUtils.isNullOrEmpty(orgIds)) {
            return referralData;
        }

        Map<Long, List<Long>> activityIdsOfOrgIdByDay =
            this.referralInfoDao.getReferralActivityIdsByOrgIds(orgIds, startTime, endTime);
        log.info("Referral ActivityIds=" + activityIdsOfOrgIdByDay);

        Map<Long, List<Long>> activityIdsOfOrgId =
            this.referralInfoDao.getReferralActivityIdsByOrgIds(orgIds, null, null);
        for (Long orgId : orgIds) {
            ReferralStatisticData referralStatisticData = new ReferralStatisticData();
            Integer referralActivityTotal = 0;
            if (GenericsUtils.notNullAndEmpty(activityIdsOfOrgIdByDay.get(orgId))) {
                
                referralActivityTotal = activityIdsOfOrgIdByDay.get(orgId).size();
            }
            List<Long> activityIds = activityIdsOfOrgId.get(orgId);
            log.info("Referral ActivityIds=" + activityIds);

            Integer browseCount = 0;
            if (way == 3) {
                browseCount = activityAccessLogDao.getActivityAccessLogTotal(activityIds,
                    TemplateTypeCategory.REFERRAL_TYPE.getType());
            } else {
                browseCount = this.activityAccessLogDao.getActivityAccessLogTotalByDay(activityIds,
                    TemplateTypeCategory.REFERRAL_TYPE.getType(), startDate, curDate);
            }

            Integer brokerNum = this.referralRecordDao.getBrokerTotal(activityIds, startTime, endTime);
            Integer customerNum = this.referralRecordDao.getReferralTotal(activityIds, startTime, endTime);

            referralStatisticData.setReferralActivityTotal(referralActivityTotal == null ? 0 :referralActivityTotal);
            referralStatisticData.setBrowseCount(browseCount == null ? 0 : browseCount);
            referralStatisticData.setBrokerNum(brokerNum);
            referralStatisticData.setCustomerNum(customerNum == null ? 0 : customerNum);
            referralData.put(orgId, referralStatisticData);
        }

        return referralData;
    }
}
