package com.baijia.tianxiao.sal.marketing.activity.service.impl;

import com.baijia.tianxiao.dal.activity.dao.ActivityConfDao;
import com.baijia.tianxiao.dal.activity.dao.ActivityDao;
import com.baijia.tianxiao.dal.activity.dao.EnrollDao;
import com.baijia.tianxiao.dal.activity.dao.TemplateDao;
import com.baijia.tianxiao.dal.activity.dao.TxActivityCommonDao;
import com.baijia.tianxiao.dal.activity.po.Activity;
import com.baijia.tianxiao.dal.activity.po.ActivityConf;
import com.baijia.tianxiao.dal.activity.po.TxActivityCommon;
import com.baijia.tianxiao.dal.org.constant.AuditStatus;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.sal.marketing.activity.dto.ActivityBaseInfo;
import com.baijia.tianxiao.sal.marketing.activity.dto.ActivityConfigDto;
import com.baijia.tianxiao.sal.marketing.activity.dto.ActivityDetailDto;
import com.baijia.tianxiao.sal.marketing.activity.dto.ActivityRequest;
import com.baijia.tianxiao.sal.marketing.activity.dto.ActivitySearchRequest;
import com.baijia.tianxiao.sal.marketing.activity.service.ActivityService;
import com.baijia.tianxiao.sal.marketing.commons.enums.TemplateTypeCategory;
import com.baijia.tianxiao.sal.marketing.commons.service.ActivityAccessLogService;
import com.baijia.tianxiao.sal.marketing.commons.service.TxActivityCommonService;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.date.DateUtil;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author Rezar
 * @createDate :Jan 10, 2016 11:00:55 PM
 * @desc : 机构活动操作的实现类
 */
@Service
public class ActivityServiceImpl implements ActivityService {

    public final Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private ActivityDao activityDao;
    @Autowired
    private ActivityConfDao activityConfDao;
    @Autowired
    private EnrollDao enrollDao;
    @Autowired
    private TemplateDao templateDao;
    @Autowired
    private ActivityAccessLogService accessLogService;
    @Autowired
    private TxActivityCommonService txActivityCommonService;
    @Autowired
    private TxActivityCommonDao txActivityCommonDao;

    /**
     * 对机构的活动进行存储,其中orgAccount 取当前登录的机构
     */
    @Override
    @Transactional("yunyingTransactionManager")
    public ActivityDetailDto saveOrgActivity(ActivitySearchRequest request, Integer orgId) {
        //判断机构类型 如果是免费版，需要判断机构已经创建活动的个数
        this.txActivityCommonService.checkoutOrgActivityVipLevelAndAmount(orgId.longValue());
        
        Activity activity = construct(request, orgId); // 创建活动对象
        activity.setSwitcher(1);
        Long aId = this.activityDao.saveActivity(activity);
        activity.setId(aId);
        ActivityConf saveActivityConf = toActivityConf(request.getConfig());
        saveActivityConf.setActivityId(aId);
        saveActivityConf.setOrgId(orgId);
        this.activityConfDao.saveActivityConf(saveActivityConf);
        //同步tx_activity_conf
        TxActivityCommon conf = new TxActivityCommon();
        txActivityCommonService.saveOrUpdateTxActivityCommon(conf, TemplateTypeCategory.COMMON_TYPE, activity, saveActivityConf);
        
        ActivitySearchRequest searchRequest = new ActivitySearchRequest();
        searchRequest.setId(aId);
        searchRequest.setOrgId(orgId.longValue());

        templateDao.updateTemplateUseCount(request.getTemplateId().longValue());

        return this.getActivityDetail(searchRequest);
    }
    
    
    /**
     * 获取到机构的活动列表
     */
    @Override
    public List<ActivityBaseInfo> getActivitiesByOrgId(ActivitySearchRequest searchRequest) {
        List<Activity> activities = this.activityDao.listActivities(searchRequest.getStatus(), searchRequest.getId(),
            searchRequest.getOrgId().intValue(), searchRequest.getPageDto());

        if (GenericsUtils.isNullOrEmpty(searchRequest)) {
            return Collections.emptyList();
        }

        Set<Long> activityIds = new HashSet<>();

        List<ActivityBaseInfo> baseInfoList = new ArrayList<>();
        for (Activity activity : activities) {
            activityIds.add(activity.getId());
        }
        Map<Long, ActivityConf> map = activityConfDao.getConfigMapByActivityIds(activityIds);
        for (Activity activity : activities) {
            ActivityConf conf = map.get(activity.getId());
            ActivityBaseInfo info = ActivityBaseInfo.getInstance(activity, conf);
            int enrollCount = enrollDao.getEnrollCount(activity.getId());
            info.setEnrollCount(enrollCount);
            baseInfoList.add(info);
        }
        return baseInfoList;
    }

    private ActivityConf toActivityConf(ActivityConfigDto config) {
        ActivityConf tempConfig = new ActivityConf();
        tempConfig.setAddress(config.getAddress());
        tempConfig.setAreaId(config.getAreaId());
        tempConfig.setCountLimit(config.getCountLimit());
        tempConfig.setDisplay(config.getDisplay());
        tempConfig.setLatitude(config.getLatitude());
        tempConfig.setLongitude(config.getLongitude());
        tempConfig.setCustomConf(config.getCustomConf());
        tempConfig.setStartTime(getTime(config.getStartTime()));
        tempConfig.setEndTime(getTime(config.getEndTime()));
        tempConfig.setReportEndTime(getTime(config.getEnrollDeadline()));
        tempConfig.setTemplateTypeId(config.getTemplateTypeId());
        tempConfig.setTemplateId(config.getTemplateId());
        return tempConfig;
    }

    private Timestamp getTime(Long time) {
        if (time == null) {
            return null;
        }
        return new Timestamp(time);
    }

    /**
     * 构造活动信息
     * 
     * @param request
     * @param orgId
     * @return
     */
    private Activity construct(ActivitySearchRequest request, Integer orgId) {
        Activity activity = new Activity();
        activity.setOrgId(orgId);
        activity.setAuditstatus(AuditStatus.PASS.getValue());
        activity.setBbType(ActivityRequest.BB_TYPE);
        activity.setContent(request.getContent());
        Date date = new Date();
        activity.setCreateTime(DateUtil.getStrByDateFormate(date, "yyyy-MM-dd HH:mm:ss"));
        activity.setUpdateTime(DateUtil.getStrByDateFormate(date, "yyyy-MM-dd HH:mm:ss"));
        activity.setcType(ActivityRequest.C_TYPE);
        activity.setStorageId(-1L); // 活动没有该字段，暂定位-1
        activity.setrType(-1); // 活动没有该字段，暂定为-1
        activity.setSync(-1); // 活动没有该字段，暂定为-1
        activity.setTitle(request.getName()); // 设置标题
        activity.setStick(1); // 活动没有该字段，暂定为1,不置顶
        activity.setSupportNum(0);
        // 0-开关关闭，1-开启
        activity.setSwitcher(request.getStatus());
        return activity;
    }

    /**
     * 删除一个机构活动
     */
    @Override
    @Transactional("yunyingTransactionManager")
    public void deleteActivity(ActivitySearchRequest searchRequest) {
        Activity activity = this.activityDao.getActivityByIdAndOrgId(searchRequest.getId(), searchRequest.getOrgId());
        if (activity != null) {
            activity.setIsdel(DeleteStatus.DELETED.getValue());
            activity.setUpdateTime(DateUtil.getStrByDateFormate(new Date(), "yyyy-MM-dd HH:mm:ss"));
            this.activityDao.update(activity);
            
            //同步tx_activity_conf
            TxActivityCommon conf = txActivityCommonDao.getTxActivityCommon(activity.getOrgId().longValue(), TemplateTypeCategory.COMMON_TYPE.getTypeId(), activity.getId());
            conf.setDelStatus(DeleteStatus.DELETED.getValue());
            conf.setUpdateTime(new Date());
            txActivityCommonService.saveOrUpdateTxActivityCommon(conf, null, null, null, "delStatus", "updateTime");;
            logger.info("[Activity] delete status set success confId:{}", conf.getId());
        } else {
            logger.info("activity is not exists {}", searchRequest.getId());
        }
    }

    /**
     * 获取机构活动的详情
     */
    @Override
    public ActivityDetailDto getActivityDetail(ActivitySearchRequest searchRequest) {
        Long orgId = searchRequest.getOrgId();
        if (orgId == null) {
            return null;
        }
        Activity activity = this.activityDao.getActivityByIdAndOrgId(searchRequest.getId(), orgId);
        logger.info("find activity {}", activity);
        if (activity == null) {
            return null;
        }
        Map<Long, ActivityConf> confMap =
            this.activityConfDao.getConfigMapByActivityIds(Arrays.asList(searchRequest.getId()));
        int enrollCount = enrollDao.getEnrollCount(activity.getId());
        logger.info("confMap is :{}", confMap);
        if (confMap != null) {
            logger.info("confMap2 is :{}", confMap);
            if (confMap.get(searchRequest.getId()) != null) {
                logger.info("build response with activity and acitvity'conf {} , {} ", activity,
                    confMap.get(searchRequest.getId()));

                ActivityDetailDto buildResponse =
                    ActivityDetailDto.buildResponse(activity, confMap.get(searchRequest.getId()));
                buildResponse.setEnrollCount(enrollCount);
                return buildResponse;
            } else {
                logger.info("not find any conf ");
                ActivityDetailDto buildResponse = ActivityDetailDto.buildResponse(activity);
                buildResponse.setCustomConf(new Object());
                buildResponse.setEnrollCount(enrollCount);
                return buildResponse;
            }
        }
        return null;
    }

    /**
     */
    @Override
    @Transactional("yunyingTransactionManager")
    public ActivityDetailDto update(ActivitySearchRequest searchRequest) {
        Long orgId = searchRequest.getOrgId();
        if (orgId == null) {
            return null;
        }
        Activity activity = this.activityDao.getActivityByIdAndOrgId(searchRequest.getId(), orgId);
        if (activity != null) {
            Activity updateActivity = construct(searchRequest, activity.getOrgId());
            // logger.info("need update : {}", updateActivity);
            updateActivity.setId(activity.getId());
            updateActivity.setOrgId(activity.getOrgId());
            updateActivity.setSupportNum(activity.getSupportNum()); // 更新的时候，这个值不应该被覆盖
            updateActivity.setUpdateTime(DateUtil.getStrByDateFormate(new Date(), "yyyy-MM-dd HH:mm:ss"));
            this.activityDao.update(updateActivity);
            ActivityConfigDto configDto = searchRequest.getConfig();
            configDto.setDisplay(searchRequest.getStatus() == null ? 1 : searchRequest.getStatus());
            ActivityConf newConf = toActivityConf(configDto);

            ActivityConf sourceConf = activityConfDao.getConfigByActivityId(activity.getId());

            if (sourceConf == null) {
                logger.error("[ActivityConf] conf is not exist.activityId = " + activity.getId());
                return null;
            } else {
                newConf.setId(sourceConf.getId());
                newConf.setActivityId(activity.getId());
                if (newConf.getStartTime() == null) {
                    newConf.setStartTime(sourceConf.getStartTime());
                }
            }

            this.activityConfDao.updateActivityConf(newConf);
            
            //同步tx_activity_conf
            TxActivityCommon conf = txActivityCommonDao.getTxActivityCommon(orgId, TemplateTypeCategory.COMMON_TYPE.getTypeId(), activity.getId());
            txActivityCommonService.saveOrUpdateTxActivityCommon(conf, TemplateTypeCategory.COMMON_TYPE, updateActivity, newConf);

            if (searchRequest.getTemplateId() != null) {
                this.templateDao.updateTemplateUseCount(searchRequest.getTemplateId().longValue());
            }

            return this.getActivityDetail(searchRequest);
        } else {
            return null;
        }
    }

    @Override
    @Transactional("yunyingTransactionManager")
    public void incrBrowseCount(Long activityId, Long orgId) {
        Activity activity = this.activityDao.getActivityByIdAndOrgId(activityId, orgId);
        if (activity != null) {
            Integer browseCount = activity.getSupportNum();
            if (browseCount == null) {
                browseCount = 1;
            } else {
                browseCount++;
            }
            activity.setSupportNum(browseCount);
            activity.setUpdateTime(DateUtil.getStrByDateFormate(new Date(), "yyyy-MM-dd HH:mm:ss"));
            this.activityDao.update(activity);
            // 设置今日访问量
            accessLogService.setTodayAccessCount(activityId, TemplateTypeCategory.COMMON_TYPE);
        }
    }

}
