
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */

package com.baijia.tianxiao.dal.comment.dao;

import com.baijia.tianxiao.dal.comment.po.CommentAudit;
import com.baijia.tianxiao.dal.comment.po.CommentStatic;
import com.baijia.tianxiao.dal.comment.po.OrgCommentAudit;
import com.baijia.tianxiao.dal.comment.po.OrgLessonComment;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.CommonDao;
import lombok.NonNull;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * @author shanyu
 * @version 1.0
 * @title OrgLessonCommentDao
 * @desc TODO
 * @date 2015年12月30日
 */

public interface OrgLessonCommentDao extends CommonDao<OrgLessonComment> {

    /**
     * 获取课节学员评价列表
     *
     * @param lessonId
     * @param queryProps
     * @param isSystem   是否是系统默认评价
     * @return
     */
    List<OrgLessonComment> getLessonStudentComments(Long lessonId, Boolean isSystem, String... queryProps);

    /**
     * 获取课节老师评语列表
     *
     * @param lessonId
     * @param queryProps
     * @param isSystem   是否是系统默认评价
     * @return
     */
    List<OrgLessonComment> getLessonTeacherComments(Long lessonId, Boolean isSystem, String... queryProps);

    /**
     * 获取课节老师评语列表
     *
     * @param lessonId
     * @param queryProps
     * @param userRole   学员角色
     * @param isSystem   是否是系统默认评价
     * @return
     */
    List<OrgLessonComment> getLessonComments(Long lessonId, Boolean isSystem, Integer userRole, String... queryProps);

    /**
     * 获取评价详情（老师，学员）
     *
     * @param userId
     * @param userRole
     * @param isSystem
     * @param queryProps
     * @return
     */
    OrgLessonComment getLessonCommentDetail(Long lessonId, Long userId, Integer userRole, Boolean isSystem,
                                            String... queryProps);

    OrgLessonComment getLessonCommentByFromAndTo(Long lessonId, Long fromId, Long toId, Integer userRole, Boolean isSystem,
                                                 String... queryProps);

    /**
     * 获取学生在本机构中的各个课程的评价信息
     *
     * @param orgId
     * @param userId
     * @param isFrom     true:表示是对别人的评价，false表示对该学生的评价
     * @param isSystem
     * @param queryPorps
     * @return
     */
    List<OrgLessonComment> getStudentComments(Long orgId, Long userId, Integer userRole, Boolean isFrom,
                                              Boolean isSystem, Collection<Long> lessonIds, PageDto pageDto, String... queryPorps);

    /**
     * 获取学员所有评价
     *
     * @param orgId
     * @param userId
     * @param isSystem
     * @param queryProps
     * @return
     */
    List<OrgLessonComment> getStudentComments(Long orgId, Long userId, Boolean isSystem, PageDto pageDto,
                                              String... queryProps);

    /**
     * 获取学员所得到的所有评价
     *
     * @param orgId
     * @param userId
     * @param isSystem
     * @param queryProps
     * @return
     */
    List<OrgLessonComment> getTeacherCommentsOfStudent(Long orgId, Long userId, Boolean isSystem, PageDto pageDto,
                                                       String... queryProps);

    /**
     * 获取课节评价数，总得分
     *
     * @param lessonId
     * @param orgId
     * @return
     */
    CommentAudit getLessonCommentAudit(Long lessonId, Long orgId, Boolean isSystem);


    /**
     * 星级评分
     *
     * @param lessonId
     * @param orgId
     * @param isSystem
     * @return
     */
    CommentAudit starScoreLessonCommentAudit(Long lessonId, Long orgId, Boolean isSystem);

    /**
     * 获取评价
     *
     * @param lessonIds
     * @param orgId
     * @param isSystem
     * @return
     */
    public CommentAudit getCommentAudit(Collection<Long> lessonIds, Long orgId, Boolean isSystem);

    /**
     * 获取机构老师的总评价数
     *
     * @param orgId
     * @param teacherIds
     * @param userRole
     * @param endTime    结束时间
     * @return <老师－－－评价数>
     */
    public Map<Long, CommentAudit> getTeacherCommentCountMap(Long orgId, Collection<Long> teacherIds, Integer userRole,
                                                             Date endTime, Boolean isSystem);

    /**
     * 获取课节学生／老师评价数量
     *
     * @param lessonIds
     * @param orgId
     * @param userRole
     * @param isSystem
     * @return
     */
    public Map<Long, Integer> getLessonCommentCountMap(Collection<Long> lessonIds, Long orgId, Long fromId, Long toId,
                                                       Integer userRole, Boolean isSystem);

    /**
     * 查看已评价学员Id
     *
     * @param lessonIds
     * @param userRole
     * @param isSystem
     * @return
     */
    public Map<Long, List<Long>> getLessonCommentStudentMap(Collection<Long> lessonIds, Boolean isSystem,
                                                            Integer userRole);

    List<CommentAudit> getLessonsCommentAudit(Collection<Long> lessonIds, Long orgId, Integer userRole,
                                              Boolean isSystem);

    List<Long> getCommentLessonIdsOfStudent(Long userId);

    Integer getStudentLessonCommentCount(Long orgId, Long userId, Boolean isSystem);

    /**
     * 获取机构评价数
     *
     * @param orgId
     * @param isSystem
     * @param lowScore
     * @return
     */
    Map<Integer, Integer> getCommentCountOfOrg(Long orgId, Boolean isSystem, Integer lowScore);

    /**
     * 按时间统计评价
     *
     * @param startTime
     * @param endTime
     * @param isSystem
     * @return
     */
    List<OrgCommentAudit> getOrgCommentAudit(Date startTime, Date endTime, Boolean isSystem);

    /**
     * 统计评价数量
     *
     * @param orgId     机构Id
     * @param lessonIds 课节ids （注意，lessonIds为空表示不过滤lessonId）
     * @param userRole  评价角色
     * @param lowSocre  评分低于某值
     * @param hasPic    是否有图片
     * @param highScore 评分高于某值
     * @param isSystem  是否是系统评价
     * @return
     */
    Integer getCommentCount(Long orgId, Collection<Long> lessonIds, Integer userRole, Integer lowSocre, Boolean hasPic,
                            Integer highScore, Boolean isSystem);

    /**
     * 统计机构某段时间评价数
     *
     * @param orgId
     * @param userRole  评价角色 2学生 6老师
     * @param lowSocre  评分低于某值
     * @param hasPic    是否有图片
     * @param highScore 评分高于某值
     * @param isSystem  是否是系统评价
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @return
     */
    Integer getPeriodComment(Long orgId, Collection<Long> lessonIds, Integer userRole, Integer lowSocre, Boolean hasPic,
                             Integer highScore, Boolean isSystem, String startDate, String endDate);

    /**
     * 统计机构某段时间评价数
     *
     * @param userRole 评价角色 2学生 0老师
     * @return
     */
    Map<Long, Integer> getComment(int userRole, Date startTime, Date endTime, List<Long> TianxiaoOrgIds);

    /**
     * 获取评价
     *
     * @param orgId     机构Id
     * @param lessonIds 课节ids （注意，lessonIds为空表示不过滤lessonId）
     * @param userRole  评价角色
     * @param hasPic    是否有图片
     * @param highScore 评分高于某值
     * @param page
     * @param isSystem  是否是系统评价
     * @return
     */
    List<OrgLessonComment> getComments(Long orgId, Collection<Long> lessonIds, Integer userRole, Integer lowScore,
                                       Boolean hasPic, Integer highScore, PageDto page, Boolean isSystem, String... queryProps);

    /**
     * 按天获取机构评价信息
     */
    Map<String, Integer> getOrgCommentMap(Date startTime, Date endTime, List<Long> TianxiaoOrgIds, Integer userRole);

    Long getSumCommentByTeacher(Long teacherUserId, String... queryProps);

    Long getCountCommentByTeacher(Long teacherUserId, String... queryProps);

    Map<Long, Long> getAveCommentByTeacherIds(Collection<Long> teacherUserIds, String... queryProps);

    List<OrgLessonComment> getComments(@NonNull Long orgId, @NonNull Long lessonId, List<Long> fromIds, List<Long> toIds,
                                       @NonNull Integer userRole);

    OrgLessonComment getByIdAndOrgId(Long commentId, Long orgId);


    /**
     * 新评价列表 筛选
     *
     * @param orgId
     * @param lessonIds
     * @param userRole
     * @param startScore
     * @return
     */
    List<OrgLessonComment> listComment(Long orgId, Collection<Long> classIds, Collection<Long> lessonIds, Integer searchTeacherUserId, Date beginDate, Date endDate, Integer userRole, Integer startScore, Long lastId, Integer pageSize);

    /**
     * 计算评价列表统计
     *
     * @param orgId
     * @param lessonIds
     * @param userRole
     * @param startScore
     * @return
     */
    CommentStatic listCommentStatic(Long orgId, Collection<Long> classIds, Collection<Long> lessonIds, Integer searchTeacherUserId, Date beginDate, Date endDate, Integer userRole, Integer startScore);


    /**
     * 搜索评价
     *
     * @param orgId
     * @param lessonIds
     * @param teacherUserId
     * @param studentUserId
     * @param searchKey
     * @param pageSize
     * @return
     */
    List<OrgLessonComment> listSearchComment(Long orgId, Collection<Long> classIds, Collection<Long> lessonIds, List<Long> teacherUserId, List<Long> studentUserId, String searchKey, Integer pageSize);


    /**
     * 根据学生id获取该课节的老师学生评价
     *
     * @param orgId
     * @param lessonId
     * @param userId
     * @return
     */
    List<OrgLessonComment> listCommentByStudent(Long orgId, Long lessonId, Long userId);
}
