package com.baijia.tianxiao.dal.org.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Repository;

import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.constant.SignStatus;
import com.baijia.tianxiao.constants.UserRoleEnum;
import com.baijia.tianxiao.dal.org.dao.OrgLessonSignDao;
import com.baijia.tianxiao.dal.org.dto.DeleteStatus;
import com.baijia.tianxiao.dal.org.po.LessonSignAudit;
import com.baijia.tianxiao.dal.org.po.OrgLessonSign;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.baijia.tianxiao.util.query.MapBatchQueryTemplate;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

@Repository
@Slf4j
public class OrgLessonSignDaoImpl extends JdbcTemplateDaoSupport<OrgLessonSign> implements OrgLessonSignDao {

    public OrgLessonSignDaoImpl() {
        super(OrgLessonSign.class);
    }

    @Override
    public List<OrgLessonSign> getStudentLessonSign(Long orgId, Long courseId, Long userId) {

        String sql =
                "select distinct ocl.id as lesson_id, ocl.org_id, ocl.number as `index`, ols.status from tts.org_class_lesson ocl "
                        + "join tts.org_student_lesson osl ON (osl.lesson_id = ocl.id and osl.org_id = :orgId) "
                        + "left join tts.org_lesson_sign ols ON (ocl.id = ols.lesson_id and ols.user_id = :userId and user_role = 2) "
                        + "where ocl.org_id = :orgId and ocl.course_id = :courseId and osl.student_id = :userId and osl.del_status=0 order by ocl.number asc";

        Map<String, Object> paramMap = Maps.newHashMap();
        paramMap.put("orgId", orgId);
        paramMap.put("courseId", courseId);
        paramMap.put("userId", userId);
        return this.getNamedJdbcTemplate().query(sql, paramMap,
                new BeanPropertyRowMapper<OrgLessonSign>(OrgLessonSign.class));
    }

    @Override
    public Map<Long, Integer> querySignedLessonStudentCountMap(final Long orgId, Collection<Long> lessonIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Collections.emptyMap();
        }
        Map<String, Object> params = Maps.newHashMap();
        StringBuilder sb = new StringBuilder("SELECT count(sign.user_id) AS numOfUser,sign.lesson_id AS lessonId ");
        sb.append("FROM tts.org_lesson_sign sign LEFT JOIN tts.org_student_lesson lesson ");
        sb.append("ON sign.lesson_id=lesson.lesson_id AND sign.user_id=lesson.student_id ");
        sb.append("WHERE sign.org_id=:orgId AND lesson.org_id=:orgId ");
        params.put("orgId", orgId);
        sb.append("AND sign.lesson_id IN (:lessonIds) ");
        params.put("lessonIds", lessonIds);

        sb.append("AND sign.user_role=:userRole ");
        params.put("userRole", UserRoleEnum.STUDENT.getCode());
        sb.append("AND sign.status=:status ");
        params.put("status", SignStatus.SIGNED.getCode());
        sb.append("AND lesson.del_status=:delStatus ");
        params.put("delStatus", DeleteStatus.NORMAL.getValue());
        sb.append("GROUP BY sign.lesson_id ");

        final Map<Long, Integer> result = Maps.newHashMap();
        this.getNamedJdbcTemplate().query(sb.toString(), params, new RowCallbackHandler() {
            @Override public void processRow(ResultSet rs) throws SQLException {
                result.put(rs.getLong("lessonId"),rs.getInt("numOfUser"));
            }
        });
        return result;
    }
    
    @Override
    public Map<Long, Integer> querySignLessonStudentCountMap(final Long orgId, Collection<Long> lessonIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Collections.emptyMap();
        }
        Map<String, Object> params = Maps.newHashMap();
        StringBuilder sb = new StringBuilder("SELECT count(sign.user_id) AS numOfUser,sign.lesson_id AS lessonId ");
        sb.append("FROM tts.org_lesson_sign sign LEFT JOIN tts.org_student_lesson lesson ");
        sb.append("ON sign.lesson_id=lesson.lesson_id AND sign.user_id=lesson.student_id ");
        sb.append("WHERE sign.org_id=:orgId AND lesson.org_id=:orgId ");
        params.put("orgId", orgId);
        sb.append("AND sign.lesson_id IN (:lessonIds) ");
        params.put("lessonIds", lessonIds);
        
        sb.append("AND sign.user_role=:userRole ");
        params.put("userRole", UserRoleEnum.STUDENT.getCode());
        sb.append("AND sign.status>:status ");
        params.put("status", SignStatus.UNSIGN.getCode());
        sb.append("AND lesson.del_status=:delStatus ");
        params.put("delStatus", DeleteStatus.NORMAL.getValue());
        sb.append("GROUP BY sign.lesson_id ");
        
        final Map<Long, Integer> result = Maps.newHashMap();
        this.getNamedJdbcTemplate().query(sb.toString(), params, new RowCallbackHandler() {
            @Override public void processRow(ResultSet rs) throws SQLException {
                result.put(rs.getLong("lessonId"),rs.getInt("numOfUser"));
            }
        });
        return result;
    }

    @Override
    public List<OrgLessonSign> getStudentLessonSign(final Long orgId, final Long courseId, Collection<Long> lessonIds,
                                                    final String... queryProps) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Lists.newArrayList();
        }
        BatchQueryTemplate<Long, List<OrgLessonSign>> batch = new ListBatchQueryTemplate<>();
        return batch.batchQuery(lessonIds, new BatchQueryCallback<Long, List<OrgLessonSign>>() {
            @Override
            public List<OrgLessonSign> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder(queryProps);
                if(orgId!=null){
                	builder.eq("orgId", orgId);
                }
                if (courseId != null) {
                    builder.eq("courseId", courseId);
                }
                builder.in("lessonId", querySet);
                builder.eq("userRole", UserRole.STUDENT.getRole());
                return queryList(builder);
            }
        });
    }

    @Override
    public List<OrgLessonSign> getStudentLessonSign(Long orgId, Long lessonId, String... queryProps) {
        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder(queryProps);
        builder.eq("lessonId", lessonId);
        builder.eq("orgId", orgId);
        builder.eq("userRole", UserRole.STUDENT.getRole());
        return queryList(builder);
    }

    @Override
    public List<OrgLessonSign> getStudentLessonByStatus(Long orgId, Long lessonId, int status, String... queryProps) {
        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder(queryProps);
        builder.eq("lessonId", lessonId);
        builder.eq("orgId", orgId);
        builder.eq("status", SignStatus.UNSIGN.getCode());
        builder.eq("userRole", UserRole.STUDENT.getRole());
        return queryList(builder);
    }

    @Override
    public List<OrgLessonSign> getLessonSignList(Long orgId, Long courseId, Long lessonId, Long userId,
                                                 Integer userRole, Integer status, boolean withUnsign, Date beginTime, Date endTime, String... props) {
        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder(props);
        if (orgId != null) {
            builder.eq("orgId", orgId);
        }
        if (courseId != null) {
            builder.eq("courseId", courseId);
        }

        if (lessonId != null) {
            builder.eq("lessonId", lessonId);
        }

        if (userId != null) {
            builder.eq("userId", userId);
        }

        if (userRole != null) {
            builder.eq("userRole", userRole);
        }

        if (status != null) {
            builder.eq("status", status);
        }

        if (beginTime != null) {
            builder.ge("createTime", beginTime);
        }

        if (endTime != null) {
            builder.le("createTime", endTime);
        }
        
        if(!withUnsign){
            builder.ne("status",SignStatus.UNSIGN.getCode());
        }

        return queryList(builder);
    }

    @Override
    public List<OrgLessonSign> getUserLessonSignList(final Long orgId, Collection<Long> userIds, final Long courseId,
                                                     final Integer userRole, final Integer status, final String... props) {

        if (CollectionUtils.isEmpty(userIds)) {
            return Collections.emptyList();
        }
        BatchQueryTemplate<Long, List<OrgLessonSign>> batchQueryTemplate = new ListBatchQueryTemplate<>();

        return batchQueryTemplate.batchQuery(userIds, new BatchQueryCallback<Long, List<OrgLessonSign>>() {

            @Override
            public List<OrgLessonSign> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder();
                if (props != null) {
                    builder.select(props);
                }
                builder.in("userId", querySet);
                if (status != null) {
                    builder.eq("status", status);
                }
                builder.eq("orgId", orgId);
                builder.eq("courseId", courseId);
                builder.eq("userRole", userRole);
                builder.asc("createTime");
                return queryList(builder);
            }
        });

    }

    @Override
    public Map<Long, Integer> getOrgLessonSignCount(Collection<Long> lessonIds, final Integer userRole) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Collections.emptyMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(lessonIds,
                new BatchQueryCallback<Long, Map<Long, Integer>>() {
                    @Override
                    public Map<Long, Integer> doQuery(Collection<Long> lessonIds) {
                        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder("userId");
                        builder.count("id", "signCount");
                        builder.in("lessonId", lessonIds);
                        builder.eq("status", SignStatus.SIGNED.getCode());
                        builder.eq("userRole", userRole);
                        builder.group("userId");
                        final Map<Long, Integer> result = Maps.newHashMap();
                        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                                new RowCallbackHandler() {
                                    @Override
                                    public void processRow(ResultSet rs) throws SQLException {
                                        result.put(rs.getLong("userId"), rs.getInt("signCount"));
                                    }
                                });
                        return result;
                    }
                });
    }

    @Override
    public void orgLessonSignBatchEdit(Long lessonId, Collection<Long> userIds, Integer status, @NonNull Long cascadeId, @NonNull Integer source) {
        Preconditions.checkArgument(lessonId != null && lessonId > 0, "lessonId is illegal");
        Preconditions.checkArgument(status != null && status >= 0, "status is illegal");
        if(CollectionUtils.isEmpty(userIds)){
            return;
        }
        String sql = "update tts.org_lesson_sign set status = :status, cascade_Id=:cascadeId, source=:source where lesson_id = :lessonId  AND  user_id in (:userId)";
        Map<String, Object> params = new HashMap<>();
        params.put("status", status);
        params.put("userId", userIds);
        params.put("lessonId", lessonId);
        params.put("cascadeId", cascadeId);
        params.put("source", source);
        log.debug("batch sign sql:{},params:{}", sql, params);
        this.getNamedJdbcTemplate().update(sql, params);
    }
    
    @Override
    public void orgLessonSignBatchEdit(Collection<Long> lessonIds, Collection<Long> userIds, Integer status, @NonNull Long cascadeId, @NonNull Integer source) {
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(lessonIds), "lessonIds is empty");
        Preconditions.checkArgument(status != null && status >= 0, "status is illegal");
        if(CollectionUtils.isEmpty(userIds)){
            return;
        }
        String sql =
                "update tts.org_lesson_sign set status = :status, cascade_Id=:cascadeId, source=:source where lesson_id in (:lessonIds)  AND  user_id in (:userId) ";
        Map<String, Object> params = new HashMap<>();
        params.put("status", status);
        params.put("userId", userIds);
        params.put("lessonIds", lessonIds);
        params.put("cascadeId", cascadeId);
        params.put("source", source);
        log.debug("batch sign sql:{},params:{}", sql, params);
        this.getNamedJdbcTemplate().update(sql, params);
    }

    @Override
    public List<OrgLessonSign> getStudentSign(Long orgId, Long userId, Integer userRole, Collection<Long> lessonIds,
                                              String... queryProps) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(userId != null && userId > 0, "userId is illegal");
        Preconditions.checkArgument(userRole != null && userRole > 0, "userRole is illegal");
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Lists.newArrayList();
        }

        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder(queryProps);
        builder.in("lessonId", lessonIds);
        builder.eq("userRole", userRole);
        builder.eq("userId", userId);
        builder.eq("orgId", orgId);
        return queryList(builder);
    }
    
    

    @Override
    public Map<Long, Integer> getOrgSignCount(Collection<Long> orgIds, Long userId, final Integer userRole) {

        if (CollectionUtils.isEmpty(orgIds)) {
            return Collections.emptyMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(orgIds,
                new BatchQueryCallback<Long, Map<Long, Integer>>() {
                    @Override
                    public Map<Long, Integer> doQuery(Collection<Long> orgIds) {
                        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder();
                        builder.distinctCount("id", "signCount");
                        builder.in("orgId", orgIds);
                        builder.eq("status", SignStatus.SIGNED.getCode());
                        builder.eq("userRole", userRole);
                        builder.group("orgId");
                        final Map<Long, Integer> result = Maps.newHashMap();
                        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                                new RowCallbackHandler() {
                                    @Override
                                    public void processRow(ResultSet rs) throws SQLException {
                                        result.put(rs.getLong("orgId"), rs.getInt("signCount"));
                                    }
                                });
                        return result;
                    }
                });

    }

    @Override
    public Map<Long, Integer> getOrgCourseSignCount(Collection<Long> courseIds, final Long userId,
                                                    final Integer userRole) {

        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(courseIds,
                new BatchQueryCallback<Long, Map<Long, Integer>>() {
                    @Override
                    public Map<Long, Integer> doQuery(Collection<Long> courseIds) {
                        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder();
                        builder.distinctCount("id", "signCount");
                        builder.in("courseId", courseIds);
                        builder.eq("status", SignStatus.SIGNED.getCode());
                        builder.eq("userRole", userRole);
                        builder.eq("userId", userId);
                        builder.group("courseId");
                        final Map<Long, Integer> result = Maps.newHashMap();
                        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                                new RowCallbackHandler() {
                                    @Override
                                    public void processRow(ResultSet rs) throws SQLException {
                                        result.put(rs.getLong("courseId"), rs.getInt("signCount"));
                                    }
                                });
                        return result;
                    }
                });

    }

    @Override
    public Map<Long, Integer> getOrgCourseStudentSignCount(Long orgId,Collection<Long> userIds, final Long courseId) {

        if (CollectionUtils.isEmpty(userIds)) {
            return Collections.emptyMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(userIds,
                new BatchQueryCallback<Long, Map<Long, Integer>>() {
                    @Override
                    public Map<Long, Integer> doQuery(Collection<Long> userIds) {
                        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder();
                        builder.count("id", "signCount");
                        builder.eq("orgId", "orgId");
                        builder.in("userId", userIds);
//                        builder.eq("status", SignStatus.SIGNED.getCode());
                        builder.ge("status", SignStatus.SIGNED.getCode());
                        builder.eq("userRole", UserRole.STUDENT.getRole());
                        builder.eq("courseId", courseId);
                        builder.group("userId");
                        final Map<Long, Integer> result = Maps.newHashMap();
                        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                                new RowCallbackHandler() {
                                    @Override
                                    public void processRow(ResultSet rs) throws SQLException {
                                        result.put(rs.getLong("userId"), rs.getInt("signCount"));
                                    }
                                });
                        return result;
                    }
                });

    }

    @Override
    public OrgLessonSign getStudentLessonSign(Long orgId, Long courseId, Long lessonId, Long userId, Integer userRole) {

        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        if (courseId != null) {
            builder.eq("courseId", courseId);
        }
        builder.eq("userRole", userRole);
        builder.eq("lessonId", lessonId);
        builder.eq("userId", userId);
        return this.uniqueResult(builder);
    }

    @Override
    public List<OrgLessonSign> getLessonStudentIds(Long orgId, Long lessonId, Collection<Long> userIds, Integer sign,
                                                   String... queryProps) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");

        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        if (lessonId != null) {
            builder.eq("lessonId", lessonId);
        }
        if (CollectionUtils.isNotEmpty(userIds)) {
            builder.in("userId", userIds);
        }
        if (sign != null) {
            builder.eq("status", sign);
        }
        builder.eq("userRole", UserRole.STUDENT.getRole());
        return queryList(builder);

    }

    @Override
    public boolean isStudentSignInLesson(Long orgId, Long lessonId, Long userId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(userId != null && userId > 0, "userId is illegal");
        Preconditions.checkArgument(lessonId != null && lessonId > 0, "lessonId is illegal");

        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder("id");
        builder.eq("orgId", orgId);
        builder.eq("lessonId", lessonId);
        builder.eq("userId", userId);
        builder.eq("userRole", UserRole.STUDENT.getRole());
        builder.gt("status", SignStatus.UNSIGN.getCode());
        Long id = queryForObject(builder, Long.class);
        if (id == null) {
            return false;
        } else {
            return true;
        }
    }

    @Override
    public Map<Long, Integer> getCourseSignCountMap(Long orgId, Collection<Long> courseIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyMap();
        }
        String sql = "select ocl.course_id as courseId,count(ols.id) as cnt "
                + "from tts.org_lesson_sign ols right join tts.org_class_lesson ocl on ocl.id = ols.lesson_id "
                + "where ocl.start_time < :now and ocl.org_id = :orgId and ocl.course_id in (:courseIds) and ols.status = 1 "
                + "group by ocl.course_id order by field(courseId,:courseIds)";

        final Map<Long, Integer> resultMap = Maps.newHashMap();
        Map<String, Object> paramMap = Maps.newHashMap();
        paramMap.put("orgId", orgId);
        paramMap.put("courseIds", courseIds);
        paramMap.put("now", new Date());
        log.debug("sql = {},param = {}", sql, paramMap);
        this.getNamedJdbcTemplate().query(sql, paramMap, new RowCallbackHandler() {

            @Override
            public void processRow(ResultSet rs) throws SQLException {
                resultMap.put(rs.getLong("courseId"), rs.getInt("cnt"));
            }

        });
        return resultMap;

    }

    @Override
    public Map<Long, Integer> getStudentSignCountMap(Long orgId, Collection<Long> userIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        if (CollectionUtils.isEmpty(userIds)) {
            return Collections.emptyMap();
        }
        String sql = "select ols.user_id as userId, count(ols.id) as cnt from tts.org_lesson_sign ols "
                + "where  ols.org_id = :orgId and ols.user_id in(:userIds) "
                + "and ols.user_role = 2 group by ols.user_id order by field(userId,:userIds)";

        final Map<Long, Integer> resultMap = Maps.newHashMap();
        Map<String, Object> paramMap = Maps.newHashMap();
        paramMap.put("orgId", orgId);
        paramMap.put("userIds", userIds);
        log.debug("sql = {},param = {}", sql, paramMap);
        this.getNamedJdbcTemplate().query(sql, paramMap, new RowCallbackHandler() {

            @Override
            public void processRow(ResultSet rs) throws SQLException {
                resultMap.put(rs.getLong("userId"), rs.getInt("cnt"));
            }

        });
        return resultMap;

    }

    @Override
    public Map<Long, LessonSignAudit> queryLessonSignAudit(Long orgId, Collection<Long> lessonIds) {
        Map<String, Object> params = Maps.newHashMap();
        params.put("orgId", orgId);
        params.put("lessonId", lessonIds);
        StringBuilder sb = new StringBuilder();
        sb.append("select s.lesson_id id, count(case when s.status = 1 then 1 else null end) signCount, ")
                .append("count(case when s.status=2 then 1 else null end) leaveCount, ")
                .append("count(case when s.status =3 then 1 else null end) absentCount ")
                .append("from tts.org_lesson_sign s ").append("where s.org_id =:orgId ")
                .append("and s.lesson_id in(:lessonId) group by s.lesson_id");
        final Map<Long, LessonSignAudit> map = Maps.newHashMap();
        this.getNamedJdbcTemplate().query(sb.toString(), params, new RowCallbackHandler() {

            @Override
            public void processRow(ResultSet rs) throws SQLException {
                LessonSignAudit lesson = new LessonSignAudit();
                lesson.setLessonId(rs.getLong("id"));
                lesson.setSignCount(rs.getInt("signCount"));
                lesson.setLeaveCount(rs.getInt("leaveCount"));
                lesson.setAbsentCount(rs.getInt("absentCount"));
                map.put(lesson.getLessonId(), lesson);
            }
        });
        return map;
    }

    @Override
    public Map<Long, Integer> getOrgSignLessonTotal(Date startTime, Date endTime, List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        String sql = "select org_id, count(distinct(lesson_id)) count from tts.org_lesson_sign where org_id in (:orgIds)";
        if (startTime != null && endTime != null) {
            sql += " and create_time between :startTime and :endTime";
            param.put("endTime", endTime);
            param.put("startTime", startTime);
        }

        sql += " group by org_id";
        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, Integer>>() {

            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Long orgId = rs.getLong("org_id");
                    Integer count = rs.getInt("count");
                    map.put(orgId, count);
                }
                return map;
            }
        });
    }

    @Override
    public Map<Long, Integer> getOrgSignStudentTotal(Date startTime, Date endTime, List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        String sql =
            "select org_id, count(id) count from tts.org_lesson_sign where user_role = 2 and status = 1 and org_id in (:orgIds)";
        if (startTime != null && endTime != null) {
            sql += " AND create_time between :startTime and :endTime";
            param.put("endTime", endTime);
            param.put("startTime", startTime);
        }

        sql += " group by org_id";
        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, Integer>>() {

            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Long orgId = rs.getLong("org_id");
                    Integer count = rs.getInt("count");
                    map.put(orgId, count);
                }
                return map;
            }
        });
    }

    @Override
    public List<OrgLessonSign> getOrgClassLesson(List<Long> TianxiaoOrgIds) {
        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder();
        builder.eq("userRole", 2);
        builder.in("orgId", TianxiaoOrgIds);
        return queryList(builder);
    }

    @Override
    public Map<String, Integer> getOrgSignMap(Date startTime, Date endTime, List<Long> TianxiaoOrgIds,
        Integer userRole) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        param.put("userRole", userRole);
        param.put("startTime", startTime);
        param.put("endTime", endTime);
        
        String sql = "SELECT count(id) count, DATE_FORMAT(create_time,'%y-%m-%d') 'createTime' FROM tts.org_lesson_sign where "
            + "user_role = :userRole and status = 1 and org_id in (:orgIds) and create_time>=:startTime and create_time<=:endTime";
        sql += " group by MONTH(create_time),DAY(create_time)";
        
        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<String, Integer>>() {

            @Override
            public Map<String, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<String, Integer> map = new HashMap<>();
                while(rs.next()){
                    String date = rs.getString("createTime");
                    Integer count = rs.getInt("count");
                    log.info("createTime===========" + rs.getString("createTime") + "count==========" + count);
                    map.put(date, count);
                }
                return map;
            }
        });
    }

    @Override
    public Map<String, List<Long>> getOrgSignLessonIdsMap(Date startTime, Date endTime, List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        param.put("startTime", startTime);
        param.put("endTime", endTime);
        
        String sql = "SELECT lesson_id, DATE_FORMAT(create_time,'%y-%m-%d') 'createTime' FROM tts.org_lesson_sign where "
            + " status = 1 and org_id in (:orgIds) and create_time>=:startTime and create_time<=:endTime group by lesson_id, MONTH(create_time),DAY(create_time)";
        
        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<String, List<Long>>>() {

            @Override
            public Map<String, List<Long>> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<String, List<Long>> map = new HashMap<>();
                while(rs.next()){
                   String createTime = rs.getString("createTime");
                   Long lessonId = rs.getLong("lesson_id");
                   List<Long> lessonIds = map.get(createTime);
                   if(GenericsUtils.isNullOrEmpty(lessonIds)){
                       lessonIds = new ArrayList<>();
                       map.put(createTime, lessonIds);
                   }
                   lessonIds.add(lessonId);
                }
                return map;
            }
        });
    }

    @Override
    public List<OrgLessonSign> getCourseLessonSignIn(List<Long> courseIds) {
        if(CollectionUtils.isEmpty(courseIds)){
            return new ArrayList<OrgLessonSign>();
        }
        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder();
        builder.eq("userRole", UserRole.STUDENT.getRole());
        builder.in("courseId", courseIds);
        return queryList(builder);
    }
    
    @Override
    public Map<Long, List<Integer>> getOrgCourseStudentSignMap(Collection<Long> userIds, final Long courseId) {
        if (CollectionUtils.isEmpty(userIds)) {
            return Collections.emptyMap();
        }
        
        return new MapBatchQueryTemplate<Long, Long, List<Integer>>().batchQuery(userIds,
                new BatchQueryCallback<Long, Map<Long, List<Integer>>>() {
                    @Override
                    public Map<Long, List<Integer>> doQuery(Collection<Long> userIds) {
                        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder("userId", "status");
                        builder.in("userId", userIds);
                        builder.eq("userRole", UserRole.STUDENT.getRole());
                        builder.eq("courseId", courseId);
                        final Map<Long, List<Integer>> result = Maps.newHashMap();
                        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                                new RowCallbackHandler() {
                                    @Override
                                    public void processRow(ResultSet rs) throws SQLException {
                                        Long userId = rs.getLong("userId");
                                        List<Integer> statusList = result.get(userId);
                                        if (statusList == null) {
                                            statusList = Lists.newArrayList();
                                            result.put(userId, statusList);
                                        }
                                        statusList.add(rs.getInt("status"));
                                    }
                                });
                        return result;
                    }
                });
    }

	@Override
	public int getVipLessonSignCount(Long orgId, Long courseId, int ruleValue) {
		SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder();
		builder.eq("orgId", orgId);
		builder.eq("courseId", courseId);
		if(ruleValue == 1) {
			builder.eq("status", 1);
		}
		if(ruleValue == 3) {
			Set<Integer> set = new HashSet<Integer>();
			set.add(1);
			set.add(2);
			builder.in("status", set);
		}
		if(ruleValue == 5) {
			Set<Integer> set = new HashSet<Integer>();
			set.add(1);
			set.add(3);
			builder.in("status", set);
		}
		if(ruleValue == 7) {
			Set<Integer> set = new HashSet<Integer>();
			set.add(1);
			set.add(2);
			set.add(3);
			builder.in("status", set);
		}
		return queryList(builder).size();
	}

    @Override
    public List<OrgLessonSign> getByCourseIdsAndUserIdsAndStatus(Long orgId, Collection<Long> courseIds,
        Collection<Long> userIds, Integer status, String...props) {
        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder(props);
        builder.eq("orgId", orgId);
        if (CollectionUtils.isNotEmpty(courseIds)) {
            builder.in("courseId", courseIds);
        }
        if (CollectionUtils.isNotEmpty(userIds)) {
            builder.in("userId", userIds);
        }
        if (null != status) {
            builder.eq("status", status);
        }
        return queryList(builder);
    }

    @Override
    public List<OrgLessonSign> getByOrgIdCourseIdLessonIdsStudentIds(@NonNull Long orgId, Long courseId,
        List<Long> lessonIds, List<Long> userIds, String...props) {
        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder(props);
        if(null != orgId){
            builder.eq("orgId", orgId);
        }
        if(null != courseId){
            builder.eq("courseId", courseId);
        }
        if(CollectionUtils.isNotEmpty(lessonIds)){
            builder.in("lessonId", lessonIds);
        }
        if(CollectionUtils.isNotEmpty(userIds)){
            builder.in("userId", userIds);
        }
        return queryList(builder);
    }

    @Override
    public List<OrgLessonSign> getByStudentLessonIds(Collection<Long> stuLessonIds,String... props) {
        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder(props);
        builder.in("lessonId", stuLessonIds);
        return queryList(builder);
    }

    
	@Override
	public Map<Long, Integer> mapKeyLessonIdValueStatus(Long orgId, Long userId) {
		final Map<Long, Integer> result = new HashMap<Long, Integer>();
		String sql = "select lesson_id,status from tts.org_lesson_sign where org_id=:orgId AND user_id=:userId";
		NamedParameterJdbcTemplate template = getNamedJdbcTemplate();
        Map<String, Object> params = new HashMap<String,Object>();
        params.put("orgId", orgId);
        params.put("userId", userId);
        
        template.query(sql, params, new RowMapper<Integer>() {
            @Override
            public Integer mapRow(ResultSet rs, int rowNum) throws SQLException {
            	result.put(rs.getLong("lesson_id") , rs.getInt("status"));
                return 1;
            }
        });
        
        return result;
	}
	
	@Override
	public Map<Integer, Integer> mapKeyStatusValueCount(Collection<Long> lessonIds, Long userId) {
		final Map<Integer, Integer> result = new HashMap<Integer, Integer>();
		
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT t1.status,count(*) AS num FROM tts.org_student_lesson t2 INNER JOIN tts.org_lesson_sign t1 ON t1.lesson_id=t2.lesson_id AND t1.user_id=t2.student_id ");
		sql.append(" WHERE t2.del_status=0 AND t1.lesson_id IN (:lessonIds) ");
	    if(userId!=null){
	    	sql.append(" AND t2.student_id=:userId ");
	    }
	    sql.append(" GROUP BY t1.status");
				   
		NamedParameterJdbcTemplate template = getNamedJdbcTemplate();
        Map<String, Object> params = new HashMap<String,Object>();
        params.put("lessonIds", lessonIds);
        params.put("userId", userId);
        
        template.query(sql.toString(), params, new RowMapper<Integer>() {
            @Override
            public Integer mapRow(ResultSet rs, int rowNum) throws SQLException {
            	result.put(rs.getInt("status") , rs.getInt("num"));
                return 1;
            }
        });
        
        return result;
	}

	@Override
	public Map<Long, Integer> mapKeyLessonIdValueCount(Collection<Long> lessonIds) {
		final Map<Long, Integer> result = new HashMap<Long, Integer>();
		
		StringBuilder builder = new StringBuilder();
		builder.append("SELECT t1.lesson_id,count(*) AS num FROM tts.org_student_lesson t2 INNER JOIN tts.org_lesson_sign t1 ON t1.lesson_id=t2.lesson_id AND t1.user_id=t2.student_id");
		builder.append(" WHERE t2.del_status=0 AND t1.lesson_id IN (:lessonIds) ");
		builder.append(" GROUP BY t1.lesson_id");

		NamedParameterJdbcTemplate template = getNamedJdbcTemplate();
        Map<String, Object> params = new HashMap<String,Object>();
        params.put("lessonIds", lessonIds);
        
        template.query(builder.toString(), params, new RowMapper<Integer>() {
            @Override
            public Integer mapRow(ResultSet rs, int rowNum) throws SQLException {
            	result.put(rs.getLong("lesson_id") , rs.getInt("num"));
                return 1;
            }
        });
        return result;
	}

	@Override
	public Map<Long,Map<Long, Map<Integer, Integer>>> mapStudentSignStatus(Collection<Long> lessonIds) {
		final Map<Long,Map<Long, Map<Integer, Integer>>> result = new HashMap<Long,Map<Long, Map<Integer, Integer>>>();
		
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT t2.course_id, t2.student_id, t1.status,count(*) AS num FROM tts.org_student_lesson t2 INNER JOIN tts.org_lesson_sign t1 ON t1.lesson_id=t2.lesson_id AND t1.user_id=t2.student_id ");
		sql.append(" WHERE t2.del_status=0 AND t1.lesson_id IN (:lessonIds) ");
	    sql.append(" GROUP BY t2.course_id, t2.student_id, t1.status");
				   
		NamedParameterJdbcTemplate template = getNamedJdbcTemplate();
        Map<String, Object> params = new HashMap<String,Object>();
        params.put("lessonIds", lessonIds);
        
        template.query(sql.toString(), params, new RowMapper<Integer>() {
            @Override
            public Integer mapRow(ResultSet rs, int rowNum) throws SQLException {
            	Long courseId = rs.getLong("course_id");
            	Long stuUserId = rs.getLong("student_id");
            	Integer status = rs.getInt("status");
            	Integer num = rs.getInt("num");
            	
            	Map<Long, Map<Integer, Integer>> stuSignStatusMap = result.get(courseId);
            	if(stuSignStatusMap==null){
            		stuSignStatusMap = new HashMap<Long, Map<Integer, Integer>>();
            		result.put(courseId, stuSignStatusMap);
            	}
            	
            	Map<Integer, Integer> signStatusMap = stuSignStatusMap.get(stuUserId);
            	if(signStatusMap==null){
            		signStatusMap = new HashMap<Integer, Integer>();
            		stuSignStatusMap.put(stuUserId, signStatusMap);
            	}
            	
            	signStatusMap.put(status, num);
            	
                return 1;
            }
        });
        
        return result;
	}


    public List<OrgLessonSign> listByPage(Date startTime, Date endTime, PageDto pageDto) {
        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder();
        if (startTime != null) {
            builder.ge("updateTime", startTime);
            builder.le("updateTime", endTime);
        }
        builder.setPage(pageDto);
        return queryList(builder);
    }

    @Override
    public List<OrgLessonSign> listByUserClass(Long orgId, Long userId, Long courseId, Date time) {
        SingleSqlBuilder<OrgLessonSign> builder = createSqlBuilder();
        builder.eq("orgId",orgId);
        builder.eq("courseId",courseId);
        builder.eq("userId",userId);
        builder.eq("status",1);
        builder.ge("updateTime",time);
        return queryList(builder);
    }
}

