
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.dal.org.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.collect.Lists;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.stereotype.Repository;

import com.baijia.tianxiao.dal.org.dao.OrgTeacherLessonDao;
import com.baijia.tianxiao.dal.org.po.OrgTeacherLesson;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.bean.impl.MatchMode;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.collection.CollectorUtil;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.baijia.tianxiao.util.query.MapBatchQueryTemplate;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * @author cxm
 * @version 1.0
 * @title OrgTeacherLessonDaoImpl
 * @desc TODO
 * @date 2015年12月29日
 */
@Repository
@Slf4j
public class OrgTeacherLessonDaoImpl extends JdbcTemplateDaoSupport<OrgTeacherLesson> implements OrgTeacherLessonDao {

    /**
     * @param
     */
    public OrgTeacherLessonDaoImpl() {
        super(OrgTeacherLesson.class);
    }

    @Override
    public List<OrgTeacherLesson> queryTeacherLessons(Long orgId, Collection<Long> teacherIds, String... queryProps) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        SingleSqlBuilder<OrgTeacherLesson> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        if (CollectionUtils.isNotEmpty(teacherIds)) {
            builder.in("teacherId", teacherIds);
        }
        return queryList(builder);
    }
    
    @Override
    public List<OrgTeacherLesson> queryTeacherLessonsByLessonIds(String query, Long orgId, Collection<Long> lessonIds, String... queryProps) {
        SingleSqlBuilder<OrgTeacherLesson> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        if (CollectionUtils.isNotEmpty(lessonIds)) {
            builder.in("lessonId", lessonIds);
        }
        if (StringUtils.isNotBlank(query)) {
            builder.like("teacherName", query, MatchMode.ANYWHERE);
        }
        return queryList(builder);
    }

    @Override
    public Map<Long, Long> queryLessonTeacherIdMap(final Long orgId, Collection<Long> lessonIds) {
        // Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Collections.emptyMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Long>().batchQuery(lessonIds,
                new BatchQueryCallback<Long, Map<Long, Long>>() {
                    @Override
                    public Map<Long, Long> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgTeacherLesson> builder = createSqlBuilder();
                        builder.select("teacherId");
                        builder.select("lessonId");
                        if (orgId != null) {
                            builder.eq("orgId", orgId);
                        }
                        builder.in("lessonId", querySet);
                        // FIXME 确保取出来的只有一个课节一个老师
                        builder.group("lessonId");
                        final Map<Long, Long> result = Maps.newHashMap();
                        log.debug("query lessonTeacher id map sql:{}, params:{}", builder.toSql(),
                                builder.collectConditionValue());
                        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                                new RowCallbackHandler() {
                                    @Override
                                    public void processRow(ResultSet rs) throws SQLException {
                                        result.put(rs.getLong("lessonId"), rs.getLong("teacherId"));
                                    }
                                });
                        return result;
                    }
                });

    }

    @Override
    public List<Long> getUserIds(Long lessonId, Long orgId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(lessonId != null && lessonId > 0, "lessonId is illegal");

        SingleSqlBuilder<OrgTeacherLesson> builder = createSqlBuilder();
        builder.select("teacherId");
        builder.eq("orgId", orgId);
        builder.eq("lessonId", lessonId);
        return queryForList(builder, Long.class);
    }

    @Override
    public List<Long> queryTeacherLessons(Long orgId, Long teacherId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        SingleSqlBuilder<OrgTeacherLesson> builder = createSqlBuilder("lessonId");
        builder.eq("orgId", orgId);
        builder.eq("teacherId", teacherId);
        return queryForList(builder, Long.class);
    }

    @Override
    public boolean isTeacherInLesson(Long lessonId, Long userId) {
        SingleSqlBuilder<OrgTeacherLesson> builder = createSqlBuilder("lessonId");
        builder.eq("lessonId", lessonId);
        builder.eq("teacherId", userId);
        List<Long> list = queryForList(builder, Long.class);
        if (CollectionUtils.isNotEmpty(list)) {
            return true;
        }
        return false;
    }

    @Override
    public OrgTeacherLesson getByLessonId(Long orgId, Long lessonId) {
        SingleSqlBuilder<OrgTeacherLesson> builder = createSqlBuilder();
        builder.eq("lessonId", lessonId);
        builder.eq("orgId", orgId);
        List<OrgTeacherLesson> list = queryList(builder);
        if (CollectionUtils.isNotEmpty(list)) {
            return list.get(0);
        }
        return null;
    }

    @Override
    public Map<Long, List<Long>> getTeacherLessonByStartTime(Date startTime, Date endTime) {
        StringBuilder sb = new StringBuilder();
        sb.append(
                "select teacher.lesson_id as lessonId, teacher.teacher_id as teacherId from tts.org_teacher_lesson teacher ");
        sb.append("join tts.org_class_lesson lesson on teacher.lesson_id = lesson.id ");
        sb.append(" and lesson.start_time >=:startTime and lesson.start_time <:endTime");
        Map<String, Object> param = Maps.newHashMap();
        param.put("startTime", startTime);
        param.put("endTime", endTime);
        final Map<Long, List<Long>> result = Maps.newHashMap();
        this.getNamedJdbcTemplate().query(sb.toString(), param, new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                Long userId = rs.getLong("teacherId");
                Long lessonId = rs.getLong("lessonId");
                if (!result.containsKey(userId)) {
                    result.put(userId, Lists.<Long>newArrayList());
                }
                result.get(userId).add(lessonId);
            }
        });
        return result;
    }

    @Override
    public Map<Long, List<Long>> getTeacherLessonByEndTime(Date startTime, Date endTime) {
        StringBuilder sb = new StringBuilder();
        sb.append(
                "select teacher.lesson_id as lessonId, teacher.teacher_id as teacherId from tts.org_teacher_lesson teacher ");
        sb.append("join tts.org_class_lesson lesson on teacher.lesson_id = lesson.id ");
        sb.append(" and lesson.end_time >=:startTime and lesson.end_time <:endTime");
        Map<String, Object> param = Maps.newHashMap();
        param.put("startTime", startTime);
        param.put("endTime", endTime);
        final Map<Long, List<Long>> result = Maps.newHashMap();
        this.getNamedJdbcTemplate().query(sb.toString(), param, new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                Long userId = rs.getLong("teacherId");
                Long lessonId = rs.getLong("lessonId");
                if (!result.containsKey(lessonId)) {
                    result.put(lessonId, Lists.<Long>newArrayList());
                }
                result.get(lessonId).add(userId);
            }
        });
        return result;
    }


    @Override
    public int countTeacherLessonByMonth(Long teacherUserId, Date startTime, Date endTime) {
        StringBuilder sb = new StringBuilder();
        sb.append(
                "select count(1) from tts.org_teacher_lesson teacher ");
        sb.append("join tts.org_class_lesson lesson on teacher.lesson_id = lesson.id ");
        sb.append(" and lesson.start_time >=:startTime and lesson.end_time <:endTime and teacher.teacher_id = :teacherId");
        Map<String, Object> param = Maps.newHashMap();
        param.put("startTime", startTime);
        param.put("endTime", endTime);
        param.put("teacherId", teacherUserId);

        return this.getNamedJdbcTemplate().queryForObject(sb.toString(), param, Integer.class);
    }
    @Override
    public Map<Long, OrgTeacherLesson> getOrgTeacherLessonMapByLessonIds(Collection<Long> lessonIds) {
        if (lessonIds.isEmpty()) {
            log.warn("lesson ids is empty.");
            return Maps.newHashMap();
        }
        BatchQueryTemplate<Long, List<OrgTeacherLesson>> queryTemplate = new ListBatchQueryTemplate<>();
        List<OrgTeacherLesson> orgTeacherLessons = queryTemplate.batchQuery(lessonIds, new BatchQueryCallback<Long, List<OrgTeacherLesson>>() {
            @Override
            public List<OrgTeacherLesson> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgTeacherLesson> builder = createSqlBuilder();
                builder.in("lessonId", querySet);
                return queryList(builder);
            }
        });
        return CollectorUtil.collectMap(orgTeacherLessons, new Function<OrgTeacherLesson, Long>() {
            @Override
            public Long apply(OrgTeacherLesson arg0) {
                return arg0.getLessonId();
            }
        });
    }

	@Override
	public List<OrgTeacherLesson> queryByLessonIds(Long orgId, Collection<Long> classLessonIds) {
		SingleSqlBuilder<OrgTeacherLesson> builder = createSqlBuilder();
		builder.eq("orgId", orgId);
		if(CollectionUtils.isNotEmpty(classLessonIds)) {
			builder.in("lessonId", classLessonIds);
		}
		return queryList(builder);
	}

	
	
	@Override
	public Map<Long, List<Long>> getTeacherLessonByStartTime(Date startTime, Date endTime, Collection<Long> teacherUserIds) {
		StringBuilder sb = new StringBuilder();
        sb.append("select teacher.lesson_id as lessonId, teacher.teacher_id as teacherId from tts.org_teacher_lesson teacher ");
        sb.append("join tts.org_class_lesson lesson on teacher.lesson_id = lesson.id ");
        sb.append(" and lesson.start_time >=:startTime and lesson.start_time <:endTime and lesson.del_status=0");
        Map<String, Object> param = Maps.newHashMap();
        param.put("startTime", startTime);
        param.put("endTime", endTime);
        
        if(CollectionUtils.isNotEmpty(teacherUserIds)){
        	sb.append(" and teacher.teacher_id IN (:teacherUserIds)");
        	param.put("teacherUserIds", teacherUserIds);
        }
        
        final Map<Long, List<Long>> result = Maps.newHashMap();
        this.getNamedJdbcTemplate().query(sb.toString(), param, new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                Long userId = rs.getLong("teacherId");
                Long lessonId = rs.getLong("lessonId");
                if (!result.containsKey(userId)) {
                    result.put(userId, Lists.<Long>newArrayList());
                }
                result.get(userId).add(lessonId);
            }
        });
        return result;
	}

	
	
	@Override
	public List<Long> getTeacherLessonIdsByStartTime(Date startTime, Date endTime, Long teacherUserId) {
		StringBuilder sb = new StringBuilder();
        sb.append("select teacher.lesson_id as lessonId from tts.org_teacher_lesson teacher ");
        sb.append("join tts.org_class_lesson lesson on teacher.lesson_id = lesson.id ");
        sb.append(" and lesson.start_time >=:startTime and lesson.start_time <:endTime and lesson.del_status=0");
        sb.append(" and teacher.teacher_id =:teacherUserId");
        Map<String, Object> param = Maps.newHashMap();
        param.put("startTime", startTime);
        param.put("endTime", endTime);
    	param.put("teacherUserId", teacherUserId);
  
        
        final List<Long> result = new ArrayList<Long>();
        this.getNamedJdbcTemplate().query(sb.toString(), param, new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                Long lessonId = rs.getLong("lessonId");
                result.add(lessonId);
            }
        });
        return result;
	}

	


	@Override
	public Map<Long, Map<Long, List<Long>>> getTeacherCourseLessonBy(Date startTime, Date endTime, Collection<Long> teacherUserIds) {
		StringBuilder sb = new StringBuilder();
        sb.append("select teacher.teacher_id, teacher.lesson_id, lesson.course_id from tts.org_teacher_lesson teacher ");
        sb.append("join tts.org_class_lesson lesson on teacher.lesson_id = lesson.id ");
        sb.append(" and lesson.start_time >=:startTime and lesson.start_time <:endTime and lesson.del_status=0");
        sb.append(" and teacher.teacher_id IN (:teacherUserIds)");
        Map<String, Object> param = Maps.newHashMap();
        param.put("startTime", startTime);
        param.put("endTime", endTime);
    	param.put("teacherUserIds", teacherUserIds);
  
        
        final Map<Long, Map<Long, List<Long>>> result = new HashMap<Long,Map<Long, List<Long>>>();
        this.getNamedJdbcTemplate().query(sb.toString(), param, new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
            	Long teacherId = rs.getLong("teacher_id");
                Long courseId = rs.getLong("course_id");
            	Long lessonId = rs.getLong("lesson_id");
            	
            	Map<Long, List<Long>> teacherCourseMap = null;
            	List<Long> lessonIds = null;
            	
            	teacherCourseMap = result.get(teacherId);
            	if(teacherCourseMap==null){
            		teacherCourseMap = new HashMap<Long, List<Long>>();
            		result.put(teacherId, teacherCourseMap);
            	}
            	
            	lessonIds = teacherCourseMap.get(courseId);
            	if(lessonIds==null){
            		lessonIds = new ArrayList<Long>();
            		teacherCourseMap.put(courseId, lessonIds);
            	}
            	
            	lessonIds.add(lessonId);
            	
            }
        });
        return result;
	}
	
	
	@Override
	public Integer countClassId(Date startTime, Date endTime, Long teacherUserId) {
		StringBuilder sb = new StringBuilder();
        sb.append("select count(distinct(lesson.course_id)) from tts.org_teacher_lesson teacher ");
        sb.append("join tts.org_class_lesson lesson on teacher.lesson_id = lesson.id ");
        sb.append(" and lesson.start_time >=:startTime and lesson.start_time <:endTime and lesson.del_status=0");
        sb.append(" and teacher.teacher_id =:teacherUserId");
        Map<String, Object> param = Maps.newHashMap();
        param.put("startTime", startTime);
        param.put("endTime", endTime);
    	param.put("teacherUserId", teacherUserId);
  
    	Integer count = this.getNamedJdbcTemplate().queryForInt(sb.toString(), param);
        return count!=null ? count:0;
	}

	
	
	@Override
	public List<Long> listClassId(Date startTime, Date endTime, Long teacherUserId, PageDto pageDto) {
		StringBuilder sb = new StringBuilder();
        sb.append("select distinct(lesson.course_id) as course_id from tts.org_teacher_lesson teacher ");
        sb.append("join tts.org_class_lesson lesson on teacher.lesson_id = lesson.id ");
        sb.append(" and lesson.start_time >=:startTime and lesson.start_time <:endTime and lesson.del_status=0");
        sb.append(" and teacher.teacher_id =:teacherUserId");
        
        Map<String, Object> param = Maps.newHashMap();
        param.put("startTime", startTime);
        param.put("endTime", endTime);
    	param.put("teacherUserId", teacherUserId);
    	
    	if(pageDto!=null){
	    	sb.append(" limit :start,:size");
	    	param.put("start", (pageDto.getPageNum()-1)*pageDto.getPageSize());
	    	param.put("size", pageDto.getPageSize());
    	}
        
        final List<Long> result = new ArrayList<Long>();
        this.getNamedJdbcTemplate().query(sb.toString(), param, new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                result.add(rs.getLong("course_id"));
            }
        });
        return result;
	}

	@Override
	public List<Long> listLessonId(Collection<Long> lessonIds, Long teacherUserId) {
		SingleSqlBuilder<OrgTeacherLesson> builder = createSqlBuilder();
		builder.select("lessonId");
        builder.in("lessonId",lessonIds);
        builder.eq("teacherId", teacherUserId);
        
        return queryList(builder,Long.class);
	}


}
