package com.baijia.tianxiao.dal.org.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Repository;

import com.baijia.tianxiao.consants.DataStatus;
import com.baijia.tianxiao.dal.org.dao.OrgStudentCourseDao;
import com.baijia.tianxiao.dal.org.dto.DeleteStatus;
import com.baijia.tianxiao.dal.org.dto.StudentClassHourStatusDocument;
import com.baijia.tianxiao.dal.org.po.OrgStudentCourse;
import com.baijia.tianxiao.enums.StudentCourseStatus;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.bean.Order;
import com.baijia.tianxiao.sqlbuilder.bean.impl.MatchMode;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.sqlbuilder.util.Expressions;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.baijia.tianxiao.util.query.MapBatchQueryTemplate;
import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

@Repository
@Slf4j
public class OrgStudentCourseDaoImpl extends JdbcTemplateDaoSupport<OrgStudentCourse> implements OrgStudentCourseDao {

    public OrgStudentCourseDaoImpl() {
        super(OrgStudentCourse.class);
    }

    @Override
    public List<Long> getStudents(Long orgId, Long courseId) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("userId");
        builder.eq("courseId", courseId);
        builder.eq("orgId", orgId);
        builder.eq("delStatus", 0);
        return queryForList(builder, Long.class);
    }

    @Override
    public OrgStudentCourse getStudentCourse(@NonNull Long orgId, @NonNull Long courseId, @NonNull Long userId) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("courseId", courseId);
        builder.eq("orgId", orgId);
        // builder.eq("delStatus", 0);
        if (userId != null) {
            builder.eq("userId", userId);
        }

        return this.uniqueResult(builder);
    }

    @Override
    public List<OrgStudentCourse> getStudentListByCourseId(Long orgId, Long courseId, String...queryProps) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("courseId", courseId);
        builder.add(Expressions.or(Expressions.eq("orgId", orgId), Expressions.eq("orgId", 0)));
        builder.eq("delStatus", 0);
        builder.select(queryProps);
        builder.desc("createTime");
        return queryList(builder);

    }

    @Override
    public List<OrgStudentCourse> getStudentListByCourseId(Long orgId, Long courseId, PageDto pageDto) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("courseId", courseId);
        builder.add(Expressions.or(Expressions.eq("orgId", orgId), Expressions.eq("orgId", 0)));
        builder.eq("delStatus", 0);
        // 学生列表不包含转班和退班的学生
        builder.eq("status", 0);
        builder.desc("createTime");
        if (pageDto != null) {
            builder.setPage(pageDto);
        }
        return queryList(builder);
    }

    @Override
    public void deleteOrgCourseStudent(Long orgId, Long courseId, Long userId, Integer status) {
        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        condition.put("courseId", courseId);
        condition.put("userId", userId);
        OrgStudentCourse obj = new OrgStudentCourse();
        obj.setStatus(status);
        this.update(condition, obj, "status");
    }

    @Override
    public void batchDeleteOrgCourseStudent(Long orgId, Long courseId, Collection<Long> userIds, Integer status) {
        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        condition.put("courseId", courseId);
        condition.put("userId", userIds);
        OrgStudentCourse obj = new OrgStudentCourse();
        obj.setStatus(status);
        this.update(condition, obj, "status");
    }

    @Override
    public List<OrgStudentCourse> getStudentListByStudentId(Long userId, String...queryProps) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("userId", userId);
        builder.select(queryProps);
        builder.desc("createTime");
        return queryList(builder);
    }

    @Override
    public Map<Long, List<OrgStudentCourse>> getStudentMapBycourseIds(Collection<Long> courseIds,
        final String...queryProps) {
        BatchQueryTemplate<Long, List<OrgStudentCourse>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudentCourse> courses =
            batchQueryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, List<OrgStudentCourse>>() {

                @Override
                public List<OrgStudentCourse> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
                    builder.in("courseId", querySet);
                    builder.eq("delStatus", 0);
                    builder.select(queryProps);
                    return queryList(builder);
                }
            });
        if (CollectionUtils.isEmpty(courses)) {
            return Collections.emptyMap();
        }
        Map<Long, List<OrgStudentCourse>> map = Maps.newHashMap();
        for (OrgStudentCourse orgStudentCourse : courses) {
            List<OrgStudentCourse> orgStudentCourses = map.get(orgStudentCourse.getCourseId());
            if (orgStudentCourses == null) {
                orgStudentCourses = Lists.newArrayList();
                map.put(orgStudentCourse.getCourseId(), orgStudentCourses);
            }
            orgStudentCourses.add(orgStudentCourse);
        }
        return map;
    }

    @Override
    public Map<Long, List<OrgStudentCourse>> getStudentMapBycourseIds(Collection<Long> courseIds, final int status,
        final String...queryProps) {
        BatchQueryTemplate<Long, List<OrgStudentCourse>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudentCourse> courses =
            batchQueryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, List<OrgStudentCourse>>() {

                @Override
                public List<OrgStudentCourse> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
                    builder.in("courseId", querySet);
                    builder.eq("status", status);
                    builder.eq("delStatus", 0);
                    builder.select(queryProps);
                    return queryList(builder);
                }
            });
        if (CollectionUtils.isEmpty(courses)) {
            return Collections.emptyMap();
        }
        Map<Long, List<OrgStudentCourse>> map = Maps.newHashMap();
        for (OrgStudentCourse orgStudentCourse : courses) {
            List<OrgStudentCourse> orgStudentCourses = map.get(orgStudentCourse.getCourseId());
            if (orgStudentCourses == null) {
                orgStudentCourses = Lists.newArrayList();
                map.put(orgStudentCourse.getCourseId(), orgStudentCourses);
            }
            orgStudentCourses.add(orgStudentCourse);
        }
        return map;
    }

    @Override
    public Map<Long, List<OrgStudentCourse>> getStudentMapToRealCourseIds(Collection<Long> courseIds, final int status,
        final String...queryProps) {
        BatchQueryTemplate<Long, List<OrgStudentCourse>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudentCourse> courses =
            batchQueryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, List<OrgStudentCourse>>() {

                @Override
                public List<OrgStudentCourse> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
                    builder.in("realCourseId", querySet);
                    builder.eq("status", status);
                    builder.eq("delStatus", 0);
                    builder.select(queryProps);
                    return queryList(builder);
                }
            });
        if (CollectionUtils.isEmpty(courses)) {
            return Collections.emptyMap();
        }
        Map<Long, List<OrgStudentCourse>> map = Maps.newHashMap();
        for (OrgStudentCourse orgStudentCourse : courses) {
            List<OrgStudentCourse> orgStudentCourses = map.get(orgStudentCourse.getRealCourseId());
            if (orgStudentCourses == null) {
                orgStudentCourses = Lists.newArrayList();
                map.put(orgStudentCourse.getRealCourseId(), orgStudentCourses);
            }
            orgStudentCourses.add(orgStudentCourse);
        }
        return map;
    }

    @Override
    public Map<Long, List<Long>> getStudentIdMapBycourseIds(Collection<Long> courseIds, final Long orgId) {
        BatchQueryTemplate<Long, List<OrgStudentCourse>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudentCourse> courses =
            batchQueryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, List<OrgStudentCourse>>() {

                @Override
                public List<OrgStudentCourse> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
                    builder.in("courseId", querySet);
                    builder.eq("delStatus", 0);
                    builder.eq("status", 0);
                    builder.eq("orgId", orgId);
                    builder.select("courseId", "userId");
                    return queryList(builder);
                }
            });
        return CollectorUtil.group(courses, new Function<OrgStudentCourse, Long>() {
            @Override
            public Long apply(OrgStudentCourse arg0) {
                return arg0.getCourseId();
            }
        }, new Function<OrgStudentCourse, Long>() {
            @Override
            public Long apply(OrgStudentCourse arg0) {
                return arg0.getUserId();
            }
        });

    }

    @Override
    public Map<Long, List<Long>> getStudentIdMapBycourseIdsNoStatus(Collection<Long> courseIds, final Long orgId) {
        BatchQueryTemplate<Long, List<OrgStudentCourse>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudentCourse> courses =
            batchQueryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, List<OrgStudentCourse>>() {

                @Override
                public List<OrgStudentCourse> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
                    builder.in("courseId", querySet);
                    builder.eq("delStatus", 0);

                    builder.eq("orgId", orgId);
                    builder.select("courseId", "userId");
                    return queryList(builder);
                }
            });
        return CollectorUtil.group(courses, new Function<OrgStudentCourse, Long>() {
            @Override
            public Long apply(OrgStudentCourse arg0) {
                return arg0.getCourseId();
            }
        }, new Function<OrgStudentCourse, Long>() {
            @Override
            public Long apply(OrgStudentCourse arg0) {
                return arg0.getUserId();
            }
        });

    }

    @Override
    public Map<Long, List<Long>> getCourseIdMapByStuIds(Collection<Long> stuIds, final Long orgId) {
        BatchQueryTemplate<Long, List<OrgStudentCourse>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        List<OrgStudentCourse> courses =
            batchQueryTemplate.batchQuery(stuIds, new BatchQueryCallback<Long, List<OrgStudentCourse>>() {

                @Override
                public List<OrgStudentCourse> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
                    builder.in("userId", querySet);
                    builder.eq("orgId", orgId);
                    builder.eq("status", 0);
                    builder.select("courseId", "userId");
                    return queryList(builder);
                }
            });
        return CollectorUtil.group(courses, new Function<OrgStudentCourse, Long>() {
            @Override
            public Long apply(OrgStudentCourse arg0) {
                return arg0.getUserId();
            }
        }, new Function<OrgStudentCourse, Long>() {
            @Override
            public Long apply(OrgStudentCourse arg0) {
                return arg0.getCourseId();
            }
        });
    }

    @Override
    public Map<Long, Integer> getCurStudentCntByCourseIds(final Long orgId, Collection<Long> courseIds,
        final String...queryProps) {

        MapBatchQueryTemplate<Long, Long, Integer> batchTemplate = new MapBatchQueryTemplate<>();
        return batchTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, Map<Long, Integer>>() {
            @Override
            public Map<Long, Integer> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
                builder.count("id");
                builder.add(Expressions.or(Expressions.eq("orgId", orgId), Expressions.eq("orgId", 0)));
                builder.eq("delStatus", 0);
                builder.group("courseId");
                builder.in("courseId", querySet);
                List<Map<String, Object>> queryList = query2ListMap(builder);

                if (CollectionUtils.isNotEmpty(queryList)) {
                    Map<Long, Integer> result = Maps.newHashMap();
                    for (Map<String, Object> objMap : queryList) {
                        result.put(((Number) objMap.get("courseId")).longValue(),
                            ((Number) objMap.get("cnt_id")).intValue());
                    }
                    return result;
                } else {
                    return Collections.emptyMap();
                }
            }
        });

    }

    @Override
    public void updateStudentName(Long userId, String studentName) {

        Map<String, Object> params = Maps.newHashMap();
        params.put("userId", userId);
        params.put("studentName", studentName);
        this.getNamedJdbcTemplate()
            .update("update tts.org_student_course set student_name = :studentName where student_id = :userId", params);
    }

    @Override
    public List<Long> getOrgCourseIds(Long orgId, List<Long> courseIds, boolean ascByStuCnt, PageDto pageDto) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyList();
        }
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("courseId");
        builder.count("id", "cnt");
        builder.in("courseId", courseIds);
        builder.eq("orgId", orgId);
        builder.eq("delStatus", 0);
        builder.eq("status", 0);
        builder.eq("tradeComplete", 1);
        builder.group("courseId");

        if (ascByStuCnt) {
            builder.asc("cnt");
        } else {
            builder.desc("cnt");
        }
        if (pageDto != null) {
            builder.setPage(pageDto);
        }
        return this.queryForList(builder, Long.class);
    }

    @Override
    public List<Long> getStudents(Long orgId, Long courseId, Integer status) {

        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("userId");
        builder.eq("courseId", courseId);
        builder.eq("orgId", orgId);
        builder.eq("status", status);
        builder.eq("delStatus", 0);

        return this.queryForList(builder, Long.class);

    }

    @Override
    public Map<Long, Integer> getStudentCount(final Long orgId, Collection<Long> courseIds) {
        return getStudentCount(orgId, courseIds, null);
    }

    @Override
    public Map<Long, Integer> getStudentCount(final Long orgId, Collection<Long> courseIds,
        final Boolean tradeComplete) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(courseIds,
            new BatchQueryCallback<Long, Map<Long, Integer>>() {
                @Override
                public Map<Long, Integer> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("courseId");
                    builder.count("id", "num");
                    builder.in("courseId", querySet);
                    builder.eq("orgId", orgId);
                    builder.eq("delStatus", 0);
                    builder.eq("status", 0);
                    if (tradeComplete != null) {
                        builder.eq("tradeComplete", tradeComplete ? 1 : 0);
                    }
                    builder.group("courseId");
                    final Map<Long, Integer> result = Maps.newHashMap();
                    getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                        new RowCallbackHandler() {
                            @Override
                            public void processRow(ResultSet rs) throws SQLException {
                                result.put(rs.getLong("courseId"), rs.getInt("num"));
                            }
                        });
                    return result;
                }
            });
    }

    @Override
    public Map<Long, Integer> getStudentCountNoStatus(final Long orgId, Collection<Long> courseIds,
        final Boolean tradeComplete) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(courseIds,
            new BatchQueryCallback<Long, Map<Long, Integer>>() {
                @Override
                public Map<Long, Integer> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("courseId");
                    builder.count("id", "num");
                    builder.in("courseId", querySet);
                    builder.eq("orgId", orgId);
                    builder.eq("delStatus", 0);

                    if (tradeComplete != null) {
                        builder.eq("tradeComplete", tradeComplete ? 1 : 0);
                    }
                    builder.group("courseId");
                    final Map<Long, Integer> result = Maps.newHashMap();
                    getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                        new RowCallbackHandler() {
                            @Override
                            public void processRow(ResultSet rs) throws SQLException {
                                result.put(rs.getLong("courseId"), rs.getInt("num"));
                            }
                        });
                    return result;
                }
            });
    }

    @Override
    public List<OrgStudentCourse> getOrgCourseIds(Long orgId, Long userId, Integer status, PageDto pageDto) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        if (userId != null) {
            builder.eq("userId", userId);
        }
        builder.eq("orgId", orgId);
        if (status != null) {
            builder.eq("status", status);
        }

        if (pageDto != null) {
            builder.setPage(pageDto);
        }
        builder.eq("delStatus", 0);
        builder.asc("createTime");
        return this.queryList(builder);
    }

    @Override
    public List<OrgStudentCourse> getOrgCourseIds(Long orgId, Long userId, PageDto pageDto) {
        return getOrgCourseIds(orgId, userId, StudentCourseStatus.NORMAL.getCode(), pageDto);
    }

    @Override
    public List<OrgStudentCourse> getOrgCourseByOrgId(Long orgId, Collection<Long> courseIds, PageDto pageDto) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("courseId");
        builder.eq("orgId", orgId);
        builder.eq("delStatus", 0);
        if (courseIds != null && !courseIds.isEmpty()) {
            builder.in("courseId", courseIds);
        }
        builder.group("courseId");
        builder.desc("createTime");
        if (pageDto != null) {
            builder.setPage(pageDto);
        }
        return this.queryList(builder);
    }

    @Override
    public List<OrgStudentCourse> getOrgCourseByOrgIdAndIds(Long orgId, Collection<Long> courseIds) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("delStatus", 0);
        builder.in("courseId", courseIds);
        builder.eq("status", 0); // 找不到常量 0不包括转班、退班
        return this.queryList(builder);
    }

    @Override
    public Integer getStudentCourseCount(Long orgId, Long userId, Integer status) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("userId", userId);
        builder.eq("orgId", orgId);
        if (status != null) {
            builder.eq("status", status);
        }
        return this.queryForObject(builder, Integer.class);
    }

    @Override
    public Integer getStudentSignupCourseCount(Long orgId, Long userId, Integer status) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.distinctCount("courseId");
        builder.eq("userId", userId);
        builder.eq("orgId", orgId);
        builder.eq("delStatus", 0);
        if (status != null) {
            builder.eq("status", status);
        }
        return this.queryForObject(builder, Integer.class);
    }

    @Override
    public List<Long> getStudentCourseIds(Long orgId, Long userId, Integer status) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("courseId");
        builder.eq("userId", userId);
        builder.eq("orgId", orgId);
        builder.eq("delStatus", 0);
        if (status != null) {
            builder.eq("status", status);
        }
        return this.queryForList(builder, Long.class);
    }

    @Override
    public List<Long> getStudentCourseIdsByStatus(Long orgId, Long userId, List<Integer> status) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("courseId");
        builder.eq("userId", userId);
        builder.eq("orgId", orgId);
        if (status != null) {
            builder.in("status", status);
        }
        return this.queryForList(builder, Long.class);
    }

    @Override
    public List<Long> getStudentIdsByCourseIds(final Long orgId, Collection<Long> courseIds) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Lists.newArrayList();
        }
        BatchQueryTemplate<Long, List<Long>> batch = new ListBatchQueryTemplate<>();
        return batch.batchQuery(courseIds, new BatchQueryCallback<Long, List<Long>>() {
            @Override
            public List<Long> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("userId");
                builder.eq("orgId", orgId);
                builder.in("courseId", querySet);
                return queryForList(builder, Long.class);
            }
        });
    }

    @Override
    public void refreshOrgStudentCourse(Long orgId, Long id) {
        StringBuilder sb =
            new StringBuilder("update tts.org_student_course set del_status =1 where id > :id and  org_id = :orgId");
        Map param = Maps.newHashMap();
        param.put("orgId", orgId);
        param.put("id", id);
        this.getNamedJdbcTemplate().update(sb.toString(), param);
    }

    @Override
    public void deleteByOrgCourseId(Integer orgId, Integer courseId) {
        StringBuilder sb = new StringBuilder(
            "update tts.org_student_course set del_status =1 where course_id = :courseId and  org_id = :orgId");
        Map param = Maps.newHashMap();
        param.put("orgId", orgId);
        param.put("courseId", courseId);
        this.getNamedJdbcTemplate().update(sb.toString(), param);
    }

    @Override
    public List<Long> getEnrollClassUserIdByUserIds(final Long orgId, Collection<Long> userIds) {

        BatchQueryTemplate<Long, List<Long>> batch = new ListBatchQueryTemplate<>();
        return batch.batchQuery(userIds, new BatchQueryCallback<Long, List<Long>>() {
            @Override
            public List<Long> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("userId");
                builder.eq("orgId", orgId);
                builder.in("userId", querySet);
                builder.eq("delStatus", 0);
                builder.group("userId");
                return queryForList(builder, Long.class);
            }
        });
    }

    @Override
    public Map<Integer, Integer> allCourseOrgIdMap() {

        Map<String, Object> param = new HashMap<>();
        String sql =
            "select org_id,course_id from tts.org_student_course where course_id <= 46763 and course_id>0 and org_id>0 and del_status=0  group by org_id,course_id";

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Integer, Integer>>() {
            @Override
            public Map<Integer, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Integer, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Integer courseId = rs.getInt("course_id");
                    Integer orgId = rs.getInt("org_id");
                    map.put(courseId, orgId);
                }
                return map;
            }
        });
    }

    @Override
    public List<OrgStudentCourse> getOrgStudentCourseByUserIds(Long orgId, Collection<Long> userIds, String...props) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder(props);
        builder.eq("orgId", orgId);
        if (userIds != null && userIds.size() > 0) {
            builder.in("userId", userIds);
        }
        builder.eq("delStatus", 0);
        return queryList(builder);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.org.dao.OrgStudentCourseDao#searchStudentWithCourseIds(java.lang.Long,
     * java.util.Set, boolean)
     */
    @Override
    public List<OrgStudentCourse> searchStudentWithCourseIds(Long orgId, Set<Long> filterIds, List<Long> noDelCourseIds,
        boolean allCourse) {
        if (orgId == null) {
            return Collections.emptyList();
        }
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("delStatus", 0);
        builder.eq("status", 0); // 不包括 退班/转班的情况
        log.info("orgId:{} , filterIds:{} , noDelCourseIds:{} ,allCourse:{}", orgId, filterIds, noDelCourseIds,
            allCourse);
        if (allCourse) {
            noDelCourseIds.removeAll(filterIds);
            if (GenericsUtils.notNullAndEmpty(noDelCourseIds)) {
                builder.in("courseId", noDelCourseIds);
            }
        } else {
            filterIds.retainAll(noDelCourseIds);
            if (GenericsUtils.isNullOrEmpty(filterIds)) {
                return Collections.emptyList();
            }
            builder.in("courseId", filterIds);
        }
        log.info("sql is : {} ", builder.toSql());
        return queryList(builder);
    }

    @Override
    public List<OrgStudentCourse> getOrgStudentCourseByOrgIds(Collection<Long> orgIds, PageDto pageDto) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.in("orgId", orgIds);
        builder.setPage(pageDto);
        return queryList(builder);
    }

    /*
     * 获取学员所有的报班记录
     */
    @Override
    public Map<Long, List<Long>> getOrgCourseIdMap(long orgId, Set<Long> userIds) {
        List<OrgStudentCourse> orgStudentCourseByUserIds =
            this.getOrgStudentCourseByUserIds(orgId, userIds, "userId", "courseId");
        Map<Long, List<Long>> emptyMap = GenericsUtils.emptyMap();
        if (GenericsUtils.isNullOrEmpty(orgStudentCourseByUserIds)) {
            return emptyMap;
        }
        for (OrgStudentCourse osc : orgStudentCourseByUserIds) {
            Long studentId = osc.getUserId();
            GenericsUtils.addListIfNotExists(emptyMap, studentId, osc.getCourseId());
        }
        return emptyMap;
    }

    @Override
    public int sumOneToOneClassHours(Collection<Long> classIds) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.in("courseId", classIds);
        builder.sum("lessonCount");
        return queryForObject(builder, Integer.class);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.org.dao.OrgStudentCourseDao#searchStudentClassHourStatus(java.util.Set,
     * java.lang.Long)
     */
    @Override
    public List<StudentClassHourStatusDocument> searchStudentClassHourStatus(Set<Long> userIds, Long orgId) {
        if (GenericsUtils.isNullOrEmpty(userIds)) {
            return GenericsUtils.emptyList();
        }
        SingleSqlBuilder<OrgStudentCourse> builder =
            createSqlBuilder("courseId", "userId", "status", "lessonCount", "chargeUnit");
        builder.eq("orgId", orgId);
        builder.in("userId", userIds);
        List<OrgStudentCourse> queryList = queryList(builder);
        List<StudentClassHourStatusDocument> ret = Lists.newArrayList();
        if (GenericsUtils.notNullAndEmpty(queryList)) {
            for (OrgStudentCourse osc : queryList) {
                StudentClassHourStatusDocument scsd = new StudentClassHourStatusDocument();
                scsd.setCourseId(osc.getCourseId());
                scsd.setLessonCount(osc.getLessonCount());
                scsd.setStatus(osc.getStatus());
                scsd.setUserId(osc.getUserId());
                scsd.setOrgId(orgId);
                scsd.setChargeUnit(osc.getChargeUnit());
                ret.add(scsd);
            }
        }
        return ret;
    }

    // @Override
    // public Double getVipClassTotalPrice(Long orgId, Long courseId) {
    // SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("payAmount");
    // builder.eq("orgId", orgId);
    // builder.eq("courseId", courseId);
    // OrgStudentCourse orgStudentCourse = queryForObject(builder, OrgStudentCourse.class);
    // return (double) (orgStudentCourse == null ? 0 : orgStudentCourse.getPayAmount());
    // }

    @Override
    public List<OrgStudentCourse> getPageByCourseIdAndMaxId(Long orgId, Long courseId, Integer limit, Long maxId) {

        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();

        builder.eq("orgId", orgId);
        builder.eq("courseId", courseId);
        if (maxId != null) {
            builder.gt("id", maxId);
        }

        builder.addOrder(Order.asc("id"));
        builder.setMaxSize(limit);

        return this.queryList(builder);

    }

    @Override
    public OrgStudentCourse getByStudentAndClassId(Long orgId, Long userId, Collection<Long> classIds) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("userId", userId);
        builder.in("courseId", classIds);
        builder.eq("status", DataStatus.NORMAL.getValue());
        builder.desc("id");
        builder.setMaxSize(1);

        return this.uniqueResult(builder);
    }

    @Override
    public OrgStudentCourse getStudentCourseByRealCourseId(Long orgId, Long realCourseId, Long userId) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("userId", userId);
        builder.eq("realCourseId", realCourseId);
        builder.desc("id");
        builder.setMaxSize(1);

        return this.uniqueResult(builder);
    }

    @Override
    public List<OrgStudentCourse> getByCourseIdsStudentId(Long orgId, Collection<Long> courseIds, Long userId) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        if (userId != null) {
            builder.eq("userId", userId);
        }
        if (CollectionUtils.isNotEmpty(courseIds)) {
            builder.in("courseId", courseIds);
        }
        builder.eq("status", StudentCourseStatus.NORMAL.getCode());// 不包括退班转班的学生
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        return this.queryList(builder);
    }

    @Override
    public List<OrgStudentCourse> getByCourseIdsStudentId(Long orgId, Collection<Long> courseIds,
        Collection<Long> userIds, Integer status, Integer delStatus) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Lists.newArrayList();
        }
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        if (CollectionUtils.isNotEmpty(userIds)) {
            builder.in("userId", userIds);
        }
        if (CollectionUtils.isNotEmpty(courseIds)) {
            builder.in("courseId", courseIds);
        }
        if (null != status) {
            builder.eq("status", status/* StudentCourseStatus.NORMAL.getCode() */);// 不包括退班转班的学生
        }
        if (null != delStatus) {
            builder.eq("delStatus", delStatus/* DeleteStatus.NORMAL.getValue() */);
        }
        return this.queryList(builder);
    }

    @Override
    public List<OrgStudentCourse> getStudentCourseByStuNameAndMobile(Long orgId, String studentName, String mobile,
        String...props) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder(props);
        builder.eq("orgId", orgId);
        if (StringUtils.isNotEmpty(studentName)) {
            builder.like("studentName", studentName, MatchMode.ANYWHERE);
        }
        if (StringUtils.isNotEmpty(mobile)) {
            builder.like("mobile", mobile, MatchMode.ANYWHERE);
        }
        builder.eq("status", StudentCourseStatus.NORMAL.getCode());// 不包括退班转班的学生
        builder.eq("delStatus", DeleteStatus.NORMAL.getValue());
        return this.queryList(builder);
    }

    @Override
    public List<OrgStudentCourse> getBetweenId(@NonNull Integer startId, @NonNull Integer endId, List<Long> orgIds,
        Integer status, Integer delStatus, String...props) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder(props);
        builder.ge("id", startId);
        builder.lt("id", endId);
        if (CollectionUtils.isNotEmpty(orgIds)) {
            builder.in("orgId", orgIds);
        }
        if (null != status) {
            builder.eq("status", status);
        }
        if (null != delStatus) {
            builder.eq("delStatus", delStatus);
        }
        return this.queryList(builder);
    }

    @Override
    public List<OrgStudentCourse> listByCourseId(Long orgId, Long courseId, Integer status, Integer delStatus) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("courseId", courseId);
        if (status != null) {
            builder.eq("status", status);
        }
        if (delStatus != null) {
            builder.eq("delStatus", delStatus);
        }
        return this.queryList(builder);
    }

    @Override
    public List<OrgStudentCourse> listByUserId(Long orgId, Long userId, Collection<Long> courseIds, Integer status) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("userId", userId);
        if (status != null) {
            builder.eq("status", status);// StudentCourseStatus.NORMAL.getCode()
        }
        if (CollectionUtils.isNotEmpty(courseIds)) {
            builder.in("courseId", courseIds);
        }

        return this.queryList(builder);
    }

    @Override
    public List<OrgStudentCourse> listByOrgId(Long orgId, Integer status) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        if (status != null) {
            builder.eq("status", status);// StudentCourseStatus.NORMAL.getCode()
        }

        return this.queryList(builder);
    }

    @Override
    public Integer countStudents(Long orgId, Long courseId, Integer status) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.count("id");
        if(orgId!=null){
        	builder.eq("orgId", orgId);
        }
        builder.eq("courseId", courseId);
        if (status != null) {
            builder.eq("status", status);
        }

        return this.queryForObject(builder, Integer.class);
    }

    @Override
    public List<Long> listReClassId(Long orgId, Long courseId, Integer status) {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public List<Long> listStudentUserIdsByCourseIds(Long orgId, Long courseId) {
        final List<Long> result = new ArrayList<Long>();
        String sql = "SELECT student_id FROM tts.org_student_course WHERE course_id=:courseId";
        if(orgId!=null){
        	sql += " AND org_id=:orgId"; 
        }

        NamedParameterJdbcTemplate template = getNamedJdbcTemplate();
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orgId", orgId);
        params.put("courseId", courseId);

        template.query(sql.toString(), params, new RowMapper<Integer>() {
            @Override
            public Integer mapRow(ResultSet rs, int rowNum) throws SQLException {
                result.add(rs.getLong("student_id"));
                return 1;
            }
        });

        return result;
    }

    @Override
    public OrgStudentCourse listByCourseIdAndUserId(Long orgId, Long courseId, Long userId) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("orgId", orgId);
        builder.eq("userId", userId);
        builder.eq("courseId", courseId);
        builder.desc("id");
        builder.setMaxSize(1);
        return uniqueResult(builder);
    }

    @Override
    public List<OrgStudentCourse> listAllNeedSyncRecord(long lastRecordId, Integer maxSize, Long...limitOrgIds) {
        SingleSqlBuilder<OrgStudentCourse> builder =
            createSqlBuilder("id", "orgId", "courseId", "userId", "realCourseId");
        builder.gt("id", lastRecordId);
        if (GenericsUtils.notNullAndEmpty(limitOrgIds)) {
            builder.in("orgId", Arrays.asList(limitOrgIds));
        }
        builder.asc("id");
        builder.setMaxSize(maxSize);
        log.info("sql is:{} ,lastRecordId {}", builder.toSql(), lastRecordId);
        List<OrgStudentCourse> retList = queryList(builder);
        log.info("retList's size is{}", retList.size());
        return retList;
    }

    @Override
    public List<OrgStudentCourse> listAllNeedSyncRecordWithRange(long firstId, long lastId) {
        SingleSqlBuilder<OrgStudentCourse> builder =
            createSqlBuilder("id", "orgId", "courseId", "userId", "realCourseId");
        builder.gt("id", firstId);
        builder.le("id", lastId);
        builder.asc("id");
        builder.setMaxSize(5000);
        log.info("sql is:{} ,lastRecordId {} and firstId :{} and lastId:{} ", builder.toSql(), firstId, lastId);
        List<OrgStudentCourse> retList = queryList(builder);
        log.info("retList's size is{}", retList.size());
        return retList;
    }

    @Override
    public Map<Long, Integer> userMapByStatus(Long orgId, Long classId, Collection<Long> userIds,
        Collection<Integer> status) {
        if (CollectionUtils.isEmpty(userIds)) {
            return Maps.newHashMap();
        }
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder("status", "userId");
        builder.eq("orgId", orgId);
        builder.eq("courseId", classId);
        builder.in("userId", userIds);
        builder.in("status", status);
        log.debug("sql:{}, params:{}", builder.toSql(), builder.collectConditionValue());
        Map<Long, Integer> result = Maps.newHashMap();
        for (OrgStudentCourse course : queryList(builder)) {
            result.put(course.getUserId(), course.getStatus());
        }
        return result;
    }

    @Override
    public List<OrgStudentCourse> listByStatus(Long orgId, Long classId, Collection<Long> userIds,
        Collection<Integer> status, String...prop) {
        if (CollectionUtils.isEmpty(userIds)) {
            return Lists.newArrayList();
        }
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder(prop);
        builder.eq("orgId", orgId);
        if (null != classId) {
            builder.eq("courseId", classId);
        }
        builder.in("userId", userIds);
        builder.in("status", status);
        return queryList(builder);
    }

    @Override
    public List<Long> selectOneToOneClassId(Long orgId) {
        String sql = "select course_id from tts.org_student_course where org_id=:orgId and course_id != real_course_id";
        Map<String, Object> map = new HashMap<>();
        map.put("orgId", orgId);
        return getNamedJdbcTemplate().queryForList(sql, map, Long.class);
    }

    @Override
    public List<OrgStudentCourse> getNoRealCourseIdData() {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.eq("realCourseId", 0);
        builder.ne("courseId", 0);
        builder.setMaxSize(50);
        return queryList(builder);
    }


	@Override
	public Map<Long, Integer> mapStudentCount(Collection<Long> classIds) {
		final Map<Long, Integer> result = new HashMap<Long, Integer>();
        String sql = "SELECT course_id,count(id) AS stuNum FROM tts.org_student_course WHERE status=0 AND del_status=0 AND course_id IN (:classIds) GROUP BY course_id";

        NamedParameterJdbcTemplate template = getNamedJdbcTemplate();
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("classIds", classIds);

        template.query(sql.toString(), params, new RowMapper<Integer>() {
            @Override
            public Integer mapRow(ResultSet rs, int rowNum) throws SQLException {
            	result.put(rs.getLong("course_id"), rs.getInt("stuNum"));
                return 1;
            }
        });

        return result;
	}

	
	@Override
	public Map<Long, List<Long>> mapCourseIdStuList(Collection<Long> classIds) {
		final Map<Long, List<Long>> result = new HashMap<Long, List<Long>>();
		if(CollectionUtils.isEmpty(classIds)){
			return result;
		}
		
        String sql = "SELECT course_id,student_id FROM tts.org_student_course WHERE course_id IN (:classIds)";
        

        NamedParameterJdbcTemplate template = getNamedJdbcTemplate();
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("classIds", classIds);

        template.query(sql.toString(), params, new RowMapper<Integer>() {
            @Override
            public Integer mapRow(ResultSet rs, int rowNum) throws SQLException {
            	Long courseId = rs.getLong("course_id");
                Long studentId = rs.getLong("student_id");
                if(result.get(courseId) == null){
                	result.put(courseId, new ArrayList<Long>());
                }
                result.get(courseId).add(studentId);
                
                return 1;
            }
        });

        return result;
	}

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.org.dao.OrgStudentCourseDao#selectMaxId()
     */

    @Override
    public int selectMaxId() {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        builder.max("id");
        return queryForObject(builder, Integer.class);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.org.dao.OrgStudentCourseDao#listByCourseIdsAndUserIds(java.util.Set, java.util.Set,
     * java.util.Set)
     */
    @Override
    public List<OrgStudentCourse> listByCourseIdsAndUserIds(Set<Long> orgIds, Set<Long> courseIds, Set<Long> userIds) {
        SingleSqlBuilder<OrgStudentCourse> builder = createSqlBuilder();
        if (GenericsUtils.isNullOrEmpty(orgIds) || GenericsUtils.isNullOrEmpty(courseIds)
            || GenericsUtils.isNullOrEmpty(userIds)) {
            return GenericsUtils.emptyList();
        }
        builder.in("orgId", orgIds);
        builder.in("userId", userIds);
        builder.in("realCourseId", courseIds);
        log.info("sql is :{} ", builder.toSql());
        return queryList(builder);
    }

}