/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */

package com.baijia.tianxiao.dal.org.dao.impl;

import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.dal.org.dao.OrgLessonCommentDao;
import com.baijia.tianxiao.dal.org.po.CommentAudit;
import com.baijia.tianxiao.dal.org.po.OrgCommentAudit;
import com.baijia.tianxiao.dal.org.po.OrgLessonComment;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.baijia.tianxiao.util.query.MapBatchQueryTemplate;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.stereotype.Repository;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author shanyu
 * @version 1.0
 * @title OrgLessonCommentDaoImpl
 * @desc TODO
 * @date 2015年12月30日
 */
@Repository
public class OrgLessonCommentDaoImpl extends JdbcTemplateDaoSupport<OrgLessonComment> implements OrgLessonCommentDao {

    public OrgLessonCommentDaoImpl() {
        super(OrgLessonComment.class);
    }

    @Override
    public List<OrgLessonComment> getLessonStudentComments(Long lessonId, Boolean isSystem, String... queryProps) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryProps);
        builder.eq("lessonId", lessonId);
        builder.eq("userRole", UserRole.STUDENT.getRole());
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        return queryList(builder);
    }

    @Override
    public List<OrgLessonComment> getLessonTeacherComments(Long lessonId, Boolean isSystem, String... queryProps) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryProps);
        builder.eq("lessonId", lessonId);
        builder.eq("userRole", UserRole.TEACHER.getRole());
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        return queryList(builder);
    }

    @Override
    public OrgLessonComment getLessonCommentDetail(Long lessonId, Long userId, Integer userRole, Boolean isSystem,
                                                   String... queryProps) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryProps);
        builder.eq("lessonId", lessonId);
        builder.eq("userRole", userRole);
        if (userRole.intValue() == UserRole.STUDENT.getRole()) {
            builder.eq("fromId", userId);
        } else {
            builder.eq("toId", userId);
        }
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        return uniqueResult(builder);
    }

    @Override
    public List<OrgLessonComment> getStudentComments(Long orgId, Long userId, Boolean isSystem, PageDto pageDto,
                                                     String... queryProps) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryProps);
        builder.eq("fromId", userId);
        builder.eq("orgId", orgId);
        builder.eq("userRole", UserRole.TEACHER.getRole());
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        builder.setPage(pageDto);
        return queryList(builder);
    }

    @Override
    public List<OrgLessonComment> getTeacherCommentsOfStudent(Long orgId, Long userId, Boolean isSystem,
                                                              PageDto pageDto, String... queryProps) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryProps);
        builder.eq("toId", userId);
        builder.eq("orgId", orgId);
        builder.eq("userRole", UserRole.TEACHER.getRole());
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        builder.setPage(pageDto);
        return queryList(builder);
    }

    @Override
    public CommentAudit getLessonCommentAudit(Long lessonId, Long orgId, Boolean isSystem) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder("lessonId");
        builder.count("id", "num");
        builder.sum("score", "totalScoce");
        builder.eq("lessonId", lessonId);
        builder.eq("orgId", orgId);
        builder.eq("userRole", UserRole.STUDENT.getRole());
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        List<CommentAudit> result = queryList(builder, CommentAudit.class);
        if (CollectionUtils.isNotEmpty(result)) {
            return result.get(0);
        }
        return null;
    }

    @Override
    public List<CommentAudit> getLessonsCommentAudit(Collection<Long> lessonIds, Long orgId, Integer userRole,
                                                     Boolean isSystem) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Lists.newArrayList();
        }
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder("lessonId");
        builder.count("id", "num");
        builder.group("lessonId");
        builder.sum("score", "totalScore");
        builder.in("lessonId", lessonIds);
        builder.eq("orgId", orgId);
        builder.eq("userRole", userRole);
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        return queryList(builder, CommentAudit.class);
    }

    @Override
    public CommentAudit getCommentAudit(Collection<Long> lessonIds, Long orgId, Boolean isSystem) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            return new CommentAudit();
        }
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();
        builder.count("id", "num");
        builder.sum("score", "totalScore");
        builder.in("lessonId", lessonIds);
        builder.eq("orgId", orgId);
        builder.eq("userRole", UserRole.STUDENT.getRole());
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        List<CommentAudit> result = queryList(builder, CommentAudit.class);
        if (CollectionUtils.isNotEmpty(result)) {
            return result.get(0);
        }
        return new CommentAudit();
    }

    @Override
    public Map<Long, Integer> getLessonCommentCountMap(Collection<Long> lessonIds, final Long orgId, final Long fromId,
                                                       final Long toId, final Integer userRole, Boolean isSystem) {
        if (CollectionUtils.isEmpty(lessonIds)) {
            return Maps.newHashMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(lessonIds,
                new BatchQueryCallback<Long, Map<Long, Integer>>() {
                    @Override
                    public Map<Long, Integer> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder("lessonId");
                        builder.count("id", "num");
                        builder.group("lessonId");
                        builder.in("lessonId", querySet);
                        builder.eq("userRole", userRole);
                        builder.eq("orgId", orgId);
                        if (fromId != null) {
                            builder.eq("fromId", fromId);
                        }
                        if (toId != null) {
                            builder.eq("toId", toId);
                        }
                        final Map<Long, Integer> result = Maps.newHashMap();
                        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                                new RowCallbackHandler() {
                                    @Override
                                    public void processRow(ResultSet rs) throws SQLException {
                                        result.put(rs.getLong("lessonId"), rs.getInt("num"));
                                    }
                                });
                        return result;
                    }
                });
    }

    @Override
    public List<OrgLessonComment> getStudentComments(Long orgId, Long userId, Integer userRole, Boolean isFrom,
                                                     Boolean isSystem, Collection<Long> lessonIds, PageDto pageDto, String... queryPorps) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryPorps);
        builder.eq("orgId", orgId);
        if (isFrom) {
            builder.eq("fromId", userId);
        } else {
            builder.eq("toId", userId);
        }
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        builder.in("lessonId", lessonIds);
        builder.eq("userRole", userRole);
        builder.setPage(pageDto);
        return queryList(builder);
    }

    @Override
    public Integer getStudentLessonCommentCount(Long orgId, Long userId, Boolean isSystem) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("fromId", userId);
        builder.eq("orgId", orgId);
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        builder.eq("userRole", UserRole.STUDENT.getRole());
        return queryForObject(builder, Integer.class);
    }

    @Override
    public List<Long> getCommentLessonIdsOfStudent(Long userId) {

        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder("lessonId");
        builder.eq("fromId", userId);
        // builder.eq("isSystem", 0);
        builder.eq("userRole", UserRole.STUDENT.getRole());
        return queryForList(builder, Long.class);

    }

    @Override
    public Map<Long, List<Long>> getLessonCommentStudentMap(Collection<Long> lessonIds, final Boolean isSystem,
                                                            final Integer userRole) {
        BatchQueryTemplate<Long, List<OrgLessonComment>> batch = new ListBatchQueryTemplate<>();
        List<OrgLessonComment> list =
                batch.batchQuery(lessonIds, new BatchQueryCallback<Long, List<OrgLessonComment>>() {
                    @Override
                    public List<OrgLessonComment> doQuery(Collection<Long> querySet) {
                        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder("lessonId", "fromId", "toId");
                        builder.in("lessonId", querySet);
                        builder.eq("userRole", userRole);
                        if (isSystem != null) {
                            builder.eq("isSystem", isSystem);
                        }
                        return queryList(builder);
                    }
                });
        return CollectorUtil.group(list, new Function<OrgLessonComment, Long>() {
            @Override
            public Long apply(OrgLessonComment arg0) {
                return arg0.getLessonId();

            }
        }, new Function<OrgLessonComment, Long>() {
            @Override
            public Long apply(OrgLessonComment arg0) {
                if (userRole.intValue() == UserRole.STUDENT.getRole()) {
                    return arg0.getFromId();
                }
                return arg0.getToId();
            }
        });
    }

    @Override
    public Map<Long, CommentAudit> getTeacherCommentCountMap(Long orgId, Collection<Long> teacherIds, Integer userRole,
                                                             Date endTime, Boolean isSystem) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        Preconditions.checkArgument(userRole != null, "userRole is null!");
        if (CollectionUtils.isEmpty(teacherIds)) {
            return Maps.newHashMap();
        }
        Map<String, Object> params = Maps.newHashMap();
        StringBuilder sb = new StringBuilder();
        sb.append("select count(comment.id) as num, sum(comment.score) as score, ");
        if (userRole.intValue() == UserRole.STUDENT.getRole()) {
            sb.append("comment.to_id as teacherId ");
        } else {
            sb.append("comment.from_id as teacherId ");
        }
        sb.append(
                "from tts.org_lesson_comment comment join tts.org_class_lesson lesson on comment.lesson_id = lesson.id ");
        sb.append("where comment.org_id =:orgId and comment.user_role =:userRole ");
        if (isSystem != null) {
            sb.append("and comment.is_system =:isSystem ");
            params.put("isSystem", isSystem);
        }
        params.put("orgId", orgId);
        params.put("userRole", userRole);
        if (endTime != null) {
            sb.append("and lesson.end_time >:endTime ");
            params.put("endTime", endTime);

        }
        if (userRole.intValue() == UserRole.STUDENT.getRole()) {
            sb.append("and comment.to_id in (:teacherIds) ");
            sb.append("group by (comment.to_id) ");
        } else {
            sb.append("and comment.from_id in (:teacherIds) ");
            sb.append("group by (comment.from_id) ");
        }
        params.put("teacherIds", teacherIds);
        final Map<Long, CommentAudit> result = Maps.newHashMap();
        getNamedJdbcTemplate().query(sb.toString(), params, new RowCallbackHandler() {

            @Override
            public void processRow(ResultSet rs) throws SQLException {
                CommentAudit audit = new CommentAudit();
                audit.setNum(rs.getInt("num"));
                audit.setTotalScore(rs.getInt("score"));
                result.put(rs.getLong("teacherId"), audit);
            }
        });
        return result;
    }

    @Override
    public Map<Integer, Integer> getCommentCountOfOrg(Long orgId, Boolean isSystem, Integer lowScore) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder("userRole");
        builder.count("id", "num");
        builder.eq("orgId", orgId);
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        if (lowScore != null) {
            builder.lt("score", lowScore);
        }
        builder.group("userRole");
        final Map<Integer, Integer> result = Maps.newHashMap();
        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(), new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                result.put(rs.getInt("userRole"), rs.getInt("num"));
            }
        });
        return result;
    }

    @Override
    public List<OrgCommentAudit> getOrgCommentAudit(Date startTime, Date endTime, final Boolean isSystem) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder("orgId");
        builder.count("id", "commentCount");
        builder.sum("score", "totalScore");
        builder.eq("userRole", UserRole.STUDENT.getRole());
        if (startTime != null) {
            builder.ge("createTime", startTime);
        }
        if (endTime != null) {
            builder.lt("createTime", endTime);
        }
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        builder.group("orgId");
        final List<OrgCommentAudit> result = Lists.newArrayList();
        getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(), new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                OrgCommentAudit audit = new OrgCommentAudit();
                if (isSystem == null) {// 包含默认评价
                    audit.setCommentCountAll(rs.getInt("commentCount"));
                    audit.setOrgId(rs.getLong("orgId"));
                    audit.setTotalScoreAll(rs.getInt("totalScore"));
                } else {// 不包含默认评价
                    audit.setCommentCount(rs.getInt("commentCount"));
                    audit.setOrgId(rs.getLong("orgId"));
                    audit.setTotalScore(rs.getInt("totalScore"));
                }
                result.add(audit);
            }
        });
        return result;
    }

    @Override
    public List<OrgLessonComment> getComments(Long orgId, Collection<Long> lessonIds, Integer userRole,
                                              Integer lowSocre, Boolean hasPic, Integer highScore, PageDto page, Boolean isSystem, String... queryProps) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        builder.eq("userRole", userRole);
        if (lowSocre != null) {
            builder.lt("score", lowSocre);
        }
        if (hasPic != null && hasPic) {
            builder.ne("storageIds", "");
        }
        if (highScore != null) {
            builder.gt("score", highScore);
        }
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        if (CollectionUtils.isNotEmpty(lessonIds)) {
            builder.in("lessonId", lessonIds);
        }
        builder.desc("createTime");
        builder.setPage(page);
        return queryList(builder);
    }

    @Override
    public List<OrgLessonComment> getLessonComments(Long lessonId, Boolean isSystem, Integer userRole,
                                                    String... queryProps) {
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder(queryProps);
        builder.eq("lessonId", lessonId);
        if (userRole != null) {
            builder.eq("userRole", userRole);
        }
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        return queryList(builder);
    }

    @Override
    public Map<Long, Integer> getComment(int userRole, Date startTime, Date endTime, List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        param.put("userRole", userRole);
        String sql =
                "select org_id, count(id) count from tts.org_lesson_comment where user_role = :userRole and org_id in (:orgIds) and is_system = 0";
        if (startTime != null && endTime != null) {
            sql += " AND create_time between :startTime and :endTime ";
            param.put("startTime", startTime);
            param.put("endTime", endTime);
        }
        sql += " group by org_id ";

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, Integer>>() {

            @Override
            public Map<Long, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Long orgId = rs.getLong("org_id");
                    Integer count = rs.getInt("count");
                    map.put(orgId, count);
                }
                return map;
            }
        });
    }


    @Override
    public Integer getCommentCount(Long orgId, Collection<Long> lessonIds, Integer userRole, Integer lowSocre,
                                   Boolean hasPic, Integer highScore, Boolean isSystem) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("orgId", orgId);
        builder.eq("userRole", userRole);
        if (lowSocre != null) {
            builder.lt("score", lowSocre);
        }
        if (hasPic != null && hasPic) {
            builder.ne("storageIds", "");
        }
        if (highScore != null) {
            builder.gt("score", highScore);
        }
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        if (CollectionUtils.isNotEmpty(lessonIds)) {
            builder.in("lessonId", lessonIds);
        }
        return queryForObject(builder, Integer.class);

    }


    @Override
    public Integer getPeriodComment(Long orgId, Collection<Long> lessonIds, Integer userRole, Integer lowSocre,
                                    Boolean hasPic, Integer highScore, Boolean isSystem, String startDate, String endDate) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("orgId", orgId);
        builder.between("createTime", startDate, endDate);
        builder.eq("userRole", userRole);
        if (lowSocre != null) {
            builder.lt("score", lowSocre);
        }
        if (hasPic != null && hasPic) {
            builder.ne("storageIds", "");
        }
        if (highScore != null) {
            builder.gt("score", highScore);
        }
        if (isSystem != null) {
            builder.eq("isSystem", isSystem);
        }
        if (CollectionUtils.isNotEmpty(lessonIds)) {
            builder.in("lessonId", lessonIds);
        }
        return queryForObject(builder, Integer.class);
    }


    @Override
    public Long getSumCommentByTeacher(Long teacherUserId, String... queryProps) {
        Preconditions.checkArgument(teacherUserId != null, "teacherUserId is null!");
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();
        builder.sum("score");
        builder.eq("toId", teacherUserId);
        return queryForObject(builder, Long.class);
    }

    @Override
    public Long getCountCommentByTeacher(Long teacherUserId, String... queryProps) {
        Preconditions.checkArgument(teacherUserId != null, "teacherUserId is null!");
        SingleSqlBuilder<OrgLessonComment> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("toId", teacherUserId);
        return queryForObject(builder, Long.class);
    }

    @Override
    public Map<Long, Long> getAveCommentByTeacherIds(Collection<Long> teacherUserIds, String... queryProps) {
        if (teacherUserIds == null || teacherUserIds.size() == 0) {
            return Maps.newHashMap();
        }
        StringBuilder sb = new StringBuilder();
        sb.append("select sum(score)/count(id) as score,to_id as tid from tts.org_lesson_comment where to_id in (:teacherIds) group by to_id");

        Map params = Maps.newHashMap();
        params.put("teacherIds", teacherUserIds);

        return this.getNamedJdbcTemplate().query(sb.toString(), params, new ResultSetExtractor<Map<Long, Long>>() {

            @Override
            public Map<Long, Long> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, Long> map = new HashMap<>();
                while (rs.next()) {
                    Long tid = rs.getLong("tid");
                    Long score = rs.getLong("score");
                    map.put(tid, score);
                }
                return map;
            }
        });
    }

    @Override
    public Map<String, Integer> getOrgCommentMap(Date startTime, Date endTime, List<Long> TianxiaoOrgIds,
                                                 Integer userRole) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        param.put("userRole", userRole);
        param.put("startTime", startTime);
        param.put("endTime", endTime);

        String sql = "SELECT count(id) count, DATE_FORMAT(create_time,'%y-%m-%d') 'createTime' FROM tts.org_lesson_comment where "
                + "user_role = :userRole and is_system = 0 and org_id in (:orgIds) and create_time>=:startTime and create_time<=:endTime";
        sql += " group by MONTH(create_time),DAY(create_time)";

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<String, Integer>>() {

            @Override
            public Map<String, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<String, Integer> map = new HashMap<>();
                while (rs.next()) {
                    String date = rs.getString("createTime");
                    Integer count = rs.getInt("count");
                    map.put(date, count);
                }
                return map;
            }
        });
    }
}