package com.baijia.tianxiao.dal.org.dao.impl;

import com.baijia.tianxiao.constants.CourseType;
import com.baijia.tianxiao.constants.PayStatus;
import com.baijia.tianxiao.dal.org.dao.CoursePurchaseDao;
import com.baijia.tianxiao.dal.org.dao.dto.CoursePurchaseDto;
import com.baijia.tianxiao.dal.org.po.CoursePurchase;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.sqlbuilder.util.Expressions;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.MapBatchQueryTemplate;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.stereotype.Repository;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

@Repository
public class CoursePurchaseDaoImpl extends JdbcTemplateDaoSupport<CoursePurchase> implements CoursePurchaseDao {

    public CoursePurchaseDaoImpl() {
        super(CoursePurchase.class);
    }

    @Override
    public List<CoursePurchase> queryByLastUpdateTime(Long orgId, Integer status, Date lastUpdateTime, int maxSize,
                                                      String... propes) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder(propes);
        if (orgId != null && orgId > 0) {
            builder.eq("orgId", orgId);
        } else {
            builder.gt("orgId", 0);
        }
        if (status != null) {
            builder.eq("status", status);
        }
        builder.gt("updateTime", lastUpdateTime);

        if (maxSize > 0) {
            builder.setMaxSize(maxSize);
        }

        builder.asc("updateTime");
        return queryList(builder);
    }

    @Override
    public Map<Long, String> queryStudentName(Collection<Long> userIds) {

        MapBatchQueryTemplate<Long, Long, String> queryTemplate = new MapBatchQueryTemplate<>();
        return queryTemplate.batchQuery(userIds, new BatchQueryCallback<Long, Map<Long, String>>() {
            @Override
            public Map<Long, String> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder("userId", "userName");
                builder.in("userId", querySet);
                List<CoursePurchase> results = queryList(builder);
                return CollectorUtil.collectMap(results, "getUserId", "getUserName");
            }
        });

    }

    @Override
    public List<Long> getLatestdPurchaseCourseIdsList(Long orgId) {
        /**
         * 用于报名时查找最近课程的ID.根据course_purchase表记录更新的时间排序。
         *
         */
        String querySql =
                "select distinct course_id from pay.course_purchase where organization_id = :orgId and course_type = 4 order by update_time desc";
        Map<String, Object> paramMap = Maps.newHashMap();
        paramMap.put("orgId", orgId);
        List<Long> list = this.getNamedJdbcTemplate().queryForList(querySql, paramMap, Long.class);
        return list;

    }

    @Override
    public List<CoursePurchase> getCoursePurchaseList(Long orgId, Long userId, Long parentPurchaseId,
                                                      Integer courseType, Integer status, String... queryProps) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder(queryProps);
        builder.eq("orgId", orgId);
        builder.eq("userId", userId);
        builder.eq("parentPurchaseId", parentPurchaseId);
        builder.eq("courseType", courseType);
        builder.eq("status", status);
        return queryList(builder);
    }

    @Override
    public List<CoursePurchase> getCousePurchaseListByMinId(Long minId, Date lastDate, int maxSize, Integer courseType, Integer parentPurchaseType,
                                                            Integer status, String... queryProps) {
        Preconditions.checkArgument(maxSize > 0, "maxSize is illegal");
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder(queryProps);
        if (minId == null || minId == 0) {
            if (lastDate == null) {
                throw new IllegalArgumentException("must have minId or lastDate");
            }
            builder.gt("updateTime", lastDate);
        } else {
            builder.gt("id", minId);
        }
        if (parentPurchaseType != null) {
            //非天校机构支付的订单
            builder.ne("parent_purchase_type", parentPurchaseType.intValue());
        }
        builder.eq("parentPurchaseId", 0);
        builder.eq("courseType", courseType);
        builder.eq("status", status);
        builder.asc("createTime");
        builder.setMaxSize(maxSize);
        return queryList(builder);
    }

    @Override
    public List<CoursePurchase> getCoursePurchaseList(Long orgId, Long userId, Integer status, String... queryPrps) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder(queryPrps);
        builder.eq("orgId", orgId);
        builder.eq("userId", userId);
        builder.eq("status", status);
        return queryList(builder);
    }

    @Override
    public List<CoursePurchase> getCoursePurchaseList(Long orgId, Integer status, String... queryPrps) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder(queryPrps);
        builder.eq("orgId", orgId);
        builder.eq("status", status);
        return queryList(builder);
    }

    @Override
    public List<CoursePurchase> getCoursePurchaseList(Date updateTime, Integer status, List<Integer> courseTypes,
                                                      PageDto page, String... queryPrps) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder(queryPrps);
        builder.gt("updateTime", updateTime);
        builder.in("courseType", courseTypes);
        builder.eq("status", status);
        builder.gt("orgId", 0);
        builder.eq("parentPurchaseId", 0);
        builder.desc("updateTime");
        builder.setPage(page);
        return queryList(builder);
    }

    @Override
    public int getTotalOrgPurchase(Long orgId) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("orgId", orgId);
        builder.eq("status", 1);
        Integer count = queryForObject(builder, Integer.class);
        if (count == null) {
            return 0;
        }
        return count;
    }

    @Override
    public int getOrgPuchaseByTime(Long orgId, Date startDate, Date endDate) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("orgId", orgId);
        builder.eq("status", 1);
        builder.ge("createTime", startDate);
        builder.le("createTime", endDate);
        return queryForObject(builder, Integer.class);
    }

    @Override
    public int getCourseTotalCount(Long orgId, Long courseNum) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("orgId", orgId);
        builder.eq("status", 1);
        builder.eq("courseNumber", courseNum);
        Integer count = queryForObject(builder, Integer.class);
        if (count == null) {
            return 0;
        }
        return count;
    }

    @Override
    public int getCourseTotalCount(Long courseNum) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("status", 1);
        builder.eq("courseNumber", courseNum);
        Integer count = queryForObject(builder, Integer.class);
        if (count == null) {
            return 0;
        }
        return count;
    }

    @Override
    public int getCourseCountByTime(Long orgId, Long courseNum, Date startDate, Date endDate) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("orgId", orgId);
        builder.eq("status", 1);
        builder.eq("courseNumber", courseNum);
        builder.ge("createTime", startDate);
        builder.le("createTime", endDate);
        Integer count = queryForObject(builder, Integer.class);
        if (count == null) {
            return 0;
        }
        return count;
    }

    @Override
    public int getCourseCountByTime(Long orgId, Date startDate, Date endDate) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("orgId", orgId);
        builder.eq("status", 1);
        builder.ge("createTime", startDate);
        builder.le("createTime", endDate);
        Integer count = queryForObject(builder, Integer.class);
        if (count == null) {
            return 0;
        }
        return count;
    }

    @Override
    public List<CoursePurchase> getUnSuccessList4CourseIdUserIdOrgId(int start, int maxSize) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder();
        builder.gt("orgId", 0);
        builder.ne("status", 1);
        builder.asc("id");
        builder.setMaxSize(maxSize);
        builder.setStart(start);
        builder.groupByNames("orgId", "courseId", "userId");
        return queryList(builder);

    }

    @Override
    public List<CoursePurchase> getSuccessList(Collection<List<Object>> groupValues) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder("orgId", "courseId", "userId");
        builder.tupleIn(Lists.newArrayList("orgId", "courseId", "userId"), groupValues);
        builder.eq("status", 1);
        return queryList(builder);
    }

    @Override
    public Map<Long, CoursePurchaseDto> getIncomeTotal(Date startTime, Date endTime, List<Long> TianxiaoOrgIds) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgIds", TianxiaoOrgIds);
        String sql = "select organization_id as org_id, count(id) count, sum(except_coupon_money) pay_money_sum from pay.course_purchase "
            + "where status = 1 and organization_id in (:orgIds) and except_coupon_money != 0 ";

        if (startTime != null && endTime != null) {
            sql += " AND create_time between :startTime and :endTime ";
            param.put("startTime", startTime);
            param.put("endTime", endTime);
        }
        sql += " group by organization_id";
        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Long, CoursePurchaseDto>>() {

            @Override
            public Map<Long, CoursePurchaseDto> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Long, CoursePurchaseDto> map = new HashMap<>();
                while (rs.next()) {
                    CoursePurchaseDto coursePurchaseDto = new CoursePurchaseDto();
                    Long orgId = rs.getLong("org_id");
                    Integer count = rs.getInt("count");
                    Double payMoneySum = rs.getDouble("pay_money_sum");
                    coursePurchaseDto.setCount(count);
                    coursePurchaseDto.setPayMoneySum(payMoneySum);
                    map.put(orgId, coursePurchaseDto);
                }
                return map;
            }
        });

    }

    @Override
    public List<CoursePurchase> getByOrgIds(Date startTime, Date endTime, List<Long> orgIds, PageDto pageDto) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder();
        builder.eq("courseType", CourseType.ORG_COURSE.getCode());
        builder.eq("status", PayStatus.SUCESS.getCode());
        builder.in("orgId", orgIds);
        builder.ge("createTime", startTime);
        builder.le("createTime", endTime);
        builder.desc("createTime");
        builder.setPage(pageDto);
        return queryList(builder);
    }

    /* (non-Javadoc)
     * @see com.baijia.tianxiao.dal.org.dao.CoursePurchaseDao#getCountByOrgIds(java.util.Date, java.util.Date, java.util.List)
     */

    @Override
    public Integer getCountByOrgIds(Date startTime, Date endTime, List<Long> orgIds) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("courseType", CourseType.ORG_COURSE.getCode());
        builder.eq("status", PayStatus.SUCESS.getCode());
        builder.in("orgId", orgIds);
        builder.ge("createTime", startTime);
        builder.le("createTime", endTime);
        return queryForObject(builder, Integer.class);
    }

    @Override
    public CoursePurchase getByPurcahseId(Long purchaseId) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder();
        builder.eq("purchaseId", purchaseId);
        builder.setMaxSize(1);
        return uniqueResult(builder);
    }


    @Override
    public CoursePurchase getByPurcahseIdCourse(Long parentPurchaseId, Long courseId) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder();
        builder.or(Expressions.eq("parentPurchaseId", parentPurchaseId),
                Expressions.eq("purchaseId",parentPurchaseId));
        
        builder.eq("courseId", courseId);
        builder.setMaxSize(1);
        return uniqueResult(builder);
    }

    @Override
    public List<CoursePurchase> getByPurcahseIds(Set<Long> purchaseIds) {
        SingleSqlBuilder<CoursePurchase> builder = createSqlBuilder();
        builder.or(Expressions.in("parentPurchaseId", purchaseIds),
            Expressions.in("purchaseId",purchaseIds));
        return queryList(builder);
    }
}
