package com.baijia.tianxiao.dal.activity.dao.article.impl;

import static com.mongodb.client.model.Filters.eq;
import static com.mongodb.client.model.Filters.in;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.bson.Document;
import org.bson.types.ObjectId;
import org.springframework.stereotype.Repository;

import com.baijia.commons.lang.utils.collection.CollectionUtils;
import com.baijia.tianxiao.dal.activity.dao.article.ArticleBaseInfoDao;
import com.baijia.tianxiao.dal.activity.mongo.MongoTemplate;
import com.baijia.tianxiao.dal.activity.po.ArticleBaseInfo;
import com.baijia.tianxiao.dal.activity.po.ArticleDetail;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.google.common.collect.Lists;
import com.google.gson.Gson;
import com.mongodb.BasicDBObject;
import com.mongodb.Block;
import com.mongodb.DBObject;
import com.mongodb.client.FindIterable;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.model.BulkWriteOptions;
import com.mongodb.client.model.InsertOneModel;
import com.mongodb.client.model.WriteModel;
import com.mongodb.util.JSON;

import lombok.extern.slf4j.Slf4j;

/**
 * Created by liuxp on 16/3/23.
 */
@Repository
@Slf4j
public class ArticleBaseInfoDaoImpl implements ArticleBaseInfoDao {

    private final static String COLLECTION = "article";

    @Override
    public List<ArticleBaseInfo> findArticleBaseInfoListByPage(int type, PageDto pageDto) {
        long totalCount = MongoTemplate.getConnection(COLLECTION).count();
        int first = (pageDto.getPageNum() - 1) * pageDto.getPageSize();
        BasicDBObject query = new BasicDBObject();
        query.put("category", type);
        query.put("articleImg", new Document("$exists", 1));
        FindIterable<Document> iterable = MongoTemplate.getConnection(COLLECTION).find(query).skip(first)
            .sort(new Document("pushTime", -1)).limit(pageDto.getPageSize());
        pageDto.setCount((int) totalCount);
        final List<ArticleBaseInfo> list = new ArrayList<>();
        iterable.forEach(new Block<Document>() {
            @Override
            public void apply(final Document document) {
                ArticleBaseInfo baseInfo = new ArticleBaseInfo();
                parseDocument(document, baseInfo);
                list.add(baseInfo);
                return;
            }
        });
        return list;
    }

    @Override
    public Map<String, ArticleDetail> findArticleBaseInfoListByIds(List<String> ids) {
        if (ids == null || ids.size() < 1) {
            return Collections.emptyMap();
        }

        List<ObjectId> objectIds = CollectionUtils.extractList(ids, new CollectionUtils.Extracter<ObjectId, String>() {
            @Override
            public ObjectId extract(String s) {
                log.info("[ObjectId] ObjectId===" + s);
                return new ObjectId(s);
            }
        });

        FindIterable<Document> iterable = MongoTemplate.getConnection(COLLECTION).find(in("_id", objectIds));

        final List<ArticleDetail> list = new ArrayList<>();
        iterable.forEach(new Block<Document>() {
            @Override
            public void apply(final Document document) {
                ArticleDetail detail = new ArticleDetail();
                parseDocument(document, detail);
                detail.setContent(document.getString("content"));
                list.add(detail);
                return;
            }
        });
        Map<String, ArticleDetail> map = new HashMap<>();
        if (list != null && list.size() > 0) {
            map = CollectionUtils.extractMap(list, new CollectionUtils.Extracter<String, ArticleDetail>() {
                @Override
                public String extract(ArticleDetail articleBaseInfo) {
                    return articleBaseInfo.getId();
                }
            });
        }
        return map;
    }

    @Override
    public ArticleDetail findArticleDetailById(String id) {
        final ArticleDetail detail = new ArticleDetail();
        FindIterable<Document> iterable = MongoTemplate.getConnection(COLLECTION).find(eq("_id", new ObjectId(id)));
        iterable.forEach(new Block<Document>() {
            @Override
            public void apply(final Document document) {
                parseDocument(document, detail);
                detail.setContent(document.getString("content"));
            }
        });
        return detail;
    }

    @Override
    public ArticleBaseInfo findArticleBaseInfoById(String id) {
        FindIterable<Document> iterable = MongoTemplate.getConnection(COLLECTION).find(eq("_id", new ObjectId(id)));

        final ArticleBaseInfo baseInfo = new ArticleBaseInfo();

        iterable.forEach(new Block<Document>() {
            @Override
            public void apply(final Document document) {
                if (document != null) {
                    parseDocument(document, baseInfo);
                }
            }
        });
        return baseInfo;
    }

    private ArticleBaseInfo parseDocument(Document document, ArticleBaseInfo baseInfo) {
        baseInfo.setId(document.getObjectId("_id").toHexString());
        baseInfo.setAbstractContent(document.getString("abstractContent"));
        baseInfo.setArticleImg(document.getString("articleImg"));
        baseInfo.setArticleTitle(document.getString("articleTitle"));
        baseInfo.setCategory(document.getInteger("category"));
        baseInfo.setArticleUrl(document.getString("articleUrl"));
        baseInfo.setPushTime(document.getString("pushTime"));
        try {
            baseInfo.setThumbUpCount(String.valueOf(document.getInteger("thumbUpCount")));
        } catch (ClassCastException e) {
            baseInfo.setThumbUpCount(document.getString("thumbUpCount"));
        }
        try {
            baseInfo.setVisitCount(String.valueOf(document.getInteger("visitCount")));
        } catch (ClassCastException e) {
            baseInfo.setVisitCount(document.getString("visitCount"));
        }
        baseInfo.setThumbUpRate(document.getString("thumbUpRate"));
        baseInfo.setAuthor(document.getString("author"));
        return baseInfo;
    }

    /**
     * just update the articleImg
     */
    @Override
    public void updateArticle(ArticleBaseInfo abi) {
        MongoTemplate.getConnection(COLLECTION).updateOne(new Document("_id", new ObjectId(abi.getId())),
            new Document("$set", new Document("articleImg", abi.getArticleImg())));
    }

    /**
     * 批量插入文章
     */
    @SuppressWarnings("unchecked")
    public void insertArticles(List<ArticleBaseInfo> articles) {
        List<WriteModel<Document>> insert = Lists.newArrayList();
        Gson gson = new Gson();
        for (Object obj : articles) {
            DBObject parse = (DBObject) JSON.parse(gson.toJson(obj));
            WriteModel<Document> e = new InsertOneModel<Document>(new Document(parse.toMap()));
            insert.add(e);
        }
        MongoCollection<Document> connection = MongoTemplate.getConnection(COLLECTION);
        BulkWriteOptions options = new BulkWriteOptions();
        options.ordered(false);
        try {
            connection.bulkWrite(insert, options);
        } catch (Exception e) {
            log.info("encount a error : {} ", e.getCause());
        }
    }
}
