/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.dal.user.dao.impl;

import com.baijia.tianxiao.constants.TianXiaoConstant;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.dal.user.dao.StudentDao;
import com.baijia.tianxiao.dal.user.po.Student;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;

import com.google.common.base.Function;
import com.google.common.collect.Maps;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Repository;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

@Repository
@Slf4j
public class StudentDaoImpl extends JdbcTemplateDaoSupport<Student> implements StudentDao {

    public StudentDaoImpl() {
        super(Student.class);
    }

    @Override
    public Student getByUserId(@NonNull Long userId, String...queryProp) {
        SingleSqlBuilder<Student> builder = createSqlBuilder();
        builder.select(queryProp);
        builder.eq("userId", userId);
        return uniqueResult(builder);

    }

    @Override
    public List<Student> getByUserIds(Collection<Long> userIds, final String...queryProp) {
        BatchQueryTemplate<Long, List<Student>> queryTemplate = new ListBatchQueryTemplate<>();
        List<Student> teachers = queryTemplate.batchQuery(userIds, new BatchQueryCallback<Long, List<Student>>() {
            @Override
            public List<Student> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<Student> builder = createSqlBuilder(queryProp);
                builder.in("userId", querySet);
                return queryList(builder);
            }
        });
        return teachers;

    }

    @Override
    public Map<Long, String> getStudentNameMap(@NonNull Collection<Long> studentIds) {
        if (studentIds.isEmpty()) {
            log.warn("teacher ids is empty.");
            return Collections.emptyMap();
        }

        BatchQueryTemplate<Long, List<Student>> queryTemplate = new ListBatchQueryTemplate<>();
        List<Student> students = queryTemplate.batchQuery(studentIds, new BatchQueryCallback<Long, List<Student>>() {
            @Override
            public List<Student> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<Student> builder = createSqlBuilder();
                builder.select("realName");
                builder.select("nickName");
                builder.select("userId");
                builder.in("userId", querySet);
                return queryList(builder);
            }
        });

        if (CollectionUtils.isEmpty(students)) {
            return Collections.emptyMap();
        }
        Map<Long, String> studentRealNameMap = Maps.newHashMap();
        for (Student student : students) {
            if (StringUtils.isNoneBlank(student.getNickName())) {
                studentRealNameMap.put(student.getUserId(), student.getNickName());
            } else if (StringUtils.isNoneBlank(student.getRealName())) {
                studentRealNameMap.put(student.getUserId(), student.getRealName());
            } else {
                studentRealNameMap.put(student.getUserId(), TianXiaoConstant.APPOINTMENT_STUDENT_NAME);
            }
        }
        return studentRealNameMap;

    }

    @Override
    public Map<Long, Integer> getAvatarsMap(Collection<Long> userIds) {
        BatchQueryTemplate<Long, List<Student>> batch = new ListBatchQueryTemplate<>();
        List<Student> students = batch.batchQuery(userIds, new BatchQueryCallback<Long, List<Student>>() {
            @Override
            public List<Student> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<Student> builder = createSqlBuilder("userId", "avatar");
                builder.in("userId", querySet);
                builder.isNotNull("avatar");
                return queryList(builder);
            }
        });
        return CollectorUtil.collectMap(students, new Function<Student, Long>() {
            @Override
            public Long apply(Student arg0) {
                return arg0.getUserId();
            }
        }, new Function<Student, Integer>() {
            @Override
            public Integer apply(Student arg0) {
                return arg0.getAvatar();
            }
        });
    }
}
