package com.baijia.tianxiao.dal.org.dao.impl;

import com.baijia.tianxiao.dal.org.dao.OrgCourseTeacherDao;
import com.baijia.tianxiao.dal.org.po.OrgCourseTeacher;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.baijia.tianxiao.util.query.MapBatchQueryTemplate;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * 
 * @title OrgCourseTeacherDaoImpl
 * @desc TODO
 * @author shizuwei
 * @date 2015年12月4日
 * @version 1.0
 */
@Repository
public class OrgCourseTeacherDaoImpl extends JdbcTemplateDaoSupport<OrgCourseTeacher> implements OrgCourseTeacherDao {

    public OrgCourseTeacherDaoImpl() {

        super(OrgCourseTeacher.class);

    }

    @Override
    public List<OrgCourseTeacher> getTeacher(Long orgId, Long courseId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        SingleSqlBuilder<OrgCourseTeacher> builder = this.createSqlBuilder();
        builder.eq("orgCourseId", courseId);
        builder.asc("createTime");
        return this.queryList(builder);
    }

    @Override
    public Integer getTeacherCourseCount(Long teacherId) {
        Preconditions.checkArgument(teacherId != null && teacherId > 0, "teacherId is illegal");
        SingleSqlBuilder<OrgCourseTeacher> builder = this.createSqlBuilder("id");
        builder.eq("userId", teacherId);
        builder.count("id");
        return this.queryForObject(builder, Integer.class);
    }

    @Override
    public List<OrgCourseTeacher> getCourseTeachers(Long userId) {
        Preconditions.checkArgument(userId != null && userId > 0, "userId is illegal");
        SingleSqlBuilder<OrgCourseTeacher> builder = this.createSqlBuilder();
        builder.eq("userId", userId);
        return this.queryList(builder);

    }

    @Override
    public List<String> getTeacherNames(Long courseId) {
        String sql =
            "select ct.nickname, ct.realname from cdb.org_course_teacher ot,cdb.teacher ct where ot.org_course_id=:orgCourseId and ot.user_id=ct.user_id";
        Map<String, Long> paramMap = Maps.newHashMap();
        paramMap.put("orgCourseId", courseId);
        List<String> names = this.getNamedJdbcTemplate().query(sql, paramMap, new RowMapper<String>() {
            @Override
            public String mapRow(ResultSet rs, int rowNum) throws SQLException {
                String nickname = rs.getString("nickname");
                String realname = rs.getString("realname");
                return StringUtils.isNotBlank(realname) ? realname : (StringUtils.isNotBlank(nickname) ? nickname : "");
            }
        });
        return names;
    }

    @Override
    public Map<Long, List<String>> getAllTeacherNames(Collection<Long> courseIds) { 
        String sql =
        		"select ot.org_course_id, ct.nickname, ct.realname from cdb.org_course_teacher ot,cdb.teacher ct where ot.org_course_id in (:orgCourseIds)  and ot.user_id=ct.user_id";
        Map<String, Collection> paramMap = Maps.newHashMap();
        paramMap.put("orgCourseIds", courseIds);
        
        final Map<Long, List<String>> map = Maps.newHashMap();
        this.getNamedJdbcTemplate().query(sql, paramMap, new RowCallbackHandler() {
        	@Override
			public void processRow(ResultSet rs) throws SQLException {
        		Long orgCourseId = rs.getLong("org_course_id");
				String name = "";
        		if(StringUtils.isNotBlank(rs.getString("realname"))) {
        				name = rs.getString("realname");
        		} else if (StringUtils.isNotBlank(rs.getString("nickname"))){
        			   name = rs.getString("nickname");
        		}		
				
				if (map.containsKey(orgCourseId)) {
					map.get(orgCourseId).add(name);
				} else {
					List<String> list = Lists.newArrayList();
					list.add(name);
					map.put(orgCourseId, list);
				}
        	};
        	
        });
        return map;
    }

    @Override
    public List<OrgCourseTeacher> getOrgCourseTeacher(Collection<Long> courseIds, PageDto pageDto) {

        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyList();
        }
        SingleSqlBuilder<OrgCourseTeacher> builder = createSqlBuilder();
        builder.in("orgCourseId", courseIds);
        builder.setPage(pageDto);
        return queryList(builder);
    }

    @Override
    public int delTeachersFromCourse(Long courseId, Collection<Long> teacherIds) {
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        SingleSqlBuilder<OrgCourseTeacher> builder = createSqlBuilder();
        builder.eq("orgCourseId", courseId);
        if (CollectionUtils.isNotEmpty(teacherIds)) {
            builder.in("userId", teacherIds);
        }
        return delete(builder);

    }

    @Override
    public Map<Long, List<Long>> getTeacherMap(Collection<Long> courseIds) {
        BatchQueryTemplate<Long, List<OrgCourseTeacher>> queryTemplate = new ListBatchQueryTemplate<>();
        List<OrgCourseTeacher> list =
            queryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, List<OrgCourseTeacher>>() {
                @Override
                public List<OrgCourseTeacher> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgCourseTeacher> builder = createSqlBuilder();
                    builder.in("orgCourseId", querySet);
                    return queryList(builder);
                }
            });
        Map<Long, List<Long>> teacherIdMap = CollectorUtil.group(list, new Function<OrgCourseTeacher, Long>() {
            @Override
            public Long apply(OrgCourseTeacher arg0) {
                return arg0.getOrgCourseId();
            }
        }, new Function<OrgCourseTeacher, Long>() {
            @Override
            public Long apply(OrgCourseTeacher arg0) {
                return arg0.getUserId();
            }
        });
        return teacherIdMap;
    }

    @Override
    public List<Long> getTeacherIdsByCourseId(Long courseId) {
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        SingleSqlBuilder<OrgCourseTeacher> builder = createSqlBuilder("userId");
        builder.eq("orgCourseId", courseId);
        return queryForList(builder, Long.class);
    }

    @Override
    public List<Long> getCourseIdByTeacherId(Long userId) {
        Preconditions.checkArgument(userId != null && userId > 0, "userId is illegal");
        SingleSqlBuilder<OrgCourseTeacher> builder = createSqlBuilder("orgCourseId");
        builder.eq("userId", userId);
        return queryForList(builder, Long.class);
    }

    @Override
    public Map<Long, Integer> getTeacherCount(Collection<Long> courseIds) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Maps.newHashMap();
        }
        return new MapBatchQueryTemplate<Long, Long, Integer>().batchQuery(courseIds,
            new BatchQueryCallback<Long, Map<Long, Integer>>() {
                @Override
                public Map<Long, Integer> doQuery(Collection<Long> querySet) {
                    SingleSqlBuilder<OrgCourseTeacher> builder = createSqlBuilder("orgCourseId");
                    builder.count("id", "num");
                    builder.in("orgCourseId", querySet);
                    builder.group("orgCourseId");
                    final Map<Long, Integer> result = Maps.newHashMap();
                    getNamedJdbcTemplate().query(builder.toSql(), builder.collectConditionValue(),
                        new RowCallbackHandler() {
                        @Override
                        public void processRow(ResultSet rs) throws SQLException {
                            result.put(rs.getLong("orgCourseId"), rs.getInt("num"));
                        }
                    });
                    return result;
                }
            });
    }

    @Override
    public List<Long> getTeacherCourseIds(Long teacherId, PageDto pageDto) {
        Preconditions.checkArgument(teacherId != null, "teacherId is illegal");
        SingleSqlBuilder<OrgCourseTeacher> builder = createSqlBuilder("orgCourseId");
        builder.eq("userId", teacherId);
        builder.setPage(pageDto);
        builder.desc("createTime");
        return queryForList(builder, Long.class);
    }

    @Override
    public List<Long> getCourseIdsByteacherIds(Collection<Long> teacherIds) {
        if (CollectionUtils.isEmpty(teacherIds)) {
            return Lists.newArrayList();
        }
        BatchQueryTemplate<Long, List<Long>> queryTemplate = new ListBatchQueryTemplate<>();
        return queryTemplate.batchQuery(teacherIds, new BatchQueryCallback<Long, List<Long>>() {
            @Override
            public List<Long> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgCourseTeacher> builder = createSqlBuilder("orgCourseId");
                builder.in("userId", querySet);
                return queryForList(builder, Long.class);
            }
        });
    }

	@Override
	public List<Long> getOrgTeacherCourseIds(Long orgId, Long userId) {
		Preconditions.checkArgument(userId != null, "teacherId is illegal");
        SingleSqlBuilder<OrgCourseTeacher> builder = createSqlBuilder("orgCourseId");
        builder.eq("userId", userId);
        return queryForList(builder, Long.class);
	}

}
