package com.baijia.tianxiao.dal.org.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.support.lob.DefaultLobHandler;
import org.springframework.jdbc.support.lob.LobHandler;
import org.springframework.stereotype.Repository;

import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.dal.util.SerialNumberUtil;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.bean.impl.MatchMode;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.baijia.tianxiao.util.query.MapBatchQueryTemplate;
import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

@Slf4j
@Repository
public class OrgCourseDaoImpl extends JdbcTemplateDaoSupport<OrgCourse> implements OrgCourseDao {

    public OrgCourseDaoImpl() {
        super(OrgCourse.class);
    }

    @Override
    public List<OrgCourse> getCoursesByOrgNumber(Long orgNumber, String key, Integer status, PageDto page,
        String...queryProp) {

        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder(queryProp);
        builder.eq("orgNumber", orgNumber);
        if (StringUtils.isNotEmpty(key)) {
            builder.like("name", key, MatchMode.ANYWHERE);
        }
        if (status != null) {
            builder.eq("status", status);
        }
        if (page != null) {
            builder.setPage(page);
        }

        builder.eq("isDel", 0);
        builder.desc("beginTime");
        return queryList(builder);
    }

    @Override
    public List<OrgCourse> getCoursesByOrgNumberAndName(Long orgNumber, String key, Integer status, PageDto page) {

        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        if (StringUtils.isNotEmpty(key)) {
            builder.eq("name", key);
        }
        if (status != null) {
            builder.eq("status", status);
        }
        if (page != null) {
            builder.setPage(page);
        }

        builder.eq("isDel", 0);
        builder.desc("beginTime");
        return queryList(builder);
    }

    @Override
    public List<OrgCourse> getCoursesByOrgNumberAndNames(Long orgNumber, List<String> courseNames, Integer status,
        PageDto page) {

        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        if (CollectionUtils.isNotEmpty(courseNames)) {
            builder.in("name", courseNames);
        }
        if (status != null) {
            builder.eq("status", status);
        }
        if (page != null) {
            builder.setPage(page);
        }

        builder.eq("isDel", 0);
        builder.desc("beginTime");
        return queryList(builder);
    }

    @Override
    public List<OrgCourse> getCourseList(Long orgNumber, Collection<Long> courseIds, String key, Integer status,
        Date startTime, Date endTime, PageDto page, String...queryProp) {
        OrgCourseListParam param = new OrgCourseListParam();
        param.setKey(key);
        param.setStatus(status);
        param.setStartTime(startTime);
        param.setEndTime(endTime);
        param.setPage(page);
        return getCourseList(orgNumber, courseIds, param, queryProp);
    }

    @Override
    public List<OrgCourse> getCourseList(Integer cascadeId, Long orgNumber, Collection<Long> courseIds, String key,
        Integer status, Date startTime, Date endTime, PageDto page, String...queryProp) {
        OrgCourseListParam param = new OrgCourseListParam();
        param.setKey(key);
        param.setStatus(status);
        param.setStartTime(startTime);
        param.setEndTime(endTime);
        param.setPage(page);
        // param在这里new，判断子帐号逻辑也得放这里
        if (cascadeId != null) {
            param.setCascadeId(cascadeId);
        }
        return getCourseList(orgNumber, courseIds, param, queryProp);
    }

    @Override
    public List<Long> getCourseIds(Long orgNumber, Integer status) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder("id");
        builder.eq("orgNumber", orgNumber);

        if (status != null) {
            builder.eq("status", status);
        }
        builder.eq("isDel", 0);
        List<OrgCourse> courses = queryList(builder);
        List<Long> resultList = Lists.newArrayList();
        for (OrgCourse course : courses) {
            resultList.add(course.getId());
        }
        return resultList;
    }

    @Override
    public List<Long> getCourseList(Long orgNumber, Collection<Long> courseIds, String key, Integer status) {
        if (courseIds != null && CollectionUtils.isEmpty(courseIds)) {
            return Lists.newArrayList();
        }

        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder("id");
        builder.eq("orgNumber", orgNumber);
        if (StringUtils.isNotEmpty(key)) {
            builder.like("name", key, MatchMode.ANYWHERE);
        }
        if (courseIds != null) {
            builder.in("id", courseIds);
        }
        if (status != null) {
            builder.eq("status", status);
        }
        builder.eq("isDel", 0);
        List<OrgCourse> courses = queryList(builder);
        List<Long> resultList = Lists.newArrayList();
        for (OrgCourse course : courses) {
            resultList.add(course.getId());
        }
        return resultList;
    }

    @Override
    public List<OrgCourse> getCoursesByIds(Long orgNumber, Collection<Long> courseIds, String key, Integer status) {
        if (courseIds != null && CollectionUtils.isEmpty(courseIds)) {
            return Lists.newArrayList();
        }

        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        if (StringUtils.isNotEmpty(key)) {
            builder.like("name", key, MatchMode.ANYWHERE);
        }
        if (courseIds != null) {
            builder.in("cascadeId", courseIds);
        }
        if (status != null) {
            builder.eq("status", status);
        }
        builder.eq("isDel", 0);
        return queryList(builder);
    }

    @Override
    public List<Long> getCourseList(Long orgNumber, Collection<Long> courseIds, Double lPrice, Double hPrice,
        Collection<Long> subjectIds, String key, Integer status) {
        if (courseIds != null && CollectionUtils.isEmpty(courseIds)) {
            return Lists.newArrayList();
        }
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder("id");
        builder.eq("orgNumber", orgNumber);
        if (StringUtils.isNotEmpty(key)) {
            builder.like("name", key, MatchMode.ANYWHERE);
        }
        if (courseIds != null) {
            builder.in("id", courseIds);
        }
        if (status != null) {
            builder.eq("status", status);
        }

        if (lPrice != null) {
            builder.ge("price", lPrice);
        }

        if (hPrice != null) {
            builder.le("price", hPrice);
        }

        if (subjectIds != null) {
            builder.in("subjectId", subjectIds);
        }
        builder.desc("beginTime");
        builder.eq("isDel", 0);
        List<OrgCourse> courses = queryList(builder);
        List<Long> resultList = Lists.newArrayList();
        for (OrgCourse course : courses) {
            resultList.add(course.getId());
        }
        return resultList;
    }

    @Override
    public List<OrgCourse> getCourseList(Long orgNumber, Collection<Long> courseIds, OrgCourseListParam param,
        String...queryProp) {
        log.debug("*********orgNumber:{},courseIds:{},param={}", orgNumber, courseIds, param);
        if (courseIds != null && CollectionUtils.isEmpty(courseIds)) {
            return Lists.newArrayList();
        }
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder(queryProp);
        builder.eq("orgNumber", orgNumber);
        if (StringUtils.isNotEmpty(param.getKey())) {
            builder.like("name", param.getKey(), MatchMode.ANYWHERE);
        }
        if (param.getCascadeId() != null && param.getCascadeId() > 0) {
            builder.eq("cascadeId", param.getCascadeId());
        }
        if (CollectionUtils.isNotEmpty(courseIds)) {
            builder.in("id", courseIds);
        }
        if (param.getStatus() != null) {
            builder.eq("status", param.getStatus());
        }
        if (param.getStartTime() != null) {
            builder.ge("createTime", param.getStartTime());
        }
        if (param.getEndTime() != null) {
            builder.le("createTime", param.getEndTime());
        }
        if (param.getPage() != null) {
            builder.setPage(param.getPage());
        }
        if (param.getQueryCondition() != null) {
            builder.add(param.getQueryCondition());
        }
        if (StringUtils.isNoneBlank(param.getOrderField(), param.getOrderType())) {
            if (param.getOrderType().equals("DESC")) {
                builder.desc(param.getOrderField(), param.getOrderField().equals("name"));
            } else if (param.getOrderType().equals("ASC")) {
                builder.asc(param.getOrderField(), param.getOrderField().equals("name"));
            }
        }
        builder.eq("isDel", param.getIsDel());
        log.debug("course list－－－－－－－－－sql={},param={}", builder.toSql(), builder.collectConditionValue());
        return queryList(builder);
    }

    @Override
    public Map<Long, OrgCourse> getOrgCourseMap(Collection<Long> courseIds) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyMap();
        }

        List<OrgCourse> courses = this.getByIds(courseIds);
        if (CollectionUtils.isNotEmpty(courses)) {
            Map<Long, OrgCourse> result = Maps.newHashMap();
            for (OrgCourse course : courses) {
                result.put(course.getId(), course);
            }
            return result;
        }
        return Collections.emptyMap();
    }

    @Override
    public List<Long> getCourseIdsByOrgNumberAndCourseName(Long orgNumber, String name, Integer status) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.select("id");
        builder.eq("orgNumber", orgNumber);
        if (status != null) {
            builder.eq("status", status);
        }
        if (StringUtils.isNotBlank(name)) {
            builder.like("name", name, MatchMode.ANYWHERE);
        }
        builder.eq("isDel", 0);
        return queryForList(builder, Long.class);
    }

    @Override
    public List<OrgCourse> getCourseByOrgNumberAndCourseName(Long orgNumber, String name, Integer status,
        Collection<Long> courseIds, Integer limit) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();

        builder.eq("orgNumber", orgNumber);
        if (status != null) {
            builder.eq("status", status);
        }
        if (StringUtils.isNotBlank(name)) {
            builder.like("name", name, MatchMode.ANYWHERE);
        }
        builder.eq("isDel", 0);
        if (courseIds != null) {
            builder.in("id", courseIds);
        }

        builder.setMaxSize(limit);
        builder.desc("createTime");
        List<OrgCourse> list = this.queryList(builder);
        log.debug("list = {}", list);
        return list;
    }

    @Override
    public OrgCourse getByCourseId(Long orgCourseId, String...queryProp) {
        return this.getById(orgCourseId, queryProp);
    }

    @Override
    public OrgCourse getByCourseNumber(Long orgCourseNumber, String...queryProp) {
        SingleSqlBuilder<OrgCourse> builder = createSqlBuilder(queryProp);
        builder.eq("number", orgCourseNumber);
        return uniqueResult(builder);
    }

    @Override
    public List<OrgCourse> getByIdsOrderByIn(Collection<Long> courseIds, Integer limit) {

        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyList();
        }
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.ne("status", 1);
        builder.in("id", courseIds);
        builder.setMaxSize(limit);
        List<OrgCourse> list = this.queryList(builder);
        log.debug("list = {}", list);
        return list;
    }

    @SuppressWarnings("deprecation")
    @Override
    public int getValidOrgCourseCount(Integer orgNumber) {
        String sql =
            "select count(*) from cdb.org_course where organization_number=? and status=0 and verify_status=1 and is_del = 0";
        return getNamedJdbcTemplate().getJdbcOperations().queryForInt(sql, new Object[] { orgNumber });
    }

    @Override
    public Map<Long, String> getCourseNameMap(Collection<Long> courseIds) {
        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyMap();
        }

        MapBatchQueryTemplate<Long, Long, String> queryTemplate = new MapBatchQueryTemplate<>();
        return queryTemplate.batchQuery(courseIds, new BatchQueryCallback<Long, Map<Long, String>>() {

            @Override
            public Map<Long, String> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgCourse> builder = createSqlBuilder("id", "name");
                builder.in("id", querySet);

                List<OrgCourse> courses = queryList(builder);

                return CollectorUtil.collectMap(courses, new Function<OrgCourse, Long>() {
                    @Override
                    public Long apply(OrgCourse from) {
                        return from.getId();
                    }
                }, new Function<OrgCourse, String>() {
                    @Override
                    public String apply(OrgCourse from) {
                        return from.getName();
                    }
                });
            }
        });
    }

    @Override
    public Map<Integer, Integer> allCourseOrgIdMap() {

        Map<String, Object> param = new HashMap<>();
        String sql =
            "select  c.id as c_course_id ,a.id as c_org_id  from cdb.org_course c,yunying.org_account a where a.number = c. organization_number";

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Integer, Integer>>() {
            @Override
            public Map<Integer, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Integer, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Integer courseId = rs.getInt("c_course_id");
                    Integer orgId = rs.getInt("c_org_id");
                    map.put(courseId, orgId);
                }
                return map;
            }
        });
    }

    @Override
    public OrgCourse getCourseByCourseNumberAndOrgNumber(Long orgNumber, Long courseNumber, String...queryProp) {
        SingleSqlBuilder<OrgCourse> builder = createSqlBuilder(queryProp);
        builder.eq("orgNumber", orgNumber);
        builder.eq("number", courseNumber);
        return uniqueResult(builder);

    }

    @Override
    public List<Integer> getOrgCourseSubjectIds(Integer orgNumber) {
        String sql = "select distinct subject_id from cdb.org_course where organization_number=? and status=0";
        List<Integer> subjectIds = this.getNamedJdbcTemplate().getJdbcOperations().query(sql,
            new Object[] { orgNumber }, new RowMapper<Integer>() {

                @Override
                public Integer mapRow(ResultSet rs, int rowNum) throws SQLException {
                    return rs.getInt("subject_id");
                }
            });
        return subjectIds;
    }

    @Override
    public List<Long> getCourseNumberByOrg(Long orgNumber) {
        String sql = "select number from cdb.org_course where organization_number=? and status=0 and is_del=0";
        List<Long> numbers = this.getNamedJdbcTemplate().getJdbcOperations().query(sql, new Object[] { orgNumber },
            new RowMapper<Long>() {

                @Override
                public Long mapRow(ResultSet rs, int rowNum) throws SQLException {
                    return rs.getLong("number");
                }
            });
        return numbers;
    }

    @Override
    public List<Long> getCourseIdsByCourseNumberAndOrgNumber(final Long orgNumber, Collection<Long> courseNumbers) {
        if (CollectionUtils.isEmpty(courseNumbers)) {
            return Lists.newArrayList();
        }
        BatchQueryTemplate<Long, List<Long>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        return batchQueryTemplate.batchQuery(courseNumbers, new BatchQueryCallback<Long, List<Long>>() {
            @Override
            public List<Long> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<OrgCourse> builder = createSqlBuilder("id");
                builder.eq("orgNumber", orgNumber);
                builder.in("number", querySet);
                builder.eq("isDel", 0);
                return queryForList(builder, Long.class);
            }
        });
    }

    @Override
    public List<OrgCourse> getAndFilterOrgCourse(Integer orgNumber, Collection<Long> excludeIds, PageDto pageDto) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        builder.eq("status", 0);
        if (excludeIds != null && !excludeIds.isEmpty()) {
            builder.notin("id", excludeIds);
        }
        builder.eq("isDel", 0);
        builder.setPage(pageDto);
        return this.queryList(builder);
    }

    @Override
    public List<OrgCourse> getAndFilterOrgCourseByIn(Integer orgNumber, Collection<Long> includeIds, PageDto pageDto) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        builder.eq("status", 0);
        builder.in("id", includeIds);
        builder.setPage(pageDto);
        builder.desc("createTime");
        builder.eq("isDel", 0);
        return this.queryList(builder);

    }

    @Override
    public List<OrgCourse> getAndFilterOrgCourseByIn(Integer orgNumber, Collection<Long> includeIds, Integer status,
        PageDto pageDto) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        if (status != null) {
            builder.eq("status", status);
        }
        builder.in("id", includeIds);
        builder.setPage(pageDto);
        builder.eq("isDel", 0);
        builder.desc("createTime");
        return this.queryList(builder);

    }

    @Override
    public int getValidOrgCourseCount(Integer orgNumber, Collection<Long> ids) {
        SingleSqlBuilder<OrgCourse> builder = createSqlBuilder();
        builder.count("id");
        builder.in("id", ids);
        builder.eq("orgNumber", orgNumber);
        builder.eq("status", 0);
        builder.eq("isDel", 0);
        return queryForObject(builder, Integer.class);
    }

    /**
     * color是以#开始的十六进制字符串 #AA33BC
     */
    @Override
    public void changeCourseColor(Integer orgNumber, Long courseId, String color) {
        OrgCourse course = this.getByCourseId(courseId);
        if (course != null) {
            course.setColor(color);
            this.update(course);
        } else {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "course not found!");
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public void changeCourseColor(Collection<OrgCourse> courses) {
        if (CollectionUtils.isNotEmpty(courses)) {
            List<Map<String, Object>> batchs = Lists.newArrayList();
            for (OrgCourse orgCourse : courses) {
                Map<String, Object> map = Maps.newHashMap();
                map.put("color", orgCourse.getColor());
                map.put("id", orgCourse.getId());
                batchs.add(map);
            }
            String sql = "update cdb.org_course set color =:color where id=:id";
            this.getNamedJdbcTemplate().batchUpdate(sql, batchs.toArray(new HashMap[courses.size()]));
        }
    }

    @Override
    public String getCourseNameById(Long orgCourseId) {

        OrgCourse course = this.getByCourseId(orgCourseId, "name");
        if (course != null) {
            return course.getName();
        }
        return null;
    }

    @Override
    public List<OrgCourse> getVisualOrgCourseByOrgNumber(Integer orgNumber, String subjectIds, int orderType) {
        // SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        // builder.eq("orgNumber", orgNumber);
        // builder.eq("status", 0);
        // builder.desc("extendStatus");
        // return this.queryList(builder);
        String sql =
            "SELECT id,number,name,subject_id,rank,extend_status,transform,page_view FROM cdb.org_course where organization_number=:orgNumber and status=0  and is_del=0 ";
        if (StringUtils.isNotEmpty(subjectIds)) {
            sql = sql + " and subject_id in(:subjectIds) ";
        }

        if (orderType == 0) {
            sql = sql + " order by extend_status desc, rank desc ";
        } else if (orderType == 1) {
            sql = sql + " order by rank desc ";
        } else if (orderType == 2) {
            sql = sql + " order by rank asc ";
        } else if (orderType == 3) {
            sql = sql + " order by page_view desc ";
        } else if (orderType == 4) {
            sql = sql + " order by page_view asc ";
        } else if (orderType == 5) {
            sql = sql + " order by transform desc ";
        } else if (orderType == 6) {
            sql = sql + " order by transform asc ";
        }
        Map<String, Object> paramMap = Maps.newHashMap();
        paramMap.put("orgNumber", orgNumber);
        if (StringUtils.isNotEmpty(subjectIds)) {
            String[] subArray = StringUtils.split(subjectIds, "|");
            List<Integer> subjectList = Lists.newArrayList();
            for (String s : subArray) {
                subjectList.add(Integer.valueOf(s));
            }
            paramMap.put("subjectIds", subjectList);
        }
        log.info("orgCourseQuery,sql=" + sql);
        final List<OrgCourse> result = Lists.newArrayList();
        getNamedJdbcTemplate().query(sql, paramMap, new ResultSetExtractor<List<OrgCourse>>() {

            @Override
            public List<OrgCourse> extractData(ResultSet rs) throws SQLException, DataAccessException {
                while (rs.next()) {
                    OrgCourse orgCourse = new OrgCourse();
                    orgCourse.setId(rs.getLong("id"));
                    orgCourse.setNumber(rs.getLong("number"));
                    orgCourse.setName(rs.getString("name"));
                    orgCourse.setSubjectId(rs.getInt("subject_id"));
                    orgCourse.setRank(rs.getInt("rank"));
                    orgCourse.setExtendStatus(rs.getInt("extend_status"));
                    orgCourse.setTransform(rs.getInt("transform"));
                    orgCourse.setPageView(rs.getInt("page_view"));
                    result.add(orgCourse);
                }
                return result;
            }
        });
        return result;
    }

    @Override
    public OrgCourse getOrgCourse(Integer orgNumber, Long courseNum) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        builder.eq("number", courseNum);
        return this.uniqueResult(builder);
    }

    @Override
    public int getAdvertiseCourseCount(Integer orgNumber) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        builder.eq("extendStatus", 1);
        builder.count("id");
        builder.eq("isDel", 0);
        Integer result = this.queryForObject(builder, Integer.class);
        if (result == null) {
            return 0;
        }
        return result;
    }

    @Override
    public int getMaxId() {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("isDel", 0);
        builder.max("id");
        Integer maxId = queryForObject(builder, Integer.class);
        if (maxId != null) {
            return maxId;
        } else {
            return 0;
        }
    }

    @Override
    public List<OrgCourse> getCourseList(Collection<Long> courseIds, Long orgNumber, PageDto page, String...queryProp) {

        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder(queryProp);
        builder.in("id", courseIds);
        if (page != null) {
            builder.setPage(page);
        }
        if (orgNumber != null) {
            builder.eq("orgNumber", orgNumber);
        }
        builder.eq("isDel", 0);
        builder.desc("createTime");
        return this.queryList(builder);

    }

    @Override
    public List<String> getBlobString(Long orgNumber, Long courseNumber) {
        String sql =
            "SELECT introduction FROM cdb.org_course where organization_number=:orgNumber and number=:courseNumber";
        Map<String, Object> paramMap = Maps.newHashMap();
        paramMap.put("orgNumber", orgNumber);
        paramMap.put("courseNumber", courseNumber);

        final LobHandler lobHandler = new DefaultLobHandler();

        return getNamedJdbcTemplate().query(sql, paramMap, new RowMapper<String>() {

            @Override
            public String mapRow(ResultSet rs, int rowNum) throws SQLException {
                byte[] bytes = lobHandler.getBlobAsBytes(rs, 1);
                if (bytes != null) {
                    String s = new String(bytes);
                    return s;
                } else {
                    return null;
                }

            }

        });

    }

    @Override
    public void repairNumber() {
        String sql = "select id,number from cdb.org_course where length(number)>10";
        List<OrgCourse> list = this.getNamedJdbcTemplate().query(sql, new RowMapper<OrgCourse>() {

            @Override
            public OrgCourse mapRow(ResultSet rs, int rowNum) throws SQLException {

                OrgCourse orgCourse = new OrgCourse();
                orgCourse.setNumber(rs.getLong("number"));
                orgCourse.setId(rs.getLong("id"));
                return orgCourse;
            }

        });
        int count = 0;
        for (OrgCourse course : list) {
            long number = SerialNumberUtil.generateNumber(course.getId().intValue(), SerialNumberUtil.COURSE_MAX, 4);
            if (!course.getNumber().equals(number)) {
                course.setNumber(number);
                this.saveOrUpdate(course, "number");
                count++;
            }
        }
        log.info("repair {} records.", count);

    }

    @Override
    public Map<Integer, Integer> getCourseTotal(Date startTime, Date endTime, List<Integer> TianxiaoOrgNumbers) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgNumbers", TianxiaoOrgNumbers);
        String sql =
            "select organization_number, count(id) count from cdb.org_course where organization_number in (:orgNumbers)";
        if (startTime != null && endTime != null) {
            sql += " and create_time between :startTime and :endTime ";
            param.put("endTime", endTime);
            param.put("startTime", startTime);
        }
        sql += " group by organization_number";

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<Integer, Integer>>() {

            @Override
            public Map<Integer, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<Integer, Integer> map = new HashMap<>();
                while (rs.next()) {
                    Integer orgNamber = rs.getInt("organization_number");
                    Integer count = rs.getInt("count");
                    map.put(orgNamber, count);
                }
                return map;
            }
        });
    }

    @Override
    public Map<String, Integer> getCourseTotalMap(Date startTime, Date endTime, List<Integer> TianxiaoOrgNumbers) {
        Map<String, Object> param = new HashMap<>();
        param.put("orgNumbers", TianxiaoOrgNumbers);
        String sql =
            "select count(id) count, DATE_FORMAT(create_time,'%y-%m-%d') createTime from cdb.org_course where organization_number in (:orgNumbers)";
        if (startTime != null && endTime != null) {
            sql += " and create_time between :startTime and :endTime ";
            param.put("endTime", endTime);
            param.put("startTime", startTime);
        }
        sql += "group by MONTH(create_time), DAY(create_time)";

        return this.getNamedJdbcTemplate().query(sql, param, new ResultSetExtractor<Map<String, Integer>>() {

            @Override
            public Map<String, Integer> extractData(ResultSet rs) throws SQLException, DataAccessException {
                Map<String, Integer> map = new HashMap<>();
                while (rs.next()) {
                    String createTime = rs.getString("createTime");
                    Integer count = rs.getInt("count");
                    map.put(createTime, count);
                }
                return map;
            }
        });
    }

    @Override
    public void refreshOrgCourse(Long organizationNumber, Long id) {
        StringBuilder sb = new StringBuilder(
            "update cdb.org_course set is_del =1 where id > :id and  organization_number = :organizationNumber");
        Map param = Maps.newHashMap();
        param.put("organizationNumber", organizationNumber);
        param.put("id", id);
        this.getNamedJdbcTemplate().update(sb.toString(), param);
    }

    @Override
    public List<OrgCourse> getOrgCourseList(Collection<Integer> orgNumbers) {
        SingleSqlBuilder<OrgCourse> builder = createSqlBuilder();
        builder.in("orgNumber", orgNumbers);

        List<OrgCourse> orgCourseList = this.queryList(builder);

        return orgCourseList;
    }

    @Override
    public List<OrgCourse> getNormalCourseList(Collection<Long> courseIds, String... queryProp) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder(queryProp);
        builder.in("id", courseIds);
        builder.eq("isDel",0);
        builder.desc("createTime");
        return this.queryList(builder);

    }

    @Override
    public List<OrgCourse> getCourseByCasCadeIds(Integer orgNumber, List<Integer> casCadeIds, String...queryProp) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder(queryProp);
        builder.eq("orgNumber", orgNumber);
        builder.in("cascadeId", casCadeIds);
        return queryList(builder);
    }

    @Override
    public List<Long> getCourseIdsByCascadeId(Integer cascadeId) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder("id");
        builder.eq("cascadeId", cascadeId);
        builder.eq("isDel", 0);
        builder.desc("createTime");
        List<Long> ids = Lists.newArrayList();
        List<OrgCourse> orgCourses = queryList(builder);
        if (CollectionUtils.isNotEmpty(orgCourses)) {
            for (OrgCourse orgCourse : orgCourses) {
                ids.add(orgCourse.getId());
            }
        }
        return ids;
    }

    @Override
    public OrgCourse getOrgCourse(Integer orgNumber, Long courseNum, Integer cascadeId) {
        SingleSqlBuilder<OrgCourse> builder = this.createSqlBuilder();
        builder.eq("orgNumber", orgNumber);
        builder.eq("number", courseNum);
        builder.eq("cascadeId", cascadeId);
        return this.uniqueResult(builder);
    }

    @Override
    public void batchUpdateHeadTeacher(Integer orgNumber, Integer oldCascadeId, Integer newCascadeId) {
        String sql =
            "update cdb.org_course set cascade_id=:newCascadeId where organization_number=:number and cascade_id=:oldCascadeId";
        Map<String, Object> map = new HashMap<>();
        map.put("number", orgNumber);
        map.put("oldCascadeId", oldCascadeId);
        map.put("newCascadeId", newCascadeId);
        log.info("[HeadTeacher] sql={},param={}", sql, map);
        getNamedJdbcTemplate().update(sql, map);
    }

    @Override
    public List<OrgCourse> searchCourses(Long number, Set<Long> filterIds, boolean allCourse, PageDto pageDto) {
        log.info("orgNumber is : {} ,filterIds is : {} ,isAllcourse is:{} ,pageDto is : {} ", number, filterIds,
            allCourse, pageDto);
        SingleSqlBuilder<OrgCourse> createSqlBuilder = this.createSqlBuilder();
        createSqlBuilder.eq("orgNumber", number);
        if (allCourse) {
            if(GenericsUtils.notNullAndEmpty(filterIds)){
                createSqlBuilder.notin("id", filterIds);
            }
        } else {
            if(GenericsUtils.isNullOrEmpty(filterIds)){
                return Collections.emptyList();
            }
            createSqlBuilder.in("id", filterIds);
        }
        if (pageDto != null) {
            createSqlBuilder.setPage(pageDto);
        }
        String sql = createSqlBuilder.toSql();
        log.info("sql is : {} ", sql);
        return this.queryList(createSqlBuilder);
    }
}
