/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.dal.course.dao.impl;

import com.baijia.tianxiao.dal.course.dao.TeacherClassCourseDao;
import com.baijia.tianxiao.dal.course.po.TeacherClassCourse;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.sqlbuilder.util.Expressions;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.support.lob.DefaultLobHandler;
import org.springframework.jdbc.support.lob.LobHandler;
import org.springframework.stereotype.Repository;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

/**
 * 班课
 * 
 * @title TeacherClassCourseDaoImpl
 * @desc TODO
 * @author zhangbing
 * @date 2015年12月5日
 * @version 1.0
 */
@Slf4j
@Repository("teacherClassCourseDao")
public class TeacherClassCourseDaoImpl extends JdbcTemplateDaoSupport<TeacherClassCourse>
    implements TeacherClassCourseDao {

    /**
     * 查询机构可报名的班课
     */
    @SuppressWarnings("deprecation")
    @Override
    public int getVliadClassCourseCount(Integer orgNumber) {
        String sql = "select count(*) from cdb.teacher_class_course where organization_number=? "
            + "and status=2 and begin_time>now()";
        return this.getNamedJdbcTemplate().getJdbcOperations().queryForObject(sql, new Object[] { orgNumber },Integer.class);
    }

    @Override
    public int getCourseCountByTeacherId(Long teacherId) {
        String sql = "select count(*) from cdb.teacher_class_course where user_id=? ";
        return this.getNamedJdbcTemplate().getJdbcOperations().queryForObject(sql, new Object[]{teacherId},Integer.class);
    }

    /**
     * 获取机构班课课程分类列表
     */
    @Override
    public List<Integer> getOrgDistinctSubject(Integer orgNumber) {
        String sql = "select distinct subject_id from cdb.teacher_class_course where organization_number=? "
            + "and status=2 and begin_time > now()";
        List<Integer> subjectIds = this.getNamedJdbcTemplate().getJdbcOperations().query(sql,
            new Object[] { orgNumber }, new RowMapper<Integer>() {
                @Override
                public Integer mapRow(ResultSet rs, int rowNum) throws SQLException {
                    return rs.getInt("subject_id");
                }
            });
        return subjectIds;
    }

    @Override
    public List<TeacherClassCourse> getCoursesByNumbers(Collection<Long> numbers, final Long orgNumber,
        final String...queryProps) {
        BatchQueryTemplate<Long, List<TeacherClassCourse>> batchQueryTemplate = new ListBatchQueryTemplate<>();
        return batchQueryTemplate.batchQuery(numbers, new BatchQueryCallback<Long, List<TeacherClassCourse>>() {
            @Override
            public List<TeacherClassCourse> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<TeacherClassCourse> builder = createSqlBuilder(queryProps);
                builder.in("number", querySet);
                builder.eq("organizationNumber", orgNumber);
                builder.eq("verifyStatus", 1);
                builder.eq("status", 2);
                builder.add(Expressions.ne("open_status", 3));
                return queryList(builder);
            }
        });
    }

    /**
     * 获取机构正在招生的班课
     */
    @Override
    public List<TeacherClassCourse> getOrgCourseList(Integer number, Collection<Long> excludeIds, PageDto pageDto) {
        SingleSqlBuilder<TeacherClassCourse> builder = createSqlBuilder();
        builder.eq("organizationNumber", number);
        builder.eq("status", 2);
        builder.gt("beginTime", new Date());
        if (excludeIds != null && !excludeIds.isEmpty()) {
            builder.notin("id", excludeIds);
        }
        builder.setPage(pageDto);
        return queryList(builder);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.course.dao.TeacherClassCourseDao#getValidClassCourseCountByIds(java.lang.Integer,
     * java.util.Collection)
     */
    @Override
    public int getValidClassCourseCountByIds(Integer orgNumber, Collection<Long> ids) {
        SingleSqlBuilder<TeacherClassCourse> builder = createSqlBuilder();
        builder.count("id");
        builder.in("id", ids);
        builder.eq("organizationNumber", orgNumber);
        builder.eq("status", 2);
        builder.gt("beginTime", new Date());
        return queryForObject(builder, Integer.class);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.dal.course.dao.TeacherClassCourseDao#getOrgCourseList(java.lang.Integer,
     * java.util.Collection)
     */

    @Override
    public List<TeacherClassCourse> getOrgCourseListByIn(Integer number, Collection<Long> inludeIds, PageDto pageDto) {
        SingleSqlBuilder<TeacherClassCourse> builder = createSqlBuilder();
        builder.in("id", inludeIds);
        builder.eq("organizationNumber", number);
        builder.eq("status", 2);
        builder.gt("beginTime", new Date());
        return queryList(builder);
    }

    @Override
    public Map<Long, TeacherClassCourse> getTeacherClassCourses(Collection<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Collections.emptyMap();
        }

        List<TeacherClassCourse> courses = this.getByIds(ids);
        if (CollectionUtils.isNotEmpty(courses)) {
            Map<Long, TeacherClassCourse> result = Maps.newHashMap();
            for (TeacherClassCourse course : courses) {
                result.put(course.getId(), course);
            }
            return result;
        }
        return Collections.emptyMap();
    }

    /**
     * 获取推广列表页数据
     */
    @Override
    public List<TeacherClassCourse> getVisualTeacherCourseByOrgNumber(Integer orgNumber, String subjectIds,
        String courseTypes, int orderType) {

        String sql =
            "SELECT id,organization_number,number,name,subject_id,extend_status,rank,transform,page_view FROM cdb.teacher_class_course ";
        sql = sql + " where organization_number=:orgNumber and end_time>now() and (status=2 or verify_status=2) ";
        if (StringUtils.isNotEmpty(subjectIds)) {
            sql = sql + " and subject_id in(:subjectIds) ";
        }
        if (StringUtils.isNotEmpty(courseTypes)) {
            sql = sql + " and lesson_way in(:courseTypes)";
        }

        if (orderType == 0) {
            sql = sql + " order by extend_status desc, rank desc";
        } else if (orderType == 1) {
            sql = sql + " order by rank desc ";
        } else if (orderType == 2) {
            sql = sql + " order by rank asc ";
        } else if (orderType == 3) {
            sql = sql + " order by page_view desc ";
        } else if (orderType == 4) {
            sql = sql + " order by page_view asc ";
        } else if (orderType == 5) {
            sql = sql + " order by transform desc ";
        } else if (orderType == 6) {
            sql = sql + " order by transform asc ";
        }

        Map<String, Object> paramMap = Maps.newHashMap();
        paramMap.put("orgNumber", orgNumber);
        if (StringUtils.isNotEmpty(subjectIds)) {
            String[] subArray = StringUtils.split(subjectIds, "|");
            List<Integer> subjectList = Lists.newArrayList();
            for (String s : subArray) {
                subjectList.add(Integer.valueOf(s));
            }
            paramMap.put("subjectIds", subjectList);
        }
        if (StringUtils.isNotEmpty(courseTypes)) {
            List<Integer> courseList = Lists.newArrayList();
            String[] subArray = StringUtils.split(courseTypes, ",");
            for (String s : subArray) {
                courseList.add(Integer.valueOf(s));
            }
            paramMap.put("courseTypes", courseList);
        }
        log.info("teacherClassCourseQuery,sql=" + sql);
        final List<TeacherClassCourse> result = Lists.newArrayList();
        getNamedJdbcTemplate().query(sql, paramMap, new ResultSetExtractor<List<TeacherClassCourse>>() {

            @Override
            public List<TeacherClassCourse> extractData(ResultSet rs) throws SQLException, DataAccessException {
                while (rs.next()) {
                    TeacherClassCourse course = new TeacherClassCourse();
                    course.setId(rs.getLong("id"));
                    course.setOrganizationNumber(rs.getInt("organization_number"));
                    course.setNumber(rs.getLong("number"));
                    course.setName(rs.getString("name"));
                    course.setSubjectid(rs.getLong("subject_id"));
                    course.setExtendStatus(rs.getInt("extend_status"));
                    course.setRank(rs.getInt("rank"));
                    course.setTransform(rs.getInt("transform"));
                    course.setPageView(rs.getInt("page_view"));
                    result.add(course);
                }
                return result;
            }
        });
        return result;
    }

    @Override
    public TeacherClassCourse getTeacherClassCourse(Integer orgNumber, Long courseNumber) {
        SingleSqlBuilder<TeacherClassCourse> builder = createSqlBuilder();
        builder.eq("organizationNumber", orgNumber);
        builder.eq("number", courseNumber);
        return uniqueResult(builder);
    }

    @Override
    public TeacherClassCourse getByCouseNumber(Long courseNumber) {
        SingleSqlBuilder<TeacherClassCourse> builder = createSqlBuilder();
        builder.eq("number", courseNumber);
        return uniqueResult(builder);
    }
    
    @Override
    public List<TeacherClassCourse> getTeacherClassCourseList(Collection<Long> courseNumbers){
        SingleSqlBuilder<TeacherClassCourse> builder = createSqlBuilder();
        builder.in("number", courseNumbers);
        return queryList(builder);
    }

    @Override
    public int getAdvertiseCourseCount(Integer orgNumber) {
        SingleSqlBuilder<TeacherClassCourse> builder = createSqlBuilder();
        builder.eq("organizationNumber", orgNumber);
        builder.eq("extendStatus", 1);
        builder.count("id");
        Integer result = queryForObject(builder, Integer.class);
        if (result == null) {
            return 0;
        }
        return result;
    }

    @Override
    public int getMaxId() {
        SingleSqlBuilder<TeacherClassCourse> builder = createSqlBuilder();
        builder.max("id");
        Integer maxId = queryForObject(builder, Integer.class);
        if (maxId != null) {
            return maxId;
        } else {
            return 0;
        }
    }

    @Override
    public List<String> getBlobString(Integer orgNumber, Long courseNumber) {
        String sql =
            "SELECT introduction FROM cdb.teacher_class_course where organization_number=:orgNumber and number=:courseNumber";
        Map<String, Object> paramMap = Maps.newHashMap();
        paramMap.put("orgNumber", orgNumber);
        paramMap.put("courseNumber", courseNumber);

        final LobHandler lobHandler = new DefaultLobHandler();

        return getNamedJdbcTemplate().query(sql, paramMap, new RowMapper<String>() {

            @Override
            public String mapRow(ResultSet rs, int rowNum) throws SQLException {
                byte[] bytes = lobHandler.getBlobAsBytes(rs, 1);
                if (bytes != null) {
                    String s = new String(bytes);
                    return s;
                } else {
                    return null;
                }

            }

        });

    }

    @Override
    public List<TeacherClassCourse> getValidClassCourseCountsByIds(Integer number, List<Integer> classIds) {
        SingleSqlBuilder<TeacherClassCourse> builder = createSqlBuilder();
        builder.in("id", classIds);
        builder.eq("organizationNumber", number);
        builder.eq("status", 2);
        builder.gt("beginTime", new Date());
        return queryList(builder);
    }
}
