package com.baijia.tianxiao.dal.user.dao.impl;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Repository;

import com.baijia.tianxiao.dal.user.dao.UserDao;
import com.baijia.tianxiao.dal.user.po.User;
import com.baijia.tianxiao.sqlbuilder.SingleSqlBuilder;
import com.baijia.tianxiao.sqlbuilder.bean.impl.MatchMode;
import com.baijia.tianxiao.sqlbuilder.support.JdbcTemplateDaoSupport;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.query.BatchQueryCallback;
import com.baijia.tianxiao.util.query.BatchQueryTemplate;
import com.baijia.tianxiao.util.query.ListBatchQueryTemplate;
import com.google.common.base.Function;
import com.google.common.collect.Maps;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Repository
public class UserDaoImpl extends JdbcTemplateDaoSupport<User> implements UserDao {
    public UserDaoImpl() {
        super(User.class);
    }

    @Override
    public User getByNumber(@NonNull Long number, String...queryProps) {
        SingleSqlBuilder<User> builder = createSqlBuilder(queryProps);
        builder.eq("number", number);

        return uniqueResult(builder);
    }

    @Override
    public List<User> getByNumbers(@NonNull Collection<Long> numbers, final String...queryProps) {
        if (CollectionUtils.isEmpty(numbers)) {
            log.warn("numbers is empty");
            return Collections.emptyList();
        }
        BatchQueryTemplate<Long, List<User>> queryTemplate = new ListBatchQueryTemplate<>();
        return queryTemplate.batchQuery(numbers, new BatchQueryCallback<Long, List<User>>() {
            @Override
            public List<User> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<User> builder = createSqlBuilder(queryProps);
                builder.in("number", querySet);
                return queryList(builder);
            }
        });
    }

    @Override
    public User getByMobile(@NonNull String mobile, String...queryProps) {
        SingleSqlBuilder<User> builder = createSqlBuilder(queryProps);
        builder.eq("mobile", mobile);

        return uniqueResult(builder);

    }

    @Override
    public List<User> getByMobiles(@NonNull final Collection<String> mobiles, final String...queryProps) {
        if (CollectionUtils.isEmpty(mobiles)) {
            log.warn("mobiles is empty");
            return Collections.emptyList();
        }
        BatchQueryTemplate<String, List<User>> queryTemplate = new ListBatchQueryTemplate<>();
        return queryTemplate.batchQuery(mobiles, new BatchQueryCallback<String, List<User>>() {
            @Override
            public List<User> doQuery(Collection<String> querySet) {
                SingleSqlBuilder<User> builder = createSqlBuilder(queryProps);
                builder.in("mobile", querySet);
                return queryList(builder);
            }
        });

    }

    @Override
    public boolean exist(@NonNull String mobile) {
        SingleSqlBuilder<User> builder = createSqlBuilder();
        builder.count("id");
        builder.eq("mobile", mobile);
        Number id = queryForObject(builder, Long.class);
        return id.intValue() > 0;

    }

    @Override
    public List<User> queryByUserIds(Collection<Long> userIds, final String...queryProps) {
        if (userIds.isEmpty()) {
            return Collections.emptyList();
        }
        BatchQueryTemplate<Long, List<User>> queryTemplate = new ListBatchQueryTemplate<>();
        return queryTemplate.batchQuery(userIds, new BatchQueryCallback<Long, List<User>>() {
            @Override
            public List<User> doQuery(Collection<Long> querySet) {
                SingleSqlBuilder<User> builder = createSqlBuilder(queryProps);
                builder.in("id", querySet);
                return queryList(builder);
            }
        });
    }

    @Override
    public Map<Long, User> getMapByIds(Collection<Long> userIds, String...queryProps) {

        if (userIds.isEmpty()) {
            // log.warn("branchIds is empty.");
            return Collections.emptyMap();
        }

        List<User> users = queryByUserIds(userIds, queryProps);

        if (CollectionUtils.isEmpty(users)) {
            return Collections.emptyMap();
        }
        Map<Long, User> result = Maps.newHashMap();
        for (User user : users) {
            result.put(user.getId().longValue(), user);
        }
        return result;
    }

    @Override
    public Map<Long, String> getUserMobile(Collection<Long> ids) {

        if (ids.isEmpty()) {
            return Collections.emptyMap();
        }

        List<User> users = queryByUserIds(ids, "id", "mobile");

        if (CollectionUtils.isEmpty(users)) {
            return Collections.emptyMap();
        }
        return CollectorUtil.collectMap(users, new Function<User, Long>() {
            @Override
            public Long apply(User from) {
                return from.getId();
            }
        }, new Function<User, String>() {
            @Override
            public String apply(User from) {
                return from.getMobile();
            }
        });
    }

    @Override
    public List<User> getUsersByMobile(String mobile, String...queryPros) {
        SingleSqlBuilder<User> builder = createSqlBuilder(queryPros);
        builder.like("mobile", mobile, MatchMode.START);
        return queryList(builder);
    }

}
