
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.biz.www.impl;

import java.io.IOException;
import java.util.List;

import com.baijia.tianxiao.enums.RedisKeyEnums;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import com.baijia.commons.dession.serialize.JavaObjectSerializer;
import com.baijia.commons.dession.serialize.Serializer;
import com.baijia.commons.lang.utils.JacksonUtil;
import com.baijia.tianxiao.biz.www.QRRedisService;
import com.baijia.tianxiao.biz.www.constant.BizConstant;
import com.baijia.tianxiao.biz.www.constant.QrLoginStatus;
import com.baijia.tianxiao.constants.LoginConstant;
import com.baijia.tianxiao.dto.mobile.AppAuthToken;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;

import lombok.extern.slf4j.Slf4j;

/**
 * @author caoliang
 * @version 1.0
 * @title QRRedisServiceImpl
 * @desc TODO
 * @date 2016年3月15日
 */
@Service
@Slf4j
public class QRRedisServiceImpl implements QRRedisService {

    //    private static final String QR_LOGIN_PREFIX = "tx_qr_login_";
//    private static final String QR_LOGIN_STATUS_PREFIX = "tx_qr_login_status_";
//    private static final String QR_LOGIN_SID_PREFIX = "tx_qr_login_sessionid_";
//    private static final String QR_LOGIN_TEMP_SESSION_PREFIX = "tx_qr_temp_session_";
//    private static final String QR_LOGIN_APP_VERSION_PREFIX = "tx_qr_app_version_";
    private static final String QR_LOGIN_SEP = "_";
    private static final String QR_LOGIN_UUID_SEP = ",";

    @Autowired
    private StringRedisTemplate redisTemplate;

    public StringRedisTemplate getRedisTemplate() {

        return redisTemplate;
    }

    public void setRedisTemplate(StringRedisTemplate redisTemplate) {

        this.redisTemplate = redisTemplate;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.baijia.tianxiao.biz.www.QRRedisService#bindUUIDSid(java.lang.String, java.lang.String)
     */
    @Override
    public void bindUUIDSid(final String uuid, final String sessionId) {
        final String key = generateUUIDSIDKey(uuid);
        redisTemplate.execute(new RedisCallback<Object>() {
            @Override
            public Object doInRedis(RedisConnection connection) throws DataAccessException {
                connection.setEx(key.getBytes(), BizConstant.BEFORE_LOGIN_EXPIRE, sessionId.getBytes());
                return null;
            }
        });
    }

    /*
     * (non-Javadoc)
     *
     * @see com.baijia.tianxiao.biz.www.QRRedisService#setUUIDStatus(java.lang.String, java.lang.Integer)
     */

    @Override
    public void setUUIDStatus(final String uuid, final Integer status) {
        final String key = generateUUIDSTATUSKey(uuid);
        redisTemplate.execute(new RedisCallback<Object>() {
            @Override
            public Object doInRedis(RedisConnection connection) throws DataAccessException {
                connection.setEx(key.getBytes(), BizConstant.BEFORE_LOGIN_EXPIRE, status.toString().getBytes());
                return null;
            }
        });
    }

    /*
     * (non-Javadoc)
     *
     * @see com.baijia.tianxiao.biz.www.QRRedisService#initOrgLoginSession(java.lang.String, java.lang.Object)
     */

    @Override
    public void initOrgLoginSession(final String sessionId, final AppAuthToken session) {
        redisTemplate.execute(new RedisCallback<Object>() {
            @Override
            public Object doInRedis(RedisConnection connection) throws DataAccessException {
                try {
                    Serializer<Object> serializer = new JavaObjectSerializer();
                    connection.hSet(sessionId.getBytes(), LoginConstant.TX_PC_LOGIN_ATTRIBUTE_NAME.getBytes(),
                            serializer.serialize(session));
                } catch (Exception e) {
                    log.error("initOrgLoginSession error!sessionId:{},session:{}", sessionId, session, e);
                }
                return null;
            }
        });
    }

    /*
     * (non-Javadoc)
     *
     * @see com.baijia.tianxiao.biz.www.QRRedisService#initOrgLoginInfo(com.baijia.tianxiao.dto.mobile.AppAuthToken,
     * java.lang.String)
     */

    @Override
    public void initOrgLoginInfo(AppAuthToken obj, final String uuid) {
        final String key = generateOrgLoginKey(obj);
        redisTemplate.execute(new RedisCallback<Object>() {
            @Override
            public Object doInRedis(RedisConnection connection) throws DataAccessException {
                // 多设备登录时，需要token和uuid一对多
                String olduuids = redisTemplate.getStringSerializer().deserialize(connection.get(key.getBytes()));
                String newuuids = null;
                if (StringUtils.isNotEmpty(olduuids)) {
                    newuuids = Joiner.on(QR_LOGIN_UUID_SEP).join(olduuids, uuid);
                }else{
                    newuuids = uuid;
                }
                connection.setEx(key.getBytes(), BizConstant.AFTER_LOGIN_EXPIRE, newuuids.getBytes());

                String uuidSidKey = generateUUIDSIDKey(uuid);
                byte[] sidValue = connection.get(uuidSidKey.getBytes());
                if (sidValue == null || sidValue.length == 0) {
                    return null;
                }
                connection.expire(uuidSidKey.getBytes(), BizConstant.AFTER_LOGIN_EXPIRE);
                return null;
            }
        });
    }

    /*
     * (non-Javadoc)
     *
     * @see com.baijia.tianxiao.biz.www.QRRedisService#isOrgLogin(com.baijia.tianxiao.dto.mobile.AppAuthToken)
     */

    @Override
    public boolean isOrgLogin(AppAuthToken obj) {

        final String key = generateOrgLoginKey(obj);
        boolean isExist = redisTemplate.execute(new RedisCallback<Boolean>() {
            @Override
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                return connection.exists(key.getBytes());
            }
        });
        return isExist;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.baijia.tianxiao.biz.www.QRRedisService#clearOrgLoginInfo(com.baijia.tianxiao.dto.mobile.AppAuthToken)
     */

    @Override
    public void clearOrgLoginInfo(final AppAuthToken obj) {
        final String key = generateOrgLoginKey(obj);
        redisTemplate.execute(new RedisCallback<Object>() {
            @Override
            public Object doInRedis(RedisConnection connection) throws DataAccessException {
                String uuidsStr = redisTemplate.getStringSerializer().deserialize(connection.get(key.getBytes()));
                if (StringUtils.isNotEmpty(uuidsStr)) {
                    List<String> uuids = Splitter.on(QR_LOGIN_UUID_SEP).splitToList(uuidsStr);
                    if (CollectionUtils.isNotEmpty(uuids)) {
                        for (String uuid : uuids) {
                            String uuidSidKey = generateUUIDSIDKey(uuid);
                            String sessionId =
                                    redisTemplate.getStringSerializer().deserialize(connection.get(uuidSidKey.getBytes()));
                            String uuidStatusKey = generateUUIDSTATUSKey(uuid);
                            connection.del(key.getBytes());
                            connection.del(uuidSidKey.getBytes());
                            connection.del(uuidStatusKey.getBytes());
                            if(StringUtils.isNotEmpty(sessionId)){
                                connection.del(sessionId.getBytes());
                            }
                        }
                    }
                }
                return null;
            }
        });
    }

    private String generateUUIDSIDKey(String uuid) {
        StringBuilder key = new StringBuilder();
        key.append(RedisKeyEnums.WWW.QR_LOGIN_SID_PREFIX.getRedisKey());
        key.append(uuid);
        return key.toString();
    }

    private String generateUUIDSTATUSKey(String uuid) {
        StringBuilder key = new StringBuilder();
        key.append(RedisKeyEnums.WWW.QR_LOGIN_STATUS_PREFIX.getRedisKey());
        key.append(uuid);
        return key.toString();
    }

    private String generateOrgLoginKey(AppAuthToken token) {
        StringBuilder key = new StringBuilder();
        key.append(RedisKeyEnums.WWW.QR_LOGIN_PREFIX.getRedisKey());
        key.append(token.getUser_id());
        key.append(QR_LOGIN_SEP);
        key.append(token.getTx_cascade_user_id());
        key.append(QR_LOGIN_SEP);
        key.append(token.getCt());
        key.append(QR_LOGIN_SEP);
        key.append(token.getSalt());
        return key.toString();
    }

    private String generateTempSessionKey(String sessionId) {
        StringBuilder key = new StringBuilder();
        key.append(RedisKeyEnums.WWW.QR_LOGIN_TEMP_SESSION_PREFIX.getRedisKey());
        key.append(sessionId);
        return key.toString();
    }

    /*
     * (non-Javadoc)
     *
     * @see com.baijia.tianxiao.biz.www.QRRedisService#getUUIDStatus(java.lang.String)
     */

    @Override
    public Integer getUUIDStatus(String uuid) {
        final String key = generateUUIDSTATUSKey(uuid);
        Integer status = redisTemplate.execute(new RedisCallback<Integer>() {
            @Override
            public Integer doInRedis(RedisConnection connection) throws DataAccessException {
                byte[] value = connection.get(key.getBytes());
                if (value == null || value.length == 0) {
                    return QrLoginStatus.EXPIRED.getCode();
                }
                String status = redisTemplate.getStringSerializer().deserialize(value);
                return Integer.valueOf(status);
            }
        });
        return status;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.baijia.tianxiao.biz.www.QRRedisService#getSidByUUID(java.lang.String)
     */

    @Override
    public String getSidByUUID(String uuid) {
        final String key = generateUUIDSIDKey(uuid);
        String sessionId = redisTemplate.execute(new RedisCallback<String>() {
            @Override
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                byte[] value = connection.get(key.getBytes());
                if (value == null || value.length == 0) {
                    return null;
                }
                String sessionId = redisTemplate.getStringSerializer().deserialize(value);
                return sessionId;
            }
        });
        return sessionId;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.baijia.tianxiao.biz.www.QRRedisService#setTempSession(java.lang.String,
     * com.baijia.tianxiao.dto.mobile.AppAuthToken)
     */

    @Override
    public void setTempSession(final String sessionId, final AppAuthToken token) {
        final String key = generateTempSessionKey(sessionId);
        redisTemplate.execute(new RedisCallback<Object>() {
            @Override
            public Object doInRedis(RedisConnection connection) throws DataAccessException {
                try {
                    String tokenStr = JacksonUtil.obj2Str(token);
                    log.info("key:{},value;{}", key, tokenStr);
                    connection.setEx(key.getBytes(), BizConstant.BEFORE_LOGIN_EXPIRE, tokenStr.getBytes());
                    log.info("tempSession:{}",
                            redisTemplate.getStringSerializer().deserialize(connection.get(key.getBytes())));
                } catch (IOException e) {
                    log.error("setTempSession error!sessionId:{},token:{}", sessionId, token, e);
                }
                return null;
            }
        });
    }

    /*
     * (non-Javadoc)
     *
     * @see com.baijia.tianxiao.biz.www.QRRedisService#getTempSession(java.lang.String)
     */

    @Override
    public AppAuthToken getTempSession(final String sessionId) {

        final String key = generateTempSessionKey(sessionId);
        AppAuthToken token = redisTemplate.execute(new RedisCallback<AppAuthToken>() {
            @Override
            public AppAuthToken doInRedis(RedisConnection connection) throws DataAccessException {
                AppAuthToken token = null;
                try {
                    String tokenStr = redisTemplate.getStringSerializer().deserialize(connection.get(key.getBytes()));
                    token = JacksonUtil.str2Obj(tokenStr, AppAuthToken.class);
                    connection.del(key.getBytes());
                } catch (IOException e) {
                    log.error("getTempSession error!sessionId:{}", sessionId, e);
                }
                return token;
            }
        });
        return token;
    }

    @Override
    public void setAppVersion(String sessionId, final String version) {

        final String key = generateAppVersionKey(sessionId);
        redisTemplate.execute(new RedisCallback<Object>() {
            @Override
            public Object doInRedis(RedisConnection connection) throws DataAccessException {
                try {
                    log.info("versionkey:{},version;{}", key, version);
                    connection.setEx(key.getBytes(), BizConstant.BEFORE_LOGIN_EXPIRE, version.getBytes());
                } catch (Exception e) {
                    log.error("setTempSession error!sessionId:{},appversion:{}", key, version, e);
                }
                return null;
            }
        });
    }

    private String generateAppVersionKey(String sessionId) {
        StringBuilder key = new StringBuilder();
        key.append(RedisKeyEnums.WWW.QR_LOGIN_APP_VERSION_PREFIX.getRedisKey());
        key.append(sessionId);
        return key.toString();
    }

    @Override
    public String getAppVersion(final String sessionId) {
        final String key = generateAppVersionKey(sessionId);
        String version = redisTemplate.execute(new RedisCallback<String>() {
            @Override
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                String version = null;
                try {
                    version = redisTemplate.getStringSerializer().deserialize(connection.get(key.getBytes()));
                    connection.del(key.getBytes());
                } catch (Exception e) {
                    log.error("getAppVersion error!sessionId:{}", sessionId, e);
                }
                return version;
            }
        });
        return version;
    }

}
