package com.baijia.tianxiao.biz.service.impl;

import java.util.*;

import javax.annotation.Resource;

import com.baijia.tianxiao.biz.constants.TransferClassType;
import com.baijia.tianxiao.biz.dto.response.RefundPayInfoDto;
import com.baijia.tianxiao.constant.TransferClassStatus;
import com.baijia.tianxiao.dal.constant.ChargeMode;
import com.baijia.tianxiao.dal.constant.ChargeUnit;
import com.baijia.tianxiao.dal.finance.dao.TxTransferClassInfoDao;
import com.baijia.tianxiao.dal.finance.po.TxTransferClassInfo;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.sal.course.exceptions.SignErrorCode;
import com.baijia.tianxiao.sal.signup.constants.SignupRefundType;
import com.baijia.tianxiao.sal.signup.dto.SignupCourseInfoDto;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.biz.dto.request.TransferClassDetailRequest;
import com.baijia.tianxiao.biz.dto.response.TransferClassDetailResponse;
import com.baijia.tianxiao.biz.dto.response.TransferClassResponse;
import com.baijia.tianxiao.biz.service.StudentSignupCourseService;
import com.baijia.tianxiao.biz.service.TransferClassService;
import com.baijia.tianxiao.constants.signup.PayResult;
import com.baijia.tianxiao.constants.signup.PayType;
import com.baijia.tianxiao.dal.finance.dao.TxTransferClassRecordDao;
import com.baijia.tianxiao.dal.finance.po.TxTransferClassRecord;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.dal.signup.dao.OrgSignupCourseDao;
import com.baijia.tianxiao.dal.signup.dao.OrgSignupCourseLessonDao;
import com.baijia.tianxiao.dal.signup.dao.OrgSignupInfoDao;
import com.baijia.tianxiao.dal.signup.po.OrgSignupCourse;
import com.baijia.tianxiao.dal.signup.po.OrgSignupCourseLesson;
import com.baijia.tianxiao.dal.signup.po.OrgSignupInfo;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.enums.SignupErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.course.dto.response.StudentSignupInfoDto;
import com.baijia.tianxiao.sal.signup.constants.TransferClassResult;
import com.baijia.tianxiao.sal.signup.dto.CheckInfoDto;
import com.baijia.tianxiao.sal.signup.dto.request.TransferClassRequest;
import com.baijia.tianxiao.sal.signup.dto.request.TransferLessonInfoDto;
import com.baijia.tianxiao.sal.signup.dto.response.OrgSingupInfoDto;
import com.baijia.tianxiao.sal.signup.dto.response.TransferClassListDto;
import com.baijia.tianxiao.sal.signup.dto.response.TransferCourseInfoDto;
import com.baijia.tianxiao.sal.signup.dto.response.TransferRefundListDto;
import com.baijia.tianxiao.sal.signup.service.SignupService;
import com.baijia.tianxiao.sal.signup.service.TransferClassInfoService;
import com.baijia.tianxiao.sal.signup.service.TransferClassOptionService;
import com.baijia.tianxiao.util.BaseUtils;
import com.baijia.tianxiao.util.bean.ObjectUtil;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * Created by wengshengli on 2017/4/21.
 */
@Service
@Slf4j
public class TransferClassServiceImpl implements TransferClassService {

    @Resource
    private SignupService signupService;

    @Resource
    private TxTransferClassRecordDao txTransferClassRecordDao;

    @Resource
    private TransferClassInfoService transferClassInfoService;

    @Resource
    private TxTransferClassInfoDao txTransferClassInfoDao;

    @Resource
    private TransferClassOptionService transferClassOptionService;

    @Resource
    private OrgStudentDao orgStudentDao;

    @Resource
    private OrgSignupCourseLessonDao orgSignupCourseLessonDao;

    @Resource
    private OrgSignupCourseDao orgSignupCourseDao;

    @Resource
    private OrgSignupInfoDao orgSignupInfoDao;

    @Resource
    private StudentSignupCourseService studentSignupCourseService;

    @Resource
    private OrgCourseDao orgCourseDao;

    @Override
    @Transactional
    public TransferClassResponse signup(TransferClassRequest transferClassRequest) {
        log.info("Transfer class signup params= {}", transferClassRequest);
        /**
         * 转班请求参数校验
         */
        transferClassRequest.validate();
        /**
         * 转班课节信息校验
         */
        transferClassOptionService.checkTransferParam(transferClassRequest);
        /**
         * 转班引用信息获取及校验
         */
        OrgStudent studentInfoDto = orgStudentDao.getById(transferClassRequest.getStudentId());
        if (studentInfoDto == null) {
            throw new BussinessException(SignupErrorCode.TRANSFER_STUDENT_ERROR);
        }
        transferClassRequest.setStudentName(studentInfoDto.getName());
        transferClassRequest.setStudentMobile(studentInfoDto.getMobile());

        /**
         * 参数初始化
         */
        transferClassRequest.init();
        /**
         * 转班报名及保存转班记录
         */
        OrgSingupInfoDto orgSingupInfoDto = signupService.signUp(transferClassRequest);
        if (orgSingupInfoDto != null) {
            fillTransferClassLesson(transferClassRequest, studentInfoDto);
            transferClassInfoService.saveTransferInfo(transferClassRequest, orgSingupInfoDto,
                    transferClassRequest.getTransferLessons());
        }
        /**
         * 转班后续学生课节关系清理
         */
        Set<Long> lessonIds = BaseUtils.getPropertiesList(transferClassRequest.getTransferLessons(), "lessonId");
        transferClassOptionService.delStudentLessons(transferClassRequest.getOrgId(), lessonIds, studentInfoDto.getUserId());

        return TransferClassResponse.build(orgSingupInfoDto);
    }

    @Override
    public void refund(Long orgId, Long inSignupPurchaseId, Integer refundType) {
        log.info("Transfer class refund = {},{},{}", orgId, inSignupPurchaseId, refundType);
        Preconditions.checkNotNull(inSignupPurchaseId, "退款订单号不能为空");
        OrgSignupInfo orgSignupInfo = orgSignupInfoDao.searchByPurchaseId(inSignupPurchaseId);
        if (orgSignupInfo == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "转班订单不存在");
        }
        if (orgSignupInfo.getPurchaseStatus() == PayResult.SUCCESS.getCode()) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "转班订单已经成功");
        }

        orgSignupInfo.checkIsExpired();

        List<TransferRefundListDto> refundListDtos = transferClassInfoService.listTransferRefund(inSignupPurchaseId);
        if (CollectionUtils.isNotEmpty(refundListDtos)) {
            transferClassInfoService.updateReundType(refundListDtos.get(0).getTransferInfoNumber(), refundType);
        }

        signupService.signupSuccess(orgId, inSignupPurchaseId, PayType.TX_NO_PAY.getCode(), refundType);
    }

    @Override
    public List<TransferClassListDto> list(Long orgId, Long signupPurchaseId, Long classId) {
        log.info("Transfer class refund = {},{},{}", orgId, signupPurchaseId, classId);

        Preconditions.checkNotNull(signupPurchaseId);
        Preconditions.checkNotNull(classId);

        List<TransferClassListDto> transferClassListDtoList =
                transferClassInfoService.listTransferInfo(orgId, signupPurchaseId, classId);
        return transferClassListDtoList;
    }

    @Override
    public TransferClassDetailResponse detail(Long orgId, TransferClassDetailRequest detailRequest) {
        log.info("Transfer class detail params = {},{}", orgId, detailRequest);
        TransferClassDetailResponse result = new TransferClassDetailResponse();
        List<TxTransferClassRecord> transferClassRecords = Lists.newArrayList();

        TxTransferClassInfo txTransferClassInfo = null;
        if (detailRequest.getInSignupPurchaseId() != null && detailRequest.getInSignupPurchaseId() > 0) {
            transferClassRecords = txTransferClassRecordDao.listByInPurchaseId(detailRequest.getInSignupPurchaseId());
            if (CollectionUtils.isNotEmpty(transferClassRecords)) {
                for (TxTransferClassRecord txTransferClassRecord : transferClassRecords) {
                    if (txTransferClassRecord.getOutPurchaseId() != null && txTransferClassRecord.getOutPurchaseId().longValue() > 0) {
                        detailRequest.setOutSignupPurchaseId(transferClassRecords.get(0).getOutPurchaseId());
                    }
                }
                txTransferClassInfo = txTransferClassInfoDao.getByTransferNumber(transferClassRecords.get(0).getTransferNumber());
            } else {
                throw new BussinessException(SignupErrorCode.TRANSFER_EXPIRED);

            }
        } else {
            transferClassRecords = txTransferClassRecordDao.listByOutPurchaseId(detailRequest.getTransferClassNumber(),
                    detailRequest.getOutSignupPurchaseId());
            if (CollectionUtils.isNotEmpty(transferClassRecords)) {
                detailRequest.setInSignupPurchaseId(transferClassRecords.get(0).getInPurchaseId());
                txTransferClassInfo = txTransferClassInfoDao.getByTransferNumber(transferClassRecords.get(0).getTransferNumber());
            } else {
                throw new BussinessException(SignupErrorCode.TRANSFER_EXPIRED);
            }
        }

        log.debug("Transfer class detail result = {},{}", txTransferClassInfo, transferClassRecords);

        OrgSingupInfoDto inSignupInfo = null;
        if (detailRequest.getInSignupPurchaseId() != null) {
            inSignupInfo = signupService.detailByPurchaseId(orgId, detailRequest.getInSignupPurchaseId());
            if (inSignupInfo != null && CollectionUtils.isNotEmpty(inSignupInfo.getCourseInfos())) {
                result.setTransferInCourseInfo(fillTransferCourseInfoDto(TransferClassType.TRANSFER_IN.getType(), inSignupInfo, txTransferClassInfo, transferClassRecords));
            }
        }

        if (detailRequest.getOutSignupPurchaseId() != null) {
            OrgSingupInfoDto outSignupInfo =
                    signupService.detailByPurchaseId(orgId, detailRequest.getOutSignupPurchaseId());
            if (outSignupInfo != null && CollectionUtils.isNotEmpty(outSignupInfo.getCourseInfos())) {
                result.setTransferOutCourseInfo(fillTransferCourseInfoDto(TransferClassType.TRANSFER_OUT.getType(), outSignupInfo, txTransferClassInfo, transferClassRecords));
            } else {
                result.setTransferOutCourseInfo(fillFreeLessonTransferCourseInfoDto(txTransferClassInfo, transferClassRecords));
            }
        } else {
            result.setTransferOutCourseInfo(fillFreeLessonTransferCourseInfoDto(txTransferClassInfo, transferClassRecords));
        }

        if (inSignupInfo != null) {
            result.setStudentName(inSignupInfo.getStudentName());
            result.setStudentMobile(inSignupInfo.getStudentMobile());
            result.setSignupPurchaseId(inSignupInfo.getSignupPurchaseId());
            buildRefundPayPurchase(result, inSignupInfo);
        }

        return result;
    }

    private void buildRefundPayPurchase(TransferClassDetailResponse result, OrgSingupInfoDto inSignupInfo) {
        RefundPayInfoDto refundPayInfoDto = new RefundPayInfoDto();
        if (inSignupInfo.getTransferRefundMoney().doubleValue() > 0) {
            refundPayInfoDto.setTransferType(TransferClassResult.NEED_REFUND.getStatus());
            refundPayInfoDto.setRefundMoney(inSignupInfo.getTransferRefundMoney());
            refundPayInfoDto.setRefundType(inSignupInfo.getTransferRefundType());
            refundPayInfoDto.setRefundTypeStr(SignupRefundType.getNameByCode(inSignupInfo.getTransferRefundType()));
            refundPayInfoDto.setRefundTime(inSignupInfo.getPayTime());
        } else if (inSignupInfo.getTotalPrice().doubleValue() + inSignupInfo.getStudentPayPrice().doubleValue() > 0) {
            refundPayInfoDto.setTransferType(TransferClassResult.NEED_PAY.getStatus());
            refundPayInfoDto.setPayMoney(inSignupInfo.getTotalPrice() + inSignupInfo.getStudentPayPrice());
            refundPayInfoDto.setPayTime(inSignupInfo.getPayTime());
            refundPayInfoDto.setPayTypeStr(inSignupInfo.getPayTypeStr());

        } else {
            refundPayInfoDto.setTransferType(TransferClassResult.SUCCESS.getStatus());
        }
        result.setRefundPayInfo(refundPayInfoDto);
    }

    private void fillTransferClassLesson(TransferClassRequest transferClassRequest, OrgStudent studentInfoDto) {
        List userList = Lists.newArrayList();
        userList.add(studentInfoDto.getUserId());

        List<TransferLessonInfoDto> transferLessons = transferClassRequest.getTransferLessons();

        Set<Long> lessonIds = BaseUtils.getPropertiesList(transferLessons, "lessonId");

        Map<Long, OrgSignupCourse> signupCourseMap = null;

        List<OrgSignupCourseLesson> courseLessonList = null;

        if (CollectionUtils.isEmpty(lessonIds)) {
            return;
        }

        log.debug("transfer class lesson = {}", transferClassRequest.getOrgId(), lessonIds, userList);
        courseLessonList = orgSignupCourseLessonDao.getLessonList(transferClassRequest.getOrgId(), lessonIds, userList);
        if (CollectionUtils.isEmpty(courseLessonList)) {
            return;
        }

        log.debug("transfer class lesson = {}", courseLessonList);
        List<Long> signupCourseIds = BaseUtils.getListFilter(courseLessonList, "signupCourseId");

        log.debug("transfer class lesson = {}", signupCourseIds);
        List<OrgSignupCourse> signupCourseList = orgSignupCourseDao.getByIds(signupCourseIds);

        log.debug("transfer class lesson = {}", signupCourseList);
        signupCourseMap = BaseUtils.listToMap(signupCourseList, "id");

        List<TransferLessonInfoDto> result = new ArrayList<TransferLessonInfoDto>();

        for (TransferLessonInfoDto transferLessonInfoDto : transferLessons) {
            if (transferLessonInfoDto.getLessonId() != null && transferLessonInfoDto.getLessonId().longValue() > 0) {
                boolean findPurchases = false;

                for (OrgSignupCourseLesson signupCourseLesson : courseLessonList) {
                    if (signupCourseLesson.getLessonId().longValue() == transferLessonInfoDto.getLessonId()
                            .longValue()) {
                        transferLessonInfoDto.setLessonType(signupCourseLesson.getLessonType());
                        OrgSignupCourse orgSignupCourse =
                                signupCourseMap.get(signupCourseLesson.getSignupCourseId().longValue());
                        if (orgSignupCourse != null) {
                            if (!findPurchases) {
                                transferLessonInfoDto.setSignupPurchaseId(orgSignupCourse.getSignupPurchaseId());
                                findPurchases = true;
                                result.add(transferLessonInfoDto);
                            } else {
                                TransferLessonInfoDto copyLessonInfo =
                                        transferLessonInfoDto.clone();
                                copyLessonInfo.setSignupPurchaseId(orgSignupCourse.getSignupPurchaseId());
                                result.add(copyLessonInfo);
                            }
                        } else {
                            result.add(transferLessonInfoDto);
                        }
                    }
                }
            } else {
                result.add(transferLessonInfoDto);
            }
        }
        log.debug("transfer class lesson = {}", result);
        transferClassRequest.setTransferLessons(result);
    }

    private TransferCourseInfoDto fillTransferCourseInfoDto(int type, OrgSingupInfoDto signupInfo, TxTransferClassInfo txTransferClassInfo, List<TxTransferClassRecord> transferClassRecords) {
        log.debug("debug Transfer class = {},{},{},{}", type, signupInfo, txTransferClassInfo, transferClassRecords);
        TransferCourseInfoDto transferCourseInfoDto = new TransferCourseInfoDto();
        if (type == TransferClassType.TRANSFER_IN.getType()) {
            ObjectUtil.copyNotNullProperties(signupInfo.getCourseInfos().get(0), transferCourseInfoDto);
        } else {
            for (SignupCourseInfoDto courseInfoDto : signupInfo.getCourseInfos()) {
                if (courseInfoDto.getClassId().longValue() == transferClassRecords.get(0).getTransferOutClassId().longValue()) {
                    ObjectUtil.copyNotNullProperties(courseInfoDto, transferCourseInfoDto);
                }
            }
        }

        log.debug("TransferCourseInfoDto transferCourseInfoDto={}", transferCourseInfoDto);

        for (TxTransferClassRecord txTransferClassRecord : transferClassRecords) {
            transferCourseInfoDto.setTransferLessonCount(transferCourseInfoDto.getTransferLessonCount() + txTransferClassRecord.getTransferLessonCount().intValue());
            transferCourseInfoDto.setTransferClassMoney(transferCourseInfoDto.getTransferClassMoney() + txTransferClassRecord.getLessonMoney().doubleValue() / 100);
            transferCourseInfoDto.setTransferTime(txTransferClassRecord.getCreateTime());
        }

        if (txTransferClassInfo != null && type == TransferClassType.TRANSFER_OUT.getType()) {
            transferCourseInfoDto.setClassLeftCount(txTransferClassInfo.getOutLeftCount());
            transferCourseInfoDto.setClassLeftMoney(txTransferClassInfo.getOutLeftMoney().doubleValue() / 100);
        }

        if (txTransferClassInfo != null && type == TransferClassType.TRANSFER_IN.getType()) {
            transferCourseInfoDto.setClassLeftCount(txTransferClassInfo.getInLeftCount());
            transferCourseInfoDto.setClassLeftMoney(txTransferClassInfo.getInLeftMoney().doubleValue() / 100);
        }

        if (transferCourseInfoDto.getChargeUnit() != null && ChargeUnit.isByTime(transferCourseInfoDto.getChargeUnit())) {
            transferCourseInfoDto.setChargeUnit(ChargeUnit.BY_MINUTE.getCode());//报名处接口统一按照chargeunit处理
            transferCourseInfoDto.setChargeMode(ChargeMode.BY_HOUR.getCode());
        }

        return transferCourseInfoDto;

    }


    private TransferCourseInfoDto fillFreeLessonTransferCourseInfoDto(TxTransferClassInfo txTransferClassInfo, List<TxTransferClassRecord> transferClassRecords) {
        log.debug("debug Transfer class =,{},{}", txTransferClassInfo, transferClassRecords);
        TransferCourseInfoDto transferCourseInfoDto = new TransferCourseInfoDto();

        Long classId = txTransferClassInfo.getTransferOutClassId();
        OrgCourse orgCourse = orgCourseDao.getByCourseId(classId);
        if (orgCourse != null && orgCourse.getParentId().intValue() > 0) {
            orgCourse = orgCourseDao.getByCourseId(orgCourse.getParentId());
        }
        transferCourseInfoDto.setChargeMode(orgCourse.getChargeUnit());
        transferCourseInfoDto.setChargeUnit(orgCourse.getChargeUnit());
        transferCourseInfoDto.setOrgCourseName(orgCourse.getName());
        transferCourseInfoDto.setCoursePrice(orgCourse.getPrice());
        transferCourseInfoDto.setOriginPrice(orgCourse.getPrice());

        log.debug("TransferCourseInfoDto transferCourseInfoDto={}", transferCourseInfoDto);

        for (TxTransferClassRecord txTransferClassRecord : transferClassRecords) {
            transferCourseInfoDto.setTransferLessonCount(transferCourseInfoDto.getTransferLessonCount() + txTransferClassRecord.getTransferLessonCount().intValue());
            transferCourseInfoDto.setTransferClassMoney(transferCourseInfoDto.getTransferClassMoney() + txTransferClassRecord.getLessonMoney().doubleValue() / 100);
            transferCourseInfoDto.setTransferTime(txTransferClassRecord.getCreateTime());
        }

        transferCourseInfoDto.setClassLeftCount(txTransferClassInfo.getOutLeftCount());
        transferCourseInfoDto.setClassLeftMoney(txTransferClassInfo.getOutLeftMoney().doubleValue() / 100);

        if (transferCourseInfoDto.getChargeUnit() != null && ChargeUnit.isByTime(transferCourseInfoDto.getChargeUnit())) {
            transferCourseInfoDto.setChargeUnit(ChargeUnit.BY_MINUTE.getCode());//报名处接口统一按照chargeunit处理
            transferCourseInfoDto.setChargeMode(ChargeMode.BY_HOUR.getCode());
        }

        return transferCourseInfoDto;

    }

    @Override
    public Map<String, Object> check(Long orgId, Long classId, Long studentId) {
        OrgStudent studentInfoDto = orgStudentDao.getById(studentId);
        if (studentInfoDto == null) {
            studentInfoDto = orgStudentDao.getStudentByUserId(orgId, studentId);//兼容有可能是userID
            if (studentInfoDto == null) {
                throw new BussinessException(SignupErrorCode.TRANSFER_STUDENT_ERROR);
            }
        }

        List<TxTransferClassRecord> data = txTransferClassRecordDao.listByUserClass(studentInfoDto.getUserId(), classId,
                null, TransferClassStatus.INIT.getCode());

        OrgCourse orgCourse = orgCourseDao.getById(classId);
        List<Long> courseIdsList = Lists.newArrayList();
        if (orgCourse != null) {
            if (orgCourse.getParentId() != null && orgCourse.getParentId().intValue() > 0) {
                courseIdsList.add(orgCourse.getParentId());
            } else {
                courseIdsList.add(classId);
            }
        }

        List<StudentSignupInfoDto> list =
                studentSignupCourseService.checkInfo(orgId.longValue(), studentInfoDto.getId(), courseIdsList);
        List<CheckInfoDto> checkInfosList = getCheckInfo(list);

        Map<String, Object> result = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(data) && data.size() > 0) {
            result.put("transferSize", data.size());
            result.put("signupPurchaseId", data.get(0).getInPurchaseId());
        } else {
            result.put("transferSize", 0);
            result.put("signupPurchaseId", null);
        }
        result.put("checkInfoList", checkInfosList);
        return result;
    }

    private List<CheckInfoDto> getCheckInfo(List<StudentSignupInfoDto> list) {
        List<CheckInfoDto> result = Lists.newArrayList();
        if (CollectionUtils.isEmpty(list)) {
            return result;
        }
        for (StudentSignupInfoDto info : list) {
            CheckInfoDto dto = new CheckInfoDto();
            dto.setCourseId(info.getCourseId());
            dto.setCourseName(info.getCourseName());
            dto.setSignupTime(info.getSignupTime());
            dto.setFinishCount(info.getFinishedLessonNum());
            dto.setPrice(info.getOriginalPrice().doubleValue());
            dto.setPayPrice(info.getPayPrice().doubleValue());
            dto.setLesssonCount(info.getFreq());
            dto.setSignupCourseId(info.getSignupCourseId());
            dto.setCourseType(info.getCourseType());
            result.add(dto);
        }
        return result;
    }

}
