
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.biz.erp.util;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import com.google.common.collect.Lists;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.time.StopWatch;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.web.multipart.MultipartFile;

import com.baijia.tianxiao.biz.erp.vo.ImportCourse;
import com.baijia.tianxiao.biz.erp.vo.ImportStudent;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.course.service.impl.ImportOrgCourseProcessServiceImpl.CourseFormat;
import com.baijia.tianxiao.sal.upload.service.UploadFileReaderService;
import com.baijia.tianxiao.sal.upload.service.impl.ExcelUploadFileReaderServiceImpl;
import com.baijia.tianxiao.util.ExcelUtils;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;

/**
 * @title ErpUploadFileUtil
 * @desc O(∩_∩)O~
 * @author caoliang
 * @date 2016年7月5日
 * @version 1.0
 */
@Slf4j
public class ErpUploadFileUtil {

    public final static Map<String, CourseFormat> colMap = Maps.newHashMap();
    public final static Map<String, StudentFormat> studentMap = Maps.newHashMap();

    private final static Map<Integer, String> courseFieldMap = Maps.newHashMap();
    private final static Map<Integer, String> studentFieldMap = Maps.newHashMap();

    @Data
    public static class StudentFormat {

        private String fieldName;
        private String formatRegex;
        private String formatMsg;

        public StudentFormat(String fieldName, String formatRegex, String formatMsg) {
            this.fieldName = fieldName;
            this.formatRegex = formatRegex;
            this.formatMsg = formatMsg;
        }

        String getFmtErrorMsg() {
            return formatMsg;
        }
    }

    static {
        colMap.put("班级名称", new CourseFormat("courseName", ".{1,20}", "【班级名称】不能为空，最多输入20个字符"));
        colMap.put("上课次数", new CourseFormat("freq", "^[1-9]\\d*$", "【上课次数】需为大于零整数"));
        colMap.put("班级人数", new CourseFormat("maxStudent", "^[1-9]\\d*$", "【班级人数】需为大于零整数"));
        colMap.put("价格", new CourseFormat("coursePrice", "^(?!0(\\d|\\.0+$|$))\\d+(\\.\\d{1,2})?$", "【价格】需为大于零的数字"));

        studentMap.put("学员姓名",
            new StudentFormat("studentName", "^[a-zA-Z0-9\u4e00-\u9fa5]{1,15}$", "【学员姓名】不能为空，最多输入15个字符，仅支持数字、汉字和字母"));
        studentMap.put("学员手机", new StudentFormat("studentMobile", "^(\\+?86\\-?)?1\\d{2}\\-?\\d{4}\\-?\\d{4}$",
            "【学员手机】不能为空，并且需为以1开头的11位数字，不能包含特殊字符和空格"));
        studentMap.put("班级名称", new StudentFormat("courseName", ".{1,20}", "【班级名称】不能为空，最多输入20个字符"));
        studentMap.put("总课次",
            new StudentFormat("lessonCount", "^[0-9]*[1-9][0-9]*$|(\\s&&[^\\f\\n\\r\\t\\v])*$", "【总课次】需为大于零整数"));
        studentMap.put("从第几个课次插班",
            new StudentFormat("insertCount", "^[0-9]*[1-9][0-9]*$|(\\s&&[^\\f\\n\\r\\t\\v])*$", "【从第几个课次插班】需为大于零整数"));

        courseFieldMap.put(0, "courseName");
        courseFieldMap.put(1, "coursePrice");
        courseFieldMap.put(2, "freq");
        courseFieldMap.put(3, "maxStudent");
        courseFieldMap.put(4, "errReason");

        studentFieldMap.put(0, "studentName");
        studentFieldMap.put(1, "studentMobile");
        studentFieldMap.put(2, "courseName");
        studentFieldMap.put(3, "lessonCount");
        studentFieldMap.put(4, "insertCount");
        studentFieldMap.put(5, "errReason");
    }

    public static UploadFileReaderService getUploadFileService(MultipartFile file) {
        UploadFileReaderService service = null;
        String fileName = file.getOriginalFilename();
        if (fileName.toLowerCase().endsWith(".xlsx")) {
            service = new ExcelUploadFileReaderServiceImpl(file);
        } else {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, fileName + "只支持模板文件导入");
        }
        return service;
    }

    public static List<String> getTemplateHeader(String fileName) {
        List<String> templateHeader = Lists.newArrayList();
        XSSFWorkbook xssfWorkbook = null;
        try {
            File file = new File(ErpUploadFileUtil.class.getClassLoader().getResource(fileName).getFile());
            xssfWorkbook = new XSSFWorkbook(file);
            XSSFSheet xssfSheet = xssfWorkbook.getSheetAt(1);
            XSSFRow xssfRow = xssfSheet.getRow(1);
            for (int i = xssfRow.getFirstCellNum(); i < xssfRow.getLastCellNum(); i++) {
                XSSFCell cell = xssfRow.getCell(i);
                templateHeader.add(cell.getStringCellValue());
            }
        } catch (Exception e) {
            log.error("getTemplateHeader", e);
        } finally {
            if (xssfWorkbook != null) {
                try {
                    xssfWorkbook.close();
                } catch (IOException e) {
                    log.error("close xssfWorkbook error!", e);
                }
            }
        }
        return templateHeader;
    }

    public static List<String> tranHeader(Object[] headerObjs) {
        Preconditions.checkArgument(ArrayUtils.isNotEmpty(headerObjs), "头信息为空");
        List<String> headers = Lists.newArrayList();
        for (Object obj : headerObjs) {
            if (obj != null) {
                headers.add(obj.toString());
            } else {
                headers.add("-");
            }
        }
        return headers;

    }

    public static XSSFWorkbook generateCourseExcel(XSSFWorkbook workbook, List<ImportCourse> data,
        boolean needBackColor) throws InvalidFormatException, IOException, IllegalAccessException,
            InvocationTargetException, NoSuchMethodException {
        if (workbook == null) {
            File file = new File(ErpUploadFileUtil.class.getClassLoader().getResource("import_course.xlsx").getPath());
            workbook = new XSSFWorkbook(file);
        }
        XSSFSheet sheet = workbook.getSheetAt(1);
        if (CollectionUtils.isEmpty(data)) {
            return workbook;
        }
        XSSFCellStyle style = null;
        if (needBackColor) {
            style = ExcelUtils.getYellowBGStyle(workbook);
        } else {
            style = ExcelUtils.getWhitewBGStyle(workbook);
        }
        ExcelUtils.fillCellStyleWithFullBorder(style, CellStyle.BORDER_THIN, IndexedColors.GREY_25_PERCENT.getIndex());
        int currentRow = sheet.getLastRowNum() + 1;
        StopWatch stopwatch = new StopWatch();
        stopwatch.start();
        for (int i = 0; i < data.size(); i++) {
            ImportCourse importCourse = data.get(i);
            XSSFRow row = sheet.getRow(currentRow);
            if (row == null) {
                row = sheet.createRow(currentRow);
            }
            for (int j = 0; j < 5; j++) {
                XSSFCell cell = row.getCell(j);
                if (cell == null) {
                    cell = row.createCell(j);
                }
                cell.setCellStyle(style);
                cell.setCellValue(BeanUtils.getProperty(importCourse, courseFieldMap.get(j)));
            }
            currentRow++;
        }
        stopwatch.stop();
        log.debug("export course stopwatch time:{}", stopwatch.getTime());
        return workbook;
    }

    public static XSSFWorkbook generateStudentExcel(XSSFWorkbook workbook, List<ImportStudent> data,
        boolean needBackColor) throws InvalidFormatException, IOException, IllegalAccessException,
            InvocationTargetException, NoSuchMethodException {
        if (workbook == null) {
            File file = new File(ErpUploadFileUtil.class.getClassLoader().getResource("import_student.xlsx").getPath());
            workbook = new XSSFWorkbook(file);
        }
        XSSFSheet sheet = workbook.getSheetAt(1);
        if (CollectionUtils.isEmpty(data)) {
            return workbook;
        }
        XSSFCellStyle style = null;
        if (needBackColor) {
            style = ExcelUtils.getYellowBGStyle(workbook);
        } else {
            style = ExcelUtils.getWhitewBGStyle(workbook);
        }
        ExcelUtils.fillCellStyleWithFullBorder(style, CellStyle.BORDER_THIN, IndexedColors.GREY_25_PERCENT.getIndex());
        int currentRow = sheet.getLastRowNum() + 1;
        StopWatch stopwatch = new StopWatch();
        stopwatch.start();
        for (int i = 0; i < data.size(); i++) {
            ImportStudent importStudent = data.get(i);
            XSSFRow row = sheet.getRow(currentRow);
            if (row == null) {
                row = sheet.createRow(currentRow);
            }
            for (int j = 0; j <= 5; j++) {
                XSSFCell cell = row.getCell(j);
                if (cell == null) {
                    cell = row.createCell(j);
                }
                cell.setCellStyle(style);
                cell.setCellValue(BeanUtils.getProperty(importStudent, studentFieldMap.get(j)));
            }
            currentRow++;
        }
        stopwatch.stop();
        log.debug("export student stopwatch time:{}", stopwatch.getTime());
        return workbook;
    }

    public static void main(String[] args) throws InvalidFormatException, IOException {
        String reg = "^[a-zA-Z0-9\u4e00-\u9fa5]{1,20}$";
        System.out.println(Pattern.matches(reg, "a9是1234567890123456%"));
    }
}
