package com.baijia.tianxiao.biz.erp.teacherCenter.service;

import com.baijia.tianxiao.biz.erp.dto.response.CourseListInfoDto;
import com.baijia.tianxiao.biz.erp.dto.response.LessonTimeTableResponseDto;
import com.baijia.tianxiao.biz.erp.teacherCenter.dto.LessonMergeParams;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.excel.dto.ExportField;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.course.dto.response.OrgTeacherLessonInfoDto;
import com.baijia.tianxiao.sal.signup.dto.SignupCourseInfoDto;
import com.baijia.tianxiao.sal.signup.dto.response.OrgSingupInfoDto;
import com.baijia.tianxiao.util.date.DateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.beanutils.PropertyUtilsBean;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.ss.util.RegionUtil;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * Created with IntelliJ IDEA.
 * User: Victor Weng
 * Date: 16/5/19
 * Time: 上午11:52
 * To change this template use File | Settings | File Templates.
 */
@Slf4j
public class ExcelCourseExportService {

    private static String HEADER_FORMAT = "@";

    public static final String EXPORT_DATE_FORMAT = "yyyy-MM-dd HH:mm:ss";

    public static final String UNIT_LESSON_MONEY_FORMULA = "IF(E%d>0,E%d*D%d/60,0)";

    public static final String LESSON_MONEY_FORMULA = "SUM(F%d:F%d)";

    public static PropertyUtilsBean propertyUtilsBean = new PropertyUtilsBean();

    private static CellStyle createCellStyle(Workbook workBook, String format) {
        CellStyle cellStyle = workBook.createCellStyle();
        DataFormat dataFormat = workBook.createDataFormat();
        cellStyle.setAlignment(CellStyle.ALIGN_CENTER);
        cellStyle.setVerticalAlignment(CellStyle.VERTICAL_CENTER);
        cellStyle.setDataFormat(dataFormat.getFormat(format));

        cellStyle.setBorderBottom(CellStyle.BORDER_THIN);
        cellStyle.setBorderLeft(CellStyle.BORDER_THIN);
        cellStyle.setBorderRight(CellStyle.BORDER_THIN);
        cellStyle.setBorderTop(CellStyle.BORDER_THIN);

        return cellStyle;
    }

    public static Sheet createSheet(Workbook workbook, String sheetName, List<String> titles,
                                    Map<String, CellStyle> cellStyleMap, List<ExportField> exportFields, List<LessonTimeTableResponseDto> datas,
                                    Date startDate) {

        Sheet sheet = workbook.createSheet(sheetName);
        int columnSize = 0;//总列数
        for (ExportField field : exportFields) {
            columnSize += field.getExportFieldSize();
        }
        int startTime = 5;//开始时间
        createTitleRow(sheet, cellStyleMap, titles, columnSize);
        int row = titles != null ? titles.size() : 0;
        createHeaderRow(sheet, cellStyleMap, exportFields, row++);
        List<LessonMergeParams> mergeList = new ArrayList<LessonMergeParams>();
        for (LessonTimeTableResponseDto ltt : datas) {
            //哪一列显示
            int index = DateUtil.getDateDiff(startDate, ltt.getStartTime()) + 1;
            //那一行开始显示
            int startHourDiff = Math.round((float) DateUtil.getMinuteDiff(startDate, ltt.getStartTime()) / 60) - startTime - (index - 1) * 24;
            //哪一行结束显示
            int endHourDiff = Math.round((float) DateUtil.getMinuteDiff(startDate, ltt.getEndTime()) / 60) - startTime - (index - 1) * 24 - 1;
            if (endHourDiff < startHourDiff) {
                endHourDiff += 1;
            }
            LessonMergeParams mergeParam = new LessonMergeParams();
            mergeParam.setIndex(index);
            mergeParam.setStartHour(startHourDiff);
            mergeParam.setEndHour(endHourDiff);
            mergeParam.setCourseName(ltt.getCourseName());
            mergeList.add(mergeParam);
        }

        for (int j = 0; j < 24 - startTime; j++) {
            int rowNum = row;
            rowNum += j;

            Row courseRow = sheet.createRow(rowNum);
            createCell(courseRow, 0, (startTime + j) + ":00", cellStyleMap, exportFields.get(0));


            for (LessonMergeParams lessonMerge : mergeList) {
                if (j >= lessonMerge.getStartHour() && j <= lessonMerge.getEndHour()) {
                    createCell(courseRow, lessonMerge.getIndex(), lessonMerge.getCourseName(), cellStyleMap, exportFields.get(lessonMerge.getIndex()));
                }
            }
        }
        for (LessonMergeParams lmr : mergeList) {
            //合并下重复的行
            mergeCell(sheet, lmr.getStartHour() + row, lmr.getEndHour() + row, lmr.getIndex(), lmr.getIndex());
        }
        return sheet;
    }


    public static Sheet createSignupSheet(Workbook workbook, String sheetName, List<String> titles,
                                          Map<String, CellStyle> cellStyleMap, List<ExportField> exportFields, List<OrgSingupInfoDto> list) {

        Sheet sheet = workbook.createSheet(sheetName);
        int columnSize = 0;//总列数
        for (ExportField field : exportFields) {
            columnSize += field.getExportFieldSize();
        }
        createTitleRow(sheet, cellStyleMap, titles, columnSize);
        int row = titles != null ? titles.size() : 0;
        createHeaderRow(sheet, cellStyleMap, exportFields, row++);
        List<LessonMergeParams> mergeList = new ArrayList<LessonMergeParams>();
        int i = 0;
        for (OrgSingupInfoDto dto : list) {
            int rowNum = row;
            rowNum += i;

            int j = 0;
            for (SignupCourseInfoDto signupCourseInfoDto : dto.getCourseInfos()) {

                Row signupRow = sheet.createRow(rowNum + j);
                createCell(signupRow, 0, dto.getStudentName(), cellStyleMap, exportFields.get(0));
                createCell(signupRow, 1, dto.getStudentMobile(), cellStyleMap, exportFields.get(1));
                createCell(signupRow, 2, dto.getTotalPrice(), cellStyleMap, exportFields.get(2));
                createCell(signupRow, 3, signupCourseInfoDto.getOrgCourseName(), cellStyleMap, exportFields.get(3));
                createCell(signupRow, 4, signupCourseInfoDto.getOriginPrice(), cellStyleMap, exportFields.get(4));
                createCell(signupRow, 5, signupCourseInfoDto.getPayPrice(), cellStyleMap, exportFields.get(5));
                createCell(signupRow, 6, DateUtil.getStrByDateFormate(dto.getUpdateTime(), "yyyy-MM-dd hh:MM:ss"), cellStyleMap, exportFields.get(6));
                String statusStr = dto.getStatus().intValue() == 1 ? "已完成" : dto.getStatus().intValue() == 2 ? "待收款" : "待报名";
                createCell(signupRow, 7, statusStr, cellStyleMap, exportFields.get(7));
                j++;
            }
            i += j;
            if (j > 1) {
                LessonMergeParams mergeParams = new LessonMergeParams();
                mergeParams.setStartHour(rowNum);
                mergeParams.setEndHour(rowNum + j - 1);
                mergeList.add(mergeParams);
            }
        }
        System.out.println(mergeList);
        for (LessonMergeParams lmr : mergeList) {
            //合并下重复的行
            mergeCell(sheet, lmr.getStartHour(), lmr.getEndHour(), 0, 0);
            mergeCell(sheet, lmr.getStartHour(), lmr.getEndHour(), 1, 1);
            mergeCell(sheet, lmr.getStartHour(), lmr.getEndHour(), 2, 2);
            mergeCell(sheet, lmr.getStartHour(), lmr.getEndHour(), 6, 6);
            mergeCell(sheet, lmr.getStartHour(), lmr.getEndHour(), 7, 7);
        }
        return sheet;
    }


    private static void mergeCell(Sheet sheet, int firstRow, int lastRow, int firstColumn, int lastColumn) {
        CellRangeAddress rangeAddress = new CellRangeAddress(firstRow, lastRow, firstColumn, lastColumn);
        sheet.addMergedRegion(rangeAddress);
        RegionUtil.setBorderBottom(CellStyle.BORDER_THIN, rangeAddress, sheet, sheet.getWorkbook());
        RegionUtil.setBorderTop(CellStyle.BORDER_THIN, rangeAddress, sheet, sheet.getWorkbook());
        RegionUtil.setBorderLeft(CellStyle.BORDER_THIN, rangeAddress, sheet, sheet.getWorkbook());
        RegionUtil.setBorderRight(CellStyle.BORDER_THIN, rangeAddress, sheet, sheet.getWorkbook());
    }

    private static void createLessonRow(Row lessonRow, int baseColumn, OrgTeacherLessonInfoDto lessonInfo,
                                        List<ExportField> exportFields, Map<String, CellStyle> cellStyleMap) {
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getLessonNumberInfo(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getRoomName(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getStartTime(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getLessonTime(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getMinutes(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getStudentCount(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getSignCount(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getLeaveCount(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getAbsentCount(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getUnsignCount(), cellStyleMap, exportFields.get(baseColumn));
        baseColumn++;
        createCell(lessonRow, baseColumn, lessonInfo.getSignRate(), cellStyleMap, exportFields.get(baseColumn));

    }

    private static Cell createCell(Row row, int column, Object value, Map<String, CellStyle> cellStyleMap,
                                   ExportField exportField) {
        Cell cell = row.createCell(column);
        String localFormat = null;
        value = value != null ? value : exportField.getDefaultVal();

        if (value != null) {
            if (value instanceof Date) {
                row.getSheet().setColumnWidth(cell.getColumnIndex(), 20 * 256);
                localFormat = EXPORT_DATE_FORMAT;
                cell.setCellValue((Date) value);
            } else {
                if (value instanceof Number && (((Number) value).longValue() < Integer.MAX_VALUE)) {
                    cell.setCellType(Cell.CELL_TYPE_NUMERIC);
                    if (value instanceof Double) {
                        localFormat = "0.00";
                        cell.setCellValue(((Number) value).doubleValue());
                    } else {
                        localFormat = "0";
                        cell.setCellValue(((Number) value).longValue());
                    }
                } else {
                    cell.setCellType(Cell.CELL_TYPE_STRING);
                    cell.setCellValue(value.toString());
                    localFormat = "@";
                }
            }
        } else {
            cell.setCellValue("");
        }
        cell.setCellStyle(getCellStyle(row.getSheet().getWorkbook(), cellStyleMap,
                exportField.getFormat() != null ? exportField.getFormat() : localFormat));
        return cell;
    }

    private static CellStyle getCellStyle(Workbook workbook, Map<String, CellStyle> cellStyleMap, String format) {
        if (!cellStyleMap.containsKey(format)) {
            cellStyleMap.put(format, createCellStyle(workbook, format));
        }
        return cellStyleMap.get(format);
    }

    /**
     * 递归创建出行头
     *
     * @param sheet
     * @param cellStyleMap
     * @param exportFields
     * @param row
     */
    private static void createHeaderRow(Sheet sheet, Map<String, CellStyle> cellStyleMap,
                                        List<ExportField> exportFields, int row) {
        Row headerRow = sheet.createRow(row);
        int column = 0;
        CellStyle style = getCellStyle(sheet.getWorkbook(), cellStyleMap, HEADER_FORMAT);
        for (ExportField field : exportFields) {
            column = createHeaderCell(headerRow, style, field, column);
        }
    }

    private static int createHeaderCell(Row row, CellStyle style, ExportField field, int column) {
        if (CollectionUtils.isEmpty(field.getChildren())) {
            row.getSheet().setColumnWidth(column, field.getWidth());
            Cell cell = row.createCell(column++);
            cell.setCellStyle(style);
            cell.setCellValue(field.getName());
        } else {
            for (ExportField child : field.getChildren()) {
                column = createHeaderCell(row, style, child, column);
            }
        }
        return column;

    }

    private static void createTitleRow(Sheet sheet, Map<String, CellStyle> cellStyleMap, List<String> titles,
                                       int headerSize) {
        int row = 0;
        // 补充头
        if (CollectionUtils.isNotEmpty(titles)) {
            for (String title : titles) {
                Row titleRow = sheet.createRow(row);
                Cell cell = titleRow.createCell(0);
                cell.setCellValue(title);
                cell.setCellStyle(getCellStyle(sheet.getWorkbook(), cellStyleMap, HEADER_FORMAT));
                mergeCell(sheet, row, row, 0, headerSize - 1);
                row++;
            }
        }
    }

    public static void exportExcel(HttpServletResponse response, Workbook workBook, String excelFileName) {
        if (StringUtils.isBlank(excelFileName)) {
            excelFileName = System.currentTimeMillis() + ".xlsx";
        }
        response.setContentType("application/vnd.ms-excel");
        try {
            response.setHeader("Content-Disposition",
                    "attachment; filename=" + URLEncoder.encode(excelFileName, "utf-8"));
        } catch (UnsupportedEncodingException e1) {
            response.setHeader("Content-Disposition", "attachment; filename=" + excelFileName);
        }

        try {
            workBook.write(response.getOutputStream());
            response.getOutputStream().flush();
        } catch (IOException e) {
            log.error("export data catch error:", e);
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "导出数据处理失败");
        }
    }

}
