package com.baijia.tianxiao.biz.erp.service.impl;

import com.baijia.tianxiao.biz.erp.service.RoomClassLessonService;
import com.baijia.tianxiao.constant.BooleanEnum;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgClassRoomDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseRoomDao;
import com.baijia.tianxiao.dal.org.po.OrgClassRoom;
import com.baijia.tianxiao.dal.solr.dto.ClassRoomQueryParam;
import com.baijia.tianxiao.dal.solr.dto.ClassRoomSolrDto;
import com.baijia.tianxiao.dal.solr.query.ClassRoomQuery;
import com.baijia.tianxiao.sal.room.dto.ClassRoomDto;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.ListUtil;
import com.baijia.tianxiao.util.date.DateUtil;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.solr.client.solrj.SolrServerException;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Author: hanlaijin
 */
@Service
@Slf4j
public class RoomClassLessonServiceImpl implements RoomClassLessonService {

    private static final int ORDER_BY_ARRANGED_MINUTE_DESC = 3;

    private static final int ORDER_BY_ARRANGED_MINUTE_ASC = 4;

    @Resource
    private OrgClassRoomDao orgClassRoomDao;

    @Resource
    private OrgCourseRoomDao orgCourseRoomDao;

    @Resource
    private OrgClassLessonDao orgClassLessonDao;

    @Resource
    private ClassRoomQuery classRoomQuery;

    @Override
    public List<ClassRoomDto> getOrgClassRoomsListFromSolr(ClassRoomQueryParam param, PageDto pageDto) {
        try {
            return getOrgClassRoomsList(param, pageDto);
        } catch (Exception e) {
            log.info("query from solr ,ClassRoomQueryParam is : {} ", param);
            List<ClassRoomSolrDto> list = null;
            try {
                list = classRoomQuery.queryClassRoom(param, pageDto);
            } catch (Exception e1) {
                log.debug("solr exception : {}",e1);
            }
            if (CollectionUtils.isEmpty(list)) {
                return Collections.EMPTY_LIST;
            }
            List<Long> roomIds = ListUtil.toKeyList(list, "roomId", ClassRoomSolrDto.class);
            Set<Long> tags = tagForClassRecord(param.getOrgId(), roomIds);
            return buildClassRoomDto(list, tags);

        }
    }

    private List<ClassRoomDto> buildClassRoomDto(List<ClassRoomSolrDto> list, Set<Long> tags) {
        List<ClassRoomDto> result = Lists.newArrayList();
        if (CollectionUtils.isEmpty(list)) {
            return result;
        }
        for (ClassRoomSolrDto solrDto : list) {
            ClassRoomDto dto = new ClassRoomDto();
            dto.setOrgId(solrDto.getOrgId());
            dto.setRoomId(solrDto.getRoomId());
            dto.setRoomName(solrDto.getRoomName());
            dto.setRoomSize(solrDto.getRoomSize());
            dto.setDelStatus(solrDto.getDelStatus() == 0 ? false : true);
            dto.setCreateTime(solrDto.getCreateTime());
            dto.setRecycleTime(solrDto.getRecycleTime());
            dto.setRecycleStatus(solrDto.getRecycleStatus());
            dto.setArrangedCount(solrDto.getArrangedCount());
            dto.setArrangedMinutes(solrDto.getArrangedMinutes());
            Integer tag =
                tags.contains(solrDto.getRoomId()) ? BooleanEnum.TRUE.getStatus() : BooleanEnum.FALSE.getStatus();
            dto.setClassRecord(tag);
            result.add(dto);
        }
        return result;
    }

    @Override
    public List<ClassRoomDto> getOrgClassRoomsList(ClassRoomQueryParam param, PageDto pageDto) {
        log.info("query from db ,ClassRoomQueryParam is : {} ", param);
        Integer order = param.getOrder();
        List<OrgClassRoom> classRooms;
        if (order != null && (order == ORDER_BY_ARRANGED_MINUTE_DESC || order == ORDER_BY_ARRANGED_MINUTE_ASC)) {
            classRooms = orgClassRoomDao.getOrgClassRoomsList(param.getOrgId(), null, param.getRecycleStatus(), order,
                param.getRoomName(), null);
        } else {
            classRooms = orgClassRoomDao.getOrgClassRoomsList(param.getOrgId(), null, param.getRecycleStatus(), order,
                param.getRoomName(), pageDto);
        }
        if (CollectionUtils.isEmpty(classRooms)) {
            return Collections.emptyList();
        }
        List<Long> allRoomId = ListUtil.toKeyList(classRooms, "id", OrgClassRoom.class);
        log.debug("[order]={}   [classRoomSize]={}", param.getOrder(), allRoomId.size());
        List<Long> allClassId = orgCourseRoomDao.getcoursIdsByRoomIds(allRoomId);
        Date start = DateUtil.getFirstDate(new Date());
        Date end = DateUtil.getNextMonthFirstDate(new Date());
        // 月排次数
        Map<Long, Integer> counts = orgClassLessonDao.getRoomArrangedCount(allClassId, start, end);
        // 月排时长
        Map<Long, Integer> minutes = orgClassLessonDao.getRoomArrangedMinute(allClassId, start, end);
        // 上课记录标签
        Set<Long> tags = tagForClassRecord(param.getOrgId(), allRoomId);
        List<ClassRoomDto> result = getRoomResutlList(classRooms, counts, minutes, tags);
        if (order != null && (order == ORDER_BY_ARRANGED_MINUTE_DESC || order == ORDER_BY_ARRANGED_MINUTE_ASC)) {
            return sortByMinutes(order, result, pageDto);
        } else {
            return result;
        }
    }

    private Set<Long> tagForClassRecord(Long orgId, List<Long> allRoomId) {
        Map<Long, Integer> map = orgClassLessonDao.getRoomClassRecordCount(orgId, allRoomId);
        Set<Long> roomLessonSet = Sets.newHashSet();
        for (Long l : map.keySet()) {
            if (map.get(l) > 0) {
                roomLessonSet.add(l);
            }
        }
        return roomLessonSet;

    }

    private List<ClassRoomDto> getRoomResutlList(List<OrgClassRoom> classRooms, Map<Long, Integer> counts,
        Map<Long, Integer> minutes, Set<Long> tags) {
        List<ClassRoomDto> result = Lists.newArrayList();
        for (OrgClassRoom classRoom : classRooms) {
            ClassRoomDto dto = ClassRoomDto.covert2Dto(classRoom);
            dto.setArrangedCount(counts.containsKey(classRoom.getId()) ? counts.get(classRoom.getId()) : 0);
            dto.setArrangedMinutes(minutes.containsKey(classRoom.getId()) ? minutes.get(classRoom.getId()) : 0);
            Integer tag =
                tags.contains(classRoom.getId()) ? BooleanEnum.TRUE.getStatus() : BooleanEnum.FALSE.getStatus();
            dto.setClassRecord(tag);
            result.add(dto);
        }
        return result;
    }

    private List<ClassRoomDto> sortByMinutes(Integer order, List<ClassRoomDto> result, PageDto pageDto) {
        if (CollectionUtils.isEmpty(result) || pageDto == null) {
            return result;
        }
        if (order == ORDER_BY_ARRANGED_MINUTE_DESC) {
            Collections.sort(result, new Comparator<ClassRoomDto>() {
                @Override
                public int compare(ClassRoomDto o1, ClassRoomDto o2) {
                    return o2.compare(o1);
                }
            });
        } else if (order == ORDER_BY_ARRANGED_MINUTE_ASC) {
            Collections.sort(result, new Comparator<ClassRoomDto>() {
                @Override
                public int compare(ClassRoomDto o1, ClassRoomDto o2) {
                    return o1.compare(o2);
                }
            });
        }
        int start = (pageDto.getPageNum() - 1) * pageDto.getPageSize();
        int end = start + pageDto.getPageSize();
        if (end > result.size()) {
            end = result.size();
        }
        if(start > result.size()){
            return null;
            //start = (result.size()/pageDto.getPageSize())*pageDto.getPageSize();
        }
        return result.subList(start, end);
    }

    public void dealClassRoomArranged(List<OrgClassRoom> list) {
        log.debug("[OrgClassRoom size]={}", list.size());
        List<Long> allRoomIds = ListUtil.toKeyList(list, "id", OrgClassRoom.class);
        List<Long> allClassId = orgCourseRoomDao.getcoursIdsByRoomIds(allRoomIds);
        Date start = DateUtil.getFirstDate(new Date());
        Date end = DateUtil.getNextMonthFirstDate(new Date());
        Map<Long, Integer> counts = orgClassLessonDao.getRoomArrangedCount(allClassId, start, end);
        Map<Long, Integer> minutes = orgClassLessonDao.getRoomArrangedMinute(allClassId, start, end);
        log.debug("[arranged minute ]={}",minutes);
        for (OrgClassRoom room : list) {
            Integer count = counts.get(room.getId());
            Integer minute = minutes.get(room.getId());
            Map<String, String> map = Maps.newHashMap();
            map.put("ID", room.getId() + "");
            map.put("ROOM_SIZE", room.getRoomSize() + "");
            map.put("ORG_ID", room.getOrgId() + "");
            map.put("ROOM_NAME", room.getRoomName());
            map.put("DEL_STATUS", room.isDelStatus() ? "1" : "0");
            map.put("RECYCLE_STATUS", room.getRecycleStatus() + "");
            map.put("CREATE_TIME", room.getCreateTime() + "");
            map.put("RECYCLE_TIME", room.getRecycleTime() == null ? null : room.getRecycleTime() + "");
            map.put("ARRANGED_COUNT", count == null ? null : count + "");
            map.put("ARRANGED_MINUTES", minute == null ? null : minute + "");
            try {
                classRoomQuery.insertNewRow(map);
            } catch (SolrServerException e) {
                log.warn("[insert into solr catch exception]={}",e);
            } catch (IOException e) {
                log.warn("[insert into solr catch exception]={}",e);
            }
        }
    }
}
