
/**
 * 
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.biz.erp.service.impl;

import com.baijia.tianxiao.biz.erp.dto.CourseSigninRecordDto;
import com.baijia.tianxiao.biz.erp.dto.CourseSigninRecordDto.StudentSigninRecord;
import com.baijia.tianxiao.biz.erp.dto.CourseSigninStsDto;
import com.baijia.tianxiao.biz.erp.dto.CourseSigninStsDto.StudentSigninSts;
import com.baijia.tianxiao.biz.erp.dto.request.FastSignInLesssonsRequestDto;
import com.baijia.tianxiao.biz.erp.dto.request.SigninStatusDto;
import com.baijia.tianxiao.biz.erp.dto.response.FastSignInViewCourseDto;
import com.baijia.tianxiao.biz.erp.dto.response.LessonResponseDto;
import com.baijia.tianxiao.biz.erp.dto.response.SignLogDto;
import com.baijia.tianxiao.biz.erp.dto.response.StudentSigninRecordDto;
import com.baijia.tianxiao.biz.erp.dto.response.StudentSigninRecordDto.CourseSigninResult;
import com.baijia.tianxiao.biz.erp.service.CourseLessonService;
import com.baijia.tianxiao.biz.erp.service.ErpSignInService;
import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.constant.OrgLessonSignLogChangeType;
import com.baijia.tianxiao.constant.OrgLessonSignSourceEnum;
import com.baijia.tianxiao.constant.SignStatus;
import com.baijia.tianxiao.dal.constant.ChargeUnit;
import com.baijia.tianxiao.dal.course.po.OrgCoursePhoto;
import com.baijia.tianxiao.dal.enums.CourseTypeEnum;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgLessonSignDao;
import com.baijia.tianxiao.dal.org.dao.OrgLessonSignLogDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentLessonDao;
import com.baijia.tianxiao.dal.org.dao.TXCascadeAccountDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgClassLesson;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.dal.org.po.OrgCourseConsumeRule;
import com.baijia.tianxiao.dal.org.po.OrgLessonSign;
import com.baijia.tianxiao.dal.org.po.OrgLessonSignLog;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.org.po.OrgStudentCourse;
import com.baijia.tianxiao.dal.org.po.TXCascadeAccount;
import com.baijia.tianxiao.dal.solr.dto.TimeRange;
import com.baijia.tianxiao.dal.storage.dao.StorageDao;
import com.baijia.tianxiao.dal.storage.po.Storage;
import com.baijia.tianxiao.dal.user.dao.TeacherDao;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.enums.StudentCourseStatus;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.filter.TianxiaoMContext;
import com.baijia.tianxiao.sal.course.dto.SignStatusRemarkDto;
import com.baijia.tianxiao.sal.course.dto.response.SigninFailedDto;
import com.baijia.tianxiao.sal.course.enums.CourseConsumeRuleEnum;
import com.baijia.tianxiao.sal.course.service.CourseStudentService;
import com.baijia.tianxiao.sal.course.service.OrgCourseConsumeRuleService;
import com.baijia.tianxiao.sal.course.service.OrgCourseListService;
import com.baijia.tianxiao.sal.course.service.OrgLessonSignService;
import com.baijia.tianxiao.sal.organization.constant.CascadeType;
import com.baijia.tianxiao.sal.organization.org.service.TxCascadeCredentialService;
import com.baijia.tianxiao.util.collection.CollectorUtil;
import com.baijia.tianxiao.util.date.DateUtil;
import com.baijia.tianxiao.util.storage.StorageUtil;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * @title ErpSignInServiceImpl
 * @desc TODO
 * @author leiruiqi
 * @date 2016年7月23日
 * @version 1.0
 */
@Service("erpSignInService")
@Slf4j
public class ErpSignInServiceImpl implements ErpSignInService {
    @Autowired
    private TeacherDao teacherDao;

    @Resource
    private OrgAccountDao orgAccountDao;

    @Resource
    private TXCascadeAccountDao txCascadeAccountDao;

    @Resource
    private OrgCourseListService orgCourseListService;

    @Resource
    private OrgLessonSignDao orgLessonSignDao;

    @Resource
    private OrgStudentCourseDao orgStudentCourseDao;

    @Resource
    private StorageDao storageDao;

    @Resource
    private OrgStudentDao orgStudentDao;

    @Resource
    private OrgCourseDao orgCourseDao;

    @Resource
    private CourseLessonService courseLessonService;

    @Resource
    private OrgLessonSignService orgLessonSignService;

    @Resource
    private CourseStudentService courseStudentService;

    @Resource
    private OrgClassLessonDao orgClassLessonDao;

    @Resource
    private OrgStudentLessonDao orgStudentLessonDao;

    @Resource
    private OrgCourseConsumeRuleService orgCourseConsumeRuleService;

    @Autowired
    private OrgLessonSignLogDao orgLessonSignLogDao;

    @Autowired
    private TxCascadeCredentialService txCascadeCredentialService;

    private static final String SIGN_STATUS_CHANGE_TEMPLATE = "%s 于 %s 将状态标记为 [%s]";
    private static final String SIGN_WITH_SOURCE_STATUS_CHANGE_TEMPLATE = "%s 于 %s 通过 %s 将状态标记为 [%s]";
    private static final String SIGN_REMARK_CHANGE_TEMPLATE = "%s 于 %s 修改备注为 %s";
    private static final String SIGN_REMARK_DELETE_TEMPLATE = "%s 于 %s 删除备注";
    private static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyy/M/d HH:mm");

    @Override
    public FastSignInViewCourseDto getSignInCourseInfo(Long orgId, Long courseId) {
        OrgCourse course = this.orgCourseDao.getById(courseId);
        if (course != null) {
            List<FastSignInViewCourseDto> dtoList = getSignInCourseInfoList(orgId, Lists.newArrayList(course));
            if (dtoList.size() > 0) {
                return dtoList.get(0);
            }
        }
        throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "课程id无效");

    }

    private List<OrgLessonSign> queryEfectLessonSignBatch(Long orgId, List<Long> courseIds) {

        List<OrgLessonSign> list = orgLessonSignDao.getCourseLessonSignIn(orgId, courseIds);

        return list;

    }

    private Map<Long, Integer> findCourseLessonSignInCountMap(List<OrgLessonSign> list) {
        Map<Long, Integer> map = Maps.newHashMap();
        Set<Long> lessonSet = new HashSet<Long>();
        for (OrgLessonSign orgLessonSign : list) {
            if (orgLessonSign.getStatus() != null && orgLessonSign.getStatus() > 0) {
                Long courseId = orgLessonSign.getCourseId();
                Long lessonId = orgLessonSign.getLessonId();
                if (!lessonSet.contains(lessonId)) {
                    lessonSet.add(lessonId);
                    Integer count = map.get(courseId);
                    if (count == null) {
                        count = new Integer(0);
                        map.put(courseId, count);
                    }
                    count++;
                    map.put(courseId, count);
                }
            }
        }
        return map;
    }

    private Map<Long, Integer> findLessonSignInCountMap(List<OrgLessonSign> list) {
        Map<Long, Integer> map = Maps.newHashMap();
        for (OrgLessonSign orgLessonSign : list) {
            if (orgLessonSign.getStatus() != null && orgLessonSign.getStatus() > SignStatus.UNSIGN.getCode()) {
                Long lessonId = orgLessonSign.getLessonId();
                Integer count = map.get(lessonId);
                if (count == null) {
                    count = new Integer(0);
                    map.put(lessonId, count);
                }
                count++;
                map.put(lessonId, count);
            }
        }
        return map;
    }

    private Map<Long, Integer> findLessonSignedCountMap(List<OrgLessonSign> list) {
        Map<Long, Integer> map = Maps.newHashMap();
        for (OrgLessonSign orgLessonSign : list) {
            if (orgLessonSign.getStatus() != null && orgLessonSign.getStatus() == SignStatus.SIGNED.getCode()) {
                Long lessonId = orgLessonSign.getLessonId();
                Integer count = map.get(lessonId);
                if (count == null) {
                    count = new Integer(0);
                    map.put(lessonId, count);
                }
                count++;
                map.put(lessonId, count);
            }
        }
        return map;
    }

    @Override
    public List<FastSignInViewCourseDto> getSignInCourseInfoList(Long orgId, List<OrgCourse> orgCourseList) {

        List<FastSignInViewCourseDto> dtoList = Lists.newArrayList();
        Set<Long> courseIdSet = Sets.newHashSet();
        for (OrgCourse couse : orgCourseList) {
            courseIdSet.add(couse.getId());
        }
        List<Long> courseIds = new ArrayList(courseIdSet);
        Map<Integer, String> coverMap = orgCourseListService.getCoverMap(orgCourseList);
        Map<Long, OrgCoursePhoto> coursePhotoMap = orgCourseListService.getPhoteMap(courseIds);
        Map<Long, Integer> studentCountMap = courseStudentService.getCourseStudentCntMap(orgId, courseIds, false);
        List<OrgLessonSign> signList = queryEfectLessonSignBatch(orgId, courseIds);
        Map<Long, Integer> courseLessonSignInCount = findCourseLessonSignInCountMap(signList);
        Map<Long, Integer> lessonCountMap =
            this.orgClassLessonDao.getLessonCount(orgId, courseIdSet, DeleteStatus.NORMAL.getValue());

        Map<Long, Map<Long, Integer>> courseLessonLengthMap =
            orgClassLessonDao.getLessonLengthMap(orgId, courseIdSet, DeleteStatus.NORMAL.getValue());
        log.debug("courseLessonLengthMap:{}", courseLessonLengthMap);
        Map<Long, Map<Long, Integer>> courseLessonSigninLengthMap =
            findCourseLessonSignInLengthMap(signList, courseLessonLengthMap);
        log.debug("courseLessonSigninLengthMap:{}", courseLessonSigninLengthMap);

        for (OrgCourse course : orgCourseList) {

            FastSignInViewCourseDto dto = new FastSignInViewCourseDto();

            dto.setOrgCourseId(course.getId());
            dto.setCourseName(course.getName());
            dto.setCourseType(course.getCourseType());

            dto.setChargeUnit(course.getChargeUnit());
            Map<Long, Integer> signMinutesMap = courseLessonSigninLengthMap.get(course.getId());
            dto.setSignMinutes(null != signMinutesMap ? intSum(signMinutesMap.values()) : 0);
            Map<Long, Integer> totalMinutesMap = courseLessonLengthMap.get(course.getId());
            dto.setTotalMinutes(null != totalMinutesMap ? intSum(totalMinutesMap.values()) : 0);

            dto.setCourseCount(null == course.getFreq() ? 0 : course.getFreq());
            dto.setMaxStudent(null == course.getMaxStudent() ? 0 : course.getMaxStudent());
            dto.setStudentCount(null == studentCountMap.get(course.getId()) ? 0 : studentCountMap.get(course.getId()));
            dto.setSignCount(
                null == courseLessonSignInCount.get(course.getId()) ? 0 : courseLessonSignInCount.get(course.getId()));
            dto.setLessonCount(null == lessonCountMap.get(course.getId()) ? 0 : lessonCountMap.get(course.getId()));
            dto.setCoverUrl(coverMap.get(course.getCover()));
            OrgCoursePhoto coursePhoto = coursePhotoMap.get(course.getId());
            if (coursePhoto != null && StringUtils.isNotBlank(coursePhoto.getCutUrl())) {
                dto.setCoverUrl(coursePhoto.getCutUrl());
            }
            dtoList.add(dto);
        }
        return dtoList;

    }

    private Long intSum(Collection<Integer> values) {
        long sum = 0;
        if (CollectionUtils.isNotEmpty(values)) {
            for (Integer value : values) {
                if (value == null) {
                    continue;
                }
                sum += value;
            }
        }
        return sum;
    }

    private Map<Long, Map<Long, Integer>> findCourseLessonSignInLengthMap(List<OrgLessonSign> signList,
        Map<Long, Map<Long, Integer>> courseLessonLengthMap) {
        Map<Long, Map<Long, Integer>> result = Maps.newHashMap();
        for (OrgLessonSign orgLessonSign : signList) {
            if (orgLessonSign.getStatus() != null && orgLessonSign.getStatus() > 0) {
                Long courseId = orgLessonSign.getCourseId();
                Long lessonId = orgLessonSign.getLessonId();
                Map<Long, Integer> innerMap = result.get(courseId);
                if (null == innerMap) {
                    innerMap = Maps.newHashMap();
                    result.put(courseId, innerMap);
                }
                innerMap.put(lessonId, courseLessonLengthMap.get(courseId).get(lessonId));
            }
        }
        return result;
    }

    @Override
    @Transactional
    public List<Long> outerCreateLessonsAndSignIn(FastSignInLesssonsRequestDto dto) {

        // 单独处理按时间的快速签到
        if (null != dto.getChargeUnit() && (dto.getChargeUnit() == ChargeUnit.BY_HOUR.getCode()
            || dto.getChargeUnit() == ChargeUnit.BY_HALF_HOUR.getCode()
            || dto.getChargeUnit() == ChargeUnit.BY_MINUTE.getCode())) {
            Preconditions.checkArgument(
                null != dto.getSignMinutes() && 0 < dto.getSignMinutes() && 1 == dto.getLessonTimes(),
                "课消类型为按时间时,传入的签到时长,或快速签到次数错误");

            List<TimeRange> timeRangeList = getTimeRangeListFromNow(dto.getLessonTimes(), dto.getSignMinutes());

            return batchCreateLessonsAndSignIn(dto.getOrgId(), dto.getCourseId(), timeRangeList,
                dto.getStudentSignStatusMap());
        }

        List<TimeRange> timeRangeList = getTimeRangeListFromNow(dto.getLessonTimes(), 60);

        int batchSize = 3;
        // 分批事物插入，每批1个课节然后再组合数据
        int batchNum = dto.getLessonTimes() / batchSize;
        int limit = dto.getLessonTimes() % batchSize;
        List<Long> newLessonIds = Lists.newArrayList();
        //
        for (int i = 0; i < batchNum; i++) {
            try {
                List<TimeRange> subTimeRangeList = timeRangeList.subList(i * batchSize, (i + 1) * batchSize);
                List<Long> tempNewLessonIds = batchCreateLessonsAndSignIn(dto.getOrgId(), dto.getCourseId(),
                    subTimeRangeList, dto.getStudentSignStatusMap());
                newLessonIds.addAll(tempNewLessonIds);
            } catch (Exception e) {
                log.warn("erpSignInService.batchCreateLessonsAndSignIn", e);
            }
        }
        if (limit > 0) {
            try {
                List<TimeRange> subTimeRangeList =
                    timeRangeList.subList(timeRangeList.size() - limit, timeRangeList.size());
                List<Long> tempNewLessonIds = batchCreateLessonsAndSignIn(dto.getOrgId(), dto.getCourseId(),
                    subTimeRangeList, dto.getStudentSignStatusMap());
                newLessonIds.addAll(tempNewLessonIds);
            } catch (Exception e) {
                log.warn("erpSignInService.batchCreateLessonsAndSignIn", e);
            }
        }
        return newLessonIds;
    }

    private List<TimeRange> getTimeRangeListFromNow(@NonNull Integer lessonTimes, Integer minutes) {
        Date now = DateUtil.getNext5MinDevide(new Date());
        Date todayEndTime =
            new DateTime(DateUtil.getStartOfDay(new DateTime(now).plusDays(1).toDate())).minusMinutes(5).toDate();
        Date end = new DateTime(now).plusMinutes(lessonTimes * minutes).toDate();
        end = end.after(todayEndTime) ? todayEndTime : end;
        LinkedList<Date> dateList = Lists.newLinkedList();
        for (int i = 0; i <= lessonTimes; i++) {
            dateList.addFirst(end);
            end = new DateTime(end).minusMinutes(minutes).toDate();
        }
        List<TimeRange> result = Lists.newArrayList();
        for (int i = 0; i < lessonTimes; i++) {
            result.add(new TimeRange(dateList.get(i), dateList.get(i + 1)));
        }
        return result;
    }

    @Transactional(rollbackFor = Exception.class)
    private List<Long> batchCreateLessonsAndSignIn(@NonNull Long orgId, @NonNull Long courseId,
        List<TimeRange> timeRangeList, Map<Long, Integer> studentSignStatusMap) {
        List<Long> lessonIds = courseLessonService.addClassLessonsBatchForFastSignIn(orgId, courseId, timeRangeList,
            studentSignStatusMap.keySet());
        if (lessonIds != null && lessonIds.size() == timeRangeList.size()) {
            orgLessonSignService.orgStudentLessonBatchFastSign(orgId, courseId, lessonIds, studentSignStatusMap);
        } else {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "新建课节数目不匹配");
        }
        return lessonIds;
    }

    private Collection<Long> getCourseIdList(List<OrgClassLesson> lessons) {
        Set<Long> courseIds = Sets.newHashSet();
        for (OrgClassLesson lesson : lessons) {
            courseIds.add(lesson.getCourseId());
        }
        return courseIds;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<LessonResponseDto> getCourseLessonSignInRecord(Long orgId, Long courseId) {
        OrgCourse course = this.orgCourseDao.getById(courseId);
        if (course == null) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "课程id无效");
        }
        /*
         * if(course.getOrgNumber().longValue() != orgId){ throw new BussinessException(CommonErrorCode.BUSINESS_ERROR,
         * "课程id无效"); }
         */

        List<Long> courseIds = new ArrayList<Long>();
        courseIds.add(courseId);
        List<OrgClassLesson> lessonList =
            orgClassLessonDao.queryLessons(orgId, courseIds, null, null, null, null, null, null);
        // 获取课节基本信息
        List<LessonResponseDto> lessonResponList = courseLessonService.buildLessonBaseInfoList(orgId, lessonList);

        // 获取课节签到信息
        List<OrgLessonSign> signList = queryEfectLessonSignBatch(orgId, courseIds);
        Map<Long, Integer> courseLessonSignInCount = findLessonSignInCountMap(signList);
        Map<Long, Integer> courseLessonSignedCount = findLessonSignedCountMap(signList);
        List<Long> lessonIds = Lists.newArrayList();
        for (OrgClassLesson lesson : lessonList) {
            lessonIds.add(lesson.getId());
        }
        // 获取课节学生数
        Map<Long, Integer> studentCountMap = orgStudentLessonDao.queryLessonStudentCountMap(orgId, lessonIds);

        for (LessonResponseDto dto : lessonResponList) {
            Long lessonId = dto.getLessonId();
            Integer signCount = courseLessonSignInCount.get(lessonId);
            Integer inClassCount = courseLessonSignedCount.get(lessonId);
            Integer studentCount = studentCountMap.get(lessonId);
            dto.setTotalSignCount(signCount != null ? signCount : 0);
            dto.setSignCount(inClassCount != null ? inClassCount : 0);
            dto.setStudentCount(studentCount != null ? studentCount : 0);
        }
        return lessonResponList;

    }

    /**
     * 获取课程封面urlmap
     *
     * @param courses
     */
    private Map<Integer, String> getCoverMap(List<OrgCourse> courses) {
        Collection<Integer> coverList = CollectorUtil.collect(courses, new Function<OrgCourse, Integer>() {
            @Override
            public Integer apply(OrgCourse arg0) {
                return arg0.getCover();
            }
        });
        List<Storage> storages = this.storageDao.getByIds(coverList);
        return CollectorUtil.collectMap(storages, new Function<Storage, Integer>() {
            @Override
            public Integer apply(Storage arg0) {
                return arg0.getId().intValue();
            }
        }, new Function<Storage, String>() {
            @Override
            public String apply(Storage arg0) {
                if (StringUtils.isNotBlank(arg0.getAttach_url())) {
                    return arg0.getAttach_url();
                }
                return StorageUtil.constructUrl(arg0.getFid(), arg0.getMimetype(), arg0.getSn());
            }
        });
    }

    private Long getOrgNumber(Long orgId) {
        OrgAccount account = this.orgAccountDao.getAccountById(orgId.intValue(), "number");
        if (account == null) {
            return null;
        }
        Integer number = account.getNumber();
        return number != null ? number.longValue() : null;
    }

    /**
     * 课程搜素
     */
    @Override
    public List<FastSignInViewCourseDto> getCourseInfo(Long orgId, String searchName, Integer courseType,
        Integer limit) {
        log.info("searchName = {}", searchName);
        Preconditions.checkArgument(searchName != null && !searchName.isEmpty(), "orgId invalid!");
        Long orgNumber = getOrgNumber(orgId);
        List<Long> courseIds = null;
        // 登录帐号为子帐号，并且是员工帐号，并且排课权限打开时，只能查看自己作为班主任的课节
        // 涉及到分页，这里取courseIds时，按照时间倒序，测试时需要关注数据是否取全
        if (TianxiaoMContext.getTXCascadeId() != null) {
            TXCascadeAccount txCascadeAccount = txCascadeAccountDao.getById(TianxiaoMContext.getTXCascadeId());
            if (txCascadeAccount == null) {
                throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "子帐号不存在");
            }

            if (txCascadeAccount.getAccountType() == CascadeType.STAFF.getValue()) {
                log.debug("before clear courseIds:{}", courseIds);
                List<Long> tempIds = orgCourseDao.getCourseIdsByCascadeId(TianxiaoMContext.getTXCascadeId(), null,
                    CourseTypeEnum.IS_CLASS_TRUE.getCode(), null);
                log.debug("user cascadeId={},courseIds={}", TianxiaoMContext.getTXCascadeId(), tempIds);

                log.debug("after set courseIds:{}", courseIds);
                if (CollectionUtils.isEmpty(tempIds)) {
                    return Collections.EMPTY_LIST;
                } else {
                    courseIds = tempIds;
                }
            }
        }

        List<OrgCourse> orgCourseList = orgCourseDao.getCourseByOrgNumberAndCourseName(orgNumber, searchName, null,
            courseIds, null, null, courseType, limit);

        List<FastSignInViewCourseDto> fastSignViewCourseDtoList = getSignInCourseInfoList(orgId, orgCourseList);

        return fastSignViewCourseDtoList;
    }

    /**
     * 学员姓名搜索
     */
    @Override
    public List<FastSignInViewCourseDto> getStudentInfo(Long orgId, String searchName, Integer courseType,
        Integer limit) {
        Long orgNumber = getOrgNumber(orgId);
        // 包含该学员的班级
        // List<OrgStudent> orgStudents = orgStudentDao.getStudents(orgId, searchName, null, null, null);
        List<OrgStudent> orgStudents = orgStudentDao.getStudentByOrgIdAndCourseName(orgId, searchName);
        Set<Long> orgStudentCoursesIds = Sets.newHashSet();

        // 每一个学生的排课情况
        Set<Long> userIds = Sets.newHashSet();
        for (OrgStudent orgStudent : orgStudents) {
            userIds.add(orgStudent.getUserId());
        }
        List<OrgStudentCourse> orgStudentCourses = null;
        if (userIds.size() > 0) {
            orgStudentCourses = orgStudentCourseDao.getOrgStudentCourseByUserIds(orgId, userIds);
            userIds.removeAll(userIds);
            for (int j = 0; j < orgStudentCourses.size(); j++) {

                userIds.add(orgStudentCourses.get(j).getUserId());// 对于没有排课的不进行搜索
                orgStudentCoursesIds.add(orgStudentCourses.get(j).getCourseId());
            }
        }

        List<Long> courseIds = null;
        // 登录帐号为子帐号，并且是员工帐号，并且排课权限打开时，只能查看自己作为班主任的课节
        // 涉及到分页，这里取courseIds时，按照时间倒序，测试时需要关注数据是否取全
        if (TianxiaoMContext.getTXCascadeId() != null) {
            TXCascadeAccount txCascadeAccount = txCascadeAccountDao.getById(TianxiaoMContext.getTXCascadeId());
            if (txCascadeAccount == null) {
                throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "子帐号不存在");
            }
            if (txCascadeAccount.getAccountType() == CascadeType.STAFF.getValue()) {
                log.debug("before clear courseIds:{}", courseIds);
                List<Long> tempIds = orgCourseDao.getCourseIdsByCascadeId(TianxiaoMContext.getTXCascadeId(), null,
                    CourseTypeEnum.IS_CLASS_TRUE.getCode(), null);
                log.debug("user cascadeId={},courseIds={}", TianxiaoMContext.getTXCascadeId(), tempIds);
                if (CollectionUtils.isEmpty(tempIds)) {
                    return Collections.EMPTY_LIST;
                } else {
                    courseIds = tempIds;
                }

            }

        }
        if (courseIds != null) {
            orgStudentCoursesIds.retainAll(courseIds);
        }

        // 学生姓名
        Map<Long, List<Long>> courseStudentUserIdMap =
            orgStudentCourseDao.getStudentIdMapBycourseIdsNoStatus(orgStudentCoursesIds, orgId);
        List<Long> orgIds = Lists.newArrayList();
        orgIds.add(orgId);

        List<OrgStudent> orgStudentsList = orgStudentDao.getStudentsByUserIdsAndOrgIds(orgIds, userIds);
        Map<Long, String> orgStudentsMap = Maps.newHashMap();
        Map<Long, List<String>> courseStudentNameMap = Maps.newHashMap();
        for (OrgStudent stu : orgStudentsList) {
            orgStudentsMap.put(stu.getUserId(), stu.getName());
        }
        for (Long courseId : orgStudentCoursesIds) {
            List<Long> courseUserIds = courseStudentUserIdMap.get(courseId);
            List<Long> tempUserIds = new ArrayList<Long>(courseUserIds);
            tempUserIds.retainAll(userIds);

            List<String> studentNameList = Lists.newArrayList();
            for (Long userId : tempUserIds) {
                String name = orgStudentsMap.get(userId);
                if (StringUtils.isNotBlank(name)) {
                    studentNameList.add(name);
                }
            }
            courseStudentNameMap.put(courseId, studentNameList);

        }

        // 每个学生的所有课程
        List<OrgCourse> orgCourseList = Lists.newArrayList();
        if (orgStudentCoursesIds.size() > 0) {
            orgCourseList = orgCourseDao.getCourseByOrgNumberAndCourseName(orgNumber, null, null, orgStudentCoursesIds,
                null, CourseTypeEnum.IS_CLASS_TRUE.getCode(), courseType, limit);
        }
        List<FastSignInViewCourseDto> fastSignViewCourseDtoList = getSignInCourseInfoList(orgId, orgCourseList);

        for (FastSignInViewCourseDto dto : fastSignViewCourseDtoList) {
            Long courseId = dto.getOrgCourseId();
            List<String> studentNameList = courseStudentNameMap.get(courseId);
            dto.setStudentNames(studentNameList);
        }
        return fastSignViewCourseDtoList;
    }

    public CourseSigninStsDto queryCourseStudentSignInInfoDto(Long orgId, Long courseId) {
        OrgCourse course = this.orgCourseDao.getById(courseId);
        if (course == null) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "课程id无效");
        }

        List<Long> courseIds = new ArrayList<Long>();
        courseIds.add(courseId);
        List<OrgClassLesson> lessonList =
            orgClassLessonDao.queryLessons(orgId, courseIds, null, null, null, null, null, null);
        Map<Long, String> courseIdNameMap = orgCourseDao.getCourseNameMap(getCourseIdList(lessonList));
        // 获取课节基本信息
        List<LessonResponseDto> lessonResponList = courseLessonService.buildLessonBaseInfoList(orgId, lessonList);

        List<Long> userIds = orgStudentCourseDao.getStudents(orgId, courseId);

        Map<Long, OrgStudent> studentMap = orgStudentDao.getStudentMap(userIds, orgId);
        // 获取课节签到信息
        List<OrgLessonSign> signList = queryEfectLessonSignBatch(orgId, courseIds);

        CourseSigninStsDto dto = new CourseSigninStsDto();
        dto.setCourseId(courseId);
        dto.setPlanNum(course.getFreq());
        dto.setCourseName(course.getName());
        dto.setLessonCourseList(lessonResponList);
        dto.setLessonNum(lessonResponList.size());
        List<StudentSigninSts> studentSigninStsList = new ArrayList<StudentSigninSts>();
        dto.setStudentSigninStsList(studentSigninStsList);

        Map<Long, StudentSigninSts> studentSigninCountMap = new HashMap<Long, StudentSigninSts>();

        Map<String, Integer> studentSignStatusMap = new HashMap<String, Integer>();
        dto.setSignInStatusMap(studentSignStatusMap);
        for (OrgLessonSign lessonSign : signList) {
            studentSignStatusMap.put(lessonSign.getLessonId() + "|" + lessonSign.getUserId(), lessonSign.getStatus());
            Long userId = lessonSign.getUserId();
            StudentSigninSts sts = studentSigninCountMap.get(userId);
            if (sts == null) {
                sts = new StudentSigninSts();
                sts.setUserId(userId);

                studentSigninCountMap.put(userId, sts);
            }

            if (lessonSign.getStatus() == 1) {
                sts.setSignedCount(sts.getSignedCount() + 1);
                sts.setSignCount(sts.getSignCount() + 1);
            } else if (lessonSign.getStatus() == 2) {
                sts.setLeaveCount(sts.getLeaveCount() + 1);
                sts.setSignCount(sts.getSignCount() + 1);
            } else if (lessonSign.getStatus() == 3) {
                sts.setAbsentCount(sts.getAbsentCount() + 1);
                sts.setSignCount(sts.getSignCount() + 1);
            }
        }

        for (OrgStudent orgStudent : studentMap.values()) {
            Long userId = orgStudent.getUserId();
            StudentSigninSts sts = studentSigninCountMap.get(userId);
            if (sts != null) {
                sts.setStudentName(orgStudent.getName());
            } else {
                sts = new StudentSigninSts();
                sts.setUserId(userId);
                sts.setStudentName(orgStudent.getName());
            }
            studentSigninStsList.add(sts);
        }

        return dto;

    }

    public CourseSigninRecordDto queryCourseStudentSignInDetailsDto(Long orgId, Long courseId) {
        OrgCourse course = this.orgCourseDao.getById(courseId);
        if (course == null) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "课程id无效");
        }

        List<Long> courseIds = new ArrayList<Long>();
        courseIds.add(courseId);
        List<OrgClassLesson> lessonList =
            orgClassLessonDao.queryLessons(orgId, courseIds, null, null, null, null, null, null);
        // 获取课节基本信息
        List<LessonResponseDto> lessonResponList = courseLessonService.buildLessonBaseInfoList(orgId, lessonList);

        List<Long> lessonIds = new ArrayList<Long>();
        for (LessonResponseDto lesson : lessonResponList) {
            lessonIds.add(lesson.getLessonId());
        }

        List<Long> userIds = orgStudentCourseDao.getStudents(orgId, courseId);

        Map<Long, OrgStudent> studentMap = orgStudentDao.getStudentMap(userIds, orgId);
        // 获取课节签到信息
        List<OrgLessonSign> signList = queryEfectLessonSignBatch(orgId, courseIds);

        CourseSigninRecordDto dto = new CourseSigninRecordDto();
        OrgCourseConsumeRule orgCourseConsumeRule = orgCourseConsumeRuleService.getRuleByCourseId(orgId, course);
        dto.setConsumRuleValue(orgCourseConsumeRule.getRuleValue());
        dto.setConsumeRuleStr(orgCourseConsumeRule.getRuleDesc());
        dto.setCourseId(courseId);
        dto.setCourseName(course.getName());
        dto.setPlanNum(course.getFreq());
        dto.setLessonCourseList(lessonResponList);
        dto.setLessonNum(lessonResponList.size());
        List<StudentSigninRecord> studentSigninRecordList = new ArrayList<StudentSigninRecord>();
        dto.setStudentSigninRecordList(studentSigninRecordList);

        Map<Long, StudentSigninRecord> studentSigninCountMap = new HashMap<Long, StudentSigninRecord>();

        Map<String, SignStatusRemarkDto> studentSignStatusMap = new HashMap<String, SignStatusRemarkDto>();
        dto.setSignInStatusMap(studentSignStatusMap);
        Set<String> studentLessonSet = Sets.newHashSet();
        dto.setStudentLessonSet(studentLessonSet);

        Map<Long, List<Long>> userIdsLessonMap = orgStudentLessonDao.getLessonIdsOfStudents(orgId, userIds, lessonIds);
        for (Long userId : userIdsLessonMap.keySet()) {
            List<Long> lessons = userIdsLessonMap.get(userId);
            for (Long lessonId : lessons) {
                studentLessonSet.add((String.valueOf(lessonId + "|" + userId)));
            }
        }

        for (OrgLessonSign lessonSign : signList) {
            studentSignStatusMap.put(lessonSign.getLessonId() + "|" + lessonSign.getUserId(),
                new SignStatusRemarkDto(lessonSign.getStatus(), lessonSign.getSignRemark()));
            Long userId = lessonSign.getUserId();
            StudentSigninRecord sts = studentSigninCountMap.get(userId);
            if (sts == null) {
                sts = new StudentSigninRecord();
                sts.setUserId(userId);

                studentSigninCountMap.put(userId, sts);
            }

            if (lessonSign.getStatus() == 1) {
                sts.setSignedCount(sts.getSignedCount() + 1);
                sts.setSignCount(sts.getSignCount() + 1);
            } else if (lessonSign.getStatus() == 2) {
                sts.setLeaveCount(sts.getLeaveCount() + 1);
                sts.setSignCount(sts.getSignCount() + 1);
            } else if (lessonSign.getStatus() == 3) {
                sts.setAbsentCount(sts.getAbsentCount() + 1);
                sts.setSignCount(sts.getSignCount() + 1);
            }
            // sts.setArrangeCount(userIdsLessonMap.get(userId).size());
        }

        for (OrgStudent orgStudent : studentMap.values()) {
            Long userId = orgStudent.getUserId();
            StudentSigninRecord ssr = studentSigninCountMap.get(userId);
            // 删除状态并且排课是0，过滤掉
            if (orgStudent.getDelStatus() == 1 && CollectionUtils.isEmpty(userIdsLessonMap.get(userId))) {
                continue;
            }
            if (ssr != null) {
                ssr.setStudentName(orgStudent.getName());
            } else {
                if (orgStudent.getDelStatus() == 1) {
                    continue;
                }
                ssr = new StudentSigninRecord();
                ssr.setUserId(userId);
                ssr.setStudentName(orgStudent.getName());
            }
            studentSigninRecordList.add(ssr);
        }

        return dto;
    }

    private Map<Long, List<OrgLessonSign>> getUserIdLessonSignMap(List<OrgLessonSign> signList) {
        return CollectorUtil.group(signList, new Function<OrgLessonSign, Long>() {
            @Override
            public Long apply(OrgLessonSign arg0) {
                return arg0.getUserId();
            }
        }, new Function<OrgLessonSign, OrgLessonSign>() {
            @Override
            public OrgLessonSign apply(OrgLessonSign arg0) {
                return arg0;
            }
        });
    }

    private Map<Long, Integer> toLessonSignMap(List<OrgLessonSign> signList) {
        Map<Long, Integer> lessonSignMap = Maps.newHashMap();
        if (CollectionUtils.isEmpty(signList)) {
            return Collections.emptyMap();
        }
        for (OrgLessonSign sign : signList) {
            lessonSignMap.put(sign.getLessonId(), sign.getStatus());
        }
        return lessonSignMap;
    }

    public StudentSigninRecordDto queryStudentSignInDetailsDto(Long orgId, Long studentId) {
        OrgStudent student = this.orgStudentDao.getStudentByUserId(orgId, studentId);
        if (student == null) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "学生id无效");
        }
        StudentSigninRecordDto dto = new StudentSigninRecordDto();
        dto.setStudentId(studentId);
        dto.setStudentName(student.getName());
        Long userId = student.getUserId();
        List<Long> userIds = Lists.newArrayList();
        userIds.add(userId);
        List<OrgStudentCourse> orgStudentCourses = orgStudentCourseDao.getStudentListByStudentId(studentId);

        List<Long> courseIds = new ArrayList<Long>();
        for (OrgStudentCourse orgStudentCourse : orgStudentCourses) {
            courseIds.add(orgStudentCourse.getCourseId());
        }
        Map<Long, OrgCourse> courseIdNameMap = orgCourseDao.getOrgCourseMap(courseIds);
        List<OrgClassLesson> lessonList =
            orgClassLessonDao.queryLessons(orgId, courseIds, null, null, null, null, null, null);

        // 1.获取课节基本信息
        Map<Long, List<LessonResponseDto>> lessonCourseListMap = new HashMap<Long, List<LessonResponseDto>>();

        List<Long> lessonIds = new ArrayList<Long>();
        for (OrgClassLesson lesson : lessonList) {
            Long lessonId = lesson.getId();
            lessonIds.add(lessonId);
            Long courseId = lesson.getCourseId();
            List<LessonResponseDto> courseLessonList = lessonCourseListMap.get(courseId);
            if (courseLessonList == null) {
                courseLessonList = Lists.newArrayList();
                lessonCourseListMap.put(courseId, courseLessonList);
            }
            LessonResponseDto lessonDto = new LessonResponseDto();
            lessonDto.setCourseId(lesson.getCourseId());
            lessonDto.setLessonId(lesson.getId());
            lessonDto.setIndex(lesson.getNumber());
            lessonDto.setLessonStartTime(lesson.getStartTime());
            lessonDto.setLessonEndTime(lesson.getEndTime());
            courseLessonList.add(lessonDto);
        }

        // 2获取课节签到信息
        List<OrgLessonSign> lessonSignList = orgLessonSignDao.getStudentSign(orgId, userId, UserRole.STUDENT.getRole(),
            lessonIds, "lessonId", "userId", "updateTime", "status", "signRemark");
        Map<String, OrgLessonSign> studentSignStatusMap = new HashMap<String, OrgLessonSign>();
        for (OrgLessonSign lessonSign : lessonSignList) {
            studentSignStatusMap.put(lessonSign.getLessonId() + "|" + lessonSign.getUserId(), lessonSign);

        }
        dto.setSignInStatusMap(studentSignStatusMap);

        // 获取学生排课信息
        Set<String> studentLessonSet = Sets.newHashSet();
        dto.setStudentLessonSet(studentLessonSet);
        Map<Long, List<Long>> userIdsLessonMap = orgStudentLessonDao.getLessonIdsOfStudents(orgId, userIds, lessonIds);
        for (Long tempuserId : userIdsLessonMap.keySet()) {
            List<Long> lessons = userIdsLessonMap.get(tempuserId);
            for (Long lessonId : lessons) {
                String tempKey = String.valueOf(lessonId + "|" + tempuserId);
                studentLessonSet.add(tempKey);

            }
        }
        List<CourseSigninResult> courseSigninResultList = Lists.newArrayList();
        dto.setCourseSigninResultList(courseSigninResultList);
        Date now = new Date();
        Map<Long, List<LessonResponseDto>> lessonCourseListMapNew = new HashMap<Long, List<LessonResponseDto>>();
        dto.setLessonCourseListMap(lessonCourseListMapNew);

        for (Long courseId : lessonCourseListMap.keySet()) {
            OrgCourse course = courseIdNameMap.get(courseId);
            if (course == null) {
                continue;
            }
            List<LessonResponseDto> tempLessonDtoList = new ArrayList<LessonResponseDto>();
            lessonCourseListMapNew.put(courseId, tempLessonDtoList);

            CourseSigninResult signinResult = new CourseSigninResult();
            courseSigninResultList.add(signinResult);

            signinResult.setCourseId(courseId);

            signinResult.setCourseName(course.getName());
            if (null == course.getFreq()) {
                signinResult.setPlanCount(0);
            } else {
                signinResult.setPlanCount(course.getFreq());
            }
            OrgCourseConsumeRule rule = orgCourseConsumeRuleService.getRuleByCourseId(orgId, course);
            signinResult.setCourseConsumeRuleStr(rule.getRuleDesc());
            signinResult.setCourseConsumeRuleValue(rule.getRuleValue());
            boolean timeEndConsume = false;
            if (signinResult.getCourseConsumeRuleValue() == null || signinResult.getCourseConsumeRuleValue() == 0) {
                timeEndConsume = true;
            }
            int arrangeCount = 0;
            int signedCount = 0;
            int leaveCount = 0;
            int absentCount = 0;
            int endCount = 0;
            for (LessonResponseDto lesson : lessonCourseListMap.get(courseId)) {
                tempLessonDtoList.add(lesson);
                String tempKey = String.valueOf(lesson.getLessonId() + "|" + userId);

                if (studentLessonSet.contains(tempKey)) {

                    arrangeCount++;
                    OrgLessonSign tempSigninStatus = studentSignStatusMap.get(tempKey);
                    if (tempSigninStatus != null) {
                        if (tempSigninStatus.getStatus() == SignStatus.SIGNED.getCode()) {
                            signedCount++;
                        } else if (tempSigninStatus.getStatus() == SignStatus.LEAVE.getCode()) {
                            leaveCount++;
                        } else if (tempSigninStatus.getStatus() == SignStatus.ABSENT.getCode()) {
                            absentCount++;
                        }
                        lesson.setSignTime(tempSigninStatus.getUpdateTime());
                    }
                    if (timeEndConsume) {
                        if (lesson.getLessonStartTime().before(now)) {
                            endCount++;
                        }
                    } else if (tempSigninStatus != null) {
                        boolean consumed = CourseConsumeRuleEnum.signStatusCodeIsConsume(
                            signinResult.getCourseConsumeRuleValue(), tempSigninStatus.getStatus());
                        if (consumed) {
                            endCount++;
                        }
                    }
                }

            }
            signinResult.setSignedCount(signedCount);
            signinResult.setLeaveCount(leaveCount);
            signinResult.setAbsentCount(absentCount);
            signinResult.setArrangeCount(arrangeCount);
            signinResult.setEndCount(endCount);

        }
        return dto;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public SigninFailedDto unionStudentsMultiLessonBatchSign(long orgId, Long courseId, List<Long> lessonIds,
        List<SigninStatusDto> list, int confirm) {

        Map<Long, Long> studentUserMap = CollectorUtil // studentId -> userId
            .collectMap(orgStudentDao.getByIds(Lists.transform(list, input -> input.getStudentId()), "id", "userId"),
                input -> input.getId(), input -> input.getUserId());

        // 退转班的学生
        Map<Long, Integer> uIdStatusWithdrawMap =
            orgStudentCourseDao.userMapByStatus(orgId, courseId, studentUserMap.values(),
                Arrays.asList(StudentCourseStatus.WITHDRAW.getCode(), StudentCourseStatus.TRANSFER.getCode()));
        Collection<Long> inValidUserIds =
            CollectionUtils.intersection(studentUserMap.values(), uIdStatusWithdrawMap.keySet());
        SigninFailedDto result = new SigninFailedDto();
        for (Long userId : inValidUserIds) {
            result.addFailedStudent(new SigninFailedDto.FailedStudent(userId, uIdStatusWithdrawMap.get(userId)));
        }
        log.info("Quit class can not modify allInvalidUserIds={}", uIdStatusWithdrawMap);

        Map<Integer, List<Long>> statusUsersMap = Maps.newHashMap();
        for (SigninStatusDto dto : list) {
            Long userId = studentUserMap.get(dto.getStudentId());
            if (uIdStatusWithdrawMap.containsKey(userId)) {
                continue;
            }
            if (statusUsersMap.containsKey(dto.getStatus())) {
                statusUsersMap.get(dto.getStatus()).add(userId);
            } else {
                statusUsersMap.put(dto.getStatus(), Lists.newArrayList(userId));
            }
        }
        // 按签到状态不同,分批次搞 有可能性能差 //FIXME
        for (Integer status : statusUsersMap.keySet()) {
            orgLessonSignService.orgUnionStudentsMultiLessonBatchSign(orgId, courseId, lessonIds,
                statusUsersMap.get(status), status, confirm);
        }
        orgLessonSignService.fillStudentInfo(orgId, result);
        return result;
    }

    @Override
    public List<SignLogDto> getLessonSignLogStrs(Long orgId, Long lessonId, Long studentId) {
        Long userId = orgStudentDao.getById(studentId).getUserId();
        List<OrgLessonSignLog> signLogs = orgLessonSignLogDao.getSignLogByLessonUserId(orgId, lessonId, userId);
        List<SignLogDto> result = Lists.newArrayList();
        List<Long> teacherIds = signLogs.stream().filter(orgLessonSignLog -> orgLessonSignLog.getTeacherId() > 0)
            .map(orgLessonSignLog -> orgLessonSignLog.getTeacherId()).collect(Collectors.toList());
        Map<Long, String> teacherMap = teacherDao.getTeacherRealNameMap(teacherIds);
        // 机构子帐号姓名map
        Map<Long, String> cascadeMap = txCascadeCredentialService.getByTxCasCadeIds(orgId.longValue());
        for (OrgLessonSignLog signLog : signLogs) {
            String str = createSignLogStr(signLog, cascadeMap, teacherMap);
            if (StringUtils.isNotEmpty(str)) {
                result.add(new SignLogDto(str));
            }
        }
        return result;
    }

    private String createSignLogStr(OrgLessonSignLog signLog, Map<Long, String> cascadeMap,
        Map<Long, String> teacherMap) {
        String name = signLog.getTeacherId() > 0L ? teacherMap.get(signLog.getTeacherId())
            : cascadeMap.get(signLog.getCascadeId());
        if (signLog.getChangeType() == OrgLessonSignLogChangeType.STATUS_CHANGE.getValue()) {
            if (signLog.getSource() == OrgLessonSignSourceEnum.UNKNOWN.getValue()) {
                return String.format(SIGN_STATUS_CHANGE_TEMPLATE, name, DATE_FORMAT.format(signLog.getCreateTime()),
                    SignStatus.getSignStatusByCode(signLog.getStatus()).getMessage());
            } else {
                return String.format(SIGN_WITH_SOURCE_STATUS_CHANGE_TEMPLATE, name,
                    DATE_FORMAT.format(signLog.getCreateTime()),
                    OrgLessonSignSourceEnum.getStrByValue(signLog.getSource()),
                    SignStatus.getSignStatusByCode(signLog.getStatus()).getMessage());
            }
        } else if (signLog.getChangeType() == OrgLessonSignLogChangeType.REMARK_CHANGE.getValue()) {
            return String.format(SIGN_REMARK_CHANGE_TEMPLATE, name, DATE_FORMAT.format(signLog.getCreateTime()),
                signLog.getRemark());
        } else if (signLog.getChangeType() == OrgLessonSignLogChangeType.REMARK_DELETE.getValue()) {
            return String.format(SIGN_REMARK_DELETE_TEMPLATE, name, DATE_FORMAT.format(signLog.getCreateTime()));
        }
        return null;
    }
}