/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.biz.erp.teacherCenter.service.impl;

import com.baijia.tianxiao.biz.erp.dto.request.ListLessonsRequestDto;
import com.baijia.tianxiao.biz.erp.dto.response.H5LessonResponseDto;
import com.baijia.tianxiao.biz.erp.dto.response.LessonResponseDto;
import com.baijia.tianxiao.biz.erp.dto.response.ListLessonResponseDto;
import com.baijia.tianxiao.biz.erp.service.CourseLessonService;
import com.baijia.tianxiao.biz.erp.teacherCenter.dto.LessonCommentStudentDto;
import com.baijia.tianxiao.biz.erp.teacherCenter.service.ErpTeacherCenterService;
import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.constant.CommentStatus;
import com.baijia.tianxiao.dal.enums.CourseTypeEnum;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgClassRoomDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseTeacherDao;
import com.baijia.tianxiao.dal.org.dao.OrgLessonCommentSmsDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgClassLesson;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.sal.comment.dto.CommentAuditDto;
import com.baijia.tianxiao.sal.comment.dto.CommentInfoDto;
import com.baijia.tianxiao.sal.comment.service.LessonCommentAudiService;
import com.baijia.tianxiao.sal.comment.service.LessonCommentService;
import com.baijia.tianxiao.sal.course.dto.response.CourseListReponseDto;
import com.baijia.tianxiao.sal.course.dto.response.OrgStudentsChooseListDto;
import com.baijia.tianxiao.sal.course.dto.response.TeacherResponseDto;
import com.baijia.tianxiao.sal.course.service.CourseStudentService;
import com.baijia.tianxiao.sal.course.service.CourseTeacherService;
import com.baijia.tianxiao.sal.course.service.OrgCourseListService;
import com.baijia.tianxiao.sal.course.service.OrgCourseListService.CourseListParam;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.collection.CollectorUtil;
import com.baijia.tianxiao.util.date.DateUtil;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;

import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

/**
 * @title ErpTeacherCenterServiceImpl
 * @desc TODO
 * @author bjhl
 * @date 2016年1月3日
 * @version 1.0
 */
// @Slf4j
@Service
public class ErpTeacherCenterServiceImpl implements ErpTeacherCenterService {

    @Resource
    private CourseTeacherService courseTeacherService;

    @Resource
    private OrgCourseTeacherDao orgCourseTeacherDao;

    @Resource
    private CourseStudentService courseStudentService;

    @Resource
    private OrgStudentDao orgStudentDao;

    @Resource
    private OrgAccountDao orgAccountDao;

    @Resource
    private CourseLessonService courseLessonService;

    @Resource
    private OrgCourseListService orgCourseListService;

    @Resource
    private OrgCourseDao orgCourseDao;

    @Resource
    private OrgClassRoomDao orgClassRoomDao;

    @Resource
    private LessonCommentAudiService LessonCommentAudiService;

    @Resource
    private LessonCommentService LessonCommentService;

    @Resource
    private OrgLessonCommentSmsDao orgLessonCommentSmsDao;

    @Resource
    private OrgClassLessonDao orgClassLessonDao;

    @Override
    @Transactional(readOnly = true)
    public TeacherResponseDto getTeacherInfo(Long teacherId, Long orgId) {
        Preconditions.checkArgument(teacherId != null, "teacher id is null!");
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        return courseTeacherService.getTeacher(teacherId, orgId);
    }

    @Override
    @Transactional(readOnly = true)
    public List<CommentAuditDto> getTeacherComment(Long orgId, Long teacherId, Boolean isSystem, Integer userRole,
        PageDto pageDto) {
        List<CommentAuditDto> courseCommentsDto =
            this.LessonCommentAudiService.getTeacherCourseCommentAudit(teacherId, orgId, isSystem, userRole, pageDto);
        Collection<Long> courseIds = CollectorUtil.collect(courseCommentsDto, new Function<CommentAuditDto, Long>() {
            @Override
            public Long apply(CommentAuditDto arg0) {
                return arg0.getId();
            }
        });
        Map<Long, String> nameMap = this.orgCourseDao.getCourseNameMap(courseIds);
        for (CommentAuditDto comment : courseCommentsDto) {
            comment.setName(nameMap.get(comment.getId()));
        }
        return courseCommentsDto;
    }

    @Override
    @Transactional(readOnly = true)
    public H5LessonResponseDto getLastLesson(List<H5LessonResponseDto> h5LessonList, Long lessonId) {
        if (lessonId != null) {
            for (H5LessonResponseDto h5LessonResponseDto : h5LessonList) {
                if (h5LessonResponseDto.getLessonId().equals(lessonId)) {
                    return h5LessonResponseDto;
                }
            }
            return null;
        } else {
            Date now = new Date();
            H5LessonResponseDto result = null;
            for (H5LessonResponseDto h5LessonResponseDto : h5LessonList) {
                if (h5LessonResponseDto.getLessonStartTime().before(now)) {
                    result = h5LessonResponseDto;
                } else {
                    return result;
                }
            }
            return result;
        }
    }

    @Override
    @Transactional(readOnly = true)
    public List<LessonCommentStudentDto> getCommentList(Long orgId, Long lessonId, Integer userRole) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        Preconditions.checkArgument(userRole != null, "userRole is null!");
        if (lessonId == null) {
            return Lists.newArrayList();
        }
        List<OrgStudentsChooseListDto> chooseList = this.courseStudentService.getLessonStudent(orgId, lessonId, false);
        Collection<Long> studentIds = CollectorUtil.collect(chooseList, new Function<OrgStudentsChooseListDto, Long>() {
            @Override
            public Long apply(OrgStudentsChooseListDto arg0) {
                return arg0.getStudentId();
            }
        });
        Map<Long, Long> userIdMap = this.orgStudentDao.getStudentIdUserIdMap(studentIds);
        Map<Long, CommentInfoDto> commentMap = getCommentMap(orgId, lessonId, userRole);
        Map<Long, Integer> smsMap =
            this.orgLessonCommentSmsDao.getSendMap(orgId, lessonId, userIdMap.values(), UserRole.STUDENT.getRole());
        List<LessonCommentStudentDto> response = Lists.newArrayList();
        OrgClassLesson lesson = orgClassLessonDao.getById(lessonId);
        for (OrgStudentsChooseListDto orgStudentsChooseListDto : chooseList) {
            LessonCommentStudentDto dto = new LessonCommentStudentDto();
            Long userId = userIdMap.get(orgStudentsChooseListDto.getStudentId());
            dto.setStudentInfo(orgStudentsChooseListDto);
            CommentInfoDto commentInfo = commentMap.get(userId);
            getAndSetCommentStatus(commentInfo, lesson, dto, userRole);
            dto.setSend(smsMap.get(userId) == null ? 0 : smsMap.get(userId));
            response.add(dto);
        }
        return response;
    }

    /**
     * 设置评价状态
     * 
     * @param commentInfo
     * @param dto
     */
    private void getAndSetCommentStatus(CommentInfoDto commentInfo, OrgClassLesson lesson, LessonCommentStudentDto dto,
        Integer userRole) {
        dto.setCommentInfo(commentInfo);
        if (commentInfo == null) {
            // OrgClassLesson lesson = orgClassLessonDao.getById(lessonId);
            dto.setCommentStatus(CommentStatus.getCommentStatusByTime(lesson.getStartTime(), lesson.getEndTime(),
                userRole.intValue() == UserRole.STUDENT.getRole()));
        } else {
            dto.setCommentStatus(CommentStatus.COMMENTED);
        }
    }

    /**
     * 获取评论map
     * 
     * @param orgId
     * @param lessonId
     * @param userRole
     * @return
     */
    private Map<Long, CommentInfoDto> getCommentMap(Long orgId, Long lessonId, final Integer userRole) {
        List<CommentInfoDto> commentInfos = this.LessonCommentService.getLessonComments(lessonId, orgId, userRole);
        return CollectorUtil.collectMap(commentInfos, new Function<CommentInfoDto, Long>() {
            @Override
            public Long apply(CommentInfoDto arg0) {
                if (userRole.intValue() == UserRole.STUDENT.getRole()) {
                    return arg0.getFromId();
                } else {
                    return arg0.getToId();
                }
            }
        });
    }

    @Override
    public List<CourseListReponseDto> getCoursesList(Long orgId, Long teacherId, PageDto pageDto) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        Preconditions.checkArgument(teacherId != null, "teacherId is null!");
        OrgAccount account = this.orgAccountDao.getById(orgId);
        List<Long> courseIds = this.orgCourseTeacherDao.getCourseIdByTeacherId(teacherId);
        if(GenericsUtils.isNullOrEmpty(courseIds)){
            return GenericsUtils.emptyList();
        }
        List<OrgCourse> courses =
            this.orgCourseDao.getAndFilterOrgCourseByIn(account.getNumber(), courseIds, null,null,CourseTypeEnum.IS_CLASS_TRUE.getCode(),null, pageDto);
        CourseListParam courseListParam = new CourseListParam();
        courseListParam.setCourses(courses).setOrgId(orgId).setTeacherId(teacherId).setNeedLesson(true)
            .setNeedStudent(true);
        List<CourseListReponseDto> response = this.orgCourseListService.getCourseList(courseListParam);

        ListLessonsRequestDto params = new ListLessonsRequestDto();
        params.setTeacherId(teacherId);
        params.setStartTime(DateUtil.getCurrentDate().getTime());
        params.setStartTime(DateUtil.getDiffDateTime(DateUtil.getCurrentDate(), 1).getTime());
        ListLessonResponseDto listLessonResponseDto = this.courseLessonService.listLessons(orgId, params, false);
        Map<Long, List<LessonResponseDto>> lessonMap =
            CollectorUtil.group(listLessonResponseDto.getList(), new Function<LessonResponseDto, Long>() {
                @Override
                public Long apply(LessonResponseDto arg0) {
                    return arg0.getCourseId();
                }
            });
        for (CourseListReponseDto courseListReponseDto : response) {
            List<LessonResponseDto> list = lessonMap.get(courseListReponseDto.getOrgCourseId());
            if (CollectionUtils.isNotEmpty(list)) {
                int count = 0;
                for (LessonResponseDto lessonResponseDto : list) {
                    if (DateUtil.isToday(lessonResponseDto.getLessonStartTime())) {
                        count++;
                    }
                }
                courseListReponseDto.setLessonToday(count);
            } else {
                courseListReponseDto.setLessonToday(0);
            }
        }
        return response;
    }

    @Override
    @Transactional(readOnly = true)
    public List<CommentAuditDto> getTeacherAuditList(Long orgId, Integer userRole, Boolean isSystem, PageDto page) {
        List<TeacherResponseDto> teacherList = courseTeacherService.listOrgTeacher(null, orgId, null, null, page);
        Map<Long, TeacherResponseDto> teacherMap =
            CollectorUtil.collectMap(teacherList, new Function<TeacherResponseDto, Long>() {
                @Override
                public Long apply(TeacherResponseDto arg0) {
                    return arg0.getTeacherId();
                }
            });
        List<CommentAuditDto> comments =
            this.LessonCommentAudiService.getTeacherAuditList(orgId, teacherMap.keySet(), userRole, isSystem);
        for (CommentAuditDto commentAuditDto : comments) {
            TeacherResponseDto teacher = teacherMap.get(commentAuditDto.getId());
            if (teacher != null) {
                commentAuditDto.setName(teacher.getTeacherName());
                commentAuditDto.setAvatarUrl(teacher.getAvatar());
            }
        }
        return comments;
    }
}
